// pick.h

// Copyright (C) 1997  Cliff Johnson                                       //
//                                                                         //
// This program is free software; you can redistribute it and/or           //
// modify it under the terms of the GNU  General Public                    //
// License as published by the Free Software Foundation; either            //
// version 2 of the License, or (at your option) any later version.        //
//                                                                         //
// This software is distributed in the hope that it will be useful,        //
// but WITHOUT ANY WARRANTY; without even the implied warranty of          //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       //
// General Public License for more details.                                //
//                                                                         //
// You should have received a copy of the GNU General Public License       //
// along with this software (see COPYING); if not, write to the        //
// Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. //


#ifndef PICK_H
#define PICK_H

#include <iostream.h>
#include "entity.h"
#include "point.h"
#include "locale_enum.h"

// the Pick class is a common container for passing entity to
// solver functions - the method is intended 
// to implement a generalized appearance in the solver class in
// a form appropriate for cad usage - hence the data
// encapsulated is a Entity* and a Point - The Entity* pointing
// to the entity selected, and the Point giving a 
// spatial coordinate, in the cad global coordinate system, 
// the selection.
// 
// This makes possible solver functions like:
// 	Point PointConstraint2D(const Pick& p1, const Pick& p2) throw (EntityException);
// 	
// This function solves a point by 2 constraints where the constraints can be combinations
// of different entity types . 2 points -> midpoint , 2 lines -> intersection, Point
// and a Circle -> intersection _nearest_ the selection point of pick 2, etc.
// 
// Similar to the selection class, Pick drops the button value, 
// and as well does not require a persistent object - a Handle -
// just the pointer to the entity.
// Of course the USER MUST THEN MANAGE THE SCOPE OF THE OBJECT		***WARNING***
// POINTERED in the container 
// 
// This is a very dangerous thing, but necessary me thinks, because sometime 
// the user selects nothing, but what he means is an "free point". In the 
// FREEdraft interface, selecting with the middle mouse button does this. 
// If the MenuHandler is needs 2 picks, it needs to keep the free
// point - which is not persistent - alive until the second pick is made
// and the "free" point is used.
// 
// Why not put the free picks into the bank as persistent objects, just
// for as long as needed?  1) this would tie the entitybank fiasco to
// the cadlib2d stuff 2) entitybank is not a particularly refined, 
// efficient, fast, optimized, or robust database system, and will 
// be changed. 
// 
// to assist in managing the scope of locally allocated entity, 
// each Pick contains a flag "local" which is set by the 
// user at pick assignment time to PERSISTENT if the object pointer is from 
// a handle, and LOCAL if locally allocated entity.
// The user can then call CleanUp() on the pick when they are finished with it.
// if the flag is LOCAL, the object pointed to is deleted. if the flag is 
// PERSISTENT or UNUSED, indicating the pick was never assigned a value, 
// the pointered object is left alone.

// DO NOT PASS THE ADDRESS OF A TEMPORARY IN A PICK. (like this)
// Point p1(x,y,z);
// Pick pk1 = Pick(&p1,p1);
// ...
// 
// ALWAYS Instead explicitly allocate all local stuff you load into a pick, (like this)
// Point* p1 = new Point(x,y,z);
// Pick pk1 = Pick(p1,*p1,LOCAL);
// ...
// 

//picks are loaded into a stack by the HandlerHandler object, and are not
//actually used in a computation until enough of them are available for the pending computation.
//i.e. a line by contraints waits until two picks are available. So if you store a temporary,
//you'll surely meet with doom.
// CleanUp is typically used by handler object sub-methods after such a computation.

// You must however provide *some* entity object. So in the case of
// a free pick, pass the coordinate (possibly rounded) point as both the 
// entity and the pick point
// 
// See the MenuHandler family for examples. 
// 
// Picks are created - converted from Selections - during calls to MenuHandler::DoPick()

class Pick
{
private:
	const Entity* entity;
	Point point;
	Locale locale;
public:
	Pick(){ locale = UNUSED; }
	Pick(const Entity* g, const Point& p,Locale l = PERSISTENT);

	Pick(const Pick& p);
	Pick& operator=(const Pick& p);

	Point GetPoint() const { return point; }
	const Entity* PointsAt() const { return entity;}

	void CleanUp();                 //for freeing non-handle (local) entity - 
					//do not call on entity created by 
				        //EntityBank::Create(). (it won't do anything) 
					//use EntityBank::Destroy(h) to delete entity objects.  
					//that.) 

	int Type() const { return entity->Type(); }

	bool IsLocal() const { return locale == LOCAL; }
	bool IsLocked() const { return locale == LOCKED; }

	bool Lock();
	bool Unlock();

	friend bool operator==(const Pick& p1, const Pick& p2);

	friend ostream& operator<<(ostream& os, const Pick& p);
};

#endif

