// point.cpp

// Copyright (C) 1997  Cliff Johnson                                       //
//                                                                         //
// This program is free software; you can redistribute it and/or           //
// modify it under the terms of the GNU  General Public                    //
// License as published by the Free Software Foundation; either            //
// version 2 of the License, or (at your option) any later version.        //
//                                                                         //
// This software is distributed in the hope that it will be useful,        //
// but WITHOUT ANY WARRANTY; without even the implied warranty of          //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       //
// General Public License for more details.                                //
//                                                                         //
// You should have received a copy of the GNU General Public License       //
// along with this software (see COPYING.LIB); if not, write to the        //
// Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. //

#include "point.h"		//	THIS
#include "iostream.h"		//	ITO
#include "math.h"		//	ITO
#include "entity_enum.h"
#include "entityexception.h"


double Point::NullDist =  0.0005;
double Point::NullAngle = 0.0001;
double Point::MaxSize =  50000.0; 

//-------------------------------------------------------------
//	Constructors
//-------------------------------------------------------------

Point::Point(): Entity(POINT)
{
	P[0]=P[1]=P[2]=0.;
}

//-------------------------------------------------------------
Point::Point(double* p): Entity(POINT)
{
	for(int i=0;i<3;i++)P[i]=p[i];
}
//-------------------------------------------------------------
// copy constructor
Point::Point(const Point& p): Entity(POINT)
{
	for(int i=0;i<3;i++)P[i]=p.P[i];
}

//-------------------------------------------------------------
Point::Point(double x, double y, double z): Entity(POINT)
{
	P[0]=x;
	P[1]=y;
	P[2]=z;
}

//-------------------------------------------------------------
//	Methods
//-------------------------------------------------------------
// assignment operator 
Point& Point::operator=(const Point& p)
{
	for(int i=0;i<3;i++)P[i]=p[i];
	return *this;
}

//-------------------------------------------------------------
istream& operator>>(istream& s , Point& p)
{
	s >> p.P[0] >> p.P[1] >> p.P[2];
	return s;
}
//-------------------------------------------------------------

ostream& operator<<(ostream& s , const Point& p)
{
  s << p[0] << ' ' << p[1] << ' ' << p[2] ;
	return s;
}
//-------------------------------------------------------------
double Point::Magnitude() const
{
	return sqrt(pow(P[0],2)+pow(P[1],2)+pow(P[2],2));
}
//-------------------------------------------------------------
Point Point::Normal() const throw (EntityException)
{
	double m = Magnitude();
	if(m < NullDist)throw EntityException("Point::Normal() : null vector cannot be normalized");
	return Point((*this) * (1/m));
}
//-------------------------------------------------------------
Point Point::Round(int power) const
{
	double p = pow(10,power);
	return Point(
		p * rint( P[0]/p ),
	        p * rint( P[1]/p ),
	        p * rint( P[2]/p )
	);
}
//-------------------------------------------------------------

Point Point::operator+(const Point& p) const
{
	double px[3];
	for(int i=0;i<3;i++)px[i]=P[i]+p[i];
	return Point(px);
}

//-------------------------------------------------------------
Point Point::operator-(const Point& p) const
{
	double px[3];
	for(int i=0;i<3;i++)px[i]=P[i]-p[i];
	return Point(px);
}
//-------------------------------------------------------------

double Point::operator*(const Point& p) const // dot product:
{
	return P[0]*p[0] + P[1]*p[1] + P[2]*p[2];
}

//-------------------------------------------------------------
Point operator-(const Point& p1)
{
	double p[3];
	for(int i=0;i<3;i++)p[i]=-p1[i];
	return Point(p);
}

//-------------------------------------------------------------
bool operator==(const Point& p1, const Point& p2)// Coordinates are equal
{
	return( p1.Distance(p2) < Point::NullDist);
}
//-------------------------------------------------------------
//Note two forms of this following:
// Member

Point Point::operator*(double ratio) const // return a vector scaled from this one.
{
	double px[3];
	for(int i=0;i<3;i++)px[i]=P[i]*ratio;
	return Point(px);
}
// friend
Point operator*(double ratio, const Point& p)	// return a vector scaled from this one.
{
	double px[3];
	for(int i=0;i<3;i++)px[i]=p[i]*ratio;
	return Point(px);
}
//-------------------------------------------------------------

Point Cross(const Point& p1, const Point& p2) // cross product
{
	double p[3];
	p[0]=p1[1]*p2[2]-p1[2]*p2[1];
	p[1]=p1[2]*p2[0]-p1[0]*p2[2];
	p[2]=p1[0]*p2[1]-p1[1]*p2[0];
	return Point(p);
}
//-------------------------------------------------------------

double Angle(const Point& v1, const Point& v2) throw (EntityException)
{
	Point vx1,vx2;
//	double l1,l2,a;
	double dot,angle;

	try
	{
		vx1 = v1.Normal();
		vx2 = v2.Normal();
	}
	catch (EntityException& ge)
	{
		throw;
	}
	dot = vx1 * vx2;

	if(dot >= 1 ) angle = 0;
	else if(dot <= -1 ) angle = M_PI;
	else angle = acos(dot);
	return angle;
}
//-------------------------------------------------------------

double Distance(const Point& p1, const Point& p2)
{
	return p1.Distance(p2);
}

//-------------------------------------------------------------
double Point::Distance(const Point& p) const
{
	Point pcopy(*this);
	Point px = p - pcopy;
	return px.Magnitude();
}


//-------------------------------------------------------------


