// geommenuhandler.cpp

// Copyright (C) 1997  Cliff Johnson                                       //
//                                                                         //
// This program is free software; you can redistribute it and/or           //
// modify it under the terms of the GNU  General Public                    //
// License as published by the Free Software Foundation; either            //
// version 2 of the License, or (at your option) any later version.        //
//                                                                         //
// This software is distributed in the hope that it will be useful,        //
// but WITHOUT ANY WARRANTY; without even the implied warranty of          //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       //
// General Public License for more details.                                //
//                                                                         //
// You should have received a copy of the GNU General Public License       //
// along with this software (see COPYING.LIB); if not, write to the        //
// Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. //


// In the constructor  relabel all the buttons and set their sensitivities. 



#include <geommenuhandler.h>

#include <iostream.h>

#include <pick.h>
#include <point.h>
#include <line.h>
#include <circle.h>
#include <segment.h>
#include <arc.h>
#include <cadlib2d.h>

#include <pickentity.h>
#include <attributes.h>

#include <data_enum.h>
#include <vdapp_enum.h>
#include <entity_enum.h>
#include <attrib_enum.h>
#include <drawable_enum.h>

#include <geommenuarcdialog.h>

#include <entityexception.h>
#include <comexception.h>
#include <bankexception.h>
#include <coreexception.h>

//**********constructor ***********************************************************
GeomMenuHandler::GeomMenuHandler(vdCmdWindow* cmdwin,int r) 
: MenuHandler(cmdwin,r)
{

// set the menu and button labels.

	commandWindow->SetString(MENULABEL0,"Points");
	commandWindow->SetString(MENUBUTTON00,"pick");
	commandWindow->SetString(MENUBUTTON01,"constr.");

	commandWindow->SetString(MENULABEL1,"Lines");
	commandWindow->SetString(MENUBUTTON10,"constr.");
	commandWindow->SetString(MENUBUTTON11,"vertical");

	commandWindow->SetString(MENULABEL2,"Lines");
	commandWindow->SetString(MENUBUTTON20,"horiz.");
	commandWindow->SetString(MENUBUTTON21,"hz+vrt");

	commandWindow->SetString(MENULABEL3,"Lines");
	commandWindow->SetString(MENUBUTTON30,"oriented");
	commandWindow->SetString(MENUBUTTON31,"orthog.");

	commandWindow->SetString(MENULABEL4,"Circles");
	commandWindow->SetString(MENUBUTTON40,"constr.");
	commandWindow->SetString(MENUBUTTON41,"rad+2  ");

	commandWindow->SetString(MENULABEL5,"Circles");
	commandWindow->SetString(MENUBUTTON50,"ctr+1");
	commandWindow->SetString(MENUBUTTON51,"ctr+rad");

	commandWindow->SetString(MENULABEL6,"Arcs");
	commandWindow->SetString(MENUBUTTON60,"closed");
	commandWindow->SetString(MENUBUTTON61,"constr");

	commandWindow->SetString(MENULABEL7,"Create");
	commandWindow->SetString(MENUBUTTON70,"trace");
	commandWindow->SetString(MENUBUTTON71,"//dist");

	commandWindow->SetString(MENULABEL8,"Create");
	commandWindow->SetString(MENUBUTTON80,"supprt");
	commandWindow->SetString(MENUBUTTON81,"group ");

	commandWindow->SetString(MENULABEL9,"Modify");
	commandWindow->SetString(MENUBUTTON90,"trim   ");
	commandWindow->SetString(MENUBUTTON91,"cut    ");

	commandWindow->SetString(MENULABEL10,"Modify");
	commandWindow->SetString(MENUBUTTON100,"fillet ");
	commandWindow->SetString(MENUBUTTON101,"champhr");

// set states pod buttons

	commandWindow->SetValue(MENUBUTTON00,isSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON01,isSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON10,isSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON11,isSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON20,isSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON21,isSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON30,notSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON31,isSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON40,notSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON41,isSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON50,isSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON51,isSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON60,isSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON61,isSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON70,isSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON71,isSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON80,isSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON81,notSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON90,isSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON91,isSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON100,notSens,Sensitive);
	commandWindow->SetValue(MENUBUTTON101,notSens,Sensitive);

// create dialog handler 
	arcDialog = new GeomMenuArcDialog(commandWindow);
}

//**********destructor ***********************************************************

GeomMenuHandler::~GeomMenuHandler()
{
	delete arcDialog;
}
//***********************************************************************************
// virtual function GeomMenuHandler
//***********************************************************************************

void GeomMenuHandler::MenuCommand(int id)
{

// map the button pressed to the corresponding command for this menu

	arcDialog->CloseDialog();

	switch(id)
	{
	case MENUBUTTON00:
		{
			PointByPicking0();
			break;
		}
	case MENUBUTTON01:
		{
			PointByConstraints0();
			break;
		}
	case MENUBUTTON10:
		{
			LineByConstraints0();
			break;
		}
	case MENUBUTTON11:
		{
			LineHorzAndVert0(LINEVERTICAL);
			break;
		}
	case MENUBUTTON20:
		{
			LineHorzAndVert0(LINEHORIZONTAL);
			break;
		}
	case MENUBUTTON21:
		{
			LineHorzAndVert0(LINEHORZANDVERT);
			break;
		}
	case MENUBUTTON30:
		{
			LineOriented0();
			break;
		}
	case MENUBUTTON31:
		{
			LineOrthoginal0();
			break;
		}
	case MENUBUTTON40:
		{
			CircleConstraints0();
			break;
		}
	case MENUBUTTON41:
		{
			CircleRadiusAndConstraints0();
			break;
		}
	case MENUBUTTON50:
		{
			CircleCenter0();
			break;
		}
	case MENUBUTTON51:
		{
			CircleRadiusAndCenter0();
			break;
		}
	case MENUBUTTON60: // arc closed
		{
			ArcClosed0();
			break;
		}

	case MENUBUTTON61: // arc constraints
		{
			ArcConstraints0();
			break;
		}
	case MENUBUTTON70:   
		{
			Trace0();
			break;
		}
	case MENUBUTTON71:
		{
			ParallelAtDistance0();
			break;
		}
	case MENUBUTTON80: // create support geom
		{
			CreateSupport0();
			break;
		}
	case MENUBUTTON90: // trim
		{
			Trim0();
			break;
		}
	case MENUBUTTON91: // cut 
		{
			Cut0();
			break;
		}

	}
// process it through the base 
	MenuHandler::MenuCommand(id);
}

//***********************************************************************************
// virtual function CommandActions
//***********************************************************************************

void GeomMenuHandler::CommandActions()
{

// CommandActions is run when the parm count decrements to 0, meaning that the current expression 
// is complete and ready to be solved.


// process pending modifiers and commands
	while(parm == 0)
	{
	// process commands
		switch(commandStack.back())
		{
// SOLUTIONS TO FUNCTIONS
			case POINTBYPICKING:
			{
				PointByPicking1(); 	// create the geom
				break;			
			}	

			case POINTBYCONSTRAINTS1:
			{
				PointByConstraints1();
				break;
			}
			case POINTBYCONSTRAINTS2:
			{
				PointByConstraints2();
				break;
			}
			case LINEVERTICAL:
	      		case LINEHORIZONTAL:	
        		case LINEHORZANDVERT:
			{
				LineHorzAndVert1();
				break;		
			}
        		case LINEBYCONSTRAINTS1:
			{
				LineByConstraints1();
				break;
			}
        		case LINEBYCONSTRAINTS2:
			{
				LineByConstraints2();
				break;
			}
        		case LINEORIENTED:
			{
				LineOriented1();
				break;
			}
			case LINEORTHOGINAL1:
			{
				LineOrthoginal1();
				break;
			}
			case LINEORTHOGINAL2:
			{
				LineOrthoginal2();
				break;
			}
			case CIRCLECENTER1:
			{
				CircleCenter1();
				break;
			}
			case CIRCLECENTER2:
			{
				CircleCenter2();
				break;
			}
			case CIRCLERADIUS:
			{
				CircleRadiusAndCenter1();
				break;
			}
			case CIRCLERADIUSANDCONSTRAINTS1:
			{
				CircleRadiusAndConstraints1();
				break;
			}
			case CIRCLERADIUSANDCONSTRAINTS2:
			{
				CircleRadiusAndConstraints2();
				break;
			}
			case CIRCLECONSTRAINTS:
			{
				CircleConstraints1();
				break;
			}
			case TRACE1:
			{
				Trace1();
				break;
			}
			case TRACE2:
			{
				Trace2();
				break;
			}
			case TRACE3:
			{
				Trace3();
				break;
			}
			case ARCCONSTRAINTS:
			{
				ArcConstraints1();
				break;
			}
			case ARCCLOSED:
			{
				ArcClosed1();
				break;
			}
			case TRIM1:
			{
				Trim1();
				break;
			}
			case TRIM2:
			{
				Trim2();
				break;
			}
			case CUT1:
			{
				Cut1();
				break;
			}
			case CUT2:
			{
				Cut2();
				break;
			}
			case PARALLELATDISTANCE:
			{
				ParallelAtDistance1();
				break;
			}
			case CREATESUPPORT:
			{
				CreateSupport1();
				break;
			}
			default: // process modifiers
			MenuHandler::CommandActions();
		}
	}
}
		

//========================================================================
void GeomMenuHandler::PointByPicking0()
{
	Initialize();
	commandStack.push_back(POINTBYPICKING); // set the command
	parm = 1;			// set the number of parameters expected
	SelectionFilter sf;	// preset the selection filter
	sf.Add(POINT);        // Define it
	SetTypeMask(sf);
	Say(NULL);
	Say("Create Points : Select a constraint >");
}
//========================================================================

void GeomMenuHandler::PointByPicking1()
{
	Pick pk;

// get last pick and button
	try
	{
		int button = MenuHandler::Get1Pick(pk);

      		switch(button){
      			case 1:
      			case 2:
              		{
              			Handle hx = BankCreate(new Point(pk.GetPoint().Round(rounding)));
              			Register(hx,Attributes(CurrentColor(),CurrentLineStyle(),CurrentPointSize(),CurrentLayer()),Drawable(FD_DRAWABLE_POINT));
              			Redraw();
              			break;
              		}
      			case 3:
              		{
              			break;
              		}
      		}
	}
	catch (ComException & ce)
	{
		cerr << "ComException raised in GeomMenuHandler::PointsByPicking() " << endl;
		cerr << "	" << ce.what() << endl;
	}

      pk.CleanUp();	// clean up the pick here. 
      parm =1;
}

//****************************************************************************8

void GeomMenuHandler::PointByConstraints0()
{
		Initialize();
		commandStack.push_back(POINTBYCONSTRAINTS1);
		parm = 1;
		SelectionFilter sf;	// preset the selection filter
		sf.Add(POINT);        // Define it
                sf.Add(LINE);
		sf.Add(CIRCLE);
		sf.Add(SEGMENT);
		sf.Add(ARC);
		SetTypeMask(sf);
		Say(NULL);
		Say("Points by  constraints: Select first constraint >");
}
//======================================================================================

void GeomMenuHandler::PointByConstraints1()
{
	Pick pk;
	int button;
// get the pick
	try
	{
		button = MenuHandler::Get1Pick(pk);
	}
	catch (ComException& ce)
	{
// bad pick, ignore
		cerr << "ComException raised in GeomMenuHandler::PointByConstraints1()" << endl;
		cerr << "           " << ce.what() << endl;
		Say(NULL);
		Say("Points by  constraints: Reselect first constraint >");
		parm = 1;
		pk.CleanUp();
		return;
	}
// all types are valid

// no new mask

// highlight the geom
	SetHighlight(pk,true);
	Redraw();

// set the new command
	commandStack.pop_back();
	commandStack.push_back(POINTBYCONSTRAINTS2);
// save the pick
	pk.Lock();	        // in case it is a local object
	PushPick(pk,button);
// set parm
	parm = 1;
// Update the message
	Say(" > Select another constraint.");
}
//========================================================================
void GeomMenuHandler::PointByConstraints2()
{
	Pick pk1,pk2;
	int button1,button2;
	Point p;
	Handle h;

// get the second pick
	try
	{
		button2 = MenuHandler::Get1Pick(pk2);
	}
	catch(ComException& ce)
	{	// pick 2 is bad - ignore it.
		cerr << "ComException raised in GeomMenuHandler::PointByConstraints2()" << endl;
		cerr << "        " << ce.what() << endl;
		Say(" > Cutting selection invalid. Reselect geom");
		pk2.CleanUp();
		parm = 1;
		return;
	}
// get the first pick again
	try
	{
		button1 = MenuHandler::Get1Pick(pk1);
	}
	catch(ComException& ce)
	{		
	// pick 1 has gone bad  - reset the function
		cerr << "ComException raised in GeomMenuHandler::PointByConstraints2()" << endl;
		cerr << "        " << ce.what() << endl;
		Say("ERROR: first constraint lost! >>> Select a constraint.");
		goto reinit;
	}

	try {
		p = CadLib2D_PointByConstraints(pk1,pk2);
	}
	catch (EntityException& ge)
	{
		cerr << "EntityException raised in GeomMenuHandler::PointByConstraints2() " << endl;
		cerr << "	" << ge.what() << endl;
		Say(" > Error: No solution >>> Select a constraint.");
		goto reinit;
	}
	{
		h = BankCreate(new Point(p));
        	Register(h,Attributes(CurrentColor(),CurrentLineStyle(),CurrentPointSize(),CurrentLayer()),Drawable(FD_DRAWABLE_POINT));
		SetHighlight(pk1,false);
       		Redraw();
	}
reinit:
	pk1.Unlock();
	pk2.Unlock();
	pk1.CleanUp();
	pk2.CleanUp();
	commandStack.pop_back();
	commandStack.push_back(POINTBYCONSTRAINTS1);
	parm = 1;
	Say(NULL);
	Say("Points by  constraints: Select first constraint >");
}

//========================================================================
void GeomMenuHandler::LineHorzAndVert0(int code)
{
	Initialize();
	commandStack.push_back(code);
	SelectionFilter sf;	
	sf.Add(POINT);        // Define it
	sf.Add(CIRCLE);
	sf.Add(ARC);
	SetTypeMask(sf);
	parm = 1;
	Say(NULL);
	switch (code)
	{
	case LINEHORIZONTAL:
		Say("> Create horizontal lines : Select a constraint");
		break;
	case LINEVERTICAL:
		Say("> Create vertical lines : Select a constraint");
		break;
	case LINEHORZANDVERT:
		Say("> Create horizontal and vertical lines : Select a constraint");
		break;
	}
}
//========================================================================
void  GeomMenuHandler::LineHorzAndVert1()
{

	Pick pk;
	Line l;
	int button;
	int command = commandStack.back();
	Attributes attr(FD_CYAN,FD_LINESTYLE_DOTTED,FD_LINETHICK_THIN,CurrentLayer()); // sketch geom has fixed color

	try
	{
		button = MenuHandler::Get1Pick(pk);
	}
	catch(ComException & ce)
	{
		cerr << "Comexception raised in GeomMenuHandler::LineHorzAndVert1()" << endl;
		cerr << "	" << ce.what() << endl;
		Say(" > ERROR: invalid selection >>> Pick the constraint.");
		goto reinit;
	}

// horizontal line
	if(command == LINEHORIZONTAL || command == LINEHORZANDVERT )
	{
		try
		{
			l = CadLib2D_LineParallel( Line(Point(),Point(1,0)), pk  );
               		Handle h = BankCreate(new Line(l));
			Register(h,attr,Drawable(FD_DRAWABLE_LINE));
		}
		catch(EntityException &  ge)
		{
			cerr << "EntityException raised in GeomMenuHandler::LineHorzAndVert1() " << endl;
			cerr << "	" << ge.what() << endl;
			goto reinit;
		}
	}
	// vertical 
	if(command == LINEVERTICAL || command == LINEHORZANDVERT) 
	{
		Line l;
		try
		{
			l = CadLib2D_LineParallel( Line(Point(),Point(0,1)), pk );
        		Handle h = BankCreate(new Line(l));
        		Register(h,attr,Drawable(FD_DRAWABLE_LINE));
		}
		catch(EntityException &  ge)
		{
			cerr << "EntityException raised in GeomMenuHandler::LineHorzAndVert1() " << endl;
			cerr << "	" << ge.what() << endl;
			goto reinit;
		}
	}
       	Redraw();
reinit:
	pk.CleanUp();
	parm = 1;
}

//=====================================================================
void GeomMenuHandler::LineByConstraints0()
{
		Initialize();
		commandStack.push_back(LINEBYCONSTRAINTS1);
		SelectionFilter sf;	
		sf.Add(POINT);        // Define it
                sf.Add(LINE);
		sf.Add(CIRCLE);
		sf.Add(SEGMENT);
		sf.Add(ARC);
		SetTypeMask(sf);
		parm = 1;
		Say(NULL);
		Say(" >>> Select a constraint.");
}
//========================================================================
void  GeomMenuHandler::LineByConstraints1()
{
	Pick pk;
	int button;
	
	// check if the first pick is valid
	try
	{
		button = (MenuHandler::Get1Pick(pk));
	}
	catch(ComException & ce)
	{
		cerr << "Comexception raised in GeomMenuHandler::LineByConstraints1()" << endl;
		cerr << "	" << ce.what() << endl;
		Say(" > ERROR: Invalid selection > Reselect constraint.");
	// ignore the pick
		pk.CleanUp();
		parm = 1;
		return;
	}
// all types are valid

// no new mask

// highlight the geom
	SetHighlight(pk,true);
	Redraw();

// set the new command
	commandStack.pop_back();
	commandStack.push_back(LINEBYCONSTRAINTS2);

// save the pick
	pk.Lock();	        // in case it is a local object
	PushPick(pk,button);

// set parm
	parm = 1;

// Update the message
	Say(" > Select another constraint");
}
//========================================================================
void  GeomMenuHandler::LineByConstraints2()
{
	Pick pk1,pk2;
	int button1,button2;
	Line l;
	Handle h;

// get the second pick
	try
	{
		button2 = MenuHandler::Get1Pick(pk2);
	}
	catch(ComException& ce)
	{	// pick 2 is bad - ignore it.
		cerr << "ComException raised in GeomMenuHandler::LineByConstraints2()" << endl;
		cerr << "        " << ce.what() << endl;
		Say(" > Selection invalid. Reselect second constraint.");
		pk2.CleanUp();
		parm = 1;
		return;
	}
// get the first pick again
	try
	{
		button1 = MenuHandler::Get1Pick(pk1);
	}
	catch(ComException& ce)
	{		
	// pick 1 has gone bad  - reset the function
		cerr << "ComException raised in GeomMenuHandler::Cut2()" << endl;
		cerr << "        " << ce.what() << endl;
		Say("ERROR: first constraint lost! >>> Select a constraint.");
		goto reinit;
	}

	try
	{
		l = CadLib2D_LineByConstraints(pk1,pk2);
	}
	catch(EntityException & ge)
	{
		cerr << "EntityException raised in GeomMenuHandler::LineByConstraints() " << endl;
		cerr << "	" << ge.what() << endl;
		Say(" > ERROR: Cannot compute line. >>> Select a contstraint.");
		goto reinit;
	}
	{
		h = BankCreate(new Line(l));
       		Register(h,Attributes(FD_CYAN,FD_LINESTYLE_DOTTED,FD_LINETHICK_THIN,CurrentLayer()),Drawable(FD_DRAWABLE_LINE));
		SetHighlight(pk1,false);
       		Redraw();
	}
reinit:
	pk1.Unlock();
	pk2.Unlock();
	pk1.CleanUp();
	pk2.CleanUp();
	commandStack.pop_back();
	commandStack.push_back(LINEBYCONSTRAINTS1);
	parm = 1;
}
//========================================================================
void GeomMenuHandler::LineOriented0()
{
	Initialize();
	commandStack.push_back(LINEORIENTED);
	SelectionFilter sf;	
	sf.Add(POINT);        // Define it
	sf.Add(LINE);
	sf.Add(CIRCLE);
	sf.Add(SEGMENT);
	sf.Add(ARC);
	SetTypeMask(sf);
	parm = 2;
	Say(NULL);
	Say("Create an oriented line.");
}
//========================================================================
void  GeomMenuHandler::LineOriented1     ()
{
	parm = 2;
}

//========================================================================
void GeomMenuHandler::LineOrthoginal0()
{
	Initialize();
	commandStack.push_back(LINEORTHOGINAL1);
	SelectionFilter sf;	
	sf.Add(LINE);
	sf.Add(CIRCLE);
	sf.Add(SEGMENT);
	sf.Add(ARC);
	SetTypeMask(sf);
	parm = 1;
	Say(NULL);
	Say("> Orthoginal line: Select a reference geom:");
}
//========================================================================
void  GeomMenuHandler::LineOrthoginal1()
{
	Pick pk;
	int button;
	
	// check if the first pick is valid
	try
	{
		button = (MenuHandler::Get1Pick(pk));
	}
	catch(ComException & ce)
	{
		cerr << "Comexception raised in GeomMenuHandler::LineOrthoginal1()" << endl;
		cerr << "	" << ce.what() << endl;
	// ignore the pick
		pk.CleanUp();
		parm = 1;
		return;
	}
	// check the type  - ignore if not valid type
	if(pk.Type() != LINE && pk.Type() != SEGMENT && pk.Type() != ARC && pk.Type() != CIRCLE)
	{
		pk.CleanUp();
		Say(" Invalid selection - pick again: ");
		parm = 1;
		return;
	}
// highlight the selection
	SetHighlight(pk,true);
	Redraw();

// change the mask
	SelectionFilter sf;
	sf.Add(POINT);
	sf.Add(CIRCLE);
	sf.Add(ARC);
	SetTypeMask(sf);

// update the command
	commandStack.pop_back();
	commandStack.push_back(LINEORTHOGINAL2);

// save the pick
	pk.Lock();
	PushPick(pk,button);
	
// update the message
	Say("Select a constraint:");

// set the parm count
	parm = 1;
}
//========================================================================
void GeomMenuHandler::LineOrthoginal2()
{
	Pick pk1,pk2;
	int button1,button2;
	Line l;

	try
	{
		button2 = MenuHandler::Get1Pick(pk2);
	}
	catch(ComException& ce)
	{	// pick 2 is bad - ignore it.
		cerr << "ComException raised in GeomMenuHandler::LineOrthoginal2()" << endl;
		cerr << "        " << ce.what() << endl;
		cerr << "Selection 2 invalid. Reselect 2nd constraint. " << endl;
		Say("> Selection invalid... reselect");
		pk2.CleanUp();
		parm = 1;
		return;
	}
	try
	{
		button1 = MenuHandler::Get1Pick(pk1);
	}
	catch(ComException& ce)
	{		
	// pick 1 has gone bad  - reset the function
		cerr << "ComException raised in GeomMenuHandler::LineOrthoginal2()" << endl;
		cerr << "        " << ce.what() << endl;
		cerr << "Pick1 gone bad.  Reinitializing function." << endl;
		goto reinit;
	}

// check for invalid second pick
	if(pk2.Type() != POINT && pk2.Type() != ARC && pk2.Type() != CIRCLE)
	{
		Say("> Selection invalid... reselect");
		pk2.CleanUp();
		parm = 1;
		return;
	}

// if pk1 is line or segment
	if(pk1.Type() == LINE || pk1.Type() == SEGMENT)
	{
// get the line
		l = GetLineFromPick(pk1);
// get a perpendicular line
		Line lx(l.Origin(),Cross(Vector(0,0,1), l.Direction()));
// get the solution
		try
		{
			l = CadLib2D_LineParallel(lx,pk2);
		}
		catch (EntityException& ge)
		{
			cerr << "Geomexception raised in GeomMenuHandler::LineOrthoginal2()" << endl;
			cerr << "	" << ge.what() << endl;
			goto reinit;
		}
	}
// pk1 is circle or arc
	else if(pk1.Type() == CIRCLE || pk1.Type() == ARC)
	{
		Circle c = GetCircleFromPick(pk1);
		Point ctr = c.Center();
		try
		{
			l = CadLib2D_LineByConstraints(Pick(&ctr,ctr,LOCAL),pk2);
		}
		catch (EntityException& ge)
		{
			cerr << "Geomexception raised in GeomMenuHandler::LineOrthoginal2()" << endl;
			cerr << "	" << ge.what() << endl;
			goto reinit;
		}
	}
	{
		Handle hx = BankCreate(new Line(l));
		Attributes attr(FD_CYAN,FD_LINESTYLE_DOTTED,FD_LINETHICK_THIN,CurrentLayer()); // sketch geom has fixed color
		Register(hx,attr,Drawable(FD_DRAWABLE_LINE));
	}
	
reinit:
	SetAllHighlights(false);
	Redraw();
	pk1.Unlock();
	pk2.Unlock();
	pk2.CleanUp();
	pk1.CleanUp();
	LineOrthoginal0();
}
//========================================================================
void GeomMenuHandler::CircleCenter0()
{
	Initialize();
	commandStack.push_back(CIRCLECENTER1);
	SelectionFilter sf;	
	sf.Add(POINT);        // Define it
	SetTypeMask(sf);
	parm = 1;
	Say(NULL);
	Say("> Select circle center point");
}
//========================================================================
void  GeomMenuHandler::CircleCenter1()
{
	Pick pk;
	int button;
	
	// check if the first pick is valid
	try
	{
		button = (MenuHandler::Get1Pick(pk));
	}
	catch(ComException & ce)
	{
		cerr << "Comexception raised in GeomMenuHandler::CircleCenter()" << endl;
		cerr << "	" << ce.what() << endl;
	// ignore the pick
		pk.CleanUp();
		parm = 1;
		return;
	}
	if(pk.Type()==POINT)	// only points make sense here 
	{
	// if local prevent deletion of object until Initialize();
		pk.Lock();
	// save the pick back on the pick stack
		PushPick(pk,button);
	// change the command mode
		commandStack.push_back(CIRCLECENTER2);
	// redefine the selection filter
		SelectionFilter sf;	
		sf.Add(POINT);        // Define it
		sf.Add(LINE);
		sf.Add(CIRCLE);
		sf.Add(SEGMENT);
		sf.Add(ARC);
		SetTypeMask(sf);
	// update messages
		Say("  > Select a constraint");
	}
	pk.CleanUp();
	parm =1;
}

//========================================================================
void  GeomMenuHandler::CircleCenter2()
{
	Pick pk1,pk2;
	int button1,button2;


	try
	{
		button2 = MenuHandler::Get1Pick(pk2);
	}
	catch(ComException& ce)
	{	// pick 2 is bad - ignore it.
		cerr << "ComException raised in GeomMenuHandler::CircleCenter2()" << endl;
		cerr << "        " << ce.what() << endl;
		cerr << "Selection 2 invalid. Reselect 2nd constraint. " << endl;
		pk2.CleanUp();
		parm = 1;
		return;
	}
	try
	{
		button1 = MenuHandler::Get1Pick(pk1);
	}
	catch(ComException& ce)
	{		
	// pick 1 has gone bad  - reset the function
		cerr << "ComException raised in GeomMenuHandler::CircleCenter2()" << endl;
		cerr << "        " << ce.what() << endl;
		cerr << "Pick1 gone bad.  Reinitializing function." << endl;
		pk1.Unlock();
		pk2.Unlock();
		pk2.CleanUp();
		pk1.CleanUp();
		CircleCenter0();
		return;
	}

	Circle c;
	Attributes attr(FD_CYAN,FD_LINESTYLE_DOTTED,FD_LINETHICK_THIN,CurrentLayer());

	try
	{
		c = CadLib2D_CircleCenter(pk1,pk2);
	}
	catch(EntityException & ge)
	{
		cerr << "EntityException raised in GeomMenuHandler::CircleCenter() " << endl;
		cerr << "	" << ge.what() << endl;
		pk1.Unlock();
		pk2.Unlock();
		pk2.CleanUp();
		pk1.CleanUp();
		CircleCenter0(); 
		return;
	}
	Handle h = BankCreate(new Circle(c));
       	Register(h,attr,Drawable(FD_DRAWABLE_CIRCLE));
       	Redraw();

// keep the same center and make more circles
//	pk1.Lock();
//	PushPick(pk1,button1);
//	pk2.CleanUp();
//	parm = 1;
//	Say(NULL);
//	Say("> Select additional constraints...");

// reinitialize the function
	pk1.Unlock();
	pk2.Unlock();
	pk2.CleanUp();
	pk1.CleanUp();
	CircleCenter0(); 
}

//========================================================================
void  GeomMenuHandler::Trace0()
{
	Initialize();
	commandStack.push_back(TRACE1);
	SelectionFilter sf;	
	sf.Add(POINT);        // Define it
	sf.Add(LINE);
	sf.Add(SEGMENT);
	sf.Add(CIRCLE);
	sf.Add(ARC);
	SetTypeMask(sf);
	parm = 1;
}
//========================================================================
void  GeomMenuHandler::Trace3()
{

	// the meaning of Trace()...
	// Trace is the basic APT algoritim. 
	// cycling Drive -> Check -> Check system
	//
	// this is how it works: ( thought exersize ) 
	// p1 , p2 -> p1 = drive, p2 = check, create segment(p1,p2)  -> load p2 as drive ->
	// l1 -> segment perp from drive p2 to check l1 -> load l1 as drive -> 
	// c1 -> segment along l1 to the intersection on c1 nearest the selection point of c1. -> 
	// load c1 as drive. (if no intersection, ignore and wait for another pick) ->
	// c2 ( is tangent to c1) -> create an arc along c1 in the direction of the c1 selection point
	// to the c1 c2 tangency point.  On so on...
		
	// Let's figure out the rules:  (each top row is before, bottom after) 
	// for each combination of point, line, and circle. 
	// 
	// | PART  | DRIVE | CHECK |        result                             
	// |-------|-------|--------------------------------------------------------------
	// |   P1  |   P1  |   P2  | segment from P1 to P2.         
	// |   P2  |   P2  |         P2 loaded as both part and drive
	// |-------|-------|--------------------------------------------------------------
	// | PART  | DRIVE | CHECK |        result                             
	// |-------|-------|--------------------------------------------------------------
	// |   P   |   P   |    L  | perpendicular segment from P to L.
	// |   Px  |   L   |         Px is current point at end of segment. 
	// |-------|-------|--------------------------------------------------------------
	// | PART  | DRIVE | CHECK |        result                             
	// |-------|-------|--------------------------------------------------------------
	// |   P   |   P   |    C  | perpendicular segment from P to C.(nearest solution to pick)
	// |   Px  |   C   |         Px is current point at end of segment. 
	// |-------|-------|--------------------------------------------------------------
	// | PART  | DRIVE | CHECK |        result                             
	// |-------|-------|--------------------------------------------------------------
	// |   P1  |   L   |    P2 | segment from projection of P1 on L to projection of P2 on L.
	// |   Px  |   Px  |         Px is current point at end of segment
	// |-------|-------|--------------------------------------------------------------
	// | PART  | DRIVE | CHECK |        result                             
	// |-------|-------|--------------------------------------------------------------
	// |   P1  |   L1  |    L2 | segment from projection of P1 on L to intersection of L1 and L2.
	// |       |       |         +++ do nothing if no solution 
	// |   Px  |   L2  |         Px is current point at end of segment
	// |-------|-------|--------------------------------------------------------------
	// | PART  | DRIVE | CHECK |        result                             
	// |-------|-------|--------------------------------------------------------------
	// |   P1  |   L   |    C  | segment from projection of P on L to intersection of L and C.
	// |   Px  |   C   |         Px is current point at end of segment
	// |-------|-------|--------------------------------------------------------------
	// | PART  | DRIVE | CHECK |        result                             
	// |-------|-------|--------------------------------------------------------------
	// |   P1  |   C   |    P2 | arc from projection of P1 on C to projection of P2 on C.
	// |       |       |       | +++ arc direction is towards C's pick point.
	// |   Px  |   Px  |         Px is current point at end of segment
	// |-------|-------|--------------------------------------------------------------
	// | PART  | DRIVE | CHECK |        result                             
	// |-------|-------|--------------------------------------------------------------
	// |   P   |   C   |    L  | arc from projection of P on C to intersection of L and C.
	// |       |       |       | +++ arc direction is towards C's pick point.
	// |       |       |       | +++ choose solution nearest to L's pick point. 
	// |       |       |       | +++ Do nothing on no intersection - leave PART and DRIVE alone. 
	// |   Px  |   L   |         Px is current point at end of arc
	// |-------|-------|--------------------------------------------------------------
	// | PART  | DRIVE | CHECK |        result                             
	// |-------|-------|--------------------------------------------------------------
	// |   P   |   C1  |   C2  | arc from projection of P on C1 to intersection of C1 and C2
	// |       |       |       | +++ arc direction is towards C1's pick point.
	// |       |       |       | +++ choose solution nearest to C2's pick point. 
	// |       |       |       | +++ Do nothing on no intersection - leave PART and DRIVE alone. 
	// |   Px  |   C2  |         Px is current point at end of arc
	// |-------|-------|--------------------------------------------------------------
	
	// observations: 
	// PART is always the current cutter location - 
	// IF DRIVE is a POINT, make a segment perpendicular to the CHECK
	// IF DRIVE is a LINE , make segment to the intersection with the CHECK
	// IF DRIVE is a CIRCLE, make arc to the intersection with the CHECK
	
	// Create an (segment / arc ) from the current point, along the current drive ,
	// to the intersection with the check. 
	


	// | PART  | DRIVE | CHECK |                            result                             |
	// |   P   |  L,S  |       | perpendicular segment from point to line / segment  |
	// |   P   |  C,A  |       | perpendicular segment from point                    |



// get the 2 picks...
	Pick pk1,pk2;
	int b1,b2;
	try
	{
		b2 = MenuHandler::Get1Pick(pk2);
	}
	catch(ComException& ce)
	{	// pick 2 is bad - ignore it.
		cerr << "ComException raised in GeomMenuHandler::Trace()3" << endl;
		cerr << "        " << ce.what() << endl;
		cerr << "Selection 2 invalid. Reselect 2nd constraint " << endl;
		pk2.CleanUp();
		parm = 1;
		return;
	}
	try
	{
		b1 = MenuHandler::Get1Pick(pk1);
	}
	catch(ComException& ce)
	{		
	// pick 1 has gone bad  - reset the function
		cerr << "ComException raised in GeomMenuHandler::Trace3()" << endl;
		cerr << "        " << ce.what() << endl;
		cerr << "Pick1 gone bad.  Reinitializing Tracing" << endl;
		goto reinit;
	}

// common part-drive-check processing  
	if(pk1.Type() == POINT)
	{       
	// status of traceStatus is irrelevent with points
		Segment s;		// if pk1 is a point this means create segments
		Point p1 = GetPointFromPick(pk1);
		Point p2; 
		switch(pk2.Type())
		{
			case POINT:
			{
				try 
				{
					p2 = GetPointFromPick(pk2);
				}
				catch (EntityException& ce)
				{
					// unable to solve, ignore second pick
					cerr << "EntityException raised in GeomMenuHandler::Trace3()" << endl;
					cerr << "        " << ce.what() << endl;
					cerr << "Invalid 2nd pick. Reselect 2nd constraint" << endl;
					goto repick;
				}
				break;
			}
			case LINE:
			case SEGMENT:
			case ARC:
			case CIRCLE:
			{
				try 
				{
					p2 = CadLib2D_PointByConstraints(pk1,pk2);
				}
				catch (EntityException& ge)
				{
					// unable to solve, ignore second pick
					cerr << "EntityException raised in GeomMenuHandler::Trace3()" << endl;
					cerr << "        " << ge.what() << endl;
					cerr << "Unable to solve for point. Reselect 2nd constraint" << endl;
					goto repick;
				}
				break;
			}
		}
	// create a segment only if points are not equal 
		try 
		{	
			s = Segment(p1,p2);
		}
		catch (EntityException & ge)
		{
			cerr << "EntityException raised in GeomMenuHandler::Trace()3" << endl;
			cerr << "        " << ge.what() << endl;
			cerr << "Can't create segment, reselect 2nd constraint" << endl;
			goto repick;
		}	

	// create the persistent segment and register it with viewer
		{
			Handle h = BankCreate(new Segment(s));
       			Register(h,Attributes(CurrentColor(),CurrentLineStyle(),CurrentThick(),CurrentLayer()),Drawable(FD_DRAWABLE_SEGMENT));
			SetHighlight(pk1,false);
			SetHighlight(pk2,true);
       			Redraw();
	// set the new "part"
 			part = p2;
		}
		goto nextpick;
	} // end else pk1 = POINT

	else if ( pk1.Type() == LINE || pk1.Type() == SEGMENT)
	{
		Line l1;
		Segment s;		// if pk1 is a line/segment this means create segments
		Point p1,p2; 
		Handle h;

		try {
			l1 = GetLineFromPick(pk1);
		}
		catch (EntityException& ge) 
		{	// can't read pick 1 ... oh oh
			cerr << "EntityException raised in GeomMenuHandler::TraceLineDrive()" << endl;
			cerr << "        " << ge.what() << endl;
			goto reinit;
		}
		p1 = l1.Project(part);
		try
		{
			p2 = CadLib2D_PointByConstraints(pk1,pk2); 
			s = Segment(p1,p2);
		}
		catch (EntityException& ge)
		{
			cerr << "EntityException raised in GeomMenuHandler::Trace()" << endl;
			cerr << "        " << ge.what() << endl;
			cerr << "Can't compute endpoint, reselect 2nd constraint" << endl;
			goto repick;
		}

// create and register
		{
			h = BankCreate(new Segment(s));
       			Register(h,Attributes(CurrentColor(),CurrentLineStyle(), CurrentThick(),CurrentLayer()),Drawable(FD_DRAWABLE_SEGMENT));
			SetHighlight(pk1,false);
			SetHighlight(pk2,true);
       			Redraw();
// set the "part" for part-drive-check
			part = p2;
		}
		goto nextpick;
	}
	else if ( pk1.Type() == CIRCLE || pk1.Type() == ARC)
	{
		Arc a1,a2;
		Circle c1;
		Point p1,p2;
		try 
		{
			c1 = GetCircleFromPick(pk1);
			p1 = c1.Project(part);
		}
		catch (EntityException& ce) 
		{	// can't read pick 1 ... oh oh
			cerr << "EntityException raised in GeomMenuHandler::Trace3()" << endl;
			cerr << "        " << ce.what() << endl;
			cerr << "Pick1 gone bad.  Reinitializing Tracing" << endl;
			goto reinit;
		}
		try
		{
			p2 = CadLib2D_PointByConstraints(pk1,pk2); 
// two possible solutions
			a1 = Arc(c1.Center(),p1,p2);
			a2 = Arc(c1.Center(),p2,p1);
		}
		catch (EntityException & ge)
		{
			cerr << "EntityException raised in GeomMenuHandler::Trace()3" << endl;
			cerr << "        " << ge.what() << endl;
			cerr << "Reselect next constraint" << endl;
			goto repick;
		}

		{
			Handle h;
			if(a1.Distance(pk1.GetPoint()) < a2.Distance(pk1.GetPoint()))
				h = BankCreate( new Arc(a1) );
			else
				h = BankCreate( new Arc(a2) );
       			Register(h,Attributes(CurrentColor(),CurrentLineStyle(), CurrentThick(),CurrentLayer()),Drawable(FD_DRAWABLE_ARC));
	// draw
			SetHighlight(pk1,false);
			SetHighlight(pk2,true);
      			Redraw();

			part = p2;
		}
		goto nextpick;
	} // end else - pk1 = CIRCLE/ARC
	cerr << "Error in GeomMenuHandler::Trace() pick is not an expected type: " << pk1.Type() << endl;
	cerr << "Reselect 2nd constraint" << endl;
repick:
	{
		pk2.CleanUp();
		pk1.Lock();
		PushPick(pk1,b1);
		parm = 1;
		return;
	}

nextpick:
	{
	// manage the picks
		pk1.Unlock();
		pk1.CleanUp();
		pk2.Lock();
		PushPick(pk2,b2);
		parm = 1;
		return;
	}
reinit:
	{
		Trace0();
	}
}

//========================================================================
void GeomMenuHandler::ArcClosed0()
{
	Initialize();
	commandStack.push_back(ARCCLOSED);
	SelectionFilter sf;	
	sf.Add(CIRCLE);        // Define it
	SetTypeMask(sf);
	parm = 1;
	arcDialog->ShowDialog("Closed Arcs");
	Say(NULL);
	Say("> Closed arcs : Set options and select a circle");
}
//========================================================================
void  GeomMenuHandler::ArcClosed1()
{
	Pick pk;
	int btn;

	try
	{
		btn = MenuHandler::Get1Pick(pk);
	}
	catch(ComException & ce)
	{
		cerr << "Comexception raised in GeomMenuHandler::ArcClosed1()" << endl;
		cerr << "	" << ce.what() << endl;
		goto cleanup;
	}
	if(btn == 1 && (pk.Type() == CIRCLE ))
	{
 		Circle c = Circle(*(reinterpret_cast<const Circle*>(pk.PointsAt())));
		Arc a(c.Center(),c.Origin(),c.Center());
		Handle h = BankCreate(new Arc(a));
       		Register(h,Attributes(CurrentColor(),CurrentLineStyle(), CurrentThick(),CurrentLayer()),Drawable(FD_DRAWABLE_ARC));

// check arc handlers
		if(arcDialog->Centerlines())
		{
// create two chain line centerline segments
			double d = 1.25 * c.Radius();
			Point dx(d,0);
			Point dy(0,d);
			Point ctr = c.Center();
			Segment s(ctr - dx, ctr + dx);
			h = BankCreate(new Segment(s));
       			Register(h,Attributes(CurrentColor(),FD_LINESTYLE_CHAIN,CurrentThick(),CurrentLayer()),Drawable(FD_DRAWABLE_SEGMENT));
			s = Segment(ctr - dy, ctr + dy);
			h = BankCreate(new Segment(s));
       			Register(h,Attributes(CurrentColor(),FD_LINESTYLE_CHAIN,CurrentThick(),CurrentLayer()),Drawable(FD_DRAWABLE_SEGMENT));
		}
		if(arcDialog->HoleStyle())
		{
			double d; 
			switch(arcDialog->HoleStyle())
			{
			case 1:  // Threaded - 3/4 thread marking on inside of arc
				d = .88 * c.Radius();
				break;
			case 2:  // Tapped - 3/4 arc outside arc
				d = 1.1  * c.Radius();
				break;
			}
			Point dx(d,0);
			Point dy(0,d);
			Point ctr = c.Center();
			Arc a(ctr, ctr + dx, ctr + dy);
			Handle h = BankCreate(new Arc(a));
       			Register(h,Attributes(CurrentColor(),CurrentLineStyle(), CurrentThick(),CurrentLayer()),Drawable(FD_DRAWABLE_ARC));
		}
       		Redraw();
	}
	
cleanup:
	parm = 1;
	pk.CleanUp();
}
			
			

//==================================================================================
void GeomMenuHandler::ArcConstraints0()
{
	Initialize();
	commandStack.push_back(ARCCONSTRAINTS);
	SelectionFilter sf;	
	sf.Add(POINT);        // Define it
	SetTypeMask(sf);
	parm = 3;
}
//========================================================================
void  GeomMenuHandler::ArcConstraints1()
{
	Pick pk1,pk2,pk3;
	int btn1,btn2,btn3;

	try 
	{
		btn3 = MenuHandler::Get1Pick(pk3);
		btn2 = MenuHandler::Get1Pick(pk2);
		btn1 = MenuHandler::Get1Pick(pk1);
	}
	catch(ComException & ce)
	{
		cerr << "Comexception raised in GeomMenuHandler::ArcConstraints()" << endl;
		cerr << "	" << ce.what() << endl;
		goto cleanup;
	}


	if(btn1 && btn2 && btn3 && 
		pk1.Type()  ==  POINT && pk2.Type() == POINT && pk3.Type() == POINT)
	{
 		Point p1 = Point(*( reinterpret_cast<const Point*>(pk1.PointsAt())));
      		Point p2 = Point(*( reinterpret_cast<const Point*>(pk2.PointsAt())));
       		Point p3 = Point(*( reinterpret_cast<const Point*>(pk3.PointsAt())));
			
		try
		{
			Arc a = CadLib2D_SolveArc3Points(p1,p2,p3);
			Handle h = BankCreate(new Arc(a));
       			Register(h,Attributes(CurrentColor(),CurrentLineStyle(), CurrentThick(),CurrentLayer()),Drawable(FD_DRAWABLE_ARC));
		}
		catch(EntityException & ge)
		{
			cerr << "EntityException raised in GeomMenuHandler::ArcConstraints() " << endl;
			cerr << "	" << ge.what() << endl;
			goto cleanup;
		}
       		Redraw();
	}
cleanup:
	parm = 3;
	pk1.CleanUp();
	pk2.CleanUp();
	pk3.CleanUp();
}

//************************************************************************

void GeomMenuHandler::Trace1()
{
	Pick pk;
	int button;

// get the pick
	try
	{
		button = MenuHandler::Get1Pick(pk);
	}
	catch (ComException& ce)
	{
// bad pick, ignore
		cerr << "ComException raised in GeomMenuHandler::Trace()" << endl;
		cerr << "           " << ce.what() << endl;
		parm = 1;
		pk.CleanUp();
		return;
	}


// check out the pick type
	switch (pk.Type())
	{
		case LINE:	  
		case SEGMENT:	 
		case CIRCLE:	
		case ARC:
		{		 // Setting TRACE2 sets the system up for a second pick which, combined
			commandStack.pop_back(); // with this pick will define the start point of the 
			commandStack.push_back(TRACE2); // tracing
			break;
		}
		case POINT:
		{	
			// start on the next selection. See... part is initialized. 
			try
			{
				part = GetPointFromPick(pk);
			}
			catch (EntityException& ce)
			{
				cerr << "EntityException raised in GeomMenuHandler::Trace()" << endl;
				cerr << "        " << ce.what() << endl;
				pk.CleanUp();
				parm = 1;
				return;
			}
				// setting trace 3 means that next pick geom will result. 
			commandStack.pop_back();
			commandStack.push_back(TRACE3);
			break;
		}
	}
// stash it back on the pick stack
// highlight the pick in the viewer
	SetHighlight(pk,true);
	Redraw();
	pk.Lock();	        // in case it is a local object
	PushPick(pk,button);
	parm = 1;
	return;
}

//********************************************************************************
void GeomMenuHandler::Trace2()
{


// get the 2 picks...
	Pick pk1,pk2;
	int button1,button2;
	try
	{
		button2 = MenuHandler::Get1Pick(pk2);
	}
	catch(ComException& ce)
	{	// pick 2 is bad - ignore it.
		cerr << "ComException raised in GeomMenuHandler::Trace()" << endl;
		cerr << "        " << ce.what() << endl;
		cerr << "Selection 2 invalid. Reselect 2nd constraint " << endl;
		pk2.CleanUp();
		parm = 1;
		return;
	}
	try
	{
		button1 = MenuHandler::Get1Pick(pk1);
	}
	catch(ComException& ce)
	{		
	// pick 1 has gone bad  - reset the function
		cerr << "ComException raised in GeomMenuHandler::Trace()" << endl;
		cerr << "        " << ce.what() << endl;
		cerr << "Pick1 gone bad.  Reinitializing Tracing" << endl;
		pk1.Unlock();
		pk2.Unlock();
		pk2.CleanUp();
		pk1.CleanUp();
		Trace0();
		return;
	}

// process the picks - line and circle as first pick 
	Point ptmp;
	try
	{
		ptmp = CadLib2D_PointByConstraints(pk1,pk2);
	}
	catch (EntityException& ge)
	{
		cerr << "EntityException raised in GeomMenuHandler::Trace()" << endl;
		cerr << "        " << ge.what() << endl;
		cerr << "Unable to solve for point. Reselect 2nd constraint" << endl;
		pk2.CleanUp();
		pk1.Lock();
		PushPick(pk1,button1);
		parm = 1;
		return;
	}	
	part = ptmp;				// save the intersection coordinate
	commandStack.pop_back();
	commandStack.push_back(TRACE3);
	SetHighlight(pk1,false);
	SetHighlight(pk2,true);
	Redraw();
	pk1.Unlock();				// unlock the drive, cleanup
	pk1.CleanUp(); 				// save the second pick (check geom) as the drive geom 
	pk2.Lock(); 				// mark the current pick	
	PushPick(pk2,button2); 			// load the "check" as the "drive"
	parm = 1;				// set parm
	return;
}

//========================================================================================
void GeomMenuHandler::Trim0()
{
	Initialize();
	commandStack.push_back(TRIM1);
	SelectionFilter sf;	
	sf.Add(SEGMENT);        // Define it
	sf.Add(ARC);        // Define it
	SetTypeMask(sf);
	parm = 1;
	Say(NULL);
	Say("> Select the geom to trim");
}
//========================================================================================
void GeomMenuHandler::Trim1()
{
	Pick pk;
	int button;
// get the pick
	try
	{
		button = MenuHandler::Get1Pick(pk);
	}
	catch (ComException& ce)
	{
// bad pick, ignore
		cerr << "ComException raised in GeomMenuHandler::Trim1()" << endl;
		cerr << "           " << ce.what() << endl;
		parm = 1;
		pk.CleanUp();
		return;
	}
// check the pick type just to be sure
	if(pk.Type() != SEGMENT && pk.Type() != ARC)
	{
		cerr <<"Invalid pick type passed to GeomMenuHandler::Trim1()" << endl;
		parm = 1;
		pk.CleanUp();
		return;
	}
// set the new mask
	SelectionFilter sf;	
	sf.Add(POINT);
	sf.Add(LINE);
	sf.Add(CIRCLE);
	sf.Add(SEGMENT);        // Define it
	sf.Add(ARC);        // Define it
	SetTypeMask(sf);
// highlight the geom
	SetHighlight(pk,true);
	Redraw();
// set the new command
	commandStack.pop_back();
	commandStack.push_back(TRIM2);
// save the pick
	pk.Lock();	        // in case it is a local object
	PushPick(pk,button);
// set parm
	parm = 1;
// Update the message
	Say("> Select relimiting geom");
}
//========================================================================================
void GeomMenuHandler::Trim2()
{
	Pick pk1,pk2;
	Point pkloc;
	int button1,button2;
	Point ptmp;
	Attributes attrib;
	Handle hx,hnew;
// get the second pick
	try
	{
		button2 = MenuHandler::Get1Pick(pk2);
	}
	catch(ComException& ce)
	{	// pick 2 is bad - ignore it.
		cerr << "ComException raised in GeomMenuHandler::Trim2()" << endl;
		cerr << "        " << ce.what() << endl;
		cerr << "Relimiting selection invalid. Reselect relimiting geom" << endl;
		pk2.CleanUp();
		parm = 1;
		return;
	}
// get the first pick again
	try
	{
		button1 = MenuHandler::Get1Pick(pk1);
	}
	catch(ComException& ce)
	{		
	// pick 1 has gone bad  - reset the function
		cerr << "ComException raised in GeomMenuHandler::Trim2()" << endl;
		cerr << "        " << ce.what() << endl;
		cerr << "Pick1 gone bad.  Reinitializing Trimming" << endl;
		pk1.Unlock();
		pk2.Unlock();
		pk2.CleanUp();
		pk1.CleanUp();
		Trim0();
		return;
	}
// get the pick location
	pkloc = pk1.GetPoint();

// get the intersection point
	try
	{
		ptmp = CadLib2D_PointByConstraints(pk1,pk2);
	}
	catch (EntityException& ge)
	{
		cerr << "EntityException raised in GeomMenuHandler::Trim2()" << endl;
		cerr << "        " << ge.what() << endl;
		cerr << "Unable to solve for point. Reselect 2nd constraint" << endl;
		pk2.CleanUp();
		pk1.Lock();
		PushPick(pk1,button1);
		parm = 1;
		return;
	}	
// get the handle to the original geom
        try 
       	{
		hx = BankGetHandleFromPointer(pk1.PointsAt());
       	}
       	catch (BankException& be)
       	{
// announce the exception - something is seriously wrong if we get this
               	cerr << "BankException raised in GeomMenuHandler::Trim2()" << endl;
               	cerr << "        " << be.what() << endl;
// reset the trim operation
		pk1.Unlock();
		pk2.Unlock();
		pk2.CleanUp();
		pk1.CleanUp();
		Trim0();
		return;
       	}
// get the current attributes so the new geom can look the same
	try
	{
		attrib = GetAttributesFromHandle(hx);
       	}
       	catch (CoreException& ce)
       	{
        // announce the exception - something is seriously wrong if we get this
               	cerr << "CoreException raised in GeomMenuHandler::Trim2()" << endl;
               	cerr << "        " << ce.what() << endl;
// reset the trim operation
		pk1.Unlock();
		pk2.Unlock();
		pk2.CleanUp();
		pk1.CleanUp();
		Trim0();
		return;
        }
// depending on the geom type - segment or arc - relimit the geom
	if(pk1.Type() == SEGMENT)
	{
		Segment oldseg,newseg;
// get a segment
		try
		{
			oldseg = GetSegmentFromPick(pk1);
		}
		catch (EntityException& ce)
		{
			cerr << "EntityException raised in GeomMenuHandler::Trim2()" << endl;
			cerr << "        " << ce.what() << endl;
			cerr << "Cannot read segment from pick - reinitializing trim" << endl;
			pk1.Unlock();
			pk2.Unlock();
			pk2.CleanUp();
			pk1.CleanUp();
			Trim0();
			return;
		}
// check if the point is at one of the endpoints
		if(ptmp == oldseg.Origin() || ptmp == oldseg.Endpoint())
		{
			pk1.Unlock();
			pk2.Unlock();
			pk2.CleanUp();
			pk1.CleanUp();
			Trim0();
			return;
		}
// determine which end of the segment to change based on the selection location of pick1 
// start point - the null arc exception should not be thrown here because of the above check :)
		if(pkloc.Distance(oldseg.Origin()) < pkloc.Distance(oldseg.Endpoint()))
			newseg = Segment(ptmp,oldseg.Endpoint());
		else	
			newseg = Segment(oldseg.Origin(),ptmp);
	
// create and register the new segment
		hnew = BankCreate(new Segment(newseg));
		Register(hnew,attrib,Drawable(FD_DRAWABLE_SEGMENT));
	}
	else if(pk1.Type() == ARC)
	{
		Arc oldarc,newarc;
		try
		{
			oldarc = GetArcFromPick(pk1);
		}
		catch (EntityException& ge)
		{
			cerr << "EntityException raised in GeomMenuHandler::Trim2()" << endl;
			cerr << "        " << ge.what() << endl;
			cerr << "Cannot read arc from pick - reinitializing trim" << endl;
			pk1.Unlock();
			pk2.Unlock();
			pk2.CleanUp();
			pk1.CleanUp();
			Trim0();
			return;
		}
// determine which end to adjust based on the pick location
		int solution = 2; // endpoint
		if(pkloc.Distance(oldarc.Origin()) < pkloc.Distance(oldarc.Endpoint())) 
			solution = 1; // origin

// check if the point is at one of the endpoints
		if((solution == 1 && ptmp == oldarc.Origin()) || 
		   (solution == 2 && ptmp == oldarc.Endpoint()))
		{
			pk1.Unlock();
			pk2.Unlock();
			pk2.CleanUp();
			pk1.CleanUp();
			Trim0();
			return;
		}
			
// determine which end to adjust based on the pick location
		if(solution == 1)
			newarc = Arc(oldarc.Center(),ptmp,oldarc.Endpoint());
		else	
			newarc = Arc(oldarc.Center(),oldarc.Origin(),ptmp);
	
// create and register the new arc 
		hnew = BankCreate(new Arc(newarc));
		Register(hnew,attrib,Drawable(FD_DRAWABLE_ARC));
	}
// unregister the old segm
	UnRegister(hx);
// delete the old segm
	BankDestroy(hx);
// clean up the picks
	pk1.Unlock();
	pk2.Unlock();
	pk1.CleanUp();
	pk2.CleanUp();
// reinitialize trim
	Trim0();
}
//========================================================================================
void GeomMenuHandler::Cut0()
{
	Initialize();
	commandStack.push_back(CUT1);
	SelectionFilter sf;	
	sf.Add(SEGMENT);        // Define it
	sf.Add(ARC);        // Define it
	SetTypeMask(sf);
	parm = 1;
	Say(NULL);
	Say("> Select geom to cut");
}
//========================================================================================
void GeomMenuHandler::Cut1()
{
	Pick pk;
	int button;
// get the pick
	try
	{
		button = MenuHandler::Get1Pick(pk);
	}
	catch (ComException& ce)
	{
// bad pick, ignore
		cerr << "ComException raised in GeomMenuHandler::Cut1()" << endl;
		cerr << "           " << ce.what() << endl;
		parm = 1;
		pk.CleanUp();
		return;
	}
// check the pick type just to be sure
	if(pk.Type() != SEGMENT && pk.Type() != ARC)
	{
		cerr <<"Invalid pick type passed to GeomMenuHandler::Cut1()" << endl;
		parm = 1;
		pk.CleanUp();
		return;
	}
// set the new mask
	SelectionFilter sf;	
	sf.Add(POINT);
	sf.Add(LINE);
	sf.Add(CIRCLE);
	sf.Add(SEGMENT);        // Define it
	sf.Add(ARC);        // Define it
	SetTypeMask(sf);
// highlight the geom
	SetHighlight(pk,true);
	Redraw();
// set the new command
	commandStack.pop_back();
	commandStack.push_back(CUT2);
// save the pick
	pk.Lock();	        // in case it is a local object
	PushPick(pk,button);
// set parm
	parm = 1;
// update message
	Say("> Select cutting geom");
}
//========================================================================================

void GeomMenuHandler::Cut2()
{
	Pick pk1,pk2;
	int button1,button2;
	Point ptmp;
	Attributes attrib;
	Handle hx,hnew1,hnew2;
// get the second pick
	try
	{
		button2 = MenuHandler::Get1Pick(pk2);
	}
	catch(ComException& ce)
	{	// pick 2 is bad - ignore it.
		cerr << "ComException raised in GeomMenuHandler::Cut2()" << endl;
		cerr << "        " << ce.what() << endl;
		cerr << "Cutting selection invalid. Reselect cutting geom" << endl;
		pk2.CleanUp();
		parm = 1;
		return;
	}
// get the first pick again
	try
	{
		button1 = MenuHandler::Get1Pick(pk1);
	}
	catch(ComException& ce)
	{		
	// pick 1 has gone bad  - reset the function
		cerr << "ComException raised in GeomMenuHandler::Cut2()" << endl;
		cerr << "        " << ce.what() << endl;
		cerr << "Pick1 gone bad.  Reinitializing Cutting" << endl;
		pk1.Unlock();
		pk2.Unlock();
		pk2.CleanUp();
		pk1.CleanUp();
		Cut0();
		return;
	}
// get the intersection point
	try
	{
		ptmp = CadLib2D_PointByConstraints(pk1,pk2);
	}
	catch (EntityException& ge)
	{
		cerr << "EntityException raised in GeomMenuHandler::Cut2()" << endl;
		cerr << "        " << ge.what() << endl;
		cerr << "Unable to solve for point. Reselect cutting constraint" << endl;
		pk2.CleanUp();
		pk1.Lock();
		PushPick(pk1,button1);
		parm = 1;
		return;
	}	
// get the handle to the original geom
        try 
       	{
               	hx = BankGetHandleFromPointer(pk1.PointsAt());
       	}
       	catch (BankException& be)
       	{
// announce the exception - something is seriously wrong if we get this
               	cerr << "BankException raised in GeomMenuHandler::Cut2()" << endl;
               	cerr << "        " << be.what() << endl;
		cerr << "Reinitializing cutting" << endl;
// reset the trim operation
		pk1.Unlock();
		pk2.Unlock();
		pk2.CleanUp();
		pk1.CleanUp();
		Cut0();
		return;
       	}
// get the current attributes so the new geom can look the same
	try
	{
		attrib = GetAttributesFromHandle(hx);
       	}
       	catch (CoreException& ce)
       	{
        // announce the exception - something is seriously wrong if we get this
               	cerr << "CoreException raised in GeomMenuHandler::Trim2()" << endl;
               	cerr << "        " << ce.what() << endl;
		cerr << "Can't get attributes. Reinitializing cutting" << endl;
// reset the trim operation
		pk1.Unlock();
		pk2.Unlock();
		pk2.CleanUp();
		pk1.CleanUp();
		Cut0();
		return;
        }
// depending on the geom type - segment or arc - relimit the geom
	if(pk1.Type() == SEGMENT)
	{
		Segment oldseg,newseg1,newseg2;
// get a segment
		try
		{
			oldseg = GetSegmentFromPick(pk1);
		}
		catch (EntityException& ge)
		{
			cerr << "EntityException raised in GeomMenuHandler::Cut2()" << endl;
			cerr << "        " << ge.what() << endl;
			cerr << "Cannot read segment from pick - reinitializing trim" << endl;
			pk1.Unlock();
			pk2.Unlock();
			pk2.CleanUp();
			pk1.CleanUp();
			Cut0();
			return;
		}
// determine if point is actually intersecting the segment
// start point
		if(ptmp.Distance(oldseg.Origin()) > oldseg.Length() || 
		   ptmp.Distance(oldseg.Endpoint()) > oldseg.Length())
		{
// nope, solution is outside segment... do nothing but reinitialize function
			pk1.Unlock();
			pk2.Unlock();
			pk2.CleanUp();
			pk1.CleanUp();
			Cut0();
			return;
		}

// create 2 new segments
		try
		{
			newseg1 = Segment(oldseg.Origin(),ptmp);
			newseg2 = Segment(ptmp,oldseg.Endpoint());
		}
// any problems here invalidate everything - basically cutting is at the end of the existing segment
		catch (EntityException & ge)
		{
			cerr << "EntityException raised in GeomMenuHandler::Cut2()" << endl;
			cerr << "        " << ge.what() << endl;
			cerr << "problems creating new segments" << endl;
			pk1.Unlock();
			pk2.Unlock();
			pk2.CleanUp();
			pk1.CleanUp();
			Cut0();
			return;
		}
		
// create and register the new segments
		hnew1 = BankCreate(new Segment(newseg1));
		hnew2 = BankCreate(new Segment(newseg2));
		Register(hnew1,attrib,Drawable(FD_DRAWABLE_SEGMENT));
		Register(hnew2,attrib,Drawable(FD_DRAWABLE_SEGMENT));
	}
	else if(pk1.Type() == ARC)
	{
		Arc oldarc,newarc1,newarc2;
		try
		{
			oldarc = GetArcFromPick(pk1);
		}
		catch (EntityException& ge)
		{
			cerr << "EntityException raised in GeomMenuHandler::Trim2()" << endl;
			cerr << "        " << ge.what() << endl;
			cerr << "Cannot read arc from pick - reinitializing cutting" << endl;
			pk1.Unlock();
			pk2.Unlock();
			pk2.CleanUp();
			pk1.CleanUp();
			Cut0();
			return;
		}
// check if point is already at one of the endpoints
		if(ptmp == oldarc.Origin() || ptmp == oldarc.Endpoint())
		{
			pk1.Unlock();
			pk2.Unlock();
			pk2.CleanUp();
			pk1.CleanUp();
			Cut0();
			return;
		}
// determine if the point is actually "inside" the arc 
		try
		{
			oldarc.Project(ptmp);
		}
		catch (EntityException& ge)
		{
			cerr << "EntityException raised in GeomMenuHandler::Trim2()" << endl;
			cerr << "        " << ge.what() << endl;
			cerr << "cannot project point onto arc - reinitializing cutting" << endl;
			pk1.Unlock();
			pk2.Unlock();
			pk2.CleanUp();
			pk1.CleanUp();
			Cut0();
			return;
		}
		
// create and register the new arcs
		newarc1 = Arc(oldarc.Center(),oldarc.Origin(),ptmp);
		newarc2 = Arc(oldarc.Center(),ptmp,oldarc.Endpoint());
		hnew1 = BankCreate(new Arc(newarc1));
		hnew2 = BankCreate(new Arc(newarc2));
		Register(hnew1,attrib,Drawable(FD_DRAWABLE_ARC));
		Register(hnew2,attrib,Drawable(FD_DRAWABLE_ARC));
	}
// unregister the old arc
	UnRegister(hx);
// delete the old arc
	BankDestroy(hx);
// clean up the picks
	pk1.Unlock();
	pk2.Unlock();
	pk1.CleanUp();
	pk2.CleanUp();
// reinitialize trim
	Cut0();
}



//****************************************************************************8
void GeomMenuHandler::CircleRadiusAndCenter0()
{
	Initialize();
	double r;

	Say(NULL);
	Say("> Enter radius: ");
// get the radius
	try 
	{ 
		r = KeyboardValue("Enter circle radius");
	}
	catch (ComException & ce)
	{
		// cancelled
		return;
	}
// save the radius for later
	dataStack.push_back(r);
// set up the command
	commandStack.push_back(CIRCLERADIUS);
// set up the filter
	SelectionFilter sf;	
	sf.Add(POINT);        // Define it
	SetTypeMask(sf);
// set the parm counter
	parm = 1;
// user message
	Say(r);
	Say(" > Select circle center point");
}
//========================================================================
void  GeomMenuHandler::CircleRadiusAndCenter1()
{
	Pick pk;
	int button;
	try
	{
		button = MenuHandler::Get1Pick(pk);
	}
	catch (ComException& ce)
	{
// bad pick, ignore
		cerr << "ComException raised in GeomMenuHandler::CircleRadiusAndCenter1()" << endl;
		cerr << "           " << ce.what() << endl;
		pk.CleanUp();
		parm = 1;
		return;
	}
// if not a point ignore
	if(pk.Type() == POINT)
	{
	// get the point
		Point p = GetPointFromPick(pk);
	// get back the radius
		double r = dataStack.back();
		Circle c(p,r);
		Handle hx = BankCreate(new Circle(c));
              	Register(hx, Attributes(FD_CYAN,FD_LINESTYLE_DOTTED,FD_LINETHICK_THIN,CurrentLayer()),Drawable(FD_DRAWABLE_CIRCLE));
              	Redraw();
	}
	pk.CleanUp();
	parm = 1;
	return;
}
//****************************************************************************8
void GeomMenuHandler::ParallelAtDistance0()
{
	Initialize();
	double d;

	Say(NULL);
	Say("> Enter distance: ");
// get the radius
	try 
	{ 
		d = KeyboardValue("Enter parallel distance");
	}
	catch (ComException & ce)
	{
		// cancelled
		return;
	}
// save the radius for later
	dataStack.push_back(d);
// set up the command
	commandStack.push_back(PARALLELATDISTANCE);
// set up the filter
	SelectionFilter sf;	
	sf.Add(LINE);
	sf.Add(CIRCLE);
	sf.Add(SEGMENT);
	sf.Add(ARC);
	SetTypeMask(sf);
// set the parm counter
	parm = 1;
// user message
	Say(d);
	Say(" > Select an geom");
}
//========================================================================
void  GeomMenuHandler::ParallelAtDistance1()
{
	Pick pk;
	int button;
	double distOffset = dataStack.back();	// remember the distance
	Handle hx;
	Attributes attrSketch(FD_CYAN,FD_LINESTYLE_DOTTED,FD_LINETHICK_THIN,CurrentLayer());
	Attributes attr(CurrentColor(),CurrentLineStyle(),CurrentThick(),CurrentLayer());

	if(distOffset < Point::NullDist)goto cleanup;

	try
	{
		button = MenuHandler::Get1Pick(pk);
	}
	catch (ComException& ce)
	{
// bad pick, ignore
		cerr << "ComException raised in GeomMenuHandler::ParallelAtDistace1()" << endl;
		cerr << "           " << ce.what() << endl;
		goto cleanup;
	}

	try
	{
		if((button == 1 && pk.Type() == LINE) || 
        	   (button == 3 && pk.Type() == SEGMENT))	// button 3 picks the support geom
		{
			Line l = GetLineFromPick(pk);
// determine the side to offset based on the pick location
			Vector dirOffset = pk.GetPoint() - l.Project(pk.GetPoint());
			dirOffset = distOffset * dirOffset.Normal() ;
			Line lx(l.Origin() + dirOffset, l.Direction());
			hx = BankCreate(new Line(lx));
              		Register(hx,attrSketch,Drawable(FD_DRAWABLE_LINE));
              		Redraw();
		}
		else if((button == 1 && pk.Type() == CIRCLE) || 
        	        (button == 3 && pk.Type() == ARC ))	// button 3 picks the support geom
      		{

			double rnew;
			Circle c  = GetCircleFromPick(pk);
			Point ctr = c.Center();
			if( ctr.Distance(pk.GetPoint()) < c.Radius()) 
			{
				rnew = c.Radius() - distOffset;
				if(rnew < 0 ) rnew = -1 * rnew;	
			}
			else
			rnew = c.Radius() + distOffset;
		
			Circle cx(c.Center(),rnew);
			hx = BankCreate(new Circle(cx));
              		Register(hx,attrSketch,Drawable(FD_DRAWABLE_CIRCLE));
              		Redraw();
		}
		else if( button == 1 && pk.Type() == SEGMENT)
		{
			Segment s= GetSegmentFromPick(pk);
			Vector dirOffset = pk.GetPoint() - s.Project(pk.GetPoint());
			dirOffset = distOffset * dirOffset.Normal() ;
			Segment sx(s.Origin() + dirOffset, s.Endpoint() + dirOffset);
			hx = BankCreate(new Segment(sx));
       			Register(hx,attr,Drawable(FD_DRAWABLE_SEGMENT));
              		Redraw();
		}
		else if( button == 1 && pk.Type() == ARC)
		{
			Arc a = GetArcFromPick(pk);	// get arc

			Point ctr = a.Center();		// get center

			double rx = ctr.Distance(pk.GetPoint()); // get distance from picked point to center

			Vector v1 = a.Origin() - a.Center(); 	// vectors from cente to start and end points
			Vector v2 = a.Endpoint() - a.Center();
			v1 = v1.Normal();
			v2 = v2.Normal();

			if(rx < a.Radius())  // offset arc inside
			{
				v1 = -v1;

				v2 = -v2;
			// inside offset more than radius requires a swap of point order
				if(distOffset > a.Radius())
				{
				 	Vector vtmp = v1;
					v1 = v2;
					v2 = vtmp;
				}
			}
// create the arc
			Arc ax(a.Center(), a.Origin() + distOffset * v1, a.Endpoint() + distOffset * v2);
			hx = BankCreate(new Arc(ax));
			Register(hx,attr,Drawable(FD_DRAWABLE_ARC));
			Redraw();
		}
	}
	catch (EntityException& ge)
	{
		cerr << "EntityException raised in GeomMenuHandler::ParallelAtDistance()" << endl;
		cerr << "        " << ge.what() << endl;
	}
cleanup:
	pk.CleanUp();
	parm = 1;
}
//****************************************************************************8

void GeomMenuHandler::CircleRadiusAndConstraints0()
{
	Initialize();
	double r ;

	Say(NULL);
	Say("> Enter radius: ");
// get the radius

	try 
	{ 
		bool valid = false;
		while(!valid)
		{
			r = KeyboardValue("Enter circle radius");
			if(r < Point::NullDist)
				Say(" > Radius value must be non-zero and positive");
			else
				valid = true;
		}
	}
	catch (ComException & ce)
	{
		// cancelled
		return;
	}
// save the radius for later
	dataStack.push_back(r);
// set up the command
	commandStack.push_back(CIRCLERADIUSANDCONSTRAINTS1);
// set up the filter
	SelectionFilter sf;	
	sf.Add(POINT);
	sf.Add(LINE);
	sf.Add(CIRCLE);
	sf.Add(SEGMENT);
	sf.Add(ARC);
	SetTypeMask(sf);
// set the parm counter
	parm = 1;
// user message
	Say(r);
	Say(" > Select first constraint:");
}

//****************************************************************************8
void GeomMenuHandler::CircleRadiusAndConstraints1()
{
	Pick pk;
	int button;
	
	// check if the first pick is valid
	try
	{
		button = MenuHandler::Get1Pick(pk);
	}
	catch(ComException & ce)
	{
		cerr << "Comexception raised in GeomMenuHandler::CircleRadiusAndConstraints1()" << endl;
		cerr << "	" << ce.what() << endl;
		Say(" > ERROR: Invalid selection > Reselect constraint.");
	// ignore the pick
		pk.CleanUp();
		parm = 1;
		return;
	}
// all types are still valid

// no new mask

// highlight the geom
	SetHighlight(pk,true);
	Redraw();

// set the new command
	commandStack.pop_back();
	commandStack.push_back(CIRCLERADIUSANDCONSTRAINTS2);

// save the pick
	pk.Lock();	        // in case it is a local object
	PushPick(pk,button);

// set parm
	parm = 1;

// Update the message
	Say(" > Select second constraint");
}

//================================================================================

void GeomMenuHandler::CircleRadiusAndConstraints2()
{
	Pick pk1,pk2;
	int button1,button2;
	Handle h;
	Circle c;

	double r = dataStack.back();

// get the second pick
	try
	{
		button2 = MenuHandler::Get1Pick(pk2);
	}
	catch(ComException& ce)
	{	// pick 2 is bad - ignore it.
		cerr << "ComException raised in GeomMenuHandler::CircleRadiusAndConstraints2()" << endl;
		cerr << "        " << ce.what() << endl;
		Say(" > Cutting selection invalid. Reselect geom");
		pk2.CleanUp();
		parm = 1;
		return;
	}
// get the first pick again
	try
	{
		button1 = MenuHandler::Get1Pick(pk1);
	}
	catch(ComException& ce)
	{		
	// pick 1 has gone bad  - reset the function
		cerr << "ComException raised in GeomMenuHandler::CircleRadiusAndConstraints2()" << endl;
		cerr << "        " << ce.what() << endl;
		Say("ERROR: first constraint lost! >>> Select first constraint.");
		goto reinit;
	}

	try {
		c = CadLib2D_CircleRadiusAndConstraints(r,pk1,pk2);
	}
	catch (EntityException& ge)
	{
		cerr << "EntityException raised in GeomMenuHandler::CircleRadiusAndConstraints2() " << endl;
		cerr << "	" << ge.what() << endl;
		Say(" > Error: No solution >>> Select a constraint.");
		goto reinit;
	}
	{
		h = BankCreate(new Circle(c));
        	Register(h,Attributes(FD_CYAN,FD_LINESTYLE_DOTTED,FD_LINETHICK_THIN,CurrentLayer()),Drawable(FD_DRAWABLE_CIRCLE));
	}
reinit:
	SetHighlight(pk1,false);
       	Redraw();
	pk1.Unlock();
	pk2.Unlock();
	pk1.CleanUp();
	pk2.CleanUp();
	commandStack.pop_back();
	commandStack.push_back(CIRCLERADIUSANDCONSTRAINTS1);
	Say(NULL);
	Say("> Enter radius: ");
	Say(r);
	Say(" > Select first constraint:");
	parm = 1;
}
//===============================================================================
void GeomMenuHandler::CircleConstraints0()
{
	Initialize();
	commandStack.push_back(CIRCLECONSTRAINTS);
	SelectionFilter sf;	
	sf.Add(POINT);        // Define it
               sf.Add(LINE);
	sf.Add(CIRCLE);
	sf.Add(SEGMENT);
	sf.Add(ARC);
	SetTypeMask(sf);
	parm = 3;
}
//===============================================================================
void GeomMenuHandler::CircleConstraints1()
{
	parm = 3;
}
//===============================================================================
void GeomMenuHandler::CreateSupport0()
{
	Initialize();
	commandStack.push_back(CREATESUPPORT);
	SelectionFilter sf;	
	sf.Add(SEGMENT);
	sf.Add(ARC);
	SetTypeMask(sf);
	parm = 1;
	Say(NULL);
	Say("> Create support geom : Select an geom");
}
//===============================================================================
void GeomMenuHandler::CreateSupport1()
{
	Pick pk;
	int button;
	Handle hx;
	Attributes attr(FD_CYAN,FD_LINESTYLE_DOTTED,FD_LINETHICK_THIN,CurrentLayer()); // sketch geom has fixed color
	
	// check if the first pick is valid
	try
	{
		button = MenuHandler::Get1Pick(pk);
	}
	catch(ComException & ce)
	{
		cerr << "Comexception raised in GeomMenuHandler::CreateSupport1()" << endl;
		cerr << "	" << ce.what() << endl;
		Say(" > ERROR: Invalid selection > Reselect constraint.");
		goto cleanup;
	}
	if(pk.Type() == SEGMENT)
	{
		Line l;
		try 
		{
			l = GetLineFromPick(pk);
		}
		catch (EntityException& ge) 
		{	// can't read pick 1 ... oh oh
			cerr << "EntityException raised in GeomMenuHandler::CreateSupport1()" << endl;
			cerr << "        " << ge.what() << endl;
			goto cleanup;
		}
		{
			hx = BankCreate(new Line(l));
			Register(hx,attr,Drawable(FD_DRAWABLE_LINE));
		}
	}
	else if(pk.Type() == ARC)
	{
		Circle c;
		try 
		{
			c = GetCircleFromPick(pk);
		}
		catch (EntityException& ge) 
		{	// can't read pick 1 ... oh oh
			cerr << "EntityException raised in GeomMenuHandler::CreateSupport1()" << endl;
			cerr << "        " << ge.what() << endl;
			goto cleanup;
		}
		{
			hx = BankCreate(new Circle(c));
			Register(hx,attr,Drawable(FD_DRAWABLE_CIRCLE));
		}
	}
	Redraw();
	
cleanup:
	pk.CleanUp();
	parm = 1;
	return;
}
