// menuhandler.cpp

// Copyright (C) 1997  Cliff Johnson                                       //
//                                                                         //
// This program is free software; you can redistribute it and/or           //
// modify it under the terms of the GNU  General Public                    //
// License as published by the Free Software Foundation; either            //
// version 2 of the License, or (at your option) any later version.        //
//                                                                         //
// This software is distributed in the hope that it will be useful,        //
// but WITHOUT ANY WARRANTY; without even the implied warranty of          //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       //
// General Public License for more details.                                //
//                                                                         //
// You should have received a copy of the GNU General Public License       //
// along with this software (see COPYING.LIB); if not, write to the        //
// Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. //

#include <menuhandler.h>

#include <v/vnotice.h>
#include <v/vreply.h>
#include <strstream.h>

#include <entity_enum.h>
#include <vdapp_enum.h>
#include <attrib_enum.h>


#include <cadlib2d.h>
#include <entityexception.h>
#include <coreexception.h>
#include <comexception.h>
#include <bankexception.h>
#include <colordialog.h>
#include <linestyledialog.h>
#include <tabledrawables.h>
#include <attributes.h>

enum { DEFAULTROUNDING 	= -1 };

//******************************************************************************

MenuHandler::MenuHandler(vdCmdWindow* cmdwin,int r) : inputString()
{
	commandWindow = cmdwin;
	rounding = r;
	colorDialog = new ColorDialog(commandWindow,this);
	lineStyleDialog = new LineStyleDialog(commandWindow,this);
}
//******************************************************************************
void MenuHandler::MenuCommand(int id)
{
// MenuCommand is called when the user selects a command button  - Menu specific behavior
// should be encapsulated in a child class. This class is intended to provide the services
// which are intended to be always available - specifically the delete, confirm, and cancel 
// buttons
// Soon I will rip the FunctionMapBehavior and TypeMaskBehavior classes out of vdCmdWindow and
// make them live in this object. 

	switch(id)
	{
		case FD_DELETE:
		{
	// clean the slate
			Initialize();

	// set the system up to select entities for deletion
			commandStack.push_back(FD_DELETE);	

	// clear the selection mask
			ClearTypeMask();
	
			Say(NULL);
			Say("Delete : Select objects to delete, then Yes or No! >");

	// set the parm count

			parm = 1;

			break;
		}
		case FD_CONFIRM:
		{
			if(commandStack.back() == FD_DELETE && deleteStack.size() > 0) ToastDeleteStack();
			break;
		}
		case FD_CANCEL:
		{
			Initialize();
			Say(NULL);
			Say("Command Cancelled.");
			break;
		}
		case m_ColorAttribute:
		case FD_COLORBUTTON:
		{	
			InitializeColor();
			break;
		}
		case m_LayerAttribute:
		case m_LineTypeAttribute:
		{
			InitializeLineStyle();
			break;
		}
	
	}
}

//*********************************************************************

void MenuHandler::InitializeColor()
{
	Initialize();

	commandStack.push_back(FD_COLORBUTTON);// onto the parmStack, and set the current parm


	// two behaviors to implement - just setting the default - and changing colors.
	// The use can change colors of existing entity after the dialog is displayed 
        // to until another action'is chosen. The default color can be manipulated as long
	// as the dialog is visible. The user can dismiss the dialog with the cancel button.

	SelectionFilter sf;
	sf.Add(SEGMENT);
	sf.Add(ARC);
	SetTypeMask(sf);

	parm = 1;			

	// display the dialog
		colorDialog->ShowDialog("Color Chooser");
}
//*********************************************************************

void MenuHandler::InitializeLineStyle()
{
	Initialize();

	commandStack.push_back(FD_LINESTYLE);// onto the parmStack, and set the current parm

	SelectionFilter sf;
	sf.Add(SEGMENT);
	sf.Add(ARC);
	SetTypeMask(sf);

	parm = 1;			

	// display the dialog
		lineStyleDialog->ShowDialog(NULL);
}

//*********************************************************************
void MenuHandler::LoadColor(int value) // loads the current color - typically called by the dialog
{

	int cx = value;

	if(cx < FD_BLACK ) cx = FD_BLACK; // 0 
	if(cx > FD_WHITE) cx = FD_WHITE; // 15

	// change the current color
	commandWindow->currentColor = cx;

	// change the button color
	commandWindow->SetValue(FD_COLORBUTTON,vStdColors[cx].r(),Red);
	commandWindow->SetValue(FD_COLORBUTTON,vStdColors[cx].g(),Green);
	commandWindow->SetValue(FD_COLORBUTTON,vStdColors[cx].b(),Blue);
}
//*********************************************************************
void MenuHandler::LoadThick(unsigned int value) 
{
	unsigned int xv = value;
	if(xv < FD_LINETHICK_THIN) xv = FD_LINETHICK_THIN;
	if(xv > FD_LINETHICK_VERYWIDE) xv = FD_LINETHICK_VERYWIDE;
	commandWindow->currentThick = xv;
}
//*********************************************************************
void MenuHandler::LoadPointSize(unsigned int value) // loads the current color - typically called by the dialog
{
	unsigned int xv = value;
	if(xv < FD_POINTSIZE_TINY) xv = FD_POINTSIZE_TINY;
	if(xv > FD_POINTSIZE_HUGE) xv = FD_POINTSIZE_HUGE;
	commandWindow->currentPointSize= xv;
}
//*********************************************************************
void MenuHandler::LoadLineStyle(unsigned short value) // loads the current color - typically called by the dialog
{
	unsigned short xv = value;
	if(xv == FD_LINESTYLE_DOTTED || xv == FD_LINESTYLE_SOLID || xv == FD_LINESTYLE_LONGDASH || xv == FD_LINESTYLE_CHAIN)
	{
		commandWindow->currentLineStyle = value;
	}
}

//*********************************************************************
void MenuHandler::ChangeAttribute()	// change the color of the selected object
{

	Pick pk;
	int button;

//cerr << "MenuHandler::ChangeAttribute()" << endl;
//cerr << "commandStack.back() = " << commandStack.back() << endl;

// get 1 pick
	try
	{
		button = Get1Pick(pk);
	}
	catch(ComException& ce)
	{
	// ignore the invalid pick
		pk.CleanUp();
		parm = 1;
		return;
	}
	
// change the object color/attribute in the viewer
// sketch entity color cannot be changed
	if(pk.Type() == LINE || pk.Type() == CIRCLE)
	{
		pk.CleanUp();
		parm = 1;
		return;
	}

	Handle hx;
	Attributes aold, anew;
// get the entity
	try 
	{
		hx = BankGetHandleFromPointer(pk.PointsAt());
	}
	catch (BankException& be)
	{
// announce the exception - something is seriously wrong if we get this
		cerr << "BankException raised in MenuHandler::ChangeAttribute()" << endl;
		cerr << "        " << be.what() << endl;
// ignore the invalid pick
		pk.CleanUp();
		parm = 1;
		return;
	}
// get the current attributes
	try
	{
		aold = GetAttributesFromHandle(hx);
	}
	catch (CoreException& ce)
	{
	// announce the exception - something is seriously wrong if we get this
		cerr << "CoreException raised in MenuHandler::ChangeAttribute()" << endl;
		cerr << "        " << ce.what() << endl;
// ignore the invalid pick
		pk.CleanUp();
		parm = 1;
		return;
	}

	switch(commandStack.back())
	{
//		case FD_LINETHICK:
//		{
//			anew = Attributes(aold.GetColor(),aold.GetLineStyle(),GetAttribute(FD_LINETHICK));
//			break;
//		}
//		case FD_POINTSIZE:
//		{
//			anew = Attributes(,aold.GetLineStyle(), aold.GetThick());
//			break;
//		}
		case FD_LINESTYLE:
		{
			anew = Attributes(aold.GetColor(),CurrentLineStyle(),CurrentThick(),aold.GetLayer());
			break;
		}
		case FD_COLORBUTTON:
		{
			anew = Attributes(CurrentColor(),aold.GetLineStyle(), aold.GetThick(),aold.GetLayer());
			break;
		}
	}

		
	commandWindow->drawregister->SetAttributes(hx,anew);
	Redraw();
		
// clean up
	pk.CleanUp();
// reset parm
	parm = 1;

}

//*********************************************************************
//void MenuHandler::CancelColor()
//{
// shut down the color modification - restore the system to the saved state. 
//	parm = parmStack.back();	// previous parm value becomes current
//	parmStack.pop_back();		// pop previous value
//	commandStack.pop_back();	// restore the current command
//	RestoreTypeMask();
//}

//*********************************************************************
void MenuHandler::Say(const int& i)
{
	inputString.AppendString(i);
}
void MenuHandler::Say(const char* sx)
{
	char* s;

	if(sx == NULL)
	{
		inputString.Clear();
		commandWindow->SetString(m_InputStatusPane,"");
	}
	else 
	{
		inputString.AppendString(sx);
		s =  const_cast<char*> (inputString.GetString());
		commandWindow->SetString(m_InputStatusPane,s);
	}
		
}

//******************************************************************************
void MenuHandler::ToastDeleteStack()
{
	Handle hx;

	while(deleteStack.size()!= 0)
	{
		hx = deleteStack.back();
// stop drawing this
		UnRegister(hx);
// toast it
		BankDestroy(hx);
// decrement the stack
		deleteStack.pop_back();

	}	
	Redraw();
	parm = 1;
	Say(NULL);
	Say("Deletion Confirmed");
}

//******************************************************************************
void MenuHandler::EmptyDeleteStack()
{
	Handle hx;

	while(deleteStack.size()!= 0)
	{
		hx = deleteStack.back();
		commandWindow->drawregister->SetVisible(hx,true);
		deleteStack.pop_back();
	}
	Redraw();
}


//******************************************************************************
void MenuHandler::LoadModifier(int code)
{


// modifier command and parm count are pushed onto the top of their respective stacks

	if(commandStack.size() == 0 )
	{			// is no active command
		commandWindow->SetValue(code,0,Value); 	// don't let them push the modifier
		return;					// no current command 
	}
// keyboard is special - it acts immediatly, does not wait for a pick. 
cout << "code = " << code << endl;
	if(code == FD_KEYBOARD)
	{
// get the point coordinates only if the point filter is active
cout << "before parm = " << parm << endl;
		if( POINT == GetTypeMask())  KeyboardPointModifier();
		else
		{
			vNoticeDialog note(commandWindow);
			note.Notice("Error: Point is not a selected input class.");
		}
cout << "after parm = " << parm << endl;
		CommandActions();
		return;
	}

	// save the current filter
	filterStack.push_back(GetTypeMask());
	// save the current parm count
	parmStack.push_back(parm);

	switch(code)
	{
		case FD_ENDPOINT:				 // for each case, push the current parm count
		{
			commandWindow->modifierBehavior->Selected(commandWindow,FD_ENDPOINT);

			SelectionFilter sf;
			sf.Add(SEGMENT);
			sf.Add(ARC);
			SetTypeMask(sf);

			commandStack.push_back(FD_ENDPOINT);// onto the parmStack, and set the current parm
			parm = 1;			
			break;
		}
		case FD_MIDPOINT:					
		{
			commandWindow->modifierBehavior->Selected(commandWindow,FD_MIDPOINT);

			SelectionFilter sf;
			sf.Add(SEGMENT);
			sf.Add(ARC);
			SetTypeMask(sf);

			commandStack.push_back(FD_MIDPOINT);
			parm = 1;
			break;
		}
		case FD_CENTER:
		{
			commandWindow->modifierBehavior->Selected(commandWindow,FD_CENTER);

			SelectionFilter sf;
			sf.Add(CIRCLE);
			sf.Add(ARC);
			SetTypeMask(sf);

			commandStack.push_back(FD_CENTER);
			parm = 1;
			break;
		}
		case FD_INTERSECTION:
		{
			InitializeIntersectionModifier();
			break;
		}
		case FD_SUPPORT:
		{
			commandWindow->modifierBehavior->Selected(commandWindow,FD_SUPPORT);
			commandStack.push_back(FD_SUPPORT);
			parm = 1;
			break;
		}
	}
}

//******************************************************************************
void MenuHandler::SimpleModifier(int modifier)
{
// replace the next selection with it's midpoint

// read the last off the pick stack- 

	Pick pk;
	try
	{
		Get1Pick(pk);


		ClearModifier();		// clear the modifier 
		parm = parmStack.back();	// previous parm value becomes current
		parmStack.pop_back();		// pop previous value
		commandStack.pop_back();	// restore the current command
// replace them with a single immediate point at the intersection (Point constraint
// result) of the original picks.
		Point p = CadLib2D_PointByModifier(pk,modifier);
			
               	Pick pkx(new Point(p),p,LOCAL);			// create a local Pick
		picks.push_back(PickAndButton(pkx,2));		// put it on the pick stack
		parm--;					// this counts as 1 parm 
	}
	catch (ComException& ce)
	{
		cerr << "Comexception raised in MenuHandler::SimpleModifier( " << modifier << " )" << endl;
		cerr << "	" << ce.what() << endl;
		ClearModifier();		// clear the modifier 
		parm = parmStack.back();	// previous parm value becomes current
		parmStack.pop_back();		// pop previous value
		commandStack.pop_back();	// restore the current command
	}
	catch (EntityException & ge )
	{
		cerr << "EntityException raised in MenuHandler::SimpleModifier()" << endl;
		cerr << "	" << ge.what() << endl;
		ClearModifier();		// clear the modifier 
		parm = parmStack.back();	// previous parm value becomes current
		parmStack.pop_back();		// pop previous value
		commandStack.pop_back();	// restore the current command
	}

// clean up the picks
	pk.CleanUp();
}



//******************************************************************************
void MenuHandler::KeyboardPointModifier()
{
// get a point
	Point p;
	try 
	{
		p = KeyboardPoint();
	}
	catch (ComException & ce)
	{
		// cancelled...
		return;
	}
        Pick pkx(new Point(p),p,LOCAL);			// create a local Pick
	picks.push_back(PickAndButton(pkx,2));		// put it on the pick stack
	parm--;			        // this counts as 1 parm 
}
//******************************************************************************

void MenuHandler::IntersectionModifier1()
{
// read the last 2 picks off the pick stack- 
	Pick pk;
	int button;
	try
	{
		button = Get1Pick(pk);
	}
	catch(ComException& ce)
	{
		cerr << "Comexception raised in MenuHandler::IntersectionModifier1()" << endl;
		cerr << "        " << ce.what() << endl;
	// ignore the pick
		pk.CleanUp();
		parm = 1;
		return;
	}
	// highlight the pick
	SetHighlight(pk,true);
	Redraw();
	// save the pick
	pk.Lock();
	PushPick(pk,button);
	// set parm
	parm = 1;
	// set command
	commandStack.pop_back();
	commandStack.push_back(INTERSECTIONMODIFIER2);
}


//************************************************************************
void MenuHandler::IntersectionModifier2()
{
// read the last 2 picks off the pick stack- 
	Pick pk1,pk2;
	int button1,button2;
	Point p;

	try
	{
		button2 = Get1Pick(pk2);
	}
	catch (ComException& ce)
	{
		// bad pick 2 --- ignore the pick
		cerr << "ComException raised in MenuHandler::IntersectionModifier2()" << endl;
		cerr << "        " << ce.what() << endl;
		pk2.CleanUp();
		parm = 1;
		return;
	}
	try
	{
		button1 = Get1Pick(pk1);
	}
	catch (ComException& ce)
	{
		cerr << "ComException raised in MenuHandler::IntersectionModifier2()" << endl;
		cerr << "        " << ce.what() << endl;
		// pick 1 no longer valid  - abort modifier - 
		ClearModifier();		// clear the modifier 
		parm = parmStack.back();	// previous parm value becomes current
		parmStack.pop_back();		// pop previous value
		commandStack.pop_back();	// restore the current command
		RestoreTypeMask();
		pk1.Unlock();
		pk2.Unlock();
		pk1.CleanUp();
		pk2.CleanUp();
		return;
	}
	
// replace them with a single immediate point at the intersection (Point constraint
// result) of the original picks.
	try
	{
		p = CadLib2D_PointByConstraints(pk1,pk2);
	}
	catch (EntityException & ge )
	{
		cerr << "EntityException raised in MenuHandler::IntersectionModifier2()" << endl;
		cerr << "	" << ge.what() << endl;
		cerr << "reselect second constraint."<< endl;
		pk2.CleanUp();
		parm = 1;
		return;
	}
        Pick pkx(new Point(p),p,LOCAL);		// fake a local Pick
	pkx.Lock();				// lock it.
// save the fake pick
	PushPick(pkx,2);	// put it on the pick stack

// clean up you mess.
	SetHighlight(pk1,false);
	Redraw();
	pk1.Unlock();
	pk2.Unlock();
	pk1.CleanUp();
	pk2.CleanUp();

// retrieve the previous command state
	ClearModifier();		// clear the modifier 
	parm = parmStack.back();	// previous parm value becomes current
	parmStack.pop_back();		// pop previous value
	commandStack.pop_back();	// restore the current command
	RestoreTypeMask();

// this counts as 1 parm 
	parm--;				
}

//******************************************************************************
void MenuHandler::Delete(/* const Selection& sx */)
{
	// remember for deletions, we need handles - not picks - 
	// if the selection actually points to something, add it to a 
	// deleteion stack, and highlight it in the viewer

	Pick pk;
	Handle hx;
	try
	{
		parm = 1;

		Get1Pick(pk);
		
		// local picks are to be ignored
		if(pk.IsLocal()) throw ComException("MenuHandler::Delete() : local pick ignored");

		// get the handle for the pick... if any
		hx = BankGetHandleFromPointer(pk.PointsAt());

		commandWindow->drawregister->SetVisible(hx,false);

		deleteStack.push_back(hx);

		Redraw();
		

		commandWindow->functionMapBehavior->Selected(commandWindow,FD_DELETE);
	}

	catch (ComException & ce)
	{
		cerr << "ComException Raised in MenuHandler::Delete()" << endl;
		cerr << "	" << ce.what() << endl;
	}

	catch (BankException & be)
	{
		cerr << "BankException Raised in MenuHandler::Delete()" << endl;
		cerr << "	" << be.what() << endl;
	}

	pk.CleanUp();
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// wrapper for a register command
	
void MenuHandler::Register(	const Handle& h, 
				const Attributes& a, 
				const DrawableEntity* d,
				bool v) const
{
	commandWindow->drawregister->Register(h,a,d,v);
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

void MenuHandler::Redraw() const
{
	commandWindow->vdglcanvas->Redraw(0,0,0,0);
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

void MenuHandler::ClearTypeMask()
{
	commandWindow->typeMaskBehavior->Clear();
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

void MenuHandler::ClearModifier()
{
	commandWindow->modifierBehavior->Clear(commandWindow);
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

void MenuHandler::ClearFunctionMap()
{
	commandWindow->functionMapBehavior->Clear(commandWindow);
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

void MenuHandler::CancelCanvasAction()
{
	commandWindow->vdglcanvas->CancelAction();
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

void MenuHandler::SetTypeMask(const SelectionFilter& sf)
{
	commandWindow->typeMaskBehavior->SetFilter(sf);
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

SelectionFilter MenuHandler::GetTypeMask() const
{
	return commandWindow->typeMaskBehavior->GetFilter();
}
//++++++++++++++++Get1Pick->++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

void MenuHandler::Initialize()
{
// a command initializes everything!!!!
	EmptyDeleteStack();
	Pick pk;
        while (commandStack.size() != 0 )commandStack.pop_back();
        while (parmStack.size() != 0 )parmStack.pop_back();
	while (dataStack.size() != 0 )dataStack.pop_back();

        while (picks.size() != 0 ){
		pk = picks.back().first;
		if(pk.IsLocked())pk.Unlock();         // garbage collection
		pk.CleanUp();		              // be sure to kill any leftover local data
		picks.pop_back();
	}
	
	parm = 0;
	
// initialize some behaviors
        ClearTypeMask();
        ClearFunctionMap();
        CancelCanvasAction();
//	SetFrameSelect(false);
	SetAllHighlights(false);
	Redraw();
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

void MenuHandler::LoadPick(const Selection& sx)
{


// convert the selection, into a pick - 
	Pick pk;
	if(sx.Button()==1 && sx.GetHandle() != Handle())
	{
		pk = Pick(sx.GetHandle().PointsAt(),sx.GetPoint());
	}
	else 
	{
		Point* px = new Point(sx.GetPoint().Round(rounding));
		pk = Pick(px,Point(*px),LOCAL);
	}

	picks.push_back(PickAndButton(pk,sx.Button()));	 // remember the pick and button

	parm--;                         // decrement the parameter counter
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void MenuHandler::DoSelection(const SelectStack& ss)
{
// adds to the current group with the window selected things
	Handle h;

	if(commandStack.size() == 0 ) return;
	if(commandStack.back() == FD_FRAME_SELECT)
	{
// get names into the current group
		for(int i = 0; i < ss.Size() ; i++)
			currentGroup.Add(ss[i].Name());
	}
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void MenuHandler::DoPick(const Selection& sx)
{
	if(commandStack.size() == 0) return;
	LoadPick(sx);
	CommandActions();
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

void MenuHandler::CommandActions()
{
// in this case, process deletions if it is active
	switch(commandStack.back())
	{
		case FD_DELETE:
		{
			Delete(/* currentSelection*/);
			break;
		}
		case FD_ENDPOINT:
		{
			SimpleModifier(CadLib2D_ENDPOINT);
			RestoreTypeMask();
			break;
		}
		case FD_MIDPOINT:					
		{
			SimpleModifier(CadLib2D_MIDPOINT);
			RestoreTypeMask();
			break;
		}
		case FD_CENTER:
		{
			SimpleModifier(CadLib2D_CENTER);
			RestoreTypeMask();
			break;
		}
		case INTERSECTIONMODIFIER1:
		{
			IntersectionModifier1();
			break;
		}
		case INTERSECTIONMODIFIER2:
		{
			IntersectionModifier2();
			break;
		}
		case FD_SUPPORT:
		{
			RestoreTypeMask();
			break;
		}
		case FD_COLORBUTTON:
		case FD_LINESTYLE:
		{
			ChangeAttribute();
			break;
		}
	}
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void MenuHandler::RestoreTypeMask()
{
	if(filterStack.size() > 0){
		SelectionFilter sf = filterStack.back();
		filterStack.pop_back();
		commandWindow->typeMaskBehavior->SetFilter(sf);
	}
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
int MenuHandler::Get1Pick(Pick& pk) throw (ComException)// returns the mouse button or 0 if invalid
{
	if(picks.size()<1)throw ComException("MenuHandler::Get1Picks() : Can't get 1 Pick");

	pk = picks.back().first;
	int button = picks.back().second;
	picks.pop_back();

	// if button is 1 picks must not be 'LOCAL' - this indicates an invalid pick -
	// i.e. picking an immediate point with button 1
	if(pk.IsLocal() && button == 1)throw
		ComException("MenuHandler::Get1Pick() : button 1 cannot select a temporary object");

	// any picking with button 3 is invalid:
	//if(button == 3)throw
		ComException("MenuHandler::Get1Pick() : button 3 is invalid for picking");

	return button;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void MenuHandler::PushPick(const Pick& pk,int button)
{
	picks.push_back(PickAndButton(pk,button)); 
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void MenuHandler::SetHighlight(const Pick& pk,bool hl) const
{
/* get handle from pick */
	if(pk.IsLocal() || pk.IsLocked())return;
	Handle hx ;
	try
	{
		hx = BankGetHandleFromPointer(pk.PointsAt());
	}
	catch (CoreException& ce)
	{
		cerr << "CoreException raised in MenuHandler::HighLight" << endl;
		cerr << "        " << ce.what() << endl;
		return;
	}
	commandWindow->drawregister->SetHighlight(hx,hl);
}
	
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void MenuHandler::SetAllHighlights(bool hl) const
{
	commandWindow->drawregister->SetAllHighlights(hl);
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void MenuHandler::SetVisible(const Pick& pk,bool v) const
{
/* get handle from pick */
	if(pk.IsLocal() || pk.IsLocked())return;
	Handle hx ;
	try
	{
		hx = BankGetHandleFromPointer(pk.PointsAt());
	}
	catch (CoreException& ce)
	{
		cerr << "CoreException raised in MenuHandler::HighLight" << endl;
		cerr << "        " << ce.what() << endl;
		return;
	}
	commandWindow->drawregister->SetVisible(hx,v);
}
	
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void MenuHandler::SetAllVisible(bool v) const
{
	commandWindow->drawregister->SetAllVisible(v);
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void MenuHandler::SwapVisible() const
{
	commandWindow->drawregister->SwapVisible();
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

void MenuHandler::InitializeIntersectionModifier()
{
	commandWindow->modifierBehavior->Selected(commandWindow,FD_INTERSECTION);
	
	SelectionFilter sf;
	sf.Add(LINE);
	sf.Add(CIRCLE);
	sf.Add(SEGMENT);
	sf.Add(ARC);
	SetTypeMask(sf);

	commandStack.push_back(INTERSECTIONMODIFIER1);
	parm = 1;
}


//==============================================================================

Attributes MenuHandler::GetAttributesFromHandle(const Handle& hx) const throw (CoreException)
{
	Attributes attr; 
	try
	{
		attr = commandWindow->drawregister->GetAttributes(hx);
	}
	catch (CoreException& ce)
	{
		throw;
	}
	return attr;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void MenuHandler::UnRegister(const Handle& hx) const
{
	commandWindow->drawregister->UnRegister(hx);
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
Handle MenuHandler::BankCreate(Entity* g) const
{
	return commandWindow->bank->Create(g);
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void MenuHandler::BankDestroy(const Handle& hx) const
{
	commandWindow->bank->Destroy(hx);
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
Handle MenuHandler::BankGetHandleFromPointer(const Entity* ptr) const throw(BankException)
{
	Handle hx;
	try
	{
		hx = commandWindow->bank->GetHandleFromPointer(ptr);
	}
	catch (BankException& be)
	{
		throw;
	}
	return hx;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
const DrawableEntity* MenuHandler::Drawable(const int& i) const throw(DrawableException)
{
	const DrawableEntity* dx;
	try
	{
		dx = commandWindow->tableDrawables->Drawable(i);
	}
	catch(DrawableException& de)
	{	
		throw;
	}
	return dx;
}


//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
string MenuHandler::CurrentFileName() const
{
	return commandWindow->currentFileName;
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
int MenuHandler::CurrentColor() const 
{
	return commandWindow->currentColor;
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
unsigned short MenuHandler::CurrentLineStyle() const
{
	return commandWindow->currentLineStyle;
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
unsigned int MenuHandler::CurrentThick() const
{
	return commandWindow->currentThick;
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
unsigned int MenuHandler::CurrentPointSize() const
{
	return commandWindow->currentPointSize;
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
unsigned int MenuHandler::CurrentLayer() const
{
	return commandWindow->currentLayer;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void MenuHandler::CanvasRegisterInit() const
{
	commandWindow->drawregister->Init();
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
bool MenuHandler::CanvasRegisterNextItem(RegisteredEntity& re) const
{
	return commandWindow->drawregister->NextItem(re);
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
Point MenuHandler::KeyboardPoint() const throw(ComException)
{
	vNoticeDialog note(commandWindow);
	vReplyDialog reply(commandWindow);
	double x,y;

	char buf[100];

	bool asking = true;

	while(asking)
	{
// create a input dialog
		if(reply.Reply("enter x and y coordinates",buf,99) == M_Cancel)
			throw ComException("MenuHandler::KeyboardPoint() : Function cancelled.");

// check the answer... 
// call it a stream
		istrstream input(buf);

// try to read two doubles -- for now, later put in a expression interpreter, or find one, oe build
// one ( i do have tha lexx/yak book ) .
		input >> x >> y;

		if(input.good())asking = false;
		else note.Notice("Error:  enter two numbers separated by a space.");
	}

	return Point(x,y);
}
		


//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// this can be used whenever a child needs to prompt the user for input of a
// single real number. 
double MenuHandler::KeyboardValue(const char* message) const throw(ComException)
{
	vNoticeDialog note(commandWindow);
	vReplyDialog reply(commandWindow);
	double x;

	char buf[100];

	bool asking = true;

	while(asking)
	{
// create a input dialog
		if(reply.Reply(message,buf,99) == M_Cancel)
			throw ComException("MenuHandler::KeyboardPoint() : Function cancelled.");

// check the answer... 
// call it a stream
		istrstream input(buf);

		input >> x ;

		if(input.good())asking = false;
		else note.Notice("Error:  enter a number.");
	}

	return x;
}
//================================================================================
