// menuhandler.h

// Copyright (C) 1997  Cliff Johnson                                       //
//                                                                         //
// This program is free software; you can redistribute it and/or           //
// modify it under the terms of the GNU  General Public                    //
// License as published by the Free Software Foundation; either            //
// version 2 of the License, or (at your option) any later version.        //
//                                                                         //
// This software is distributed in the hope that it will be useful,        //
// but WITHOUT ANY WARRANTY; without even the implied warranty of          //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       //
// General Public License for more details.                                //
//                                                                         //
// You should have received a copy of the GNU General Public License       //
// along with this software (see COPYING); if not, write to the        //
// Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. //


#ifndef MENUHANDLER_H
#define MENUHANDLER_H

#include <vector>
#include <string>

#include <selection.h>
#include <pick.h>
#include <pair.h>
#include <drawableentity.h>
#include <vdcmdwin.h>
#include <menuhandler_enum.h>
#include <group.h>

class ComException;
class DrawableException;
class ColorDialog;
class LineStyleDialog;
class Attributes;

class MenuHandler
{

protected:

	vdCmdWindow* commandWindow;

	typedef pair<Pick,int> PickAndButton;
	vector<PickAndButton> picks;
	int parm;
	vector<int> parmStack;
	vector<int> commandStack;
	vector<SelectionFilter> filterStack;
	vector<double>dataStack;
	vector<Handle>deleteStack;
	int rounding; 			// rounding constant - PLEASE SEE POINT.H !!!!!
	Selection currentSelection;
	InputString inputString;
	
	Group currentGroup;
				// just one unnamed group for now.. in the old school.
				// tracking names instead of handles because handles cannot
				// be trusted to exist persistently. Names are kept persistently,
				// and are easier treated by retrieving the current handle 
				// from the name when it is required. If the object a name
				// refers to does not exist, it is less risky then finding
				// out by the handle itself. 

public:
	// The selection handler needs to know the entitybank, canvas, and rounding constant.

	MenuHandler(vdCmdWindow* cmdwin,int r);

	virtual ~MenuHandler() {}

//	void SetFrameSelect(const bool&); // tell the interface that frame selection is 
					  // expected for this function  - like delete - 
					  // the interface should then make sensitive 
					  // the FS button. ( if bool is true of course) 
	virtual void DoSelection(const SelectStack& ss);// frame selections from viewer
	virtual void DoPick(const Selection& s);	// provide to deal
							// with derived windows
							// selections
	virtual void CommandActions(); // process pending modifiers and commands

	virtual void MenuCommand(int);                 // provide to handle 
							// input and processing
							// of commands
//	void ExecModifier();

	void LoadPick(const Selection& sx);		// load a pick from a selection

	void LoadModifier(int code);	// not changable behavior

	void Initialize(); // set all stacks to zero. 

	void Rounding(int r) {rounding = r; }		// managment of the rounding constant
	int Rounding() const { return rounding; }

	// simple is for midpoint, endpoint, and center... those with 1 pick.
	// parameter should be FD_MIDPOINT, FD_CENTER, or FD_ENDPOINT.
	void SimpleModifier(int);

	// intersection uses 2 functions to process 2 picks
	void InitializeIntersectionModifier();
	void IntersectionModifier1();
	void IntersectionModifier2();
	void KeyboardPointModifier();

	void Delete(/* const Selection& */ );

// bank functions
	Handle BankCreate(Entity*) const; 	// create a new entity in the bank
	void BankDestroy(const Handle& ) const;   // destroy an existing entity from the bank
	Handle BankGetHandleFromPointer(const Entity*) const throw (BankException);

// drawregister and vdcmdwindow manipulation functions
	void Register(const Handle&, const Attributes&, const DrawableEntity* ,bool v = true)const;
	void UnRegister(const Handle&) const;
	Attributes GetAttributesFromHandle(const Handle&) const throw (CoreException);
	void Redraw() const;
	void SetHighlight(const Pick&,bool hl) const;
	void SetAllHighlights(bool hl) const;
	void SetVisible(const Pick&, bool v) const;
	void SetAllVisible(bool v) const;
	void SwapVisible() const;
	

	void ClearTypeMask();
	void ClearModifier();
        void ClearFunctionMap();
        void CancelCanvasAction();
	void SetTypeMask(const SelectionFilter&);
	SelectionFilter GetTypeMask() const;
	void RestoreTypeMask();

// external informatives 
	bool IsActive() const { return (commandStack.size()>0); }

	int Get1Pick(Pick&) throw (ComException); // returns the mouse button, loads the pick into the supplied reference
	void PushPick(const Pick&,int); // push a pick and button back onto the stack

	void ToastDeleteStack();
	void EmptyDeleteStack();

	void Say(const int&);
	void Say(const char*);

public: // calls for the dialogs
	void LoadColor(int value);
	void LoadLineStyle(unsigned short style);
	void LoadThick(unsigned int   thick);
	void LoadPointSize(unsigned int   size);
	

protected: // calls for the children - access to data members of the command window
	int CurrentColor() const ;
	unsigned short CurrentLineStyle() const;
	unsigned int CurrentThick() const;
	unsigned int CurrentPointSize() const;
	unsigned int CurrentLayer() const;
	string CurrentFileName() const;
	const DrawableEntity* Drawable(const int&) const throw(DrawableException);
	void CanvasRegisterInit() const;
	bool CanvasRegisterNextItem(RegisteredEntity&) const;

private:
// initialize the color change function
	void InitializeColor();
	void InitializeLineStyle();
	
// process a Pick and change the color of the entity
	void ChangeAttribute();

	ColorDialog* colorDialog;
	LineStyleDialog* lineStyleDialog;
	
protected:
	Point KeyboardPoint() const throw(ComException); // get a keyboard entered point 
	double KeyboardValue(const char* message) const throw(ComException); // get a keyboard entered real number
};

#endif
