// namehandler.h
// 
// copyright cliff johnson - 1998  
//
// This is free software. It is licensed under the terms of 
// the GNU GENERAL PUBLIC LICENSE. Read the file COPYING.
//



// 
// The handle class is designed to have a static pointer to a namehandler object. 
// The function the namehandler is to provide automatic naming services
// for the creation of handles. 
// 
// The NameHandler also provides services to the public at large - it contains a list 
// of all the created names handles plus those that are manually assigned can be
// recorded.  This mechanism is to insure that duplicate names are not issued, 
// and not allowed to be assigned within a model. 
// 
// The automatic names consist of a string prefix, 
// and a counter unique to each entity type.
// 
// For example, a possible name might be %P142 indicating the 142'nd instance of a 
// point in a local context. 
// 
// The local context flag, '%', indicates that the entity name may be compacted,
// usually at save time, so as to have sequential numbering of entities. 
// 
// a name without this flag, ie. P142, is treated differently in that, when saved, 
// the name will be save as is, and that when recovered, the same name will be
// preserved - if the name is already in use, it will be renamed according to the
// renaming rules.
// 
// All entities are initially constructed in local context.

// Other things that remain to be considered / implemented:  Allowing the user to change the prefixes, 
// Allowing user changing counter values, Dynamic adding of types, deletion of names
// from the list - currently this is not provided for... perhaps just
// have a way to parse the bank, and re-read all the names in use... name compaction .. etc. 


#ifndef NAMEHANDLER_H
#define NAMEHANDLER_H

#include <string>
#include <map>
#include <set>

class Handle;


class NameHandler
{

private:

	string unknown;
	unsigned int unknown_nextID;
	unsigned int mangle_nextID;

	map < int , unsigned int , less< int > > nextID ;       // each entity type has next ID counter
	map < int , string , less< int > > prefixes;   // each entity type has a prefix for naming
	set< string > names;			       // list of already assigned names
	

public:
	
	NameHandler();
	
	
	string AutoAssignName(Handle&);         // assigns the next available name to the handle
						// and returns it as a string. If the type is 
						// recognized by the NameHandler ( and all should
						// be ) it increments the appropriate counter. 
						// if the type is not recognized, '?'
						// is used as a prefix and the unknown counter
						// is updated. Also updates the global name list. 
						// 
						
						
	string AssignName(Handle& h, const string& name);
						// Attempts to assign 'name' to 'h'
						// returns string which is what actually
						// _was_ assigned. Hopefully the returned
						// string is the same as the supplied, but
						// if the supplied is already in the list, 
						// then the function will mangle it, in a 
						// friendly sort of way, and return the 
						// mangled string. It also updates the 
						// current list. 

	void SaveName(Handle&);			// This one simply removes the local context
						// flag from a handle. It also updates the 
						// current list. If the first character of the name
						// is not a local context flag, it does nothing. 
	
	bool NameExists(const string&) const;	// user function to check for existing names.
	
	string MangleName(const string&);       // friendly mangling of duplicate names
	
};

#endif
