/* $Id: getaxfr.c,v 0.7 1998/09/27 05:18:15 gaius Exp $ */

#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <arpa/nameser.h>
#include <arpa/inet.h>
#include <resolv.h>
#include <netdb.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <zlib.h>

#define PRF_DEF 0x2ff9
#define PRF_MIN 0xA930
#define PRF_ZONE 0x24f9
#define	packet  (packet_.packet_)
#define ZONE_PREFIX "zone_"
#define HOSTS_PREFIX "hosts_"
#define HINFO_PREFIX "hinfo_"
#define GZ_SUFFIX "gz"

#include "error.h"
#include "getaxfr.h"
#include "res.h"
union
{
  HEADER header_;
  u_char packet_[PACKETSZ];
}
packet_;

typedef union
{
  HEADER qb1;
  u_char qb2[PACKETSZ];
}
querybuf;

extern int debug;
extern int dont_recurse;
extern char *directory;
extern char *hosts_d;
extern char *zone_d;

void 
axfr (struct domain_s *domain);

const u_char *
srvy_p_rr (struct domain_s *domain, const u_char * cp, u_char * msg);

struct domain_s *new_domain();
void delete_domain(struct domain_s *domain);
inline void all_lowercase(char *string);

static const char *deproto (int protonum);

static const char *dewks (int wks);

static const u_char *
sp_fqname(const u_char *cp, const u_char *msg, FILE *file);


static const u_char *
sp_fqnname(const u_char *cp, const u_char *msg, int msglen, 
char *name, int namelen);

static int
printZone (struct domain_s *domain, struct sockaddr_in *sin);

static int
printRR (struct domain_s *domain, u_char * msg, u_char * eom);


static const u_char *
gzp_fqname(const u_char *cp, const u_char *msg, gzFile file)
{
        char name[MAXDNAME];
        const u_char *n;

        n = __p_fqnname(cp, msg, MAXCDNAME, name, sizeof name);
        if (n == NULL)
                return (NULL);
        gzputs(file, name);
        return (n);
}

void getaxfrbydomain (struct domain_s *domain)
{ 
  int index;
  char *parent;
  char *pathname;
  if (debug)
    (void) fprintf (stderr, "Doing an authoritative zone transfer for domain %s\n", domain->name);
  
  /* Set parent name */
  
  if (strcmp (".", domain->name) > 0)
  {
    char *parent = (strchr (domain->name, '.') + sizeof (char));
    domain->parent = (char *) malloc (strlen (parent) + 1);
    if (!domain->parent)
      fatal ("malloc(domain->parent)");

    (void) strcpy (domain->parent, parent);

    if (debug)
      (void) fprintf (stderr, "Parent domain is %s \n", domain->parent);
  }

  /* set dns list structure */
    setnslistfordomain (domain);
    fflush(stderr);
    fflush(stdout);
if (domain->nscount) {
    (void) fprintf (stderr, "Found %d name server%s for domain \'%s':\n", domain->nscount, domain->nscount > 1 ? "s" : "", domain->name);

    for (index = 0; index < domain->nscount; index++)
    {
      (void) fprintf (stderr, "%s\n", inet_ntoa (*domain->nsaddr_list[index]));
    }
   
  }
  
  if ((pathname = (char *) alloca(strlen(domain->name)+1)) == NULL)
    fatal("allocating pathname %s", domain->name);
  
  strcpy(pathname, domain->name);
   pathname[strlen(pathname)-1] = 0;

  mkdir(pathname, DIR_PERM);
    if (chdir(pathname))
      fatal("Can't change dir to %s", pathname);

  axfr (domain);
  if (!(dont_recurse) && domain->subdomaincount)
  {
    int index = domain->subdomaincount;
    while (index--)
	{
	getaxfrbydomain(domain->subdomains[index]);
    }
  }
  (void) chdir("..");
}


inline void all_lowercase(char *string)
{
  while (*string)
  {
    *string = (char) tolower((int) *string);
    string++;
  }
}

void mk_base_dir(char *domainname)
{
  char *parent;
  if ((parent = strchr (domainname, '.')) != NULL)
  {
    parent += sizeof (char);
    mk_base_dir(parent);
  mkdir(parent, DIR_PERM);
    if (chdir(parent))
      fatal("Can't change dir to %s", parent);
  }
}


inline struct domain_s 
*new_domain()
{
  struct domain_s *domain;

  domain = (struct domain_s *) malloc (sizeof (struct domain_s));
  if (!domain)
    fatal ("memory allocation domain structure");
  domain->name = NULL;
  domain->parent = NULL;
  domain->subdomaincount = 0;
  domain->subdomains = NULL;
  domain->nscount = 0;
  domain->nsaddr_list = 0;
  return(domain);
}

inline void 
delete_domain(struct domain_s *domain)
{
  if (debug) (void) fprintf(stderr, "Free structure %s\n", domain->name);
  free (domain->name);
  free (domain->parent);
  while (domain->subdomaincount)
  {
    domain->subdomaincount--;
    delete_domain (domain->subdomains[domain->subdomaincount]);
    free (domain->subdomains[domain->subdomaincount]);
  }
  free (domain->subdomains);
  
  while (domain->nscount)
  {
    free (domain->nsaddr_list[--domain->nscount]);
  }
  free (domain->nsaddr_list);
}



const u_char *
short_prr (struct domain_s *domain, const u_char * cp, const u_char * msg)
{
  int type, class, dlen;
  int skip;
  struct in_addr inaddr;
  const u_char *cp1;
  u_int32_t tmpttl, t;
  char rrname[MAXDNAME];	/* The fqdn of this RR */

  if ((_res.options & RES_INIT) == 0 && res_init () == -1)
  {
    h_errno = NETDB_INTERNAL;
    return (NULL);
  }
  cp = __p_fqnname (cp, msg, MAXCDNAME, rrname, sizeof rrname);
  if (!cp)
    return (NULL);		/* compression error */
  all_lowercase(rrname);
  if (debug) fputs (rrname, stderr);

  type = _getshort ((u_char *) cp);
  cp += INT16SZ;
  class = _getshort ((u_char *) cp);
  cp += INT16SZ;
  tmpttl = _getlong ((u_char *) cp);
  cp += INT32SZ;
  dlen = _getshort ((u_char *) cp);
  cp += INT16SZ;
  cp1 = cp;
  if (debug)
    (void) fprintf (stderr, "\t%s", __p_type (type));
  /*
   * Print type specific data, if appropriate
   */
  switch (type)
  {
  case T_A:
    switch (class)
    {
    case C_IN:
    case C_HS:
      memcpy (&inaddr, cp, INADDRSZ);
      if (dlen == 4)
      {
	skip = 0;
	if (domain->nscount)
	{
	  int index = domain->nscount;
	  while (index--)
	  {
	    if (memcmp (domain->nsaddr_list[index], &inaddr, sizeof (struct in_addr)) == 0)
	    {
	      if (debug)
            (void) fprintf (stderr, "\t%s", inet_ntoa (inaddr));
	      skip = 1;
	      break;
	    }
	  }
	}
	if (!skip)
	{
	  if (domain->nscount)
	    domain->nsaddr_list = realloc (domain->nsaddr_list, (sizeof (struct in_addr *) * (domain->nscount + 1)));	/*XXX check */
	  else
	    domain->nsaddr_list = malloc (sizeof (struct in_addr *));	/*XXX check */
	  domain->nsaddr_list[domain->nscount] = (struct in_addr *) malloc (sizeof (struct in_addr));

	  memcpy (domain->nsaddr_list[domain->nscount], (const void *) &inaddr, sizeof (struct in_addr));
	  if (debug)
        (void) fprintf (stderr, "\t%s", inet_ntoa (*domain->nsaddr_list[domain->nscount]));
	  domain->nscount++;
	}

	cp += dlen;
      }
      else if (dlen == 7)
      {
	char *address;
	u_char protocol;
	u_short port;

	address = inet_ntoa (inaddr);
	cp += INADDRSZ;
	protocol = *(u_char *) cp;
	cp += sizeof (u_char);
	port = _getshort ((u_char *) cp);
	cp += INT16SZ;
	if (debug)
      (void) fprintf (stderr, "\t%s\t; proto %d, port %d",
		 address, protocol, port);
      }
      break;
    default:
      cp += dlen;
    }
    break;
  case T_NS:
     if (debug)  
      if (debug)  putc ('\t', stderr);
    if ((cp = sp_fqname (cp, msg, stderr)) == NULL)
      return (NULL);
    break;

  case T_SOA:
       if (debug)  putc ('\t', stderr);
    if ((cp = sp_fqname (cp, msg, stderr)) == NULL)
      return (NULL);
    if (debug)  putc (' ', stderr);
    if ((cp = sp_fqname (cp, msg, stderr)) == NULL)
      return (NULL);
    if (debug) fputs (" (\n", stderr);
    t = _getlong ((u_char *) cp);
    cp += INT32SZ;
    if (debug)
      (void) fprintf (stderr, "\t\t\t%lu\t; serial\n", (u_long) t);
    t = _getlong ((u_char *) cp);
    cp += INT32SZ;
    if (debug)
      (void) fprintf (stderr, "\t\t\t%lu\t; refresh (%s)\n",
	     (u_long) t, __p_time (t));
    t = _getlong ((u_char *) cp);
    cp += INT32SZ;
    if (debug)
      (void) fprintf (stderr, "\t\t\t%lu\t; retry (%s)\n",
	     (u_long) t, __p_time (t));
    t = _getlong ((u_char *) cp);
    cp += INT32SZ;
    if (debug)
      (void) fprintf (stderr, "\t\t\t%lu\t; expire (%s)\n",
	     (u_long) t, __p_time (t));
    t = _getlong ((u_char *) cp);
    cp += INT32SZ;
        if (debug)
      (void) fprintf (stderr, "\t\t\t%lu )\t; minimum (%s)",
	     (u_long) t, __p_time (t));
    break;


  default:
    if (debug)
      (void) fprintf (stderr, "\t?%d?", type);
    cp += dlen;
  }
#if 0
  (void) fprintf (stderr, "\t; dlen=%d, ttl %s\n", dlen, __p_time (tmpttl));
#else
  if (debug)  putc ('\n', stderr);
#endif
  if (cp - cp1 != dlen)
  {
    (void) fprintf (stderr, ";; packet size error (found %d, dlen was %d)\n",
	     cp - cp1, dlen);
    cp = NULL;
  }
  return (cp);
}

const u_char *
do_rrset (struct domain_s *domain, const u_char * msg, int len, const u_char * cp, int cnt, int pflag)
{
  int n;
  int sflag;

  /*
   * Print answer records.
   */
  sflag = (_res.pfcode & pflag);
  if ((n = ntohs (cnt)))
  {
    while (--n >= 0)
    {
      if ((!_res.pfcode) || sflag)
      {
	cp = short_prr (domain, cp, msg);
      }
      else
      {
	unsigned int dlen;
	cp += __dn_skipname (cp, cp + MAXCDNAME);
	cp += INT16SZ;
	cp += INT16SZ;
	cp += INT32SZ;
	dlen = _getshort ((u_char *) cp);
	cp += INT16SZ;
	cp += dlen;
      }
      if ((cp - msg) > len)
	return (NULL);
    }
  }
  return (cp);
}


void 
setnslistfordomain (struct domain_s *domain)
{
  u_char answer[8 * 1024];
  char msg[120];
  register const u_char *cp = NULL, *endMark;
  register const HEADER *hp;
  register int count;
  char *srvmsg;
  int index, length;
  struct servent *domain_entry;
  domain_entry = getservbyname ("domain", "udp");

  for (index = 0; index < _res.nscount; index++)
  {
    _res.nsaddr_list[index].sin_family = AF_INET;
    _res.nsaddr_list[index].sin_port = domain_entry->s_port;
  }
  _res.id += _res.retry;
  _res.pfcode = PRF_MIN;
  _res.options &= RES_DEBUG;
  length = res_mkquery (QUERY, domain->name, C_IN, T_SOA, NULL, 0, NULL, packet, sizeof (packet));
  if (length < 0)
  {
    fflush (stderr);
    fatal ("res_mkquery: buffer too small\n\n");
  }
  if ((length = res_send (packet, length, answer, sizeof (answer))) < 0)
  {
    fflush (stdout);
    length = 0 - length;
    msg[0] = 0;
    return;
  }
  hp = (HEADER *) answer;
  cp = answer + HFIXEDSZ;
  endMark = answer + PACKETSZ;

  if ((count = ntohs (hp->qdcount)))
  {
    while (--count >= 0)
    {
      int length;
      char name[MAXDNAME];

      if ((length = dn_expand (answer, answer + PACKETSZ, cp, name, sizeof name)) < 0)
	cp = NULL;
      else
	cp += length;
      cp += INT16SZ;
      cp += INT16SZ;
    }
  }
  cp = do_rrset (domain, answer, PACKETSZ, cp, hp->ancount, RES_PRF_ANS);
  cp = do_rrset (domain, answer, PACKETSZ, cp, hp->nscount, RES_PRF_AUTH);

  cp = do_rrset (domain, answer, PACKETSZ, cp, hp->arcount, RES_PRF_ADD);
}

struct domain_s
*getaxfrbydomainname (char *domainname, struct in_addr *nsaddr_list[])
{
  int index;
  char *pathname; 
  char **token;
  char *tokc;
  struct domain_s *domain;
  domain = new_domain();
 /* set domain name */
 all_lowercase(domainname);
 domain->name = (char *) malloc (strlen (domainname) + 1);
 if (!domain->name)
   fatal ("alloca(domain)");
   (void) strcpy (domain->name, domainname);

 if (chdir(directory)) /* print WARNING message XXX */
   fatal("Can't change dir to %s", directory);

 domainname[strlen(domainname)-1] = 0;
 mk_base_dir(domainname);
 
  
  getaxfrbydomain(domain);
  fflush (stderr);
  return domain;
}

static const u_char *
sp_fqname(const u_char *cp, const u_char *msg, FILE *file)
{
	char name[MAXDNAME];
	const u_char *n;

	n = sp_fqnname(cp, msg, MAXCDNAME, name, sizeof name);
	if (n == NULL)
		return (NULL);
	if (debug) fputs(name, file);
	return (n);
}


static const u_char *
sp_fqnname(const u_char *cp, const u_char *msg, int msglen, char *name, int namelen)
{
	int n, newlen;

	if ((n = dn_expand(msg, cp + msglen, cp, name, namelen)) < 0)
		return (NULL);
	newlen = strlen (name);
	if (newlen == 0 || name[newlen - 1] != '.')
		if (newlen+1 >= namelen)	/* Lack space for final dot */
			return (NULL);
		else
			strcpy(name + newlen, ".");
	return (cp + n);
}


void 
axfr (struct domain_s *domain)
{
  int index;
  int hosts_dir_l = 0;
  int zone_dir_l = 0;
  int hosts_l = 0;
  int zone_l = 0;
  struct sockaddr_in ns_sockaddr;
  char *pathname;
  char *oldpath;
  char *newpath;
  char *cwd;

  cwd = (char *) get_current_dir_name();
  _res.pfcode = PRF_ZONE;
  ns_sockaddr.sin_family = AF_INET;
  ns_sockaddr.sin_port = htons(53);
  
  hosts_l = strlen(HOSTS_PREFIX)+ strlen(domain->name) + strlen(GZ_SUFFIX) + 1;
  zone_l = strlen(ZONE_PREFIX) + strlen(domain->name) + strlen(GZ_SUFFIX) + 1;

  /* Open and symlink hosts file */
  pathname = (char *) malloc(hosts_l); 
  if (!pathname)                                                                                
    fatal("allocating memory for pathnames %s", domain->name);                                  
  snprintf (pathname, hosts_l, "%s%s%s", HOSTS_PREFIX, domain->name, GZ_SUFFIX); 
  if ((domain->HOSTS = gzopen(pathname, "wb9")) == NULL)                                        
    fatal("Can't create file %s", pathname);                                                    
  newpath = (char *) malloc(strlen(hosts_d) + strlen(domain->name)+ strlen(GZ_SUFFIX)+ 2);
  if (!newpath)                                                                                
    fatal("allocating memory for pathnames %s", domain->name);                                  
  sprintf (newpath, "%s/%s%s", hosts_d, domain->name, GZ_SUFFIX);
  oldpath = (char *) malloc(strlen(cwd) + strlen(pathname) + 2);                          
  if (!oldpath)                                                                                
    fatal("allocating memory for pathnames %s", domain->name);                                  
  sprintf (oldpath, "%s/%s", cwd, pathname);
  symlink(oldpath, newpath);                                                              

  /* Open and symlink zone file */
  pathname = (char *) realloc(pathname, zone_l); 
  if (!pathname)
    fatal("allocating memory for pathnames %s", domain->name);
  snprintf (pathname, zone_l, "%s%s%s",ZONE_PREFIX, domain->name, GZ_SUFFIX);
  if ((domain->ZONE = gzopen(pathname, "wb9")) == NULL)
    fatal("Can't create file %s", pathname);
  newpath = (char *) realloc(newpath, strlen(zone_d) + strlen(domain->name) +  strlen(GZ_SUFFIX) + 2 );
  if (!newpath)
    fatal("allocating memory for pathnames %s", domain->name);
  sprintf (newpath, "%s/%s%s", zone_d, domain->name, GZ_SUFFIX);
  oldpath = (char *) realloc(oldpath, strlen(cwd) + strlen(pathname) + 2);
  if (!oldpath)
    fatal("allocating memory for pathnames %s", domain->name);
  sprintf (oldpath, "%s/%s", cwd, pathname); 
  symlink(oldpath, newpath);  
  free(pathname);
  free(oldpath);
  free(newpath);
   
  for (index = 0; index < domain->nscount; index++)
  {
    int x;
    ns_sockaddr.sin_addr = *domain->nsaddr_list[index];
    x = printZone (domain, &ns_sockaddr);
    if (!x)
      break;			/* success */
  }

  gzclose(domain->HOSTS);
  gzclose(domain->ZONE);
  
}

static int
printZone (struct domain_s *domain, struct sockaddr_in *sin)
{
  querybuf buf;
  int msglen;
  int amtToRead;
  int numRead;
  int numAnswers = 0;
  int numRecords = 0;
  int result;
  int soacnt = 0;
  int sockFD;
  int count, type, class, rlen, done, n;
  u_short len;
  u_char *cp;
  char dname[2][256];
  static u_char *answer = NULL;
  static int answerLen = 0;
  enum
  {
    NO_ERRORS,
    ERR_READING_LEN,
    ERR_READING_MSG,
    ERR_PRINTING
  }
  error = NO_ERRORS;

  /*
   *  Create a query packet for the requested zone name.
   */
  msglen = res_mkquery (QUERY, domain->name, C_IN, T_AXFR, NULL,
			0, 0, buf.qb2, sizeof (buf));
  if (msglen < 0)
  {
    if (_res.options & RES_DEBUG)
    {
      (void) fprintf (stderr, ";; res_mkquery failed\n");
    }
    return (-3);
  }

  /*
   *  Set up a virtual circuit to the server.
   */
  if ((sockFD = socket (sin->sin_family, SOCK_STREAM, 0)) < 0)
  {
    int e = errno;
    perror (";; socket");
    return (e);
  }
  if (connect (sockFD, (struct sockaddr *) sin, sizeof (*sin)) < 0)
  {
    int e = errno;
    perror (";; connect");
    (void) close (sockFD);
    sockFD = -1;
    return e;
  }

  /*
   * Send length & message for zone transfer
   */

  __putshort (msglen, (u_char *) & len);

  if (write (sockFD, (char *) &len, INT16SZ) != INT16SZ ||
      write (sockFD, (char *) &buf, msglen) != msglen)
  {
    int e = errno;
    perror (";; write");
    (void) close (sockFD);
    sockFD = -1;
    return (e);
  }

  dname[0][0] = '\0';
  for (done = 0; !done; NULL)
  {
    u_int16_t tmp;

    /*
     * Read the length of the response.
     */

    cp = (u_char *) & tmp;
    amtToRead = INT16SZ;
    while (amtToRead > 0 && (numRead = read (sockFD, cp, amtToRead)) > 0)
    {
      cp += numRead;
      amtToRead -= numRead;
    }
    if (numRead <= 0)
    {
      error = ERR_READING_LEN;
      break;
    }

    if ((len = _getshort ((u_char *) & tmp)) == 0)
    {
      break;			/* nothing left to read */
    }

    /* The server sent too much data to fit in buffer */
    if (len > (u_int) answerLen)
    {
      if (answerLen != 0)
      {
	free (answer);
      }
      answerLen = len;
      answer = (u_char *) malloc (answerLen);
    }

    /*
     * Read the response.
     */

    amtToRead = len;
    cp = answer;
    while (amtToRead > 0 && (numRead = read (sockFD, cp, amtToRead)) > 0)
    {
      cp += numRead;
      amtToRead -= numRead;
    }
    if (numRead <= 0)
    {
      error = ERR_READING_MSG;
      break;
    }

    result = printRR (domain, answer, cp);
    if (result != 0)
    {
      error = ERR_PRINTING;
      break;
    }
    numRecords += htons (((HEADER *) answer)->ancount);
    numAnswers++;

    /* Header. */
    cp = answer + HFIXEDSZ;
    /* Question. */
    for (count = ntohs (((HEADER *) answer)->qdcount);
	 count > 0;
	 count--)
      cp += dn_skipname (cp, answer + len) + QFIXEDSZ;
    /* Answer. */
    for (count = ntohs (((HEADER *) answer)->ancount);
	 count > 0;
	 count--)
    {
      n = dn_expand (answer, answer + len, cp,
		     dname[soacnt], sizeof dname[0]);
      if (n < 0)
      {
	error = ERR_PRINTING;
	done++;
	break;
      }
      cp += n;
      GETSHORT (type, cp);
      GETSHORT (class, cp);
      cp += INT32SZ;		/* ttl */
      GETSHORT (rlen, cp);
      cp += rlen;
      if (type == T_SOA && soacnt++ &&
	  !strcasecmp (dname[0], dname[1]))
      {
	done++;
	break;
      }
    }
  }

  printf ("Received %d answer%s (%d record%s).\n",
	  numAnswers, (numAnswers != 1) ? "s" : "",
	  numRecords, (numRecords != 1) ? "s" : "");

  (void) close (sockFD);
  sockFD = -1;

  switch (error)
  {
  case NO_ERRORS:
    return (0);

  case ERR_READING_LEN:
    return (EMSGSIZE);

  case ERR_PRINTING:
    return (result);

  case ERR_READING_MSG:
    return (EMSGSIZE);

  default:
    return (EFAULT);
  }
}
static int
printRR (struct domain_s *domain, u_char * msg, u_char * eom)
{
  register u_char *cp;
  HEADER *headerPtr;
  int nameLen;
  int n;

  /*
   * Read the header fields.
   */
  headerPtr = (HEADER *) msg;
  cp = msg + HFIXEDSZ;
  if (headerPtr->rcode != NOERROR)
  {
    return (headerPtr->rcode);
  }

  /*
   *  We are looking for info from answer resource records.
   *  If there aren't any, return with an error. We assume
   *  there aren't any question records.
   */

  if (ntohs (headerPtr->ancount) == 0)
  {
    return (NO_INFO);
  }
  for (n = ntohs (headerPtr->qdcount); n > 0; n--)
  {
    nameLen = dn_skipname (cp, eom);
    if (nameLen < 0)
      return (ERROR);
    cp += nameLen + QFIXEDSZ;
  }
#ifdef PROTOCOLDEBUG
  printf (";;; (message of %d octets has %d answers)\n",
	  eom - msg, ntohs (headerPtr->ancount));
#endif
  for (n = ntohs (headerPtr->ancount); n > 0; n--)
    cp = (u_char *) srvy_p_rr (domain, cp, msg);
  return (SUCCESS);
}

const u_char *
srvy_p_rr (struct domain_s *domain, const u_char * cp, u_char * msg)
{
  int type, class, dlen, n, c, skip;
  struct in_addr inaddr;
  const u_char *cp1, *cp2;
  u_int32_t tmpttl, t;
  int lcnt;
  u_int16_t keyflags;
  char rrname[MAXDNAME];	/* The fqdn of this RR */
  char base64_key[MAX_KEY_BASE64];

  if ((_res.options & RES_INIT) == 0 && res_init () == -1)
  {
    h_errno = NETDB_INTERNAL;
    return (NULL);
  }
  cp = __p_fqnname (cp, msg, MAXCDNAME, rrname, sizeof rrname);
  if (!cp)
    return (NULL);		/* compression error */
  all_lowercase(rrname);
  gzputs (domain->ZONE, rrname);
  type = _getshort ((u_char *) cp);
  cp += INT16SZ;
  class = _getshort ((u_char *) cp);
  cp += INT16SZ;
  tmpttl = _getlong ((u_char *) cp);
  cp += INT32SZ;
  dlen = _getshort ((u_char *) cp);
  cp += INT16SZ;
  cp1 = cp;
  if ((!_res.pfcode) || (_res.pfcode & RES_PRF_TTLID))
    gzprintf (domain->ZONE, "\t%lu", (u_long) tmpttl);
  if ((!_res.pfcode) || (_res.pfcode & RES_PRF_CLASS))
    gzprintf (domain->ZONE, "\t%s", __p_class (class));
  gzprintf (domain->ZONE, "\t%s", __p_type (type));
  /*
   * Print type specific data, if appropriate
   */
  switch (type)
  {
  case T_A:
    switch (class)
    {
    case C_IN:
    case C_HS:
      bcopy (cp, (char *) &inaddr, INADDRSZ);
      if (dlen == 4)
      {
	gzprintf (domain->ZONE, "\t%s", inet_ntoa (inaddr));
	gzprintf (domain->HOSTS, "%s\t%s\n", inet_ntoa(inaddr), rrname);
	cp += dlen;
      }
      else if (dlen == 7)
      {
	char *address;
	u_char protocol;
	u_short port;

	address = inet_ntoa (inaddr);
	cp += INADDRSZ;
	protocol = *(u_char *) cp;
	cp += sizeof (u_char);
	port = _getshort ((u_char *) cp);
	cp += INT16SZ;
	gzprintf (domain->ZONE, "\t%s\t; proto %d, port %d",
		 address, protocol, port);
      }
      break;
    default:
      cp += dlen;
    }
    break;
  case T_CNAME:
  case T_MB:
  case T_MG:
  case T_MR:
  case T_PTR:
    gzputc (domain->ZONE, '\t');
    if ((cp = gzp_fqname (cp, msg, domain->ZONE)) == NULL)
      return (NULL);
    break;
  case T_NS:
    gzputc (domain->ZONE, '\t');
    if ((cp = gzp_fqname (cp, msg, domain->ZONE)) == NULL)
      return (NULL);
    if (strcmp(domain->name, rrname) != 0)
     {
      skip = 0;
      if (domain->subdomaincount)
	  {
	    int index = domain->subdomaincount;
	    while (index--)
	    {
	      if (strcmp ((domain->subdomains[index])->name, rrname) == 0)
	      {
	        skip = 1;
	        break;
	      }
	    }
      }
      if (!skip)
      {
	    if (domain->subdomaincount)
	      domain->subdomains = realloc (domain->subdomains, (sizeof (struct domain_s *) * (domain->subdomaincount + 1)));	/*XXX check */
	  else
	    domain->subdomains = malloc (sizeof (struct domain_s *));	/*XXX check */
	    
	  domain->subdomains[domain->subdomaincount] = new_domain();
      
       (domain->subdomains[domain->subdomaincount])->name = (char *) malloc (strlen (rrname) + 1);
  if (!domain->subdomains[domain->subdomaincount]->name)
    fatal ("malloc() for %s->name", rrname);

	  strcpy ((domain->subdomains[domain->subdomaincount])->name, rrname);
	  if (debug)
        gzprintf (stderr, "Subdomain %s added.\n", (domain->subdomains[domain->subdomaincount])->name);
	  domain->subdomaincount++;
	  }
      
    }

    break;
  case T_HINFO:
  case T_ISDN:
    cp2 = cp + dlen;
    (void) gzputs (domain->ZONE, "\t\"");
    if ((n = (unsigned char) *cp++) != 0)
    {
      for (c = n; c > 0 && cp < cp2; c--)
      {
	if (strchr ("\n\"\\", *cp))
	  (void) gzputc (domain->ZONE,'\\');
	(void) gzputc (domain->ZONE, *cp++);
      }
    }
    gzputc (domain->ZONE, '"');
    if (cp < cp2 && (n = (unsigned char) *cp++) != 0)
    {
      (void) gzputs (domain->ZONE, "\t\"");
      for (c = n; c > 0 && cp < cp2; c--)
      {
	if (strchr ("\n\"\\", *cp))
	  (void) gzputc (domain->ZONE, '\\');
	(void) gzputc (domain->ZONE, *cp++);
      }
      gzputc (domain->ZONE, '"');
    }
    else if (type == T_HINFO)
    {
      (void) gzputs (domain->ZONE, "\"?\"\n");
    }
    break;

  case T_SOA:
    gzputc (domain->ZONE, '\t');
    if ((cp = gzp_fqname (cp, msg, domain->ZONE)) == NULL)
      return (NULL);
    gzputc (domain->ZONE, ' ');
    if ((cp = gzp_fqname (cp, msg, domain->ZONE)) == NULL)
      return (NULL);
    gzputs (domain->ZONE, " (\n");
    t = _getlong ((u_char *) cp);
    cp += INT32SZ;
    gzprintf (domain->ZONE, "\t\t\t%lu\t; serial\n", (u_long) t);
    t = _getlong ((u_char *) cp);
    cp += INT32SZ;
    gzprintf (domain->ZONE, "\t\t\t%lu\t; refresh (%s)\n",
	     (u_long) t, __p_time (t));
    t = _getlong ((u_char *) cp);
    cp += INT32SZ;
    gzprintf (domain->ZONE, "\t\t\t%lu\t; retry (%s)\n",
	     (u_long) t, __p_time (t));
    t = _getlong ((u_char *) cp);
    cp += INT32SZ;
    gzprintf (domain->ZONE, "\t\t\t%lu\t; expire (%s)\n",
	     (u_long) t, __p_time (t));
    t = _getlong ((u_char *) cp);
    cp += INT32SZ;
    gzprintf (domain->ZONE, "\t\t\t%lu )\t; minimum (%s)",
	     (u_long) t, __p_time (t));
    break;

  case T_MX:
  case T_AFSDB:
  case T_RT:
    gzprintf (domain->ZONE, "\t%d ", _getshort ((u_char *) cp));
    cp += INT16SZ;
    if ((cp = gzp_fqname (cp, msg, domain->ZONE)) == NULL)
      return (NULL);
    break;

  case T_PX:
    gzprintf (domain->ZONE, "\t%d ", _getshort ((u_char *) cp));
    cp += INT16SZ;
    if ((cp = gzp_fqname (cp, msg, domain->ZONE)) == NULL)
      return (NULL);
    gzputc (domain->ZONE, ' ');
    if ((cp = gzp_fqname (cp, msg, domain->ZONE)) == NULL)
      return (NULL);
    break;

  case T_X25:
    cp2 = cp + dlen;
    (void) gzputs (domain->ZONE, "\t\"");
    if ((n = (unsigned char) *cp++) != 0)
    {
      for (c = n; c > 0 && cp < cp2; c--)
      {
	if (strchr ("\n\"\\", *cp))
	  (void) gzputc (domain->ZONE, '\\');
	(void) gzputc (domain->ZONE, *cp++);
      }
    }
    gzputc (domain->ZONE, '"');
    break;

  case T_TXT:
    (void) gzputc (domain->ZONE, '\t');
    cp2 = cp1 + dlen;
    while (cp < cp2)
    {
      gzputc (domain->ZONE, '"');
      if ((n = (unsigned char) *cp++))
      {
	for (c = n; c > 0 && cp < cp2; c--)
	{
	  if (strchr ("\n\"\\", *cp))
	    (void) gzputc (domain->ZONE, '\\');
	  (void) gzputc (domain->ZONE, *cp++);
	}
      }
      gzputc (domain->ZONE, '"');
      if (cp < cp2)
	gzputc (domain->ZONE, ' ');
    }
    break;

  case T_NSAP:
    (void) gzprintf (domain->ZONE, "\t%s", inet_nsap_ntoa (dlen, cp, NULL));
    cp += dlen;
    break;


  case T_LOC:
    {
      char t[255];

      gzprintf (domain->ZONE, "\t%s", loc_ntoa (cp, t));
      cp += dlen;
      break;
    }

  case T_NAPTR:
    {
      u_int order, preference;

      order = _getshort (cp);
      cp += INT16SZ;
      preference = _getshort (cp);
      cp += INT16SZ;
      gzprintf (domain->ZONE, "\t%u %u ", order, preference);
      /* Flags */
      n = *cp++;
      gzprintf (domain->ZONE, "\"%.*s\" ", (int) n, cp);
      cp += n;
      /* Service */
      n = *cp++;
      gzprintf (domain->ZONE, "\"%.*s\" ", (int) n, cp);
      cp += n;
      /* Regexp */
      n = *cp++;
      gzprintf (domain->ZONE, "\"%.*s\" ", (int) n, cp);
      cp += n;
      if ((cp = gzp_fqname (cp, msg, domain->ZONE)) == NULL)
	return (NULL);
      break;
    }

  case T_SRV:
    {
      u_int priority, weight, port;

      priority = _getshort (cp);
      cp += INT16SZ;
      weight = _getshort (cp);
      cp += INT16SZ;
      port = _getshort (cp);
      cp += INT16SZ;
      gzprintf (domain->ZONE, "\t%u %u %u ", priority, weight, port);
      if ((cp = gzp_fqname (cp, msg, domain->ZONE)) == NULL)
	return (NULL);
      break;
    }

  case T_MINFO:
  case T_RP:
    gzputc (domain->ZONE, '\t');
    if ((cp = gzp_fqname (cp, msg, domain->ZONE)) == NULL)
      return (NULL);
    gzputc (domain->ZONE, ' ');
    if ((cp = gzp_fqname (cp, msg, domain->ZONE)) == NULL)
      return (NULL);
    break;

  case T_UINFO:
    gzputc (domain->ZONE, '\t');
    gzputs (domain->ZONE, (char *) cp);
    cp += dlen;
    break;

  case T_UID:
  case T_GID:
    if (dlen == 4)
    {
      gzprintf (domain->ZONE, "\t%u", _getlong ((u_char *) cp));
      cp += INT32SZ;
    }
    break;

  case T_WKS:
    if (dlen < INT32SZ + 1)
      break;
    bcopy (cp, (char *) &inaddr, INADDRSZ);
    cp += INT32SZ;
    gzprintf (domain->ZONE, "\t%s %s ( ",
	     inet_ntoa (inaddr),
	     deproto ((int) *cp));
    cp += sizeof (u_char);
    n = 0;
    lcnt = 0;
    while (cp < cp1 + dlen)
    {
      c = *cp++;
      do
      {
	if (c & 0200)
	{
	  if (lcnt == 0)
	  {
	    gzputs (domain->ZONE, "\n\t\t\t");
	    lcnt = 5;
	  }
	  gzputs (domain->ZONE, dewks (n));
	  gzputc (domain->ZONE, ' ');
	  lcnt--;
	}
	c <<= 1;
      }
      while (++n & 07);
    }
    gzputc (domain->ZONE, ')');
    break;

  case T_KEY:
    gzputc (domain->ZONE, '\t');
    keyflags = _getshort (cp);
    cp += 2;
    gzprintf (domain->ZONE, "0x%04x", keyflags);		/* flags */
    gzprintf (domain->ZONE, " %u", *cp++);	/* protocol */
    gzprintf (domain->ZONE, " %u (", *cp++);	/* algorithm */

    n = b64_ntop (cp, (cp1 + dlen) - cp,
		  base64_key, sizeof base64_key);
    for (c = 0; c < n; ++c)
    {
      if (0 == (c & 0x3F))
	gzprintf (domain->ZONE, "\n\t");
      gzputc (domain->ZONE,base64_key[c]);	/* public key data */
    }

    gzprintf (domain->ZONE, " )");
    if (n < 0)
      gzprintf (domain->ZONE, "\t; BAD BASE64");
    fflush (domain->ZONE);
    cp = cp1 + dlen;
    break;

  case T_SIG:
    type = _getshort ((u_char *) cp);
    cp += INT16SZ;
    gzprintf (domain->ZONE, " %s", p_type (type));
    gzprintf (domain->ZONE, "\t%d", *cp++);	/* algorithm */
    /* Check label value and print error if wrong. */
    n = *cp++;
    c = dn_count_labels (rrname);
    if (n != c)
      gzprintf (domain->ZONE, "\t; LABELS WRONG (%d should be %d)\n\t",
	       n, c);
    /* orig ttl */
    n = _getlong ((u_char *) cp);
    if (n != tmpttl)
      gzprintf (domain->ZONE, " %u", n);
    cp += INT32SZ;
    /* sig expire */
    gzprintf (domain->ZONE, " (\n\t%s",
	     __p_secstodate (_getlong ((u_char *) cp)));
    cp += INT32SZ;
    /* time signed */
    gzprintf (domain->ZONE, " %s", __p_secstodate (_getlong ((u_char *) cp)));
    cp += INT32SZ;
    /* sig footprint */
    gzprintf (domain->ZONE, " %u ", _getshort ((u_char *) cp));
    cp += INT16SZ;
    /* signer's name */
    cp = gzp_fqname (cp, msg, domain->ZONE);
    n = b64_ntop (cp, (cp1 + dlen) - cp,
		  base64_key, sizeof base64_key);
    for (c = 0; c < n; c++)
    {
      if (0 == (c & 0x3F))
	gzprintf (domain->ZONE, "\n\t");
      gzputc (domain->ZONE, base64_key[c]);	/* signature */
    }
    /* Clean up... */
    gzprintf (domain->ZONE, " )");
    if (n < 0)
      gzprintf (domain->ZONE, "\t; BAD BASE64");
    fflush (domain->ZONE);
    cp = cp1 + dlen;
    break;

#ifdef ALLOW_T_UNSPEC
  case T_UNSPEC:
    {
      int NumBytes = 8;
      u_char *DataPtr;
      int i;

      if (dlen < NumBytes)
	NumBytes = dlen;
      gzprintf (domain->ZONE, "\tFirst %d bytes of hex data:",
	       NumBytes);
      for (i = 0, DataPtr = cp; i < NumBytes; i++, DataPtr++)
	gzprintf (domain->ZONE, " %x", *DataPtr);
      cp += dlen;
    }
    break;
#endif /* ALLOW_T_UNSPEC */

  default:
    gzprintf (domain->ZONE, "\t?%d?", type);
    cp += dlen;
  }
#if 0
  gzprintf (domain->ZONE, "\t; dlen=%d, ttl %s\n", dlen, __p_time (tmpttl));
#else
  gzputc (domain->ZONE, '\n');
#endif
  if (cp - cp1 != dlen)
  {
    gzprintf (domain->ZONE, ";; packet size error (found %d, dlen was %d)\n",
	     cp - cp1, dlen);
    cp = NULL;
  }
  return (cp);
}

static const char *
deproto (int protonum)
{
  struct protoent *proto_entry;
  static char nbuf[20];

  if ((proto_entry = getprotobynumber (protonum)) == NULL)
  {
    (void) sprintf (nbuf, "%d", protonum);
    return (nbuf);
  }
  else
  {
    return (proto_entry->p_name);
  }
}

static const char *
dewks (int wks)
{
  struct servent *serv_entry;
  static char nbuf[20];

  if ((serv_entry = getservbyport (wks, NULL)) == NULL)
  {
    (void) sprintf (nbuf, "%d", wks);
    return (nbuf);
  }
  else
  {
    return (serv_entry->s_name);
  }
}
