/*
 *          Copyright (c) mjh-EDV Beratung, 1996-1998
 *     mjh-EDV Beratung - 63263 Neu-Isenburg - Rosenstrasse 12
 *          Tel +49 6102 328279 - Fax +49 6102 328278
 *                Email info@mjh.teddy-net.com
 *
 *    Author: Jordan Hrycaj <jordan@mjh.teddy-net.com>
 *
 *   $Id: cbc-frame.c,v 1.21 1999/01/31 17:57:01 jordan Exp $
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License for more details.
 */

#include "common-stuff.h"

#ifdef HAVE_SYS_SOCKET_H
# include <sys/socket.h>
#else
# include <socket.h> /* ????? */
#endif

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#ifdef HAVE_ASSERT_H
#include <assert.h>
#else
#define assert(x)
#endif

#define __IOSTREAM_PRIVATE__ /* prevent from using send/recv auto-replaces */
#include "cbc-frame.h"
#include "crandom.h"
#include "messages.h"
#include "rnd-pool.h"

#if CHAR_BIT != 8
#error Not supporting this architecture: CHAR_BIT != 8
#endif

#ifdef DUMP_IOLAYER
#if 1
#define DUMP_SENDER
#define DUMP_READER 
#endif
#define DUMP_DATA    0x0001
#define DUMP_ERROR   0x0002
#define DUMP_THREAD  0x0004
#define DUMP_DELAY   0x0100
#define DUMP_KEYS    0x0200
#define DUMP_CONTROL 0x0400
#define DUMP_COOKIE  0x0800
#endif

/* ---------------------------------------------------------------------- *
 *              helper definitions                                        *
 * ---------------------------------------------------------------------- */

/* Input buffer to allow reading the total block size, all at once. */
typedef struct _cipher_iocache {

  unsigned short max_threads;	/* maximal number of threads, allowed */

  unsigned limit_maxblock;	/* limits auto blocksize change */
  unsigned         start ;	/* data [start]        is the first byte */
  unsigned          fill ;	/* date [start+fill-1] is the last byte */
  unsigned           dim ;	/* max size of cache data */

  unsigned char  got_embedded;	/* tmp flag to notify about embedded cmds */
  unsigned char stop_on_empty;	/* always return at 0 byte frames, 
				   even with an embedded exec command */
  unsigned char      eof ;	/* marks end of file */
  unsigned char data [1] ;	/* the data cache, following */
  /*VARIABLE SIZE*/

} cipher_iocache ;


/* embedded commands */
typedef struct _cipher_command {
  
  /* adjustment date to face the 1/19/2038 problem: time (0) values 
     smaller than time_adjust are considered wraped. */
  time_t     time_adjust ; 

  /* handling threads */
  unsigned  char    id_counter; /* used to generate thread ids */
  unsigned short active_thread;	/* the thread to be used, 0 == no thread */

  /* pending commands */
  unsigned     maxblock ;	/* request to change the block size */
  unsigned char bitlist ;	/* embedded commands not executed, yet */ 

  /* sub commands */
  unsigned char exe ;		/* sending a sub command */
  unsigned char buf [16] ;	/* sub command argument */

} cipher_command ;


/* internal state of the cipher data stream */
typedef struct _cipher_state {

  unsigned char cookie [8] ;	/* id hash for threaded protocol version */
  unsigned short thread_id ;	/* current thread id (if any) */

  pid_t        creator_pid ;	/* the process id */

  /* key change schedule */
  unsigned long next_change ;	/* next time when the key needs a change */
  unsigned long   key_sched ;	/* max time interval to the next change */

  /* data transfer encrypted <-> plain */
  cipher_desc   *cipher;	/* current cipher */
  unsigned char *block ;	/* cbc context: last chained block */
  unsigned       blen  ;	/* cbc block length */

  /* stream data integrity check: crc/hash */
  frame_desc       *frame ;	/* current frame */
  unsigned char chain [4] ;	/* crc is chained with the last block */

} cipher_state ;


/* ---------------------------------------------------------------------- *
 *  the internal data type that carries all the information at run time   *
 * ---------------------------------------------------------------------- */

/* threaded io channels */
typedef struct _cipher_thread {

  cipher_state          state ;	/* internal state of this thread */
  struct _cipher_thread *next ;	/* chain */
  
} cipher_thread ;


typedef struct _ioCipher {

  /* do some accounting */
  unsigned long  payload ;	/* the user data without protocol overhead, only */
  unsigned long    total ;	/* all bytes tranfered, including proto overhead */

  pid_t    synthetic_pid ;	/* set by somebody who does not like getpid () */

  unsigned            fd ;	/* read/write file handle */
  cipher_state     state ;	/* internal state */
  unsigned      maxblock ;	/* maximal (non-fragmented) io block length */

  cipher_command *sender ;	/* handles embedded sender commands */
  cipher_iocache  *cache ;	/* input is read and encrypted blockwise */

  /* handling threads */
  unsigned char public_destroy; /* everybody may destroy the receiver threads */
  unsigned char enable_threads;	/* always send/receive a thread identifier */
  unsigned short   act_threads; /* current number of threads in the list */
  cipher_thread        *thread;	/* table used when threading enabled */

} ioCipher ;

/* ---------------------------------------------------------------------------- *
 *                      debugging                                               *
 * ---------------------------------------------------------------------------- */

#if defined (DUMP_READER) || defined (DUMP_SENDER)
#include "baseXX.h"
#include "peks.h"

static void
_debug_notify 
  (unsigned      line,
   char           *fn,
   char         *type,
   char         *info,
   char       *cookie,
   char     *isthread,
   unsigned thread_id,
   char        *ismax,
   unsigned max_thrds)
{
  static char nullstr [8];
  char *c = 0, *s = 0;
  if (type == 0) type =  "pid" ;
  if   (fn == 0)   fn = "info" ;
  if (info == 0) info =     "" ;
  if (cookie != 0 && memcmp (cookie, nullstr, 8) != 0) {
    c = " cookie=" ;
    s = bin2base64 (cookie, 8);
  }
  fprintf (stderr, 
	   "DEBUG(%s=%u.%04u): %s(%s):%s%s %s%u %s%d",
	   type, line, getpid (), info, fn, c ? c : "", s ? s : "", 
	   isthread, thread_id, ismax, max_thrds) ;
  if (strcmp (ismax, "info=") == 0)
    fprintf (stderr, "=(%u.%u.%u)=(%u.%u)",
	     max_thrds>>16, (max_thrds>>8)&0xff, max_thrds&0xff,
	     max_thrds>>8, max_thrds&0xff);
  fputs ("\n", stderr);
  fflush (stderr);
  if (s != 0)
    XFREE (s);
}

static void
_debug_text
  (unsigned   line,
   char        *fn,
   char      *type,
   char      *info,
   unsigned    len)
{
  unsigned char *s, *t;

  if (type == 0) type =  "pid" ;
  if   (fn == 0)   fn = "info" ;
  if (info == 0) {info = "*" ; len = 1; }

  memcpy (s = ALLOCA (len + 1), info, len);
  while (len > 0 && s [len-1] < ' ') len -- ;
  s [len] = '\0';
  for (t = s; len --; t++) if (*t < ' ' || *t > 127) *t = '.' ;
  fprintf (stderr, "DEBUG(%s=%u.%04u): (%s) data=%s.\n",
	   type, line, getpid (), fn, s);
  fflush (stderr);
  DEALLOCA (s);
}

static void
_debug_errno
  (unsigned   line,
   char        *fn,
   char      *type,
   unsigned    err)
{
  if (type == 0) type =  "pid" ;
  if   (fn == 0)   fn = "info" ;

  fprintf (stderr, "DEBUG(%s=%u.%04u): %s: error(%u) %s\n",
	   type, line, getpid (), fn, err, peks_strerr (err));
  fflush (stderr);
}
#endif

#if defined (DUMP_SENDER)
static void
_send_notify 
  (unsigned      line,
   char           *fn,
   char         *info,
   ioCipher     *desc,
   cipher_state *stat,
   unsigned       num)
{
  char   * txt = "pid=" ;
  unsigned   n = (stat != 0) ? stat->creator_pid : 0;
  unsigned tid = (stat != 0) ? stat->thread_id   : 0;
  char *cookie = (stat != 0) ? stat->cookie      : 0;

  if (stat == 0 && num) {
    txt = "info=" ;
    n   = num ;
  }
  _debug_notify (line, fn, "send", info, cookie, "thread=", tid, txt, n);
}

static void
_send_text
  (unsigned   line,
   char        *fn,
   char      *info,
   unsigned    len)
{
  _debug_text (line, fn, "send", info, len);
}

static void
_send_drain_delay (void)
{
  fflush (stderr) ;
  /* sleep (1) ; */
}

#define IF_SEND(x)                if (dump_send && ((x) & iodebug_flags))
#define SEND_NOTIFY(x,f,i,d,s,n) {IF_SEND(x) _send_notify (__LINE__, f,i,d,s,n);}
#define SEND_DRAIN_DELAY() {IF_SEND(DUMP_DELAY) _send_drain_delay ();}
#define SEND_TEXT(f,i,l)   {IF_SEND(DUMP_DATA) _send_text   (__LINE__,f,i,l);}
#endif


#if defined (DUMP_READER)   
static void
_recv_notify 
  (unsigned      line,
   char           *fn,
   char         *info,
   ioCipher     *desc,
   cipher_state *stat,
   unsigned       num)
{
  char   * txt = "left=" ;
  unsigned   n = desc->cache->max_threads - desc->act_threads ;
  unsigned tid = (stat != 0) ? stat->thread_id : 0;
  char *cookie = (stat != 0) ? stat->cookie    : 0;

  if (stat == 0 && num) {
    txt = "info=" ;
    n   = num ;
  }
  
  _debug_notify (line, fn, "recv", info, cookie, "thread=", tid, txt, n);
}

static void
_recv_text
  (unsigned   line,
   char        *fn,
   char      *info,
   unsigned    len)
{
  _debug_text (line, fn, "recv", info, len);
}

static void
_recv_errno
  (unsigned   line,
   char        *fn,
   unsigned    err)
{
  _debug_errno (line, fn, "recv", err);
}

#define IF_RECV(x)                if (dump_recv && ((x) & iodebug_flags))
#define RECV_NOTIFY(x,f,i,s,t,v) {IF_RECV(x) _recv_notify (__LINE__, f,i,s,t,v);}
#define RECV_TEXT(f,i,l)     {IF_RECV(DUMP_DATA) _recv_text   (__LINE__, f,i,l);}
#define RECV_ERRNO(f,e)      {IF_RECV(DUMP_ERROR) _recv_errno  (__LINE__, f,e);}
#endif

/* --------------------------------------------------------------------------- */

#ifndef SEND_DRAIN_DELAY
#define SEND_DRAIN_DELAY() 
#endif
#ifndef SEND_NOTIFY
#define SEND_NOTIFY(x,f,i,c,t,p)
#endif
#ifndef SEND_TEXT
#define SEND_TEXT(f,i,l)
#endif
#ifndef RECV_NOTIFY
#define RECV_NOTIFY(x,f,i,s,t,v)
#endif
#ifndef RECV_TEXT
#define RECV_TEXT(f,i,l)
#endif
#ifndef RECV_ERRNO
#define RECV_ERRNO(f,e)
#endif

/* ---------------------------------------------------------------------------- *
 *                      private helpers                                         *
 * ---------------------------------------------------------------------------- */

#define is_sender(d)     ((d)->cache == 0)

static void
destroy_ioState_links 
  (cipher_state *s)
{
  if (s->block != 0)
    XFREE (s->block) ;
  if (s->cipher != 0)
    destroy_cipher (s->cipher) ;
  if (s->frame != 0)
    destroy_frame (s->frame) ;
}


static void
destroy_ioCipher_links 
  (void *c)
{
  ioCipher *desc = c;
  /* destroy threads */
  while (desc->thread != 0) {
    cipher_thread *t = desc->thread ;
    desc->thread = t->next ;
    destroy_ioState_links (& t->state) ;
    XFREE (t) ;
  }
  /* destroy the basic stream */
  if (desc->cache != 0)
    XFREE (desc->cache) ;
  if (desc->sender != 0)
    XFREE (desc->sender) ;
  destroy_ioState_links (& desc->state) ;
}


#define buffer2ulong(b)    ( (((b) [0] << 24) & 0xff000000) |\
			     (((b) [1] << 16) & 0x00ff0000) |\
			     (((b) [2] <<  8) & 0x0000ff00) |\
		 	     (((b) [3]      ) & 0x000000ff) )

#define  ulong2buffer(b,u)  ( (b) [0] = ((u) >> 24) & 0xff ,\
                              (b) [1] = ((u) >> 16) & 0xff ,\
                              (b) [2] = ((u) >>  8) & 0xff ,\
                              (b) [3] = ((u)      ) & 0xff )

#define buffer2ushort(b)   ( (((b) [0] << 8) & 0xff00) |\
			     (((b) [1]     ) & 0x00ff) )

#define ushort2buffer(b,u) ( (b) [0] = ((u) >> 8) & 0xff ,\
                             (b) [1] = ((u)     ) & 0xff )

#define bufferXbuffer2ulong(b1, b2) ( ((((b1) [0] ^ (b2) [0]) << 24) & 0xff000000) |\
				      ((((b1) [1] ^ (b2) [1]) << 16) & 0x00ff0000) |\
				      ((((b1) [2] ^ (b2) [2]) <<  8) & 0x0000ff00) |\
				      ((((b1) [3] ^ (b2) [3])      ) & 0x000000ff) )

#define ulongXbuffer2buffer(t, u, b) ( (t) [0] = (((u) >> 24) ^ (b) [0]) & 0xff ,\
				       (t) [1] = (((u) >> 16) ^ (b) [1]) & 0xff ,\
				       (t) [2] = (((u) >>  8) ^ (b) [2]) & 0xff ,\
				       (t) [3] = (((u)      ) ^ (b) [3]) & 0xff )


#ifndef HAVE_MEMMOVE
/* redefine as prototypes may hang around */
#define memmove(x,y,z) movemem (x, y, z)

static void 
movemem 
  (unsigned char *trg,
   unsigned char *src,
   unsigned       len)
{
  unsigned n ;
  
  if (trg + len <= src ||
      src + len <= trg) {
    /* ranges do not overlap */
    if (len != 0)
      memcpy (trg, src, len) ;
    return ;
  } 

  /* shift left */
  if (trg < src) {

    /* do the non-overlapping part */
    memcpy (trg, src, n = src - trg) ;
    trg += n ;
    src += n ;
    len -= n ;
  
    /* do the rest */
    while (len --)
      * trg ++ = * src ++ ;
    return ;
  }

  /* shift right (if there is something to do, at all) */
  if (src == trg)
    return;
  
  /* do the non-overlapping part */
  n = trg - src ;
  memcpy (trg + len - n, src + len - n, n) ;
  len -= n ;

  /* do the rest */
  while (len --)
    trg [len] = src [len] ;
}
#endif /* HAVE_MEMMOVE */


#ifndef XREALLOC
static void *
xrealloc (void *p, unsigned n)
{
  void *q = realloc (p, n) ;
  if (q == 0) {
    fprintf (stderr, "Could not re-allocate to %u bytes !\n", n);
    exit (1);
  }
  return q ;
}
#define XREALLOC_DOES_NOT_INITALIZE
#define XREALLOC(p,n)	xrealloc (p, n)
#endif

/* ---------------------------------------------------------------------------- *
 *                      private input cache handling                            *
 * ---------------------------------------------------------------------------- */

static int
extract_from_io_cache
 (cipher_iocache *ioc,   /* beware of ioc == NULL */
  char           *trg,
  unsigned        len)
{
  /* prevent from a cache underflow */
  if (ioc->fill < len)
    len = ioc->fill ;

  if (len) {
    /* move to target buffer */
    if (len == 1)
      trg [0] = ioc->data [ioc->start] ;
    else
      memcpy (trg, ioc->data + ioc->start, len) ;
    if ((ioc->fill -= len) == 0)
      ioc->start = 0 ;
    else
      ioc->start += len ;
  }

  return len ;
}


static void
append_to_io_cache 
  (cipher_iocache *ioc,
   char           *buf,
   unsigned        len)
{
  /* beware of ioc == NULL */

  unsigned max_buf_usage = (ioc->dim >> 1) ;

  POINT_OF_RANDOM_STACK (1) ;

  /* Appending a text [0..len-1] to the input cache data, we need to 
     check whether we must compact the input cache. This is done by
     a shift left of the text already stored. */
  
  if (max_buf_usage < ioc->start + ioc->fill + len) { 
    
    assert (ioc->fill + len <= ioc->dim) ;

    if (ioc->fill) 
      /* there is no more space for len bytes, so shift left */
      memmove (ioc->data, ioc->data + ioc->start, ioc->fill) ;
    
    /* ok, define the new start of the input cache */ 
    ioc->start = 0 ; 
  }
  
  /* append the input block to the cache */
  memcpy (ioc->data + ioc->start, buf, len) ;
  ioc->fill += len;

  POINT_OF_RANDOM_STACK (7) ;
}

/* note, that this function has another descriptor type */
static int
resize_ioCache
  (ioCipher *desc,
   unsigned   new_size)
{
  cipher_iocache *ioc = desc->cache ;

  POINT_OF_RANDOM_STACK (3) ;
  
  /* prevent from a cache underflow */
  if (ioc->fill >= new_size) {
    errno = errnum (CBC_CACHERES_BLOCKED) ;
    return -1;
  }

  /* shift data to the left */
  if (ioc->start) {
    memmove (ioc->data, ioc->data + ioc->start, ioc->fill) ;
    ioc->start = 0 ;
  }

  /* adjust */
  ioc->dim    = (new_size << 1);
  desc->cache = XREALLOC (ioc, sizeof (cipher_iocache) - 1 + ioc->dim) ;

  POINT_OF_RANDOM_VAR (desc->cache) ;

  return new_size ;
}


/* ---------------------------------------------------------------------------- *
 *                      private cbc functions                                   *
 * ---------------------------------------------------------------------------- */

#define supported_block_length(n) ((n) == 8 || (n) == 16)

/* 
 * Instead of transfering 4 times one byte, we use a long (aka 4 byte) variable
 * to transfer 4 bytes all at once. This worke under the following condition:
 *
 * The buffers used are aligned on a pointer boundary and encrypted or decrypted 
 * 8 byte (== 64 bit) wise, and the u32 alignement necessary is a 4 byte boundary. 
 *
 * The first hypothesis is provided for by the code using ALLOCA (), the second
 * is trivially true. So we need to check, that we align on a 4 byte boundary.
 */

#if defined (HAVE_U32_ALIGN)
#define p32(p,n,m)            ((u32*)(p)) [2*(n)+(m)]

#define p64xassign( p,q,  n)  (p32 (p,n,0) ^= p32 (q,n,0),\
			       p32 (p,n,1) ^= p32 (q,n,1))

#define p64xyassign(p,q,r,n)  (p32 (p,n,0) = (p32 (q,n,0) ^ p32 (r,n,0)),\
			       p32 (p,n,1) = (p32 (q,n,1) ^ p32 (r,n,1)))

#elif defined (HAVE_U64_ALIGN)

/* FIXME: somebody should check that code and define u64 */
#define p64(p,n)              ((u64*)(p)) [n]
#define p64xassign( p,q,  n)   (p64 (p,n) ^= p64 (q,n))
#define p64xyassign(p,q,r,n)   (p64 (p,n) = (p64 (q,n) ^ p64 (r,n)))

#else

#define p8(p,n,m)             ((unsigned char*)(p)) [8*(n)+(m)]

#define p64xassign( p,q,  n)  ( p8 (p,n,0) ^= p8 (q,n,0) ,\
			        p8 (p,n,1) ^= p8 (q,n,1) ,\
			        p8 (p,n,2) ^= p8 (q,n,2) ,\
			        p8 (p,n,3) ^= p8 (q,n,3) ,\
			        p8 (p,n,4) ^= p8 (q,n,4) ,\
			        p8 (p,n,5) ^= p8 (q,n,5) ,\
			        p8 (p,n,6) ^= p8 (q,n,6) ,\
			        p8 (p,n,7) ^= p8 (q,n,7) )

#define p64xyassign(p,q,r,n)  ( p8 (p,n,0) = (p8 (q,n,0) ^ p8 (r,n,0)) ,\
				p8 (p,n,1) = (p8 (q,n,1) ^ p8 (r,n,1)) ,\
				p8 (p,n,2) = (p8 (q,n,2) ^ p8 (r,n,2)) ,\
				p8 (p,n,3) = (p8 (q,n,3) ^ p8 (r,n,3)) ,\
				p8 (p,n,4) = (p8 (q,n,4) ^ p8 (r,n,4)) ,\
				p8 (p,n,5) = (p8 (q,n,5) ^ p8 (r,n,5)) ,\
				p8 (p,n,6) = (p8 (q,n,6) ^ p8 (r,n,6)) ,\
				p8 (p,n,7) = (p8 (q,n,7) ^ p8 (r,n,7)) )
#endif

static void
cbc_encrypt
  (cipher_state       *desc,
   unsigned       char *out,
   const unsigned char  *in)
{
  /* cbc-encrypt (E (), block [], in []) -> out []
   *
   *	block [] := E (block [] XOR in [])
   *	out   [] := block []
   */

  /* block [] := block [] XOR in [] */
  p64xassign (desc->block, in, 0) ;
  if (desc->blen > 8)
    p64xassign (desc->block, in, 1) ;
  
  /* out [] := ENCRYPT (block []) */
  XCRYPT (desc->cipher, out, desc->block);

  /* block [] := out [] */
  memcpy (desc->block, out, desc->blen);
}


static void
cbc_decrypt
  (cipher_state       *desc,
   unsigned       char *out,
   const unsigned char  *in)
{
  /* cbc-decrypt (D (), block [], in []) -> out []
   *
   *	out   [] := D (in []) XOR block []
   *	block [] := in []
   */

  unsigned char buf [16] ;
  XCRYPT (desc->cipher, buf, in);

  /* out [] := D (in []) XOR block [] */
  p64xyassign (out, buf, desc->block, 0) ;
  if (desc->blen > 8)
    p64xyassign (out, buf, desc->block, 1) ;
  
  /* block [] := in [] */
  memcpy (desc->block, in, desc->blen);
}


/* cbc decrypt the first 16 bytes without saving the 
   internal state */

static void
cbc_peep_decrypt16
  (const cipher_state  *desc,
   unsigned       char  *out,
   const unsigned char   *in)
{
  /* cbc-peep-decrypt (D (), block [], in []) -> out []
   *
   * if blen == 16:
   *
   *	out   [] := D (in   []) XOR block []
   *
   * if blen == 8:
   *
   *	out   [] := D (in   []) XOR block []
   *	out+8 [] := D (in+8 []) XOR in    []
   */
  
  cipher_desc *hold = duplicate_cipher (desc->cipher) ;
  XCRYPT (hold, out, in);
  
  /* out [] := D (in []) XOR block [] */
  p64xassign (out, desc->block, 0) ;

  if (desc->blen > 8)

    p64xassign (out, desc->block, 1) ;

  else {

    /*     out+8 [] := D (in+8 []) XOR in [] 
     * <=> out+8 [] := D (in+8 [])
     *     out+8 [] := out+8 [] XOR in []
     */
    XCRYPT (hold, out + 8, in + 8);
    p64xassign (out+8, in, 0) ;
  }

  destroy_cipher (hold);
}

#undef p8
#undef p32
#undef p64
#undef p64xassign
#undef p64xyassign

/* ---------------------------------------------------------------------------- *
 *                  private functions: io function helpers                      *
 * ---------------------------------------------------------------------------- */

/*
 * The protocol block frame looks like:
 *
 * +-..-+--+--+-..-+--+--+-..-+---+- .. -+-  ... --+--+- .. -+--+-..-+--+
 * |  COOKIE  | LENX  | LENY  | P | args |  padding   |  data   |  CRC  |
 * +-..-+--+--+-..-+--+--+-..-+---+- .. -+-  ... --+--+- .. -+--+-..-+--+
 *
 *  <-- 8 ---> <- 4 -> <- 4 ->            <-- P&15 -->           <- 4 ->
 *  <optional> <--------- encryption distance ------------------------->
 *             <------------ LENX ^ LENY ------------------------------>
 *             <------------ CRC DISTANCE ----------------------------->
 *
 *
 * where (LENX ^ LENY) % 8 == 0 for 64 bit cbc mode, and (LENX ^ LENY) % 16
 * == 0 for 128 bit cbc mode.  The current CRC is hashed while the last four 
 * bytes of the protocol block are filled with the CRC from the previous block.  
 *
 * All numbers are stored in big endian.
 *
 * The value C = P & ~15 is reserved as embedded command with:
 */

#define SEND_CHANNEL_ID     0x80
#define EXEC_COMMAND        0x40
#define CHANGE_BUFFER_SIZE  0x20
#define EMBD_COMMAND_MASK   0xf0 /* masking all command bits */


/* sub commands passed as EXEC_COMMAND */
#define EXEC_DESTROY_THREAD       1
#define EXEC_CREATE_THREAD        2
#define EXEC_CHANGE_SESSION_KEY  10


/* 13 bytes static plus at most 15 padding bytes,        ... 28   
   possibly 2 bytes  channel ID,                         ...  2
   possibly 17 bytes key change request,                 ... 17
   possibly 4 bytes buffer change arg                    ...  4
                                                    -------------  */
#define PROTO_OVERHEAD                                       51


/* We know, that a cmd is most likely 0, or SEND_CHANNEL_ID. So the following
   macro optimizes the function call determining the argument  length  */

#define get_embedded_cmd_arglen(cmd) \
	((cmd) ? ((cmd) == SEND_CHANNEL_ID ? 2 : _get_emb_cmd_arglen (cmd)) : 0)

static unsigned
_get_emb_cmd_arglen
  (unsigned char cmd)
{
  unsigned len = 0 ;

  if (cmd & SEND_CHANNEL_ID)    len +=  2;
  if (cmd & EXEC_COMMAND)       len += 17;
  if (cmd & CHANGE_BUFFER_SIZE) len +=  4;

  return len ;
}

/* ---------------------------------------------------------------------------- *
 *                  private functions: threaded io                              *
 * ---------------------------------------------------------------------------- */

static cipher_thread *
duplicate_thread
  (cipher_state *stat)
{
  cipher_thread *new = XMALLOC (sizeof (cipher_thread)) ;

  /* copy the current channel state */
  new->state         = *stat ;
  new->state.cipher  = duplicate_cipher (stat->cipher) ;
  new->state.frame   = duplicate_frame  (stat->frame);
  new->state.block   = memcpy (XMALLOC (stat->blen), stat->block, stat->blen) ;

  return new ;
}

static cipher_thread **
thread_ptr_by_id
  (ioCipher *desc,
   int         id)
{
  cipher_thread *t, **T = &desc->thread ;

  if (id > 0) {

    /* find that thread id */
    while ((t = *T, t != 0) && t->state.thread_id != id)
      T = & t->next ;

    if (t != 0) 
      return T ;
  }

  errno = errnum (CBC_NOSUCH_THREADID);
  return 0 ;
}


static cipher_thread **
thread_ptr_by_pid
  (ioCipher *desc,
   pid_t      pid)
{
  cipher_thread *t, **T = &desc->thread ;

  if (pid) {
    /* find that thread id */
    while ((t = *T, t != 0) && t->state.creator_pid != pid)
      T = & t->next ;
  } else
    /* get the first one, if pid == 0 */
    t = *T ;

  if (t != 0) 
    return T ;

  errno = errnum (CBC_NOSUCH_THREADID);
  return 0 ;
}


static cipher_thread *
unlink_thread
  (ioCipher *desc,
   int         id,
   cipher_thread **(*find) (ioCipher *,int))
{
  cipher_thread *t, **T = (*find) (desc, id) ;

  if (T == 0) 
    return 0;

  /* unlink */
  t = *T ;
  *T = t->next ;
  return t;
}


static int
destroy_thread
  (ioCipher *desc, 
   unsigned    id,
   cipher_thread **(*find) (ioCipher *,int))
{
  cipher_thread *t = unlink_thread (desc, id, find);

  if (t == 0) {
    errno = errnum (CBC_ILL_THREADID) ;
    return -1 ;
  }

  destroy_ioState_links (& t->state) ;
  XFREE (t) ;

  /* set id to something that exists */
  if (desc->sender != 0 &&
      desc->sender->active_thread == id)
    desc->sender->active_thread = 0 ;

  return 0 ;
}



static void
rotate_cookie
 (unsigned char *cookie,
  unsigned char    *buf,
  frame_desc      *hash)
{

  /* the last 4 bytes of the cookie [] are shifted left, and replaced by 
     the 4 bytes of buf []. Finally, this new cookie is hashed. */
  
  memcpy (cookie, cookie + 4, 4);
  memcpy (cookie,        buf, 4);
  XCRCFIRST (hash, cookie, 8);
  
  /* store the new cookie in the lookup table */
  memcpy (cookie, XCRCRESULT0 (hash), 8) ;
}

/* ---------------------------------------------------------------------------- *
 *                private functions: threaded receiver io                       *
 * ---------------------------------------------------------------------------- */

static unsigned
receiver_thread_id_matches
  (cipher_state        *state,
   const unsigned char *inbuf)
{ 
  unsigned char plainbuf [16] ;

  cbc_peep_decrypt16 (state, plainbuf, inbuf) ;

  return  
    (plainbuf [8] & SEND_CHANNEL_ID) &&
    buffer2ushort (plainbuf + 9) == state->thread_id ;
}


static cipher_state *
get_receiver_thread
  (ioCipher *desc,
   char   *cookie)
{
  cipher_thread *t, *u ;
  unsigned matches, master_thread_matches ;

  if (!desc->enable_threads) 
    return &desc->state ;

  /* Search through the linked list of thread records to find matching 
     cookies. Return the corresponding state record if there is exactly 
     one match */
  
  t       = desc->thread ;
  u       = 0 ;		/* holds the first match */
  matches = 0 ;		/* counts the number of matches */

  /* start with the basic thread */
  if ((master_thread_matches =
      (memcmp (desc->state.cookie, cookie, 8) == 0)) != 0)
    matches ++ ;

  POINT_OF_RANDOM_VAR (cookie) ;

  /* continue with the list of derived threads */
  while (t != 0) {
    if (memcmp (t->state.cookie, cookie, 8) == 0) {
      if (u == 0)
	u = t ;
      matches ++ ;
    }
    t = t->next ;
  }

  switch (matches) {
  case 0:	/* no match at all */
    return 0 ;
  case 1:	/* exatly one match, return that thread */
    return master_thread_matches ? &desc->state : &u->state ;
  }

  /* Now, decode the first part of the next 16 bytes looking
     for a thread-ID */

  if (master_thread_matches) {
    if (receiver_thread_id_matches (&desc->state, cookie + 8))
      return &desc->state ;
    matches -- ;
  }

  /* look through the other instances */
  if (u == 0) 
    u = desc->thread ;
  
  /* continue with the rest of the list of threads */
  while (u != 0 && matches) {
    if (memcmp (u->state.cookie, cookie, 8) == 0) {
      if (receiver_thread_id_matches (&u->state, cookie + 8))
	return &u->state ;
      matches -- ;
    }
    u = u->next ;
  }

  /* mo match */
  return 0;
}


static int
make_recv_thread
  (ioCipher        *desc, 
   cipher_state    *stat, 
   unsigned short     id, 
   unsigned char *cookie,
   unsigned char    *buf)
{
  cipher_thread *t ;

  POINT_OF_RANDOM_VAR (buf) ;

  if (desc->act_threads >= desc->cache->max_threads) {
    errno = errnum (CBC_NO_MORE_THREADS) ;
    return -1;
  }
  
  if (desc->enable_threads == 0) {	/* tag basic thread */
    memset (desc->state.cookie, 0, 8) ;
    rotate_cookie (desc->state.cookie, buf, desc->state.frame);
    RECV_NOTIFY /* for debugging */
      (DUMP_CONTROL, "make_recv_thread", "initializing basic thread", 
       desc, stat, 0) ;
    desc->enable_threads ++ ;
  }

  POINT_OF_RANDOM_VAR (cookie) ;

  /* copy the current channel state, assign thread id */
  t = duplicate_thread (stat) ;

  /* assign cookie and thread id */
  memcpy (t->state.cookie, cookie, 8) ;
  t->state.thread_id = id ;

  /* insert in list */
  t->next      = desc->thread ;
  desc->thread = t ;
  desc->act_threads ++ ;

  POINT_OF_RANDOM_STACK (7) ;

  RECV_NOTIFY /* for debugging */
    (DUMP_THREAD, "make_recv_thread", "creating a new thread", 
     desc, &t->state, 0) ;
  
  return 0;
}


/* ---------------------------------------------------------------------------- *
 *          private functions: receiver embedded command handling               *
 * ---------------------------------------------------------------------------- */

/* We know, that a cmd is most likely 0, or SEND_CHANNEL_ID. So the following
   macro optimizes the function call evaluationg the argument */

#define do_embedded_recv_cmds(desc,stat,cmd,buf,p) \
	((cmd) ? ((cmd) == SEND_CHANNEL_ID && stat->thread_id == buffer2ushort (p) \
		  ? (desc->cache->got_embedded ++, 2)				   \
		  : _do_emb_recv_cmds (desc, stat, cmd, buf, p)) : 0)

static int
_do_emb_recv_cmds 
  (ioCipher     *desc, 
   cipher_state *stat,
   unsigned char  cmd,
   unsigned char *buf,
   unsigned char   *p)
{
  unsigned processed = 0, id = -1 ;

  if (cmd & SEND_CHANNEL_ID) {

    /* check for valid channel id */
    if ((id = stat->thread_id) != buffer2ushort (p)) {
      errno = errnum (CBC_REC_ILL_THREADID) ;
      RECV_NOTIFY /* for debugging */
	(DUMP_ERROR, "_do_emb_recv_cmds", "got wrong thread id", 
	 desc, 0, buffer2ushort (p)) ;
      return -1 ;
    }
    desc->cache->got_embedded ++ ;
    processed += 2 ;
    p         += 2 ;
  }

  if (cmd & EXEC_COMMAND) {
    switch (* p ++) {
    case EXEC_CREATE_THREAD:
      if (make_recv_thread (desc, stat, buffer2ushort (p), p + 2, buf) < 0)
	return -1;
      POINT_OF_RANDOM_STACK (3) ;
      break ;

    case EXEC_DESTROY_THREAD:
      /* Who is allowed to destroy a thead ?. OK, I agree on the 
	 owner of the thread, but what about the rest ? */
      if (desc->public_destroy == 0 && id && id != buffer2ushort (p)) {
	errno = errnum (CBC_CANT_KILL_NOTOWN) ;
	return -1 ;
      }
      if (destroy_thread (desc, buffer2ushort (p), thread_ptr_by_id) < 0) {
	RECV_NOTIFY /* for debugging */
	  (DUMP_ERROR, "_do_emb_recv_cmds", "unable to destroy thread id", 
	   desc, 0, buffer2ushort (p)) ;
	return -1;
      }
      POINT_OF_RANDOM_STACK (5) ;
      desc->act_threads -- ;
      RECV_NOTIFY /* for debugging */
	(DUMP_THREAD, "_do_emb_recv_cmds", "destroyed thread id", 
	 desc, 0, buffer2ushort (p)) ;
      break ;

    case EXEC_CHANGE_SESSION_KEY:
      RECV_NOTIFY /* for debugging */
	(DUMP_CONTROL, "_do_emb_recv_cmds", "changeing the session key",
	 desc, stat, 0);
      if (change_decryption_key (stat->cipher, p) < 0)
	return -1;
      POINT_OF_RANDOM_STACK (4) ;
      break ;
      
    default:
      errno = errnum (CBC_ILL_SUBCMD) ;
      return -1;
    }
    
    desc->cache->got_embedded ++ ;
    processed += 17 ;
    p         += 16 ;
  }

  if (cmd & CHANGE_BUFFER_SIZE) {
    unsigned long l = buffer2ulong (p) ;
  
    /* adjust to something sensible */
    if (l < CBC_IO_BUFLEN_MIN) {
      l = CBC_IO_BUFLEN_MIN ;
    } else {
      if (l > CBC_IO_BUFLEN_MAX)
	l = CBC_IO_BUFLEN_MAX ;
    }

    /* limit_maxblock imposes a limit on the buffer size */
    if (desc->cache->limit_maxblock != 0) 
      
      /* block size ranges from limit_maxblock to state.maxblock */
      if (desc->cache->limit_maxblock < desc->maxblock) {
	if (l < desc->cache->limit_maxblock)
	  l = desc->cache->limit_maxblock ;
	else
	  if (desc->maxblock < l)
	    l = desc->maxblock ;
      } else {

	/* block size ranges from state.maxblock to limit_maxblock */
	if (desc->maxblock <= desc->cache->limit_maxblock) {
	  if (l < desc->maxblock)
	    l = desc->maxblock ;
	  else
	    if (desc->cache->limit_maxblock < l)
	      l = desc->cache->limit_maxblock ;
 	}
      }
    
    /* adjust input cache */
    if (l != desc->maxblock)
      resize_ioCache (desc, desc->maxblock = l) ;

    desc->cache->got_embedded ++ ;
    processed += 4 ;
    /* p      += 4 ; */
  }

  return processed ;
}

/* ---------------------------------------------------------------------------- *
 *                    private functions: receiver io functions                  *
 * ---------------------------------------------------------------------------- */

static int
recfrom_ioCipher_block
  (ioCipher     *desc,
   unsigned char   *p,
   unsigned       cnt,
   int          flags)
{
  int n, len ;
  unsigned char crc [4] ;
  unsigned processed, padding, mask, cmd ;
  cipher_state *thread;

    /* allocate the data bufers */
  unsigned char *inbuf    = ALLOCA (desc->maxblock + PROTO_OVERHEAD) ;
  unsigned char *plainbuf = ALLOCA (desc->maxblock + PROTO_OVERHEAD) ;

  /* check, whether we run the protocol in threaded mode in which
     case we need lo load the 8 byte cookie followed by the first
     16 byte of the data block */
  unsigned start = desc->enable_threads ? 8 : 0 ;

  /* read the 16 bytes from the data block, possibly there is a leading
     8 bytes thread cookie */
  unsigned arglen = 16 + start ;

  /* use this buffer to read the block header */
  unsigned char *data_start = start ? ALLOCA (arglen) : inbuf ;

  /* get the first packet which has the minimum size 16 bytes */
  for (processed = 0; processed < arglen; processed += n, desc->total += n)
    if ((n = OS_RECV 
	 (desc->fd, data_start + processed, arglen - processed, flags)) <= 0) {
      /* take care of interrupts */
      if (n < 0 && errno == EINTR) {
	RECV_NOTIFY (DUMP_ERROR, "recfrom_ioCipher_block", 
		     "signal while reading first packet", desc, 0, processed);
	n = 0 ;
	continue ;
      }
      /* get ready to leave, otherwise */
      if (start) 
	DEALLOCA (data_start);
      DEALLOCA (plainbuf);
      DEALLOCA    (inbuf);
      if (n == 0 && processed == 0) { /* still the first try ? */
	desc->cache->eof = 1 ;       /* mark end of file */
	return 0 ;
      } 
      errno = errnum (CBC_RECV_NULL_BLOCK) ;
      return -1 ;
    }

  /* get the current thread */
  if (start) {
    
    if ((thread = get_receiver_thread (desc, data_start)) == 0) {
      DEALLOCA (data_start);
      errno = errnum (CBC_NOSUCH_COOKIE) ;
      goto error_return ;
    }

    RECV_NOTIFY /* for debugging */
      (DUMP_COOKIE, "recfrom_ioCipher_block", "got threaded block", 
       desc, thread, 0);

    /* place the data header without the cookie in the input buffer */
    memcpy (inbuf, data_start + start, 16);
    DEALLOCA (data_start);	/* done with this buffer */

  } else {
    /* non-threaded mode */
    thread = & desc->state ;
  }

  /* Calculate the mask corresponding to the block size 8, or 16. So
     x % blen is equivalent to x & mask. */
  mask = (thread->blen - 1) ;
  
  /* As we are on a packet boundary, we can decode the first part. Note
     that the cbc block size is either 16, or 8. In the latter case
     we need to decrypt twice */
  cbc_decrypt (thread, plainbuf, inbuf) ;
  if (thread->blen == 8) 
    cbc_decrypt (thread, plainbuf + 8, inbuf + 8) ;

  /* read the total input block size */
  len = bufferXbuffer2ulong (plainbuf, plainbuf + 4) ;

  if (len < 16) {
    /* total input block completely rubbish */
    errno = errnum (CBC_RECV_LEN_2SMALL) ;
    goto error_return ;
  }
  if (len > desc->maxblock + PROTO_OVERHEAD) {
    /* total input block size too large */
    errno = errnum (CBC_RECV_LEN_2LARGE) ;
    goto error_return ;
  }
  if (len & mask) {
    /* len not on block boundary */
    errno = errnum (CBC_RECV_LEN_NOBNDRY) ;
    goto error_return ;
  }

  /* get the argument size for embedded commands */
  cmd    = (plainbuf [8] & EMBD_COMMAND_MASK) ;
  arglen = get_embedded_cmd_arglen (cmd) ;
  
  /* read the padding length */
  padding = (plainbuf [8] & mask) ;

  /* read the rest of data into the temporary input buffer */
  for (processed = 16; processed < len; processed += n, desc->total += n)
    if ((n = OS_RECV (desc->fd, inbuf + processed, len - processed, flags)) <= 0) {
      if (n < 0 && errno == EINTR) {
	RECV_NOTIFY /* for debugging */
	  (DUMP_ERROR, "recfrom_ioCipher_block", 
	   "got signal while reading data packets", desc, thread, 0);
	n = 0;
	continue ;
      }
      if (n < 0) 
	goto error_return ;
      desc->cache->eof = 1 ;  /* n = 0: read only some bytes (less than expected) */
      errno = errnum (CBC_RECV_UNEXP_EOF) ;
      goto error_return ;
    }

  /* decode that input to the plain text buffer */
  for (len = 16; len < processed; len += thread->blen)
    cbc_decrypt (thread, plainbuf + len, inbuf + len) ;
    
  /* read crc from the input block */
  memcpy (crc, plainbuf + len - 4, 4);

  /* store crc from previos block and recalculate crc */ 
  memcpy (plainbuf + len - 4, thread->chain, 4) ;

  /* compare */
  XCRCFIRST (thread->frame, plainbuf, len);
  if (memcmp (crc, XCRCRESULT (thread->frame), 4) != 0) {
    /* read invalid crc */
    errno = errnum (CBC_RECV_CRCERR) ;
    goto error_return ;
  }
  memcpy (thread->chain, crc, 4) ;

  if (start) {

    /* we need to update the cookie in the lookup table */
    rotate_cookie (thread->cookie, plainbuf, thread->frame) ;

    RECV_NOTIFY /* for debugging */
      (DUMP_COOKIE, "recfrom_ioCipher_block", "updating to new cookie", 
       desc, thread, 0);

    if ((cmd & SEND_CHANNEL_ID) == 0) {
      /* protocol error, id is required in threaded mode */
      errno = errnum (CBC_REC_NO_THREADID) ;
      goto error_return ;
    }
  }
  
  /* start with the embedded commands section */
  data_start = plainbuf + 9 ;

  /* read the embedded command arguments and process it */
  if ((n = do_embedded_recv_cmds (desc, thread, cmd, plainbuf, data_start)) < 0)
    goto error_return ;

  /* get the start of data payload */
  data_start += n + padding ;

  RECV_TEXT /* for debugging */
    ("recfrom_ioCipher_block", data_start, len-13-padding-arglen);
  
  /* calculate the size of the data payload */
  if ((n = len - 13 - padding - arglen) == 0)
    return 0 ;

  /* just a shortcut - not really necessary */
  if (desc->cache->fill == 0 && cnt >= n) {
    memcpy (p, data_start, n) ;
    DEALLOCA (plainbuf);
    DEALLOCA    (inbuf);
    return n ;
  }
    
  /* append data block to io buffer */
  append_to_io_cache (desc->cache, data_start, n); 

  DEALLOCA (plainbuf);
  DEALLOCA    (inbuf);

  /* move input from the cache to the request buffer */
  return extract_from_io_cache (desc->cache, p, cnt) ;
  /*@NOTREACHED@*/
  
 error_return:
  DEALLOCA (plainbuf);
  DEALLOCA    (inbuf);
  RECV_ERRNO /* for debugging */
    ("recfrom_ioCipher_block", errno);
  return -1 ;
}


static int
recfrom_ioCipher
  (void      *c,
   char    *buf,
   unsigned cnt,
   int    flags)
{
  ioCipher *desc = c;
  int n ;

  /* check, whether there is some old stuff, left over */
  if (desc->cache->fill)
    return extract_from_io_cache (desc->cache, buf, cnt) ;
  
  /* check whether we reached end-of-file, already */
  if (desc->cache->eof)
    return 0 ;

  do /* skip over embedded commands */
    desc->cache->got_embedded = 0 ;
  while ((n = recfrom_ioCipher_block (desc, buf, cnt, flags)) == 0 &&
	 !desc->cache->stop_on_empty && desc->cache->got_embedded) ;

  POINT_OF_RANDOM_STACK (7) ;

  if (n > 0)
    desc->payload += n ; /* accounting */

  return n ;
}


/* ---------------------------------------------------------------------------- *
 *                private functions: threaded sender io                         *
 * ---------------------------------------------------------------------------- */

static cipher_state * 
get_current_sender_thread
  (ioCipher *desc)
{
  unsigned id = desc->sender->active_thread ;
  cipher_thread *t ;
  
  if (id == 0) 
    return &desc->state ;

  POINT_OF_RANDOM_VAR (t);

  t = desc->thread ;
  while (t != 0) {
    if (t->state.thread_id == id)
      return &t->state ;
    t = t->next ;
  }

  POINT_OF_RANDOM_STACK (7) ;
  
  return 0 ;
}


/* ---------------------------------------------------------------------------- *
 *          private functions: sender embedded command handling               *
 * ---------------------------------------------------------------------------- */


static unsigned
store_embedded_commands 
  (ioCipher    *desc,
   unsigned char cmd,
   unsigned char  *p)
{
  unsigned processed = 0;

  if (cmd & SEND_CHANNEL_ID) {
    ushort2buffer (p, desc->sender->active_thread) ;
    SEND_NOTIFY /* for debugging */
      (DUMP_CONTROL, "store_embedded_commands", 
       "sending threaded", desc, 0, buffer2ushort (p));
    p         += 2 ;
    processed += 2 ;
  }

  POINT_OF_RANDOM_VAR (p) ;

  if (cmd & EXEC_COMMAND) {
    * p ++ = desc->sender->exe ;
    memcpy (p, desc->sender->buf, 16);
    p         += 16 ;
    processed += 17 ;
  }

  if (cmd & CHANGE_BUFFER_SIZE) {
    ulong2buffer (p, desc->sender->maxblock) ;
    processed += 4 ;
    /* p      += 4 ; */
  }

  return processed ;
}


static void 
post_process_sender_cmds 
  (ioCipher       *desc,
   cipher_state *thread,
   unsigned char    cmd,
   unsigned char   *buf)
{
  /* post process embedded command args */

  POINT_OF_RANDOM_VAR (buf) ;

  if (cmd & EXEC_COMMAND) {

    switch (desc->sender->exe) {
    case EXEC_CHANGE_SESSION_KEY:
      SEND_NOTIFY /* for debugging */
	(DUMP_KEYS, "post_process_sender_cmds", "changeing session keys", 
	 desc, thread, 0);
      change_encryption_key (thread->cipher, desc->sender->buf) ;
      break ;

    case EXEC_CREATE_THREAD:
      /* do we need to switch from the non-threaded to the threaded mode ? */
      if (! desc->enable_threads) {
	memset (desc->state.cookie, 0, 8) ;
	rotate_cookie (desc->state.cookie, buf, desc->state.frame);
	desc->enable_threads ++ ;
	SEND_NOTIFY /* for debugging */
	  (DUMP_THREAD, "post_process_sender_cmds", 
	   "creating basic thread", desc, thread, 0);
      }
    }
  }

  POINT_OF_RANDOM_STACK (11) ;

  if (cmd & CHANGE_BUFFER_SIZE) {
    desc->maxblock = desc->sender->maxblock ;
    desc->sender->maxblock = 0 ;
  }

  desc->sender->bitlist = 0 ;
}


/* ---------------------------------------------------------------------------- *
 *                      private functions: sender io functions                  *
 * ---------------------------------------------------------------------------- */

static int
sendby_ioCipher_block
  (ioCipher       *desc,
   cipher_state *thread,
   const char        *p,
   unsigned         cnt,
   int            flags)
{
  int n, processed ;
  char *data_start ;

  /* check whether we need to send a cookie header */
  unsigned start = desc->enable_threads ? 8 : 0 ;

  /* get embedded command bits (a threaded sender always sends the channel id) */
  unsigned char cmd = ((desc->sender->bitlist & EMBD_COMMAND_MASK) |
		       (start ? SEND_CHANNEL_ID : 0)) ;

  /* allocate the io buffer */
  unsigned char *plainbuf = ALLOCA (desc->maxblock + PROTO_OVERHEAD) ;
  unsigned char  *sendbuf = ALLOCA (desc->maxblock + PROTO_OVERHEAD + start) ;

  /* get the argument size for embedded commands */
  unsigned arglen = get_embedded_cmd_arglen (cmd) ;
  
  /* Calculate the mask corresponding to the block size 8, or 16. So
     x % blen is equivalent to x & mask. */
  unsigned mask = thread->blen - 1 ;

  /* Calculate the padding length so that the totel block length is a
     multiple of blen, i.e. find x so that 13 + (cnt+arglen) + x == 0 
     (mod blen), so x == - 13 - (cnt+arglen) (mod blen).  Using the fact,
     that blen is 8, or 16 we gain x == 19 - (cnt+arglen) (mod blen) */
  unsigned padding = (19 - cnt - arglen) & mask ;

  /* Calculate the total block length rounded up to the next multiple
     of blen (the value of padding has been figured out, already.) */
  int len = 13 + padding + arglen + cnt ;

  POINT_OF_RANDOM_VAR (data_start) ;

  /* there first 4 bytes are random followed by the xor-part of the
     4 byte length header */
  fast_random_bytes (plainbuf,  4) ;
  ulongXbuffer2buffer (plainbuf + 4, len, plainbuf) ;
  
  /* the least significant four bits carry the padding length */
  plainbuf [8]  = padding ;

  /* store the most significant 4 bits for embedded commands */
  plainbuf [8] |= cmd ;

  /* add the embedded command arguments */
  data_start = plainbuf + 9 ;
  if (cmd)
    data_start += store_embedded_commands (desc, cmd, data_start) ;
  
  /* add the padding data, already initialized */
  if (padding) {
    fast_random_bytes (data_start, padding) ;
    data_start += padding ;
  }
  
  /* add the argument text */
  if (cnt)
    memcpy (data_start, p, cnt);

  POINT_OF_RANDOM_STACK (9) ;

  /* add (temporary) CRC from last block */
  memcpy (plainbuf + len - 4, thread->chain, 4) ;

  /* Calculate the crc trailer for this block while saving the new crc */
  XCRCFIRST (thread->frame, plainbuf, len);
  memcpy (thread->chain, XCRCRESULT (thread->frame), 4) ;

  /* overwrite serial count by crc trailor */
  memcpy (plainbuf + len - 4, thread->chain, 4) ;
  
  /* check whether we need to send a cookie header */
  if (start) {

    SEND_NOTIFY /* for debugging */
      (DUMP_COOKIE, "sendby_ioCipher_block", "sending threaded", 
       desc, thread, 0);

    /* store the current cookie in the send buffer */
    memcpy (sendbuf, thread->cookie, 8) ;
    
    /* get some new values for the cookie and hash that */
    rotate_cookie (thread->cookie, plainbuf, thread->frame) ;

    SEND_NOTIFY /* for debugging */
      (DUMP_COOKIE, "sendby_ioCipher_block", "generated new cookie", 
       desc, thread, 0);
  }
  
  POINT_OF_RANDOM_VAR (sendbuf) ;

  /* encrypt the whole lot blockwise, place it after the thread cookie */
  for (processed = 0; processed < len; processed += thread->blen)
    cbc_encrypt (thread, sendbuf + start + processed, plainbuf + processed) ;

  /* do some post processing and clean up */
  post_process_sender_cmds (desc, thread, cmd, plainbuf) ;

  POINT_OF_RANDOM_VAR (plainbuf) ;

  SEND_TEXT ("sendby_ioCipher_block", data_start, cnt);

  /* send that stuff including the thread cookie */
  for (processed = 0, len += start; len > 0; processed += n, len -= n) 
    if ((n = OS_SEND (desc->fd, sendbuf + processed, len, flags)) <= 0) {
      if (n < 0 && errno == EINTR) {
	RECV_NOTIFY /* for debugging */
	  (DUMP_ERROR, "recfrom_ioCipher_block", 
	   "got signal while sending data packets", desc, thread, 0);
	n = 0;
	continue ;
      }
      DEALLOCA (plainbuf);
      DEALLOCA  (sendbuf);
      return n;
    } else
      desc->total += n ; /* accounting */

  POINT_OF_RANDOM_STACK (3) ;

  SEND_DRAIN_DELAY (); /* for debugging */
  DEALLOCA (plainbuf);
  DEALLOCA  (sendbuf);
  return cnt ;
}


static int
sendby_ioCipher
  (void      *c,
   char    *msg,
   unsigned len,
   int    flags)
{
  ioCipher *desc = c;
  int n, bytes = 0 ;

  /* get the current thread */
  cipher_state * c_stat = get_current_sender_thread (desc) ;

  if (c_stat == 0) {
    errno = errnum (CBC_NOSUCH_THREADID) ;
    return -1 ;
  }

  POINT_OF_RANDOM_STACK (9) ;

  /* test, whether we need to change the session key while we can send an
     embedded command (if the automatic change feature is enabled, at all) */
  if ((desc->sender->bitlist & EXEC_COMMAND) == 0 && c_stat->key_sched) {
    
    time_t now = time (0) ;
    unsigned next ;

    /* check for a time wrap, all we do here is forcing a key change */
    if (desc->sender->time_adjust > now) {
      cipher_thread *t = desc->thread;
      desc->sender->time_adjust = 
	desc->state.next_change = now ;
      while (t) {
	t->state.next_change = now ;
	t = t->next ;
      }
      POINT_OF_RANDOM_VAR (t) ;
    }

    if (now >= c_stat->next_change) {

      SEND_NOTIFY /* for debugging */
	(DUMP_KEYS, "sendby_ioCipher", "forcing a key change", desc, 0, 0);
      
      /* generate random key, make embedded command */
      prime_random_bytes (desc->sender->buf, desc->state.cipher->keylen) ;
      desc->sender->exe      = EXEC_CHANGE_SESSION_KEY ;
      desc->sender->bitlist |= EXEC_COMMAND ;

      POINT_OF_RANDOM_STACK (5) ;

      /* set time for the next key change */
      fast_random_bytes ((void *)&next, sizeof (next)) ;
      c_stat->next_change = now + (next % c_stat->key_sched);
      if (c_stat->next_change < desc->sender->time_adjust)
	c_stat->next_change = desc->sender->time_adjust ;
    }
  }
    
  /* need to fragment ? */
  while (len > desc->maxblock) {
    if ((n = sendby_ioCipher_block 
	 (desc, c_stat, msg, desc->maxblock, flags)) < 0)
      return n ;
    bytes         += n ;
    desc->payload += n ; /* accounting */
    len -= desc->maxblock ;
  }

  POINT_OF_RANDOM_VAR (msg) ;
  
  if ((n = sendby_ioCipher_block (desc, c_stat, msg, len, flags)) < 0)
    return n ;
  desc->payload += n ; /* accounting */

  POINT_OF_RANDOM_VAR (c_stat) ;

  return bytes + n ;
}

/* ---------------------------------------------------------------------------- *
 *                   private functions: io layer management                     *
 * ---------------------------------------------------------------------------- */

/* The sender stores a new session key. If no argument is given, the session key
   is generated, automatically. */ 
    
static int
activate_session_key
  (ioCipher *desc,
   int       *arg)
{
  cipher_state *stat ;
  int n;

  POINT_OF_RANDOM_VAR (arg) ;

  if (!is_sender (desc)) {
    errno = errnum (CBC_CTL_SENDER_ONLY) ;
    return -1;
  }
  /* get the current protocol thread */
  if ((stat = get_current_sender_thread (desc)) == 0) { 
    /* should not happen */
    errno = errnum (CBC_NOSUCH_THREADID) ;
    return -1;
  }

  POINT_OF_RANDOM_VAR (stat) ;

  if (arg == 0) { /* change now ? */

    desc->sender->exe      = EXEC_CHANGE_SESSION_KEY ;
    desc->sender->bitlist |= EXEC_COMMAND ;

    /* generate random key */
    prime_random_bytes (desc->sender->buf, desc->state.cipher->keylen) ;
    
    /* send that key to the receiver */
    if (sendby_ioCipher_block (desc, stat, 0, 0, 0) < 0) 
      return -1 ;

    return 0 ;
  }

  POINT_OF_RANDOM_STACK (7) ;

  if (*arg > 0) {
    /* set the time when the next change is due */
    time_t now = time (0) ;
    n = stat->next_change ;
    stat->next_change = now + (*arg) ;
    return n < now ? 0 : now - n ;
  }
  
  /* set key schedule */
  n = stat->key_sched ;
  stat->key_sched = - (*arg) ;
  return n ;
}



/* Resizing a buffer is actively done on the sender, and passed to the client 
   as an embedded command.  If the receiver is passed the resizing command, it 
   imposes a limit value for the next buffer change to be received */
  
static int
resize_iobuffer
  (ioCipher *desc,
   int       *arg)
{
  unsigned len ;

  POINT_OF_RANDOM_VAR (arg) ;

  if (arg == 0)
    return desc->maxblock ;

  /* adjust automatically to something sensible */
  if ((len = *arg) > CBC_IO_BUFLEN_MAX)
    len = CBC_IO_BUFLEN_MAX ;
  else
    if (len < CBC_IO_BUFLEN_MIN)
      len = CBC_IO_BUFLEN_MIN ;

  /* the command is activated only on the sender side, the receiver
     side sees the resizing argument as a limit when the request comes
     through the cannel */
  
  POINT_OF_RANDOM_STACK (9) ;

  if (!is_sender (desc)) 
    return desc->cache->limit_maxblock = len ;

  desc->sender->bitlist |= CHANGE_BUFFER_SIZE ;

  return desc->sender->maxblock = len ;
}



/* A thread implies an independent instance of the sender descriptor.  
   There is always just one active thread.  The thread ID is the return 
   code passwd back from the registry service (unless negative.) */

static int
register_thread
  (ioCipher *desc,
   int    *unused)
{
  cipher_thread    *t ;
  cipher_state  *stat ;
  cipher_command *cmd ;
  unsigned    pid, id ;

  POINT_OF_RANDOM_VAR (t) ;

  if (!is_sender (desc)) {
    errno = errnum (CBC_CTL_SENDER_ONLY) ;
    return -1;
  }
  /* get the current protocol thread */
  if ((stat = get_current_sender_thread (desc)) == 0) { 
    /* should not happen */
    errno = errnum (CBC_NOSUCH_THREADID) ;
    return -1;
  }
  POINT_OF_RANDOM_VAR (unused) ;

  /* get pid */
  if ((pid = desc->synthetic_pid) == 0)
    pid = getpid () ;

  POINT_OF_RANDOM_VAR (cmd) ;

  /* create a new thread id on the receiver */
  cmd = desc->sender ;
  id = (pid << CHAR_BIT) + (++ cmd->id_counter) ;
  
  /* set up a command structure to be sent */
  cmd->bitlist |= EXEC_COMMAND ;
  cmd->exe      = EXEC_CREATE_THREAD ;
  ushort2buffer (cmd->buf, id);

  /* make a cookie (+ 6 bytes padding) */
  fast_random_bytes (cmd->buf + 2, 14) ;

  /* consider the situation, that a forking process has the same
     context but a different process id, so we make the cookie 
     depend on the pid and the current time */
  XCRCFIRST (desc->state.frame, cmd->buf + 2, 14);
  XCRCNEXT  (desc->state.frame, (char*)&pid, sizeof (pid)) ;
#ifdef HAVE_GETTIMEOFDAY
  { struct timeval tv ; gettimeofday (&tv, 0);
  XCRCNEXT (desc->state.frame, (char*)&tv, sizeof (tv)); }
#else
  { time_t ti = time (0);
  XCRCNEXT (desc->state.frame, (char*)&ti, sizeof (ti)); }
#endif
  memcpy (cmd->buf + 2, XCRCRESULT0 (desc->state.frame), 8) ;
  
  SEND_NOTIFY /* for debugging */
    (DUMP_CONTROL, "register_thread", "thread generation request", 
     desc, 0, id);

  /* send that registration request to the receiver */
  if (sendby_ioCipher_block (desc, stat, 0, 0, 0) < 0) {
    SEND_NOTIFY /* for debugging */
      (DUMP_ERROR, "register_thread", "Error: cannot register on receiver", 
       desc, 0, 0);
    return -1 ;
  }

  POINT_OF_RANDOM_STACK (5) ;
  
  /* duplcate the current state */
  t = duplicate_thread (stat) ;
  t->state.thread_id   =  id ;
  t->state.creator_pid = pid ;

  POINT_OF_RANDOM_VAR (pid) ;

  /* we sent the cookie, assign it now */
  memcpy (t->state.cookie, cmd->buf + 2, 8) ;

  SEND_NOTIFY /* for debugging */
    (DUMP_THREAD, "register_thread", "thread has been generated", 
     desc, &t->state, 0);

  POINT_OF_RANDOM_VAR (t) ;

  /* push that thread */
  t->next      = desc->thread ;
  desc->thread = t ;
  desc->act_threads ++ ;

  return t->state.thread_id ;
}



static int
activate_thread_id
  (ioCipher *desc,
   int       *arg)
{
  unsigned id, pid ;

  POINT_OF_RANDOM_VAR (arg) ;

  if (!is_sender (desc)) {
    errno = errnum (CBC_CTL_SENDER_ONLY) ;
    return -1;
  }

  if (arg == 0)
    return desc->sender->active_thread ;

  SEND_NOTIFY /* for debugging */
    (DUMP_CONTROL, "activate_thread_id", "thread activation request", 
     desc, 0, *arg);

  POINT_OF_RANDOM_VAR (id) ;

  /* get pid */
  if ((pid = desc->synthetic_pid) == 0)
    pid = getpid () ;

  if ((id = *arg) == 0) {
    desc->sender->active_thread = 0 ;
    desc->state.creator_pid = pid ;
  } else {

    /* move the thread to be activated onto the top of the queue */
    if (desc->thread == 0 || /* not allowed, creates error condition, below */
	desc->thread->state.thread_id != id) {
      cipher_thread *t = unlink_thread (desc, id, thread_ptr_by_id) ;
      if (t == 0)
	return -1;
      t->next      = desc->thread ;
      desc->thread = t ;
    }

    POINT_OF_RANDOM_STACK (11) ;

    /* register the process id */
    if ((desc->thread->state.creator_pid = desc->synthetic_pid) == 0)
      desc->thread->state.creator_pid = pid ;
    
    desc->sender->active_thread = id ;
  }
  
  SEND_NOTIFY /* for debugging */
    (DUMP_THREAD, "activate_thread_id", "thread activation", desc, 0, *arg);
  
  return desc->sender->active_thread ;
}



static int
unlink_thread_id
  (ioCipher *desc,
   int       *arg)
{
  int id = (arg == 0) ? desc->sender->active_thread : *arg ;

  POINT_OF_RANDOM_VAR (arg) ;

  if (!is_sender (desc)) {
    errno = errnum (CBC_CTL_SENDER_ONLY) ;
    return -1;
  }

  SEND_NOTIFY /* for debugging */
    (DUMP_THREAD, "unlink_thread_id", "thread unlink request", desc, 0, id);

  if (destroy_thread (desc, id, thread_ptr_by_id) < 0)
    return -1 ;
  desc->act_threads -- ;

  POINT_OF_RANDOM_STACK (11) ;

  return id ;
}



static int
unlink_thread_pid
  (ioCipher *desc,
   int       *arg)
{
  pid_t pid ;
  unsigned n = 0 ;

  POINT_OF_RANDOM_VAR (arg) ;

  if (!is_sender (desc)) {
    errno = errnum (CBC_CTL_SENDER_ONLY) ;
    return -1;
  }
  if (arg == 0) {
    if ((pid = desc->synthetic_pid) == 0) 
      pid = getpid () ;
  } else
    pid = *arg ;

  SEND_NOTIFY /* for debugging */
    (DUMP_CONTROL, "unlink_thread_pid", "thread PID unlink request",
     desc, 0, pid);

  while (destroy_thread (desc, pid, thread_ptr_by_pid) >= 0)
    (n ++, desc->act_threads -- ) ;

  POINT_OF_RANDOM_STACK (7) ;

  SEND_NOTIFY /* for debugging */
    (DUMP_THREAD, "unlink_thread_pid", "thread PIDs unlinked", desc, 0, n);

  return n ;
}



static int
destroy_thread_id
  (ioCipher *desc,
   int       *arg)
{
  cipher_state *stat ;
  int id = (arg == 0) ? desc->sender->active_thread : *arg ;

  POINT_OF_RANDOM_VAR (arg) ;

  if (!is_sender (desc)) {
    errno = errnum (CBC_CTL_SENDER_ONLY) ;
    return -1;
  }
  /* get the current protocol thread */
  if ((stat = get_current_sender_thread (desc)) == 0) { 
    /* should not happen */
    errno = errnum (CBC_NOSUCH_THREADID) ;
    return -1;
  }

  SEND_NOTIFY /* for debugging */
    (DUMP_THREAD, "destoy_thread_id", "thread destroy request", 
     desc, 0, id);

  /* set up a command structure to be sent, later on */
  desc->sender->bitlist |= EXEC_COMMAND ;
  desc->sender->exe      = EXEC_DESTROY_THREAD ;
  ushort2buffer  (desc->sender->buf, id) ;
  fast_random_bytes (desc->sender->buf + 2, 14) ;

  /* send that destruction request to the receiver */
  if (sendby_ioCipher_block (desc, stat, 0, 0, 0) < 0) 
    return -1 ;

  POINT_OF_RANDOM_STACK (7) ;

  if (destroy_thread (desc, id, thread_ptr_by_id) < 0)
    return -1 ;
  desc->act_threads -- ;
  return id ;
}


static void
destroy_all_notifying_threads
  (void *c)
{
  ioCipher *desc = c;

  POINT_OF_RANDOM_STACK (5) ;

  while (desc->thread != 0) {
    int id = desc->thread->state.thread_id ;
    destroy_thread_id (desc, &id) ;
  }
  destroy_ioCipher_links (desc) ;

  POINT_OF_RANDOM_STACK (3) ;
}


static int
destroy_thread_pid
  (ioCipher *desc,
   int       *arg)
{
  unsigned n = 0;

  pid_t pid ;
  cipher_thread **T ; 

  POINT_OF_RANDOM_VAR (arg) ;

  if (!is_sender (desc)) {
    errno = errnum (CBC_CTL_SENDER_ONLY) ;
    return -1;
  }
  if (arg == 0) {
    if ((pid = desc->synthetic_pid) == 0) 
      pid = getpid () ;
  } else
    pid = *arg ;

  POINT_OF_RANDOM_VAR (T) ;

  SEND_NOTIFY /* for debugging */
    (DUMP_THREAD, "destoy_thread_pid", "thread PID destroy request", 
     desc, 0, pid);
  
  /* destroy id wise */
  while ((T = thread_ptr_by_pid (desc, pid)) != 0) {
    int id = (*T)->state.thread_id ;
    if (destroy_thread_id (desc, &id) < 0)
      return -1;
    n ++ ;
  }

  POINT_OF_RANDOM_STACK (9) ;

  SEND_NOTIFY /* for debugging */
    (DUMP_THREAD, "destoy_thread_pid", "thread PIDs destroyed", desc, 0, n);

  return n ;
}


static int
set_destroy_flag
  (ioCipher *desc,
   int       *arg)
{
  int n ;
  
  POINT_OF_RANDOM_VAR (arg) ;
  
  if (is_sender (desc)) {
    errno = errnum (CBC_CTL_RECV_ONLY) ;
    return -1;
  }

  POINT_OF_RANDOM_VAR (n) ;
  
  n = desc->public_destroy ;
  if (arg != 0)
    desc->public_destroy = (*arg != 0) ;
  return n;
}


static int
set_synthetic_pid
  (ioCipher *desc,
   int       *arg)
{
  pid_t n, pid = (arg == 0) ? getpid () : *arg ;

  POINT_OF_RANDOM_VAR (arg) ;
  
  if (!is_sender (desc)) {
    errno = errnum (CBC_CTL_SENDER_ONLY) ;
    return -1;
  }
  
  SEND_NOTIFY /* for debugging */
    (DUMP_CONTROL, "set_synthetic_pid", "activating a synthetic PID", 
     desc, 0, pid);
 
  n = desc->synthetic_pid ;
  desc->synthetic_pid = pid ;

  POINT_OF_RANDOM_STACK (3) ;

  return n;
}


static int
set_max_threads
  (ioCipher *desc,
   int       *arg)
{
  unsigned n ;
  
  if (is_sender (desc)) {
    errno = errnum (CBC_CTL_RECV_ONLY) ;
    return -1;
  }

  POINT_OF_RANDOM_VAR (arg) ;
  
  n = desc->cache->max_threads ;
  if (arg != 0)
    if (*arg > CBC_THREADS_LIMIT)
      desc->cache->max_threads = CBC_THREADS_LIMIT ;
    else
      if (*arg >= desc->act_threads) 
	desc->cache->max_threads = *arg ;
      else {
	errno = errnum (CBC_NO_MORE_THREADS) ;
	return -1;
      }
  
  RECV_NOTIFY /* for debugging */
    (DUMP_CONTROL, "set_max_threads", "maximum threads request", 
     desc, 0, n);

  POINT_OF_RANDOM_STACK (7) ;

  return n ;
}



static int
flush_cache
  (ioCipher *desc,
   int       *arg)
{
  unsigned n ;

  POINT_OF_RANDOM_VAR (arg) ;

  if (is_sender (desc)) {
    errno = errnum (CBC_CTL_RECV_ONLY) ;
    return -1;
  }
  n = desc->cache->fill ;
  desc->cache->fill =
    desc->cache->start = 0; 

  POINT_OF_RANDOM_STACK (3) ;

  return n;
}


static int
io_control
  (void                 *c,
   io_ctrl_request request,
   int                *arg)
{
  ioCipher *desc = c;
  unsigned n ;

  POINT_OF_RANDOM_STACK (3) ;
  
  switch (request) {

  case IO_RESIZE_BUF:         return resize_iobuffer      (desc, arg) ;
  case IO_MAX_THREADS:        return set_max_threads      (desc, arg) ;
  case IO_CHANGE_KEY:         return activate_session_key (desc, arg) ;
  case IO_REGISTER_THREAD:    return register_thread      (desc, arg) ;
  case IO_ACTIVATE_THREAD:    return activate_thread_id   (desc, arg) ;
  case IO_DESTROY_THREAD:     return destroy_thread_id    (desc, arg) ;
  case IO_DESTROY_THREAD_PID: return destroy_thread_pid   (desc, arg) ;
  case IO_UNLINK_THREAD:      return unlink_thread_id     (desc, arg) ;
  case IO_UNLINK_THREAD_PID:  return unlink_thread_pid    (desc, arg) ;
  case IO_FLUSH_CACHE:        return flush_cache          (desc, arg) ;
  case IO_SYNTHETIC_PID:      return set_synthetic_pid    (desc, arg) ;
  case IO_PUBLIC_DESTROY:     return set_destroy_flag     (desc, arg) ;
  
  case IO_TOTAL_COUNTER:
    n = desc->total ;
    POINT_OF_RANDOM_STACK (2) ;
    if (arg != 0) desc->total = *arg ;
    return n;

  case IO_PAYLOAD_COUNTER:
    n = desc->payload ;
    if (arg != 0) desc->payload = *arg ;
    POINT_OF_RANDOM_STACK (3) ;
    return n;

  case IO_EOF_STATE:
    POINT_OF_RANDOM_STACK (1) ;
    if (is_sender (desc)) { errno = errnum (CBC_CTL_RECV_ONLY) ; return -1; }
    n = (desc->cache->eof != 0) ;
    if (arg != 0) desc->cache->eof = (*arg != 0) ;
    return n;

  case IO_STOPONEMPTY_STATE:
    if (is_sender (desc)) { errno = errnum (CBC_CTL_RECV_ONLY) ; return -1; }
    n = (desc->cache->stop_on_empty != 0) ;
    POINT_OF_RANDOM_STACK (2) ;
    if (arg != 0) desc->cache->stop_on_empty = (*arg != 0) ;
    return n;
  }
  errno = errnum (CBC_UNKNOWN_CTL) ;
  return -1 ;
}


/* ---------------------------------------------------------------------------- *
 *                   pivate functions: io layer management                      *
 * ---------------------------------------------------------------------------- */
 
static int
cbc_initialize_any
  (ioCipher     *layer,
   unsigned         fd,
   cipher_desc *cipher,
   frame_desc   *frame,
   char      chain [4])
{
  if (!supported_block_length (cipher->blocklen)) {
    errno = errnum (CBC_UNSUPPORTED_BLEN) ;
    return -1;
  }

  POINT_OF_RANDOM_STACK (7) ;

  if (cipher->keylen > 16) {
    errno = errnum (CBC_UNSUPPORTED_KLEN) ;
    return -1;
  }

  /* install cbc function initialized cbc buffer */
  layer->state.cipher = cipher ;
  layer->state.blen   = cipher->blocklen ;
  layer->state.block  = XMALLOC (layer->state.blen) ;

  POINT_OF_RANDOM_VAR (fd) ;
  
  /* install frame */
  layer->state.frame = frame ;
  memcpy (layer->state.chain, chain, 4) ;
  
  /* set maximal block length */
  layer->maxblock = CBC_IO_BUFLEN ;

  /* save current file handle */
  layer->fd = fd ;

  POINT_OF_RANDOM_STACK (2) ;

  return 0;
}


static int
cbc_initialize_receiver
  (void       *context,
   unsigned         fd,
   cipher_desc *cipher,
   frame_desc   *frame,
   char      chain [4])
{
  ioCipher *layer = context;

  POINT_OF_RANDOM_STACK (1) ;

  if (cbc_initialize_any (layer, fd, cipher, frame, chain) < 0)
    return -1;

  /* is receiver, install input cache */
  layer->cache = XMALLOC (sizeof (cipher_iocache) + 2*CBC_IO_BUFLEN - 1);
  layer->cache->dim = 2*CBC_IO_BUFLEN ;

  POINT_OF_RANDOM_STACK (3) ;

  /* limit the maximal number of threads (apart from the main stream) */
  layer->cache->max_threads = CBC_THREADS_MAX ;

  return 0 ;
}

static int
cbc_initialize_sender
  (void       *context,
   unsigned         fd,
   cipher_desc *cipher,
   frame_desc   *frame,
   char      chain [4])
{
  ioCipher *layer = context;

  POINT_OF_RANDOM_STACK (3) ;

  if (cbc_initialize_any (layer, fd, cipher, frame, chain) < 0)
    return -1;

  /* allocate the command buffer */
  layer->sender = XMALLOC (sizeof (cipher_command)) ;

  /* store the current time, needed to face the 1/19/2038 problem */
  layer->sender->time_adjust = time (0) ;
  layer->state.key_sched     = CBC_KEYTTL_SECS ;

  POINT_OF_RANDOM_STACK (5) ;
  
  return 0 ;
}

/* ---------------------------------------------------------------------------- *
 *                   public functions: call back functions                      *
 * ---------------------------------------------------------------------------- */


char *
cbc_get_info 
  (unsigned is_sender,
   size_t *contextsize,
   int (**cbc_open) (void *, unsigned, cipher_desc*, frame_desc*, char[4]),
   int (**cbc_rdwr) (void *, char *, unsigned, int),
   int (**cbc_ioctl)(void *, io_ctrl_request, int*),
   void(**cbc_close)(void *))
{
  if (is_sender) {
    *cbc_open  = cbc_initialize_sender ;
    *cbc_rdwr  = sendby_ioCipher  ;
    *cbc_close = destroy_all_notifying_threads ;
  } else {
    *cbc_open  = cbc_initialize_receiver ;
    *cbc_rdwr  = recfrom_ioCipher  ;
    *cbc_close = destroy_ioCipher_links ;
  }

  *contextsize = sizeof (ioCipher) ;
  *cbc_ioctl   = io_control ;

  return is_sender ? "cbc-send" : "cbc-recv" ;
}
