/*
 *          Copyright (c) mjh-EDV Beratung, 1996-1998
 *     mjh-EDV Beratung - 63263 Neu-Isenburg - Rosenstrasse 12
 *          Tel +49 6102 328279 - Fax +49 6102 328278
 *                Email info@mjh.teddy-net.com
 *
 *   Author: Jordan Hrycaj <jordan@mjh.teddy-net.com>
 *
 *   $Id: iostream.c,v 1.10 1999/01/31 12:36:43 jordan Exp $
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 *   This module sets a generic layer on top of the standard io functions
 *   like send, recv, read and write.  It replaces the channel numbers by
 *   a generic descriptor and calls the approriate io functions.
 */

#include "common-stuff.h"

#ifdef HAVE_SYS_SOCKET_H
# include <sys/socket.h>
#else
# include <socket.h> /* ????? */
#endif

#include "messages.h"

#define __IOSTREAM_PRIVATE__ /* prevent from using send/recv auto-replaces */
#include "iostream.h"

typedef
struct _io_desc {
  void    *desc_ptr ;
  unsigned disabled ;
  int  (*fn) (void *, char *msg, unsigned len, int flags) ;
  int  (*ctl)(void *, io_ctrl_request, int *arg);
  void (*df) (void *);
} io_desc ;


typedef
struct _io_table {
  io_desc rd ;
  io_desc wr ;
} io_table ;


static io_table *rw_table ;
static unsigned  rw_table_dim = 0 ;

/* ---------------------------------------------------------------------------- *
 *                      debugging                                               *
 * ---------------------------------------------------------------------------- */

#ifdef DUMP_IOLAYER /* for debugging */
int dump_send = 0, dump_recv = 0, iodebug_flags = 0xff;

#if 0
#undef DUMP_SENDER
#undef DUMP_READER
#endif
#endif

#if defined (DUMP_READER) || defined (DUMP_SENDER)
#include "baseXX.h"
#include "peks.h"

static void
_debug_notify 
  (unsigned      line,
   char           *fn,
   char         *type,
   char         *info,
   int             fd)
{
  if (type == 0) type =  "pid" ;
  if   (fn == 0)   fn = "info" ;
  if (info == 0) info =     "" ;
  fprintf (stderr, "DEBUG(%s=%u.%04u): fd=%d %s(%s).\n",
	   type, getpid (), line, fd, info, fn);
}

static void
_debug_text
  (unsigned   line,
   char        *fn,
   char      *type,
   char      *info,
   int         len,
   int          fd)
{
  unsigned char *s, *t;

  if (type == 0) type =  "pid" ;
  if   (fn == 0)   fn = "info" ;
  if (info == 0) {info = "*" ; len = 1; }

  if (len < 0) {
    sprintf (s = ALLOCA (400), "<error %u: %s>", errno, peks_strerr (errno));
  } else {
    memcpy (s = ALLOCA (len + 1), info, len);
    while (len > 0 && s [len-1] < ' ') len -- ;
    s [len] = '\0';
  }
  for (t = s; len --; t++) if (*t < ' ' || *t > 127) *t = '.' ;
  fprintf (stderr, "DEBUG(%s=%u.%04u): %s fd=%d data=%s.\n", 
	   type, getpid (), line, fn, fd, s);
  DEALLOCA (s);
}
#endif

#if defined (DUMP_SENDER)
static void
_send_notify 
  (unsigned      line,
   char           *fn,
   char         *info,
   int             fd)
{
  _debug_notify (line, fn, "send", info, fd);
}

static void
_send_text
  (unsigned   line,
   char        *fn,
   char      *info,
   int         len,
   int          fd)
{
  _debug_text (line, fn, "send", info, len, fd);
}

#define SEND_NOTIFY(f,i,n)   _send_notify (__LINE__, f, i, n)
#define SEND_TEXT(  f,i,l,n) _send_text   (__LINE__, f, i, l, n)
#endif


#if defined (DUMP_READER)   
static void
_recv_notify 
  (unsigned      line,
   char           *fn,
   char         *info,
   int             fd)
{
  _debug_notify (line, fn, "recv", info, fd);
}

static void
_recv_text
  (unsigned   line,
   char        *fn,
   char      *info,
   int         len,
   int          fd)
{
  _debug_text (line, fn, "recv", info, len, fd);
}

#define RECV_NOTIFY(f,i,n)   _recv_notify (__LINE__, f, i, n)
#define RECV_TEXT(  f,i,l,n) _recv_text   (__LINE__, f, i, l, n)
#endif

/* --------------------------------------------------------------------------- */

#ifndef SEND_NOTIFY
#define SEND_NOTIFY(f,i,n)
#endif
#ifndef SEND_TEXT
#define SEND_TEXT(  f,i,l,n)
#endif
#ifndef RECV_NOTIFY
#define RECV_NOTIFY(f,i,n)
#endif
#ifndef RECV_TEXT
#define RECV_TEXT(  f,i,l,n)
#endif

/* ---------------------------------------------------------------------------- *
 *                      private helpers                                         *
 * ---------------------------------------------------------------------------- */

#ifndef XREALLOC
static void *
xrealloc (void *p, unsigned n)
{
  void *q = realloc (p, n) ;

  if (q == 0) {
    fprintf (stderr, "Could not re-allocate to %u bytes !\n", n);
    exit (1);
  }
  
  return q ;
}

#define XREALLOC_DOES_NOT_INITALIZE
#define XREALLOC(p,n)	xrealloc (p, n)
#endif

/* ---------------------------------------------------------------------------- *
 *                      private functions: layer management                     *
 * ---------------------------------------------------------------------------- */

static io_table *
get_io_entry
 (unsigned fd)
{
  /* make table entry fd in rw_table [] */
  if (fd >= rw_table_dim) {
    if (rw_table == 0) {
      rw_table = XMALLOC ((fd+1) * sizeof (io_table)) ;
    } else {
      rw_table = XREALLOC (rw_table, (fd+1) * sizeof (io_table)) ;
#     ifdef XREALLOC_DOES_NOT_INITALIZE
      memset (rw_table + rw_table_dim, 0, (fd+1 - rw_table_dim) * sizeof (io_table)) ;
#     endif
    }
    /* set to new dimension */
    rw_table_dim = fd + 1;
  }

  return rw_table + fd ;
}


static void
readjust_table_dim 
  (void)
{
  unsigned n = rw_table_dim ;

  /* find least index so that all entries above are unused */
  while (n --  &&
	 rw_table [n].rd.desc_ptr == 0 &&
	 rw_table [n].wr.desc_ptr == 0)
    ;
  
  /* so rw_table [n] is the first one, used */
  if (++ n < rw_table_dim) {
    if (n == 0) {
      XFREE (rw_table) ;
      rw_table = 0 ;
    } else
      rw_table = XREALLOC (rw_table, n * sizeof (io_table)) ;
    rw_table_dim = n ;
  }
}


static void
pop_any
  (io_desc  *d)
{
  if (d->desc_ptr != 0) {
    if (d->df != 0)
      (*d->df) (d->desc_ptr) ;
    XFREE (d->desc_ptr) ;
  }
  memset (d, 0, sizeof (io_desc)) ;
}


static void *
push_any
  (io_desc    *d,
   unsigned size,
   int  (*fn) (void *, char *, unsigned, int),
   int  (*ctl)(void *, io_ctrl_request, int*),
   void (*df) (void *),
   unsigned excl)
{
  if (d->desc_ptr != 0) {
    if (excl) {
      errno = errnum (IOST_EEXISTS) ;
      return 0 ;
    }
    pop_any (d) ;
  }
  d->fn  = fn ;
  d->df  = df ;
  d->ctl = ctl ;
  if (size == 0)
    size ++ ;
  return
    d->desc_ptr = XMALLOC (size) ;
}


/* ---------------------------------------------------------------------------- *
 *                      public functions: layer management                      *
 * ---------------------------------------------------------------------------- */

void *
io_push
  (unsigned   fd,
   unsigned size,
   int  (*fn) (void *, char *, unsigned, int),
   int  (*ctl)(void *, io_ctrl_request, int*),
   void (*df) (void *),
   unsigned   how)
{
  unsigned excl = how & IO_PUSH_EXCLUSIVELY ;
  io_table *t = get_io_entry (fd) ;

  switch (how & 3) {
  case 0:
    return push_any (& t->rd, size, fn, ctl, df, excl) ;
  case 1:
    return push_any (& t->wr, size, fn, ctl, df, excl) ;
  }
  errno = errnum (IOST_SEND_RECV_ONLY) ;
  return 0 ;
}



void
io_pop
  (unsigned  fd,
   unsigned how)
{
  if (fd < rw_table_dim) 
    switch (how) {
    case 2:
    case 0:
      pop_any (& rw_table [fd].rd) ;
      if (!how)
	return ;
    case 1:
      pop_any (& rw_table [fd].wr) ;
    }
}


int 
io_shutdown
  (unsigned  fd,
   unsigned how)
{
  io_pop (fd, how);
  return shutdown (fd, how);
}



int 
io_close
  (unsigned  fd)
{
  if (fd < rw_table_dim) {
    pop_any (& rw_table [fd].rd) ;
    pop_any (& rw_table [fd].wr) ;
    readjust_table_dim () ;
  }
  return close (fd);
}


int 
io_ctrl 
  (unsigned         fd,
   io_ctrl_request req, 
   int            *arg, 
   unsigned        how)
{
  void *desc;
  int (*ctl)(void *, io_ctrl_request, int *arg) ;
    
  switch (how) {
  case 0:
    if (fd >= rw_table_dim ||
	(desc = rw_table [fd].rd.desc_ptr) == 0) {
      errno = EBADF ;
      return -1 ;
    }
    if ((ctl = rw_table [fd].rd.ctl) == 0) {
      errno = errnum (IOST_NO_RECV_CTLFN) ;
      return -1 ;
    }
    errno = 0 ;
    return (*ctl) (desc, req, arg) ;
  case 1:
    if (fd >= rw_table_dim ||
	(desc = rw_table [fd].wr.desc_ptr) == 0) {
      errno = EBADF ;
      return -1 ;
    }
    if ((ctl = rw_table [fd].wr.ctl) == 0) {
      errno = errnum (IOST_NO_SEND_CTLFN) ;
      return -1 ;
    }
    errno = 0 ;
      return (*ctl) (desc, req, arg) ;
  }
  errno = errnum (IOST_SEND_RECV_ONLY) ;
  return -1 ;
}

void
io_enable
  (unsigned  fd,
   unsigned how)
{
  if (fd < rw_table_dim) 
    switch (how) {
    case 2:
    case 0:
      rw_table [fd].rd.disabled = 0 ;
      if (!how)
	return ;
    case 1:
      rw_table [fd].wr.disabled = 0 ;
    }
}


void
io_suspend
  (unsigned  fd,
   unsigned how)
{
  if (fd < rw_table_dim) 
    switch (how) {
    case 2:
    case 0:
      rw_table [fd].rd.disabled = 1 ;
      if (!how)
	return ;
    case 1:
      rw_table [fd].wr.disabled = 1 ;
    }
}


/* ---------------------------------------------------------------------------- *
 *                      public functions: io functions                          *
 * ---------------------------------------------------------------------------- */

int
io_send
  (unsigned     fd,
   const char *buf,
   unsigned    len,
   unsigned  flags)
{
  void *desc ;
  int (*fn) (void *, char *msg, unsigned len, int flags);
  
  SEND_TEXT ("io_send", buf, len, fd);

  if (fd < rw_table_dim &&
      rw_table [fd].wr.disabled == 0 &&
      (desc = rw_table [fd].wr.desc_ptr) != 0) {

    if ((fn = rw_table [fd].wr.fn) == 0) {
      errno = errnum (IOST_NO_SEND_IOLFN) ;
      return -1 ;
    }
  
    return (*fn) (desc, (char *)buf, len, flags);
  }
  
  return OS_SEND (fd, (char *)buf, len, flags) ;
}


int
io_recv
  (unsigned    fd,
   char      *buf,
   unsigned   len,
   unsigned flags)
{
  void *desc ;
  int n, (*fn) (void *, char *msg, unsigned len, int flags);

  if (fd < rw_table_dim &&
      rw_table [fd].rd.disabled == 0 &&
      (desc = rw_table [fd].rd.desc_ptr) != 0) {

    if ((fn = rw_table [fd].rd.fn) == 0) {
      errno = errnum (IOST_NO_RECV_IOLFN) ;
      RECV_TEXT ("io_recv", buf, -1, fd);
      return -1 ;
    } 

    n =  (*fn) (desc, buf, len, flags);

  } else
    
    n = OS_RECV (fd, buf, len, flags) ;

  
  RECV_TEXT ("io_recv", buf, n, fd);

  return n;
}
