/*
 *          Copyright (c) mjh-EDV Beratung, 1996-1998
 *     mjh-EDV Beratung - 63263 Neu-Isenburg - Rosenstrasse 12
 *          Tel +49 6102 328279 - Fax +49 6102 328278
 *                Email info@mjh.teddy-net.com
 *
 *   Author: Jordan Hrycaj <jordan@mjh.teddy-net.com>
 *
 *   $Id: peks-client.c,v 1.10 1999/01/31 12:12:05 jordan Exp $
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   PEKS - private exponent key stuff for Diffie Hellman and El Gamal
 */

#include "common-stuff.h"

#ifdef HAVE_SYS_SOCKET_H
# include <sys/socket.h>
#else
# include <socket.h> /* ????? */
#endif

#define __IOSTREAM_PRIVATE__ /* prevent from using send/recv auto-replaces */
#include "peks-internal.h"
#include "crandom.h"
#include "messages.h"
#include "cbc-frame.h"
#include "baseXX.h"
#include "version.h"
#include "rnd-pool.h"


/* ---------------------------------------------------------------------------- *
 *                  private helpers                                             *
 * ---------------------------------------------------------------------------- */

static peks_key *
peks_client_setup
  (const char       *in,
   const char     *host,
   const char     *file,
   int abort_on_new_key)
{
  peks_key *key ;
  int n ;

  if ((key = accept_public_key_str (in)) == 0) {
    errno = errnum (ACCEPT_PUBKEY_ERR) ;
    return 0;
  }
  
  POINT_OF_RANDOM_STACK (7);

  /* check whether we know the server key, already */
  if (host != 0 && file != 0) {
    n = check_peks_server_key (key, host, file) ;
    if (n < 0 || (n > 0 && abort_on_new_key)) {
      end_peks_key (key) ;
      return 0 ;
    }
  }

  /* update the key structure for sending */
  update_accepted_public_key_str (in, key) ;

  POINT_OF_RANDOM_VAR (key);
  
  return key ;
}


static char *
peks_client_encode_receiver_key
  (peks_key      *k,
   const char    *s,
   unsigned     len,
   const char *type)
{
  char *t, *sk, *sl ;

  POINT_OF_RANDOM_STACK (9);

  if (len == 0)
    len = strlen (s) ;
  if (type == 0)
    type = "" ;

  /*
   * make the public peks encrypted receiver session key as
   *
   *	s [] := ("key:" <base64-key-bits> <cipher-type>)
   */
  sk = bin2base64 (s, len) ;
  sl  = ALLOCA (8 + strlen (sk) + strlen (type));
  sprintf (sl, "key: %s %s\n", sk, type) ;
  POINT_OF_RANDOM_VAR (sk);
  XFREE (sk) ;

  t = make_elg_response_key_str (k, sl, strlen (sl), 0) ;
  POINT_OF_RANDOM_VAR (sl);
  DEALLOCA (sl) ;

  POINT_OF_RANDOM_VAR (t);
  return t ;
}


static char *
peks_client_unwrap_sender_key
  (unsigned    *N,
   const char **T,
   const char *in)
{
  char *s, *sk, *sl;
  unsigned dummy ;

  if (N == 0)
    N = &dummy ;

  POINT_OF_RANDOM_VAR (in);

  /* must be editbale in order to use strtok () */
  s = ALLOCASTRDUP (in) ;

  /* parse the sender session key from the server:
   *
   *	key: <base64-key-bits> <cipher-type>
   */

  if ((sk = strtok (s, PEKS_FS)) == 0 ||
      strcmp (sk, "key:")         != 0 ||
      (sk = strtok  (0, PEKS_FS)) == 0 ||
      (sl = strtok  (0, PEKS_FS)) == 0) {
    DEALLOCA (s);
    errno = errnum (DECODE_SRVKEY_ERR) ;
    return 0 ;
  }

  POINT_OF_RANDOM_VAR (s);

  /* extract the binary key */
  sk = base64toBin (sk, N) ;

  /* dup the cypher type */
  if (T != 0)
    *T = XSTRDUP (sl) ;

  POINT_OF_RANDOM_VAR (sk);

  /* clean up and return */
  DEALLOCA (s);
  return sk;
}


static int
push_client_io_layer
  (unsigned        fd,
   const char   *what,
   const char    *key,
   unsigned       len,
   unsigned is_sender)

{
  void *cbc, *cc, *fc ;
  cipher_desc *cp ;
  frame_desc  *fp ;

  size_t c_size ;
  int  (*c_open) (void *, unsigned, cipher_desc*, frame_desc*, char[4]) ;
  int  (*c_rdwr) (void *, char *, unsigned, int);
  int  (*c_ioctl)(void *, io_ctrl_request, int*);
  void (*c_close)(void *) ;

  /* get new cipher and frame classes to be used, here */
  if (find_classes_by_name (&cc, &fc, what) == 0)
    return -1;

  /* create an cipher instance and a frame */
  if ((cp = (*(is_sender?create_encryption:create_decryption)) (cc, key, len)) < 0)
    return -1;
  fp = create_frame (fc, key [len ? len-1 : 0]) ;
  
  /* push that new io layer */
  cbc_get_info (is_sender, &c_size, &c_open, &c_rdwr, &c_ioctl, &c_close);
  if ((cbc = io_push (fd, c_size, c_rdwr, c_ioctl, c_close, is_sender)) == 0) {
    destroy_cipher (cp);
    destroy_frame  (fp);
    return -1;
  }
  
  /* initialize that io layer */
  return (*c_open) (cbc, fd, cp, fp, (char*)key + (len<5 ? 0 : len-5)) ;
}

/* ---------------------------------------------------------------------------- *
 *                  public key client handshake function                        *
 * ---------------------------------------------------------------------------- */

int
client_negotiate_session_key
  (const char  * cipher,
   int           socket,
   const char  * server,
   const char * keyfile)
{
  peks_key *key ;
  char *s, *u, *buf ;
  int n, l ;

  /* initialize prime random generator */
  init_random_gen ((char*)&s, sizeof (s)) ;

  /* read the public key from the server host */
  buf = ALLOCA (CBC_IO_BUFLEN_MAX+1) ;
  if ((n = OS_RECV (socket, buf, CBC_IO_BUFLEN_MAX, 0)) < 0) {
    DEALLOCA (buf) ;
    return -1 ;
  }
  buf [n] = 0 ;

  POINT_OF_RANDOM_VAR (buf);

  /* arg 4 set means: return an error if a new key has been stored */
  if ((key = peks_client_setup (buf, server, keyfile, 1)) == 0) {
    /* send a termination message to the server */
    OS_SEND (socket, ".", 2, 0) ;
    goto error_code_return;
  }

  POINT_OF_RANDOM_STACK (13);
  
  /* generate the receiver session key */
  if ((l = cipher_keylen (cipher) + 5) == 5) {
    end_peks_key (key) ;
    goto error_code_return;
  }
  prime_random_bytes (s = ALLOCA (l), l) ;
  
  /* push a new io layer for reading */
  if (push_client_io_layer (socket, cipher, s, l, 0) < 0) {
    end_peks_key (key) ;
    DEALLOCA (s);
    goto error_code_return;
  }

  POINT_OF_RANDOM_VAR (s);
  
  /*  wrap the client receiver session key and send it */
  u = peks_client_encode_receiver_key (key, s, l, cipher) ;
  end_peks_key (key) ;			/* not needed, any longer */
  DEALLOCA (s);
  if (u == 0) 
    goto error_code_return;
  OS_SEND (socket, u, strlen (u), 0);
  POINT_OF_RANDOM_VAR (u);
  XFREE (u) ;
  
  /* receive the sender session key s from the server */
  if ((n = io_recv (socket, buf, CBC_IO_BUFLEN, 0)) < 0)
    goto error_code_return;
  buf [n] = '\0' ;
  if ((s = peks_client_unwrap_sender_key (&l, (const char **)&u, buf)) == 0)
    goto error_code_return;

  POINT_OF_RANDOM_VAR (s);
    
  /* push a new io layer for sending */
  n = push_client_io_layer (socket, u, s, l, 1) ;
  XFREE (s);
  XFREE (u);
  if (n != 0)
    goto error_code_return;
  
  /* send a magig string in order to test the connection */
  if (io_send (socket, PEKS_MAGIC, strlen (PEKS_MAGIC)+1, 0) !=
      strlen (PEKS_MAGIC)+1) {
    errno = errnum (CLNT_MAGIC_FAILED) ;
    goto error_code_return;
  }
  
#if 0
  /*********************************************/
  buf = ALLOCA (1024) ;
  n = io_recv (socket, buf, 1024, 0) ;
  n = io_recv (socket, buf, 1024, 0) ;
  n = io_recv (socket, buf, 1024, 0) ;
  n = io_recv (socket, buf, 1024, 0) ;
  n = io_recv (socket, buf, 1024, 0) ;
  n = io_recv (socket, buf, 1024, 0) ;
  n = io_recv (socket, buf, 1024, 0) ;
  n = io_recv (socket, buf, 1024, 0) ;
  n = io_recv (socket, buf, 1024, 0) ;
  n = io_ctrl (socket, IO_CHANGE_KEY, 0, 1);
  /*********************************************/
#endif

  POINT_OF_RANDOM_VAR (buf);
  DEALLOCA (buf) ;
  return 0;
  /*@NOTREACHED@*/

 error_code_return:
  io_pop (socket, 2) ; /* its's save to always execute it */
  DEALLOCA (buf) ;
  return -1 ;
}
