/*
 *          Copyright (c) mjh-EDV Beratung, 1996-1998
 *     mjh-EDV Beratung - 63263 Neu-Isenburg - Rosenstrasse 12
 *          Tel +49 6102 328279 - Fax +49 6102 328278
 *                Email info@mjh.teddy-net.com
 *
 *   Author: Jordan Hrycaj <jordan@mjh.teddy-net.com>
 *
 *   $Id: peks-file.c,v 1.11 1999/01/31 12:36:45 jordan Exp $
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "common-stuff.h"

#ifdef HAVE_CTYPE_H
#include <ctype.h>
#endif

#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#elif defined (HAVE_STAT_H)
# include <stat.h>
#else
  extern int stat ();
#endif
#ifndef HAVE_LSTAT
# define lstat(x,y) stat(x,y)
#endif

#include "peks-internal.h"
#include "peks-baseXX.h"
#include "messages.h"
#include "rnd-pool.h"
#include "crandom.h"

/* the maximal line length in a key file */
#define MAX_LINE_LENGTH (5 * 1024)

/* ---------------------------------------------------------------------------- *
 *                        private functions                                     *
 * ---------------------------------------------------------------------------- */

static int
private_access
 (const char *name,
  int     strictly)
{
   struct stat stbuf;

   if (lstat (name, &stbuf))
     return -1;
#ifdef HAVE_LSTAT
   /* check for regular file or symlink */
   if (S_ISLNK(stbuf.st_mode)) {
     errno = errnum (NO_SYMLINK_WANTED) ;
     return -1;
   }
#endif
   if (S_ISREG(stbuf.st_mode) == 0) {
     errno = errnum (REG_FILE_WANTED) ;
     return -1;
   }

   POINT_OF_RANDOM_STACK (21) ;

   if (stbuf.st_mode & (S_IWGRP|S_IWOTH)) {
     errno = errnum (NOT_GROUP_WRITABLE) ;
     return -1;
   }
   if (strictly && (stbuf.st_mode & S_IROTH)) {
     errno = errnum (NOT_WORLD_READABLE) ;
     return -1;
   }
   if (strictly > 1 && (stbuf.st_mode & S_IRGRP)) {
     errno = errnum (NOT_GROUP_READABLE) ;
     return -1;
   }

   POINT_OF_RANDOM (0,0);
   return 0;
}


/* ---------------------------------------------------------------------------- *
 *                  raw public key file i/o functions                           *
 * ---------------------------------------------------------------------------- */

/*
 * parse a file linewise to find
 *
 *	<tag>: <key defs> ...
 *
 * and load that line into a key structure.
 */

peks_key *
read_peks_key
  (const char  *tag,
   strncmp_fn   cmp,
   const char *file)
{
  FILE *inp;
  unsigned len, n ;
  char read_buf [MAX_LINE_LENGTH], *s, *p ;

  /* open input file if it exists */
  if (tag == 0 || file == 0 || cmp == 0) {
    errno = errnum (FUNCTION_ARG_ERROR) ;
    return 0;
  }
  if (private_access (file, 2) < 0) {
    if (errno == ENOENT)
      errno = 0 ;
    return 0;
  }

  POINT_OF_RANDOM_STACK (13) ;

  if ((inp = fopen (file, "r")) == 0)
    return 0 ;

  /* parse input file up until the tag line */
  while (fgets (read_buf, sizeof (read_buf), inp)) {
	   
    /* skip leading spaces */
    for (s = read_buf, n = strlen (read_buf); 
	 isspace (*s) && n > 0 ; 
	 s ++, n--) 
      ;
      
    /* find end of tag */
    if ((p = strchr (s, ':')) != 0)
      len = p - s ;
    else
      len = 0 ;

    /* stop at the pattern |^<tag>: | */
    if (len &&
	s [len    ] == ':' &&
	s [len + 1] == ' ' &&
	(* cmp) (tag, s, len) == 0) {

      fclose (inp) ;
      return get_peks_key_from_str (s + len + 2) ;
    }
  }
    
  /* did not find that tag */
  errno = 0 ;
  fclose (inp);
  return 0;
}



int
save_peks_key
  (peks_key    *key,
   const char  *tag,
   strncmp_fn   cmp,
   const char *file)
{
  FILE *inp, *outp = 0;
  char read_buf [MAX_LINE_LENGTH], *new, *old, *s, *p ;
  unsigned len, n, r_code = -1 ;
  int omask ;

  if (key == 0 || tag == 0 || file == 0) {
    errno = errnum (FUNCTION_ARG_ERROR) ;
    return -1 ;
  }

  POINT_OF_RANDOM_VAR (key);

  /* check file permissions */
  if (private_access (file, 2) < 0 && errno != ENOENT)
    return -1;

  /* set private file creation mask */
  omask = umask (0177) ;

  /* open input file if it exists */
  if ((inp = fopen (file, "r")) == 0 && errno != ENOENT) {
    errno = errnum (CANT_OPEN_KEY_RFILE) ;
    goto get_ready_for_return;
  }
  if (inp != 0 && cmp == 0) {
    errno = errnum (KEY_FILE_EXISTS) ;
    goto get_ready_for_return;
  }

  POINT_OF_RANDOM_VAR (*inp) ;

  /* create output file names for "new" and "old" */
  len = strlen (file) ;
  new = ALLOCA (len + 4 + 1) ;
  strcpy (new, file);
  strcat (new, ".new") ;

  POINT_OF_RANDOM_STACK (37) ;

  old = ALLOCA (len + 4 + 1) ;
  strcpy (old, file);
  strcat (old, ".old") ;

  POINT_OF_RANDOM_VAR (new) ;

  /* create output file */
  unlink (new) ;
  if ((outp = fopen (new, "w")) == 0) {
    errno = errnum (CANT_OPEN_KEY_WFILE) ;
    goto get_ready_for_return;
  }

  if (inp != 0) {

    /* parse input file up until the tag line */
    while (fgets (read_buf, sizeof (read_buf), inp) > 0) {
      
      /* skip leading spaces */
      for (s = read_buf, n = strlen (read_buf); 
	   isspace (*s) && n > 0 ; 
	   s ++, n--) 
	;

      /* find end of tag */
      if ((p = strchr (s, ':')) != 0)
	len = p - s ;
      else
	len = 0 ;

      /* stop at the pattern |^<tag>: | */
      if (cmp != 0 && len &&
	  s [len    ] == ':' &&
	  s [len + 1] == ' ' &&
	  (* cmp) (tag, s, len) == 0) {
	break ;
      }
      
      /* check for input line overflow */
      if (s [n-1] != '\n' &&
	  s [n-1] != '\r') {
	errno = errnum (KEYFILE_PARSER_ERR) ;
	goto get_ready_for_return;
      }

      /* copy to the output file */
      fputs (read_buf, outp) ;
    }
  } /* if (inp != 0) */ 
  
  POINT_OF_RANDOM_VAR (old) ;

  /* make string from private key, make whole keys line */
  p = mpz2base64 (&key->private) ;
  s = make_peks_key_line (tag, key, p, "\n") ;
  POINT_OF_RANDOM_VAR (p) ;

  /* clean up after writing the keys string to file */
  fputs (s, outp);
  XFREE (p);
  POINT_OF_RANDOM_VAR (s) ;
  XFREE (s);

  if (inp != 0) {

    /* dump the rest onto file */
    while (fgets (read_buf, sizeof (read_buf), inp) > 0) {
      
      /* check for input line overflow */
      n = strlen (read_buf) ;
      if (read_buf [n-1] != '\n' &&
	  read_buf [n-1] != '\r') {
	errno = errnum (KEYFILE_PARSER_ERR) ;
	goto get_ready_for_return;
      }

      /* copy to the output file */
      fputs (read_buf, outp) ;
    }
  
    fclose (inp);
  } /* if (inp != 0) */ 

  fclose (outp);

  /* files have been closed, now */
  inp = outp = 0 ;

  /* save current version of the key file */
  if (inp != 0) {
    unlink (old) ;
    rename (file, old) ;
  }

  /* install new key file */
  r_code = rename (new, file) ;

 get_ready_for_return:

  if (inp != 0)  fclose (inp);
  if (outp != 0) fclose (outp);
  DEALLOCA (new) ;
  DEALLOCA (old) ;
  umask (omask) ; /* restore file creation mode */
  return r_code ;
}

/* ---------------------------------------------------------------------------- *
 *               public key file convenience functions                          *
 * ---------------------------------------------------------------------------- */

peks_key *
read_peks_server_keyfile
  (const char *file)
{
  peks_key *key ;

  /* initialize prime random generator (unless done, yet) */
  init_random_gen ((char*)&key, sizeof (key)) ;

  key = read_peks_key (PEKS_SERVER_KEY_TAG, strncasecmp, file) ;
  
  POINT_OF_RANDOM_VAR (key) ;

  /* some non key entries have 0 or 1 generators */
  if (key == 0 || key->generator > 1)
    return key ;

  POINT_OF_RANDOM_VAR (file) ;

  /* so this is never a private server key */
  errno = errnum (MODIFIED_KEY_TAG) ;
  end_peks_key (key);
  return 0;
}

int
create_peks_server_keyfile
  (peks_key    *key,
   const char *file)
{
  POINT_OF_RANDOM_VAR (file) ;
  return save_peks_key (key, PEKS_SERVER_KEY_TAG, strncasecmp, file);
}

int
check_peks_server_key
  (peks_key    *key,
   const char *host,
   const char *file)
{
  char *p = 0;
  int r_code = -1 ;

  peks_key *fkey ;

  /* initialize prime random generator (unless done, yet) */
  init_random_gen ((char*)&fkey, sizeof (fkey)) ;

  if (key == 0 || file == 0) {
    errno = errnum (FUNCTION_ARG_ERROR) ;
    return -1;
  }

  POINT_OF_RANDOM_VAR (file) ;

  /* prepend by HOME variable */
  if (file [0] == '~' && file [1] == '/') {

    char *s = getenv ("HOME") ;

    if (s == 0) {
      errno = errnum (CANT_EXPAND_FNAME) ;
      return -1;
    }

    POINT_OF_RANDOM_VAR (p) ;

    p = strcpy (ALLOCA (strlen (s) + strlen (file)), s) ;
    file = strcat (p, file + 1) ;
  }

  POINT_OF_RANDOM_VAR (file) ;

  /* check, whether the key file exists, already */
  if ((fkey = read_peks_key (host, hostcmp, file)) == 0 && errno)
    goto done ;

  POINT_OF_RANDOM_VAR (fkey) ;
  
  /* check the key in the file against the key, given */
  if (fkey != 0) {
    if (mpz_cmp (key->modulus, fkey->modulus) == 0 &&
	mpz_cmp (key->private, fkey->private) == 0 &&
	key->generator == key->generator)
      r_code = 0 ;
    else
      errno = errnum (SERVER_SPOOF_ALERT) ;
    end_peks_key (fkey) ;
    goto done ;
  }

  /* append the key in the key file */
  if (save_peks_key (key, host, hostnomatch, file) >= 0) {
    r_code = 1 ;
    errno = errnum (STORED_SERVER_KEY) ;
  }
  
 done:

  if (p != 0)
    DEALLOCA (p);
 
  return r_code ;
}
