/*
 *          Copyright (c) mjh-EDV Beratung, 1996-1998
 *     mjh-EDV Beratung - 63263 Neu-Isenburg - Rosenstrasse 12
 *          Tel +49 6102 328279 - Fax +49 6102 328278
 *                Email info@mjh.teddy-net.com
 *
 *   Author: Jordan Hrycaj <jordan@mjh.teddy-net.com>
 *
 *   $Id: peks-handshake.c,v 1.7 1999/01/31 12:36:45 jordan Exp $
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   PEKS - private exponent key stuff for Diffie Hellman and El Gamal
 */

#include "common-stuff.h"

#ifdef HAVE_CTYPE_H
#include <ctype.h>
#endif

#include "peks-internal.h"
#include "crandom.h"
#include "make-primes.h"
#include "peks-baseXX.h"
#include "messages.h"
#include "baseXX.h"
#include "version.h"
#include "rnd-pool.h"

#if CHAR_BIT != 8
#error Not supporting this architecture: CHAR_BIT != 8
#endif

#define __diffie_hellmann_currently_unused__

/* ---------------------------------------------------------------------------- *
 *                  public key handshake functions                              *
 * ---------------------------------------------------------------------------- */

/*
 * make a public server key string ready for export
 *
 * "peks/0.0:" <p> <g> <g^a> <crc>
 *
 * ready to initiate Diffie Hellmann or El Gamal
 */

char *
make_public_key_str
  (peks_key      *k,
   const char *term)
{
  POINT_OF_RANDOM_STACK (5);
  return make_peks_key_line (PEKS_VERSION, k, k->export_key_str, term) ;
}

/*
 * Accept a public key as generated with make_public_key_str ()
 * and return an initialized client key.  This is done in two
 * steps to allow a key check against a host key data base, in
 * beween .
 */

peks_key *
accept_public_key_str
  (const char *line)
{
  char *s ;
  const char *proto [] = {"peks", 0} ;
  unsigned version = peks_split_ident (proto, line, 0) ;

  POINT_OF_RANDOM_VAR (version);

  /* check for version compatibility */
  switch (version) {
  case (100 + PEKSMAJOR) * 100 + PEKSMINOR:
    break;
  default:
    errno = errnum (PEKS_PROTO_ERROR) ;
    return 0;
  }

  POINT_OF_RANDOM_STACK (11);

  /* skip over version string */
  if ((s = strchr (line, ':')) == 0 || * ++ s != ' ') {
    errno = errnum (PEKS_SYNTAX_ERROR) ;
    return 0;
  }
  
  return get_peks_key_from_str (s) ;
}


void
update_accepted_public_key_str
  (const char *line,
   peks_key    *key)
{
  mpz_t OP, pub_key, G;
  unsigned n; ;

  POINT_OF_RANDOM_VAR (OP);
  
  /* reorganize the peks key structure as the function read_peks_key_line ()
     assumes the third line item as the private key */
  mpz_init_set (OP, key->private) ;

  POINT_OF_RANDOM_STACK (13);
  
  /* generate the private key: b */
  n = mpz_sizeinbase (key->modulus, 2) ;
  get_random_num (&key->private, (n + 1) >> 1) ;

  POINT_OF_RANDOM_VAR (G);

  /* calculate the public response key g^b (mod p) */
  mpz_init (pub_key) ;
  mpz_init_set_ui (G, key->generator) ;
  mpz_powm (pub_key, G, key->private, key->modulus) ;  /* g^a (mod m) */
  key->export_key_str = mpz2base64 (&pub_key) ;
  mpz_clear (pub_key) ;
  mpz_clear (G) ;

  /* calculate the internal common key (g^a)^b == g^ab (mod p) */
  mpz_powm (key->common, OP, key->private, key->modulus) ;
  mpz_clear (OP) ;
}

#ifndef __diffie_hellmann_currently_unused__
/*
 * make a response key string after accepting the public key
 *
 *	"dh/0.0:" <g^b> <plain_text> <crc>
 *
 * ready to complete the Diffie Hellmann key negotiation
 */

char *
make_dh_response_key_str (key, text, term)
     peks_key *key;
     const char *text;
     const char *term;
{
  char *s, *crc_str ;
  unsigned n ;
  
  while (isspace (text))
    text ++ ;
  if (text == 0 || *text == '\0')
    text = "." ;

  POINT_OF_RANDOM_VAR (s);

  /* calculate crc for this line */
  crc_str = seqB64_md (key->export_key_str, text, 0);

  n = (term == 0) ? 0 : strlen (term) ;

  /* make the response line */
  s = XMALLOC (sizeof (DH_VERSION) + 2 +
	       strlen (key->export_key_str) + 1 +
	       strlen (text) + 1 +
	       strlen (crc_str) + 1 + n) ;

  POINT_OF_RANDOM_VAR (s);

  sprintf (s, "%s: %s %s %s%s", 
	   DH_VERSION, key->export_key_str, text, crc_str, n?term:"");
  
  XFREE (crc_str);
  return s ;
}
#endif  

/*
 * make a response key string after accepting the public key
 *
 *	"elg/0.0:" <g^b> <text * g^ab> <crc>
 *
 * ready to complete the El Gamal key negotiation;
 *
 * here text is repesented as the sequence
 *
 *	<text> := (<len_high>, <len_low>, bytes ...>
 */

char *
make_elg_response_key_str
  (peks_key      *k,
   const char *text,
   unsigned     len,
   const char *term)
{
  /* calculate maximal text len */
  unsigned lt = (mpz_sizeinbase (k->modulus, 2) + 7) >> 3 ; 
  
  char *s, *t ;
  mpz_t bin_arg ;
 
  /* check whether the arg text is not too long */
  if (len >= lt) {
    errno = errnum (ELG_TEXT_2LONG) ;
    return 0;
  }

  POINT_OF_RANDOM_STACK (11) ;

  /* 
   * assemble text buffer: we set the high order bit so that we can 
   * make  sure about the length of the number when converted to an
   * mpz_t type number and back; leading zeros may savely be dropped
   * when converting back from the mpz_t type to the text
   */
  t = ALLOCA (lt + 3) ;
  t [0] = ((len >> 8) & 0xff) | 0x80 ;
  t [1] =   len       & 0xff  ;
  memcpy (t + 2, text, len) ;

  POINT_OF_RANDOM_VAR (t);

  /* append some random noise if the text length is small */
  if (len < (lt >> 2)) {
    fast_random_bytes (t + 2 + len, (lt >> 4) - len) ;
    len = (lt >> 2) ;
  }

  /* make base64 string */
  text = bin2base64 (t, len + 2) ;

  POINT_OF_RANDOM_VAR (text);

  /* initialize numerics */
  mpz_init (bin_arg) ;

  if (base64toMpz (&bin_arg, text) == 0) {

    errno = errnum (ELG_B64TEXT_ERR) ;
    s     = 0 ;

  } else {

    unsigned n ;
    char *scrambler_str, *crc_str ;

    /* multiply the argument text against the public key */

    mpz_mul (bin_arg, bin_arg, k->common);	/* x*g^ab */
    mpz_mod (bin_arg, bin_arg, k->modulus) ;	/* x*g^ab (mod p) */
    scrambler_str = mpz2base64 (&bin_arg) ;

    /* calculate crc for this line */
    crc_str = seqB64_md (k->export_key_str, scrambler_str, 0);

    POINT_OF_RANDOM_VAR (crc_str);

    n = (term == 0) ? 0 : strlen (term) ;

    /* make the response line */
    s = XMALLOC (sizeof (ELG_VERSION) + 2 +
		 strlen (k->export_key_str) + 1 +
		 strlen (scrambler_str) + 1 +
		 strlen (crc_str) + 1 + n) ;
    sprintf (s, "%s: %s %s %s%s", 
	     ELG_VERSION, k->export_key_str, scrambler_str, crc_str, n?term:"");

    POINT_OF_RANDOM_VAR (scrambler_str);
    XFREE (scrambler_str) ;
    XFREE (crc_str);
  }
  
  XFREE (text) ;
  DEALLOCA (t) ;
  mpz_clear (bin_arg) ;
  return s ;
}


/*
 * accept a response key 
 *
 *      xx/#.#: <pub_key> <value> <crc>
 *
 * as generated with make_xx_response_key_str () and update 
 * the server key structure
 */

int
accept_response_key_str
  (peks_key    *key,
   const char *line)
{
  int             rval = -1 ;	/* error return code */
  const char *proto [] = {"dh", "elg", 0};
  unsigned     version = peks_split_ident (proto, line, 0) ;
  
  char *t, *pub_key_str, *text,* crc_str ;
  mpz_t OP, pub_key ;

  /* check for version compatibility */
  switch (version) {
#ifndef __diffie_hellmann_currently_unused__
  case (100 + PEKSMAJOR) * 100 + PEKSMINOR: /* Diffie Hellmann */
    version = 1 ;
    break;
#endif
  case (200 + PEKSMAJOR) * 100 + PEKSMINOR: /* El Gamal */
    version = 2 ;
    break;
  default:
    errno = errnum (RESP_VERS_MISMATCH) ;
    return 0;
  }

  POINT_OF_RANDOM_VAR (version);

  /* skip over version string */
  if ((line = strchr (line, ':')) == 0 || * ++ line != ' ') {
    errno = errnum (RESP_VERS_MISMATCH) ;
    return 0;
  }
  /* dup argument string: we need to edit */
  t = ALLOCASTRDUP (line) ;

  /* parse line entries: get response key */
  if ((pub_key_str = strtok (t, PEKS_FS)) == 0) {
    errno = errnum (RESP_PUBKEY_MISSING) ;
    goto return_rval ;
  }
  /* parse line entries: get text */
  if ((text = strtok (0, PEKS_FS)) == 0) {
    errno = errnum (RESP_CODE_MISSING) ;
    goto return_rval ;
  }
  /* parse line entries: get crc */
  if ((crc_str = strtok (0, PEKS_FS)) == 0) {
    errno = errnum (RESP_CRC_MISSING) ;
    goto return_rval ;
  }
  /* verify line entries */
  if (comp_seqB64_md (crc_str, pub_key_str, text, 0) != 0) {
    errno = errnum (RESP_CRCVAL_ERR) ;
    goto return_rval ;
  }
  /* parse line entries: no more fields */
  if ((crc_str = strtok (0, PEKS_FS)) != 0) {
    errno = errnum (RESP_2MANY_FIELDS) ;
    goto return_rval ;
  }

  POINT_OF_RANDOM_VAR (pub_key);

  /* convert to internal number representation */
  mpz_init (pub_key);
  if (base64toMpz (&pub_key, pub_key_str) == 0) {
    errno = errnum (RESP_PUBKEY_ERR) ;
    goto mpz_error_return;
  }

  /* calculate secret common key */
  mpz_powm (key->common, pub_key, key->private, key->modulus) ;

  POINT_OF_RANDOM_VAR (key);

  /* assign payload: text */
  switch (version) {

#ifndef __diffie_hellmann_currently_unused__
  case 1:	/* diffie hellmann */
    ?????????????
#endif

  case 2:	/* El Gamal encrypted: text*g^ab (mod p) */

    mpz_init (OP) ;
    if (base64toMpz (&OP, text) == 0) {
      errno = errnum (RESP_PAYLOAD_ERR) ;

    } else {		/* OP := text*g^ab (mod p) */
      unsigned char *s;
      unsigned n ;

      mpz_invert (pub_key, key->common, key->modulus);
      mpz_mul (OP, pub_key, OP);	/* OP := g^(-ab)*text*g^ab */
      mpz_mod (OP, OP, key->modulus);	/* OP := text (mod p) */

      /* convert numeric text to binary text */
      text = mpz2base64 (&OP) ;
      s    = base64toBin (text, &n) ;
      POINT_OF_RANDOM_VAR (text);
      XFREE (text) ;
      
      /* now, s = (<len_high>, <len_low>, bytes ...) */
      if (n >= 2) {
	int len = ((s [0] & 0x7f) << 8) | s [1] ;
	if (len <= n - 2) {
	  key->import_str  = bin2base64 (s + 2, len) ;
	  key->import_type = elg_encrypted ;
	  rval = 0;
	}
      }
      
      XFREE (s) ;
    }

    POINT_OF_RANDOM_VAR (OP);
    mpz_clear (OP);
    break;

  default:				/* unencrypted plain text */
    key->import_str  = XSTRDUP (text) ;
    key->import_type = plain_text ;
    rval = 0 ;
  }
  
 mpz_error_return:
  mpz_clear (pub_key);

 return_rval:  
  DEALLOCA (t) ;
  return rval ;
}
