/*
 *          Copyright (c) mjh-EDV Beratung, 1996-1998
 *     mjh-EDV Beratung - 63263 Neu-Isenburg - Rosenstrasse 12
 *          Tel +49 6102 328279 - Fax +49 6102 328278
 *                Email info@mjh.teddy-net.com
 *
 *       Author: Jordan Hrycaj <jordan@mjh.teddy-net.com>
 *
 *   $Id: peks-server.c,v 1.9 1999/01/31 12:12:05 jordan Exp $
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   PEKS - private exponent key stuff for Diffie Hellman and El Gamal
 */

#include "common-stuff.h"

#ifdef HAVE_SYS_SOCKET_H
# include <sys/socket.h>
#else
# include <socket.h> /* ????? */
#endif

#define __IOSTREAM_PRIVATE__ /* prevent from using send/recv auto-replaces */
#include "peks-internal.h"
#include "crandom.h"
#include "messages.h"
#include "cbc-frame.h"
#include "baseXX.h"
#include "version.h"
#include "rnd-pool.h"

/* ---------------------------------------------------------------------------- *
 *                  private key handshake functions                             *
 * ---------------------------------------------------------------------------- */

static const char *
peks_server_setup
  (peks_key       **K,
   const char   *host,
   const char   *file,
   peks_key *template)
{
  /* we use a jump table, here */
  POINT_OF_RANDOM_STACK (13);

  if (host != 0 && file != 0 &&
      (*K = read_peks_key (host, hostcmp, file)) != 0)
    goto return_public_key ;
  
  /* get default key: dup_peks_key () always returns a value */
  if (template != 0)  {
    *K = dup_peks_key (template) ;
    goto return_public_key ;
  }

  if ((*K = new_peks_key (0)) != 0)
    goto return_public_key ;

  return 0 ;
  /*@NOTREACHED@*/

 return_public_key:
  return make_public_key_str (*K, 0) ;
}




static char *
peks_server_decode_sender_key
  (peks_key    *k,
   unsigned    *N,
   const char **T,
   const char *in)
{
  char *s, *sk, *sl;
  unsigned n, dummy ;

  if (N == 0)
    N = &dummy ;

  POINT_OF_RANDOM_STACK (7);

  /* in case of an error, errno will be set */
  if (accept_response_key_str (k, in) < 0)
    return 0 ;
  
  if (k->import_type != elg_encrypted) {
    errno = errnum (DECODE_NO_CIPHER) ;
    return 0 ;
  }

  /* extract session key s of length n */
  s = base64toBin (k->import_str, &n) ;
  
  /* 
   * parse the session key details:
   *
   *	key: <base64-key-bits> <cipher-type>
   */
  s [n-1] = '\0' ;
  if ((sk = strtok (s, PEKS_FS)) == 0 ||
      strcmp (sk, "key:")        != 0 ||
      (sk = strtok (0, PEKS_FS)) == 0 ||
      (sl = strtok (0, PEKS_FS)) == 0) {
    XFREE (s);
    errno = errnum (DECODE_CLNTKEY_ERR) ;
    return 0 ;
  }
  sk = base64toBin (sk, N) ;

  POINT_OF_RANDOM_VAR (sk);

  if (T != 0)
    *T = XSTRDUP (sl) ;
  
  XFREE (s);
  return sk;
}




static char *
peks_server_wrap_receiver_key
  (const char    *s,
   unsigned     len,
   const char *type)
{
  char *t, *sk ;

  POINT_OF_RANDOM_STACK (7);

  if (len == 0)
    len = strlen (s) ;
  if (type == 0)
    type = "." ;

  /* make reciever session key as
   *
   *	key: <base64-key-bits> <cipher-type>
   */
  sk = bin2base64 (s, len) ;
  t  = XMALLOC (8 + strlen (sk) + strlen (type));

  POINT_OF_RANDOM_VAR (sk);

  sprintf (t, "key: %s %s\n", sk, type) ;
  XFREE (sk) ;

  return t;
}



static int
push_server_io_layer
  (unsigned        fd,
   const char   *what,
   const char    *key,
   unsigned       len,
   unsigned is_sender)

{
  void *cbc, *cc, *fc ;
  cipher_desc *cp ;
  frame_desc  *fp ;

  size_t c_size ;
  int  (*c_open) (void *, unsigned, cipher_desc*, frame_desc*, char[4]) ;
  int  (*c_rdwr) (void *, char *, unsigned, int);
  int  (*c_ioctl)(void *, io_ctrl_request, int*);
  void (*c_close)(void *) ;

  /* get new cipher and frame classes to be used, here */
  if (find_classes_by_name (&cc, &fc, what) == 0)
    return -1;

  /* create an cipher instance and a frame */
  if ((cp = (*(is_sender?create_encryption:create_decryption)) (cc, key, len)) < 0)
    return -1;
  fp = create_frame (fc, key [len ? len-1 : 0]) ;
  
  /* push that new io layer */
  cbc_get_info (is_sender, &c_size, &c_open, &c_rdwr, &c_ioctl, &c_close);
  if ((cbc = io_push (fd, c_size, c_rdwr, c_ioctl, c_close, is_sender)) == 0) {
    destroy_cipher (cp);
    destroy_frame  (fp);
    return -1;
  }

  POINT_OF_RANDOM_STACK (11);
  
  /* initialize that io layer */
  return (*c_open) (cbc, fd, cp, fp, (char*)key + (len<5 ? 0 : len-5)) ;
}


/* ---------------------------------------------------------------------------- *
 *                  public key server handshake function                        *
 * ---------------------------------------------------------------------------- */

int
server_negotiate_session_key
  (peks_key       * key,
   int           socket,
   const char  * client,
   const char * keyfile)
{
  const char *cipher, *t ;
  char *s, *buf;
  int l, n;

  POINT_OF_RANDOM_STACK (14);

  /* initialize prime random generator */
  init_random_gen ((char*)&s, sizeof (s)) ;

  /* set up the public peks server key */
  if ((t = peks_server_setup (&key, client, keyfile, key)) == 0) {
    errno = errnum (SRV_SETUP_SKEY) ;
    return -1;
  }
  
  /* send the public peks key to the client */
  OS_SEND (socket, t, strlen (t), 0);
  POINT_OF_RANDOM_VAR (t);
  XFREE (t);

  /* receive the peks wrapped answer from the client */
  buf = ALLOCA (CBC_IO_BUFLEN_MAX+1) ;
  if ((n = OS_RECV (socket, buf, CBC_IO_BUFLEN_MAX, 0)) <= 2) {
    /* check for a termination message from the client */
    if (n == 2 && buf [0] == '.' && buf [1] == '\0')
      errno = errnum (SRV_CLIENT_KNOCKING);
    else if (n >= 0)
      errno = errnum (SRV_CLIENT_RESPERR);
    goto error_code_return ;
  }
  buf [n] = '\0' ;

  /* unpack the session key from the peks wrapper */
  s = peks_server_decode_sender_key (key, &l, &cipher, buf) ;
  end_peks_key (key) ;
  if (s == 0)
    goto error_code_return ;
 
  /* install it as sender session key s of length l and type t */
  n = push_server_io_layer (socket, cipher, s, l, 1) ;
  POINT_OF_RANDOM_VAR (s);
  XFREE (s);
  if (n < 0)
    goto error_code_return ;
  
  /* generate the receiver session key */
  if ((l = cipher_keylen (cipher) + 5) == 5)
    goto error_code_return;
  prime_random_bytes (s = ALLOCA (l), l) ;
   
  /* push a new io layer for reading */
  if (push_server_io_layer (socket, cipher, s, l, 0) < 0) {
    DEALLOCA (s);
    goto error_code_return ;
  }

  /* peks_server_wrap_receiver_key () always returns a value, send it */
  buf = peks_server_wrap_receiver_key (s, l, cipher) ;
  POINT_OF_RANDOM_VAR (cipher);
  XFREE (cipher);
  POINT_OF_RANDOM_VAR (s);
  DEALLOCA (s);
  if (io_send (socket, buf, strlen (buf), 0) < 0)
    goto error_code_return ;
  
  /* check magic string */
  if ((n = io_recv (socket, buf, CBC_IO_BUFLEN, 0)) < 0) {
    errno = errnum (SRV_MAGIC_FAILED) ;
    goto error_code_return ;
  }
  buf [n] = '\0' ;
  if (strcmp (buf, PEKS_MAGIC) != 0) {
    errno = errnum (SRV_WRONG_MAGIC) ;
    goto error_code_return ;
  }

#if 0
  /*********************************************/
  n = io_send (socket, "1234567890", 10, 0);
  l = io_ctrl (socket, IO_CHANGE_KEY,       0, 1);
  n = io_send (socket, "abcdefghij", 10, 0) ;

  l = io_ctrl (socket, IO_REGISTER_THREAD,  0, 1);
  n = io_ctrl (socket, IO_ACTIVATE_THREAD, &l, 1);
  l = io_ctrl (socket, IO_REGISTER_THREAD,  0, 1);
  n = io_ctrl (socket, IO_ACTIVATE_THREAD, &l, 1);
  l = io_ctrl (socket, IO_REGISTER_THREAD,  0, 1);
  n = io_ctrl (socket, IO_ACTIVATE_THREAD, &l, 1);
  n = io_send (socket, "1234567890", 10, 0);

  l = io_ctrl (socket, IO_CHANGE_KEY,       0, 1);
  n = io_ctrl (socket, IO_ACTIVATE_THREAD, &l, 1);

  l = io_ctrl (socket, IO_REGISTER_THREAD,  0, 1);
  n = io_ctrl (socket, IO_ACTIVATE_THREAD, &l, 1);
  n = io_send (socket, "abcdefghij", 10, 0) ;
  n = io_ctrl (socket, IO_DESTROY_THREAD,  &l, 1);
  /*********************************************/
#endif

  /* done */
  DEALLOCA (buf) ;
  return 0;
  /*@NOTREACHED@*/ 

 error_code_return:
  io_pop (socket, 2) ; /* its's always save to execute it */
  DEALLOCA (buf) ;
  return -1 ;
}

