/*
 *          Copyright (c) mjh-EDV Beratung, 1996-1998
 *     mjh-EDV Beratung - 63263 Neu-Isenburg - Rosenstrasse 12
 *          Tel +49 6102 328279 - Fax +49 6102 328278
 *                Email info@mjh.teddy-net.com
 *
 *    Author: Jordan Hrycaj <jordan@mjh.teddy-net.com>
 *
 *   $Id: peks-setup.c,v 1.8 1999/01/31 12:36:45 jordan Exp $
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 *   PEKS - private exponent key stuff for Diffie Hellman and El Gamal
 */

#include "common-stuff.h"

#include "make-primes.h"
#include "version.h"
#include "peks-internal.h"
#include "peks-baseXX.h"
#include "messages.h"
#include "cipher.h"
#include "rnd-pool.h"

/* ---------------------------------------------------------------------------- *
 *                  public functions: utilities                                 *
 * ---------------------------------------------------------------------------- */

char *
peks_version (void)
{
  return PEKS_VERSION ;
}

/*
 * For a given list of strings [] = {<ident>, ... } and two non negative
 * numbers <major> < 100, <minor> < 100 parse a string 
 *
 *	<ident> "/" <major> "." <minor>
 *
 * and return the unique identifier
 *
 *	(index {<ident>} + 1) * 10000 + <major> * 100	+ <minor>
 *
 * if such an index exists, and 0 otherwise.
 */

unsigned
peks_split_ident
  (const char *ident_table [], /* NULL terminated array */
   const char           *text,
   unsigned               len)
{
  char *t, *s ;
  int maj, min, n = 0, result = 0;

  POINT_OF_RANDOM_STACK (14);

  /* get a writable copy of the argument string */
  if (len == 0) 
    len = strlen (text) ;
  (s = strncpy (ALLOCA (len + 1), text, len + 1)) [len] = '\0' ;

  if ((t = strtok (s, "/")) != 0)
    
    while (ident_table [n] != 0) {
      
      /* get a match in the lookup table */
      if (strcmp (ident_table [n ++], t) != 0)
	continue ;

      /* ok, get the rest */
      if ((t = strtok (0, PEKS_FS)) != 0 &&
	  sscanf (t, "%u.%u", &maj, &min) == 2 &&
	  maj < 100 && min < 100)
	result = (n * 100 + maj) * 100 + min ;
      break;
    }
  
  DEALLOCA (s) ;
  return result ;
}

/*
 * For a given key structure, return a text
 *
 *   <tag>: <key->modulus_str> <key->generator_str> <key_str> <crc> <term>
 *
 * where CRC is an hash over the preceeding three text fields.
 */

char *
make_peks_key_line
  (const char     *tag,
   const peks_key *key,
   const char *key_str, 
   const char    *term)
{
  /* calculate crc for this line */
  char *crc_str = seqB64_md (key->modulus_str, key->generator_str, key_str);

  /* there may be a prefix and a postfix */
  unsigned e =  (tag == 0) ? 0 : strlen (tag) ;
  unsigned f = (term == 0) ? 0 : strlen (term) ;

  unsigned a, b, c, d ;

  char *t = XMALLOC ( e + f + 2 +
		    (a = strlen (key->modulus_str)) + 1 +
		    (b = strlen (key->generator_str)) + 1 +
		    (c = strlen (key_str)) + 1 +
		    (d = strlen (crc_str)) + 1) ;
  char *s = t ;

  if (tag != 0) {
    strcpy (s, tag)              ; s += e ;
    strcpy (s, ": ")             ; s += 2 ;
  }

  POINT_OF_RANDOM_STACK (11);

  strcpy (s, key->modulus_str)   ; s += a ;
  strcpy (s, " ")                ; s ++ ;
  strcpy (s, key->generator_str) ; s += b ;
  strcpy (s, " ")                ; s ++ ;
  strcpy (s, key_str)            ; s += c ;
  strcpy (s, " ")                ; s ++ ;
  strcpy (s, crc_str) ;          ; s += d ;
  if (term != 0)	/* no space delimiter, here */
    strcpy (s, term) ;

  XFREE (crc_str);

  return t ;
}

/* ---------------------------------------------------------------------------- *
 *                     public initialization functions                          *
 * ---------------------------------------------------------------------------- */

peks_key *
dup_peks_key
  (const peks_key *old)
{
  peks_key *new = XMALLOC (sizeof (peks_key));
  
  mpz_init_set (new->modulus, old->modulus);
  mpz_init_set (new->private, old->private);
  mpz_init_set (new->common,  old->common);

  new->generator  = old->generator ;

  POINT_OF_RANDOM_STACK (13);
  
  if (old->modulus_str   != 0)
    new->modulus_str =   XSTRDUP (old->modulus_str);
  if (old->generator_str != 0)
    new->generator_str = XSTRDUP (old->generator_str);
  if (old->export_key_str != 0) 
    new->export_key_str = XSTRDUP (old->export_key_str) ;
  if (old->import_str     != 0)
    new->import_str =     XSTRDUP (old->import_str);

  POINT_OF_RANDOM_VAR (new);

  return new;
}


peks_key *
new_peks_key
  (unsigned bits)
{
  void (*prt) (char*) = xprint1 ;
  peks_key         *k = XMALLOC (sizeof (peks_key));

  unsigned n;
  mpz_t G, PUB ;

  mpz_init (k->modulus) ;
  mpz_init (k->private) ;
  mpz_init (k->common) ;

  if (bits == 0) bits = DEFAULT_PRIME_BITS ;

  POINT_OF_RANDOM_VAR (k);

  /* generate prime module and generator  */
  n = get_generated_prime_module
    (&k->modulus, &k->generator, &k->private, bits, prt) ;

  POINT_OF_RANDOM_VAR (k);

  if (prt != 0) (*prt) ("\n") ; 
  if (n == 0) {
    errno = errnum (TOO_MANY_PRIME_TRIES) ;
    end_peks_key (k) ;
    return 0 ;
  }

  /* overwrite private with proper key */
  get_random_num (&k->private, bits);			/* prv key as big as p */

  /* don't let the random number become too small */
  if (mpz_sizeinbase (k->private, 2) < bits-4)		/* set the msb to 1 */
    mpz_setbit (k->private, bits) ;			/* num |= 1 makes it big */

  /* calculate public key */
  mpz_init (PUB);
  mpz_init_set_ui (G, k->generator) ;
  mpz_powm (PUB, G, k->private, k->modulus);		/* pub key := g^a (mod p) */
 
  /* make printables, re-assign strings */
  k->modulus_str = mpz2base64 (&k->modulus) ;		/* prime modulus p */
  k->generator_str = uint2base64 (k->generator);	/* generator g (mod p) */
  k->export_key_str = mpz2base64 (&PUB) ;		/* public key g^a (mod p) */

  POINT_OF_RANDOM_VAR (PUB);

  /* clean up */
  mpz_clear (PUB) ;
  mpz_clear (G) ;
  return k;
}

/*
 * parse a line
 *
 *	<p> <g> <a> <crc>
 *
 * and initialize a peks key structure 
 */

peks_key *
get_peks_key_from_str
  (const char *l)
{
  peks_key *key = 0;
  char *line = ALLOCASTRDUP (l) ;

  char *prv_str, *crc_str, *s ;
  mpz_t PUB, G;

  POINT_OF_RANDOM_VAR (line);

  /* create key structure */
  key = XMALLOC (sizeof (peks_key));

  /* parse line entries: get modulus */
  if ((key->modulus_str = strtok (line, PEKS_FS)) == 0) {
    errno = errnum (LINE_PARSER_ERR);
    goto error_return;
  }
    /* parse line entries: get generator */
  if ((key->generator_str = strtok (0, PEKS_FS)) == 0) {
    errno = errnum (LINE_PARSER_ERR);
    goto error_return;
  }
  /* parse line entries: get private key */
  if ((prv_str = strtok (0, PEKS_FS)) == 0) {
    errno = errnum (LINE_PARSER_ERR);
    goto error_return;
  }
  /* parse line entries: get crc */
  if ((crc_str = strtok (0, PEKS_FS)) == 0) {
    errno = errnum (LINE_PARSER_ERR);
    goto error_return;
  }
  /* no more entries ! */
  if ((s = strtok (0, PEKS_FS)) != 0) {
    errno = errnum (LINE_PARSER_ERR);
    goto error_return;
  }
  /* verify line entries */
  if (comp_seqB64_md (crc_str, key->modulus_str, key->generator_str, prv_str) != 0) {
    errno = errnum (LINE_PARSER_CRCERR);
    goto error_return;
  }
  /* convert to internal number representation */
  mpz_init (key->modulus) ;
  mpz_init (key->private) ;
  if (base64toMpz (&key->modulus, key->modulus_str) == 0) {
    errno = errnum (LINE_PARSER_B64ERR);
    goto mpz_error_return;
  }
  if (base64toMpz (&key->private, prv_str) == 0) {
    errno = errnum (LINE_PARSER_B64ERR);
    goto mpz_error_return;
  }
  if ((key->generator = base64toUint (key->generator_str)) == UINT_MAX) {
    errno = errnum (LINE_PARSER_B64ERR);
    goto mpz_error_return;
  }
  /* accept line entries, allocate string spaces */
  key->modulus_str   = XSTRDUP (key->modulus_str) ;
  key->generator_str = XSTRDUP (key->generator_str) ;

  POINT_OF_RANDOM_STACK (9);

  /* calculate public key, set export string */
  mpz_init (PUB);
  mpz_init_set_ui (G, key->generator) ;
  mpz_powm (PUB, G, key->private, key->modulus);	/* pub key := g^a (mod p) */
  key->export_key_str = mpz2base64 (&PUB) ;		/* public key g^a (mod p) */
  mpz_clear (PUB) ;
  mpz_clear (G) ;

  return key;
  /*@NOTREACHED@*/

 mpz_error_return:

  mpz_clear (key->modulus) ;
  mpz_clear (key->private) ;

 error_return:
  
  XFREE    (key) ;
  DEALLOCA (line) ;
  return 0 ;
}

/* ---------------------------------------------------------------------------- *
 *                     public destroy function                                  *
 * ---------------------------------------------------------------------------- */

void
end_peks_key
  (peks_key *key)
{
  if (key == 0)
    return ;

  POINT_OF_RANDOM_VAR (key) ;

  mpz_clear (key->modulus) ;
  mpz_clear (key->private) ;
  mpz_clear (key->common) ;

  POINT_OF_RANDOM_STACK (13);

  if (key->modulus_str    != 0)
    XFREE (key->modulus_str) ;
  if (key->generator_str  != 0)
    XFREE (key->generator_str) ;
  if (key->export_key_str != 0) 
    XFREE (key->export_key_str);
  if (key->import_str != 0)
    XFREE (key->import_str) ;
  
  memset (key, 0, sizeof (*key)) ;
  XFREE (key) ;
}


