/*
 * Copyright (c) 1993,1994,1995,1997,1998
 *      Texas A&M University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Texas A&M University
 *      and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Developers:
 *     Russell Neeper, David K. Hess, Douglas Lee Schales, David R. Safford
 */

#include <sys/types.h>
#include <stdio.h>
#include <sys/stat.h>
#include <string.h>
#ifdef __FreeBSD__
#include <stdlib.h>
#else
#include <malloc.h>
#endif

#include "chario.h"

struct file_str {
     struct file_str *next;
     dev_t dev;
     ino_t inode;
     FILE *fp; /* NULL for internal buffer pointed to by buffer */
     int linenum;
     char *filename;
     unsigned char *buffer;
     int bufptr;
};

static struct file_str *current = (struct file_str *)0;
static struct file_str *tail = (struct file_str *)0;

int __lastchar;
int __pushed = 0;

int
fetchar(void)
{
     int c = EOF;

     if(__pushed){
	  c = __lastchar;
	  __pushed = 0;
	  return c; /* short circuit to avoid line inc */
     }
     else do {
	  struct file_str *old = current;
	  if(!current)
	       return EOF;
	  else if(!current->fp){
	       if(current->buffer[current->bufptr])
		    c = current->buffer[current->bufptr++];
	  }
	  else if(current->fp){
	       c = getc(current->fp);
	  }

	  if(c == EOF){
	       if(current->fp){
		    if(current->filename)
			 free(current->filename);
		    fclose(current->fp);
	       }
	       current = current->next;
	       free(old);
	       if(!current){
		    tail = 0;
		    break;
	       }
	  }
     } while(c == EOF);

     if(c == '\n')
	  current->linenum++;
     return c;
}

int
getlinenum(void)
{
     if(current)
	  return current->linenum;
     else
	  return -1;
}

char *
getfilename(void)
{
     if(current)
	  return current->filename;
     else
	  return "*last*";
}

int
addmembuf(char *buffer, char *name)
{
     struct file_str *fs;

     fs = (struct file_str *)malloc(sizeof(struct file_str));
     if(!fs){
	  fprintf(stderr, "%s, line %d: malloc failed\n",
		  __FILE__, __LINE__);
	  exit(1);
     }
     fs->linenum = 1;
     fs->buffer = (unsigned char *)buffer;
     fs->bufptr = 0;
     fs->fp = (FILE *)0;
     fs->filename = name;
     fs->next = (struct file_str *)0;
     if(tail)
	  tail->next = fs;
     tail = fs;
     if(!current)
	  current = fs;
     return NOERR;
}

int
includemembuf(char *buffer, char *name)
{
     struct file_str *fs;

     fs = (struct file_str *)malloc(sizeof(struct file_str));
     if(!fs){
	  fprintf(stderr, "%s, line %d: malloc failed\n",
		  __FILE__, __LINE__);
	  exit(1);
     }
     fs->linenum = 1;
     fs->buffer = (unsigned char *)buffer;
     fs->bufptr = 0;
     fs->fp = (FILE *)0;
     fs->filename = name;
     fs->next = current;
     if(!tail)
	  tail = fs;
     current = fs;
     return NOERR;
}

int
includefile(char *filename)
{
     struct stat statbuf;
     struct file_str *rove, *fs;

     if(stat(filename, &statbuf) == -1){
	  return FILEERR;
     }
     for(rove=current;rove;rove=rove->next)
	  if(rove->dev == statbuf.st_dev &&
	     rove->inode == statbuf.st_ino)
	       return RECURSE;
     fs = (struct file_str *)malloc(sizeof(struct file_str));
     if(!fs){
	  fprintf(stderr, "%s, line %d: malloc failed\n",
		  __FILE__, __LINE__);
	  exit(1);
     }
     if(!(fs->fp = fopen(filename, "r"))){
	  free(fs);
	  return FILEERR;
     }
     else {
	  fs->dev = statbuf.st_dev;
	  fs->inode = statbuf.st_ino;
	  fs->linenum = 1;
	  fs->filename = strdup(filename);
	  fs->buffer = 0;
	  fs->next = current;
	  if(!tail)
	       tail = fs;
	  current = fs;
     }
     return NOERR;
}

int
addfile(char *filename)
{
     struct stat statbuf;
     struct file_str *fs;

     if(stat(filename, &statbuf) == -1){
	  return FILEERR;
     }
     fs = (struct file_str *)malloc(sizeof(struct file_str));
     if(!fs){
	  fprintf(stderr, "%s, line %d: malloc failed\n",
		  __FILE__, __LINE__);
	  exit(1);
     }
     if(!(fs->fp = fopen(filename, "r"))){
	  free(fs);
	  return FILEERR;
     }
     else {
	  fs->dev = statbuf.st_dev;
	  fs->inode = statbuf.st_ino;
	  fs->linenum = 1;
	  fs->filename = strdup(filename);
	  fs->buffer = 0;
	  fs->next = 0;
	  if(tail)
	       tail->next = fs;
	  tail = fs;
	  if(!current)
	       current = fs;
     }
     return NOERR;
}
