%{
/*
 * Copyright (c) 1993,1995
 *	Texas A&M University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Texas A&M University
 *	and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Developers:
 *             David K. Hess, Douglas Lee Schales, David R. Safford
 */
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <netdb.h>
#include <netinet/in.h>
#include "chario.h"
#include "hosts.h"
#include "groups.h"
#include "services.h"

extern int parse_error;

%}
     
%union {
     unsigned int int32;
     char *str;
     struct hostrange hostrange;
     struct service service;
}

%token DEFINE HOST NETWORK INCLUDE ACCEPT REJECT OVERRIDE
%token STRING IN OUT INOUT BYTEVAL INTEGER
%token SRC DST
%type <int32> netmask ip_address hostspec
%type <str> STRING IN OUT INOUT
%type <int32> inout dirspec
%type <int32> BYTEVAL
%type <int32> INTEGER integer
%type <hostrange> machine_spec machine_range
%type <service> service_spec service service_range
%type <int32> protocol
%%

file
        : stmt ';' file
        | stmt ';'
        | error {
	     fprintf(stderr, "\"%s\", line %d: syntax error.\n",
		     getfilename(), getlinenum());
	}
;

stmt
	: /* */
        | host_def
	| group_def
	| network_def
	| include
        | accept
        | reject
        | override
;

accept
        : ACCEPT ip_address netmask {
	     add_accept($2, $3, NORMAL);
	}
        | ACCEPT '~' ip_address netmask {
	     add_accept($3, $4, INVERT);
	}
;

reject
        : REJECT ip_address netmask {
	     add_reject($2, $3, NORMAL);
	}
	| REJECT '~' ip_address netmask {
	     add_reject($3, $4, INVERT);
	}
;

override
        : OVERRIDE ip_address netmask service_list {
	     add_override($2, $3);
	}
;

group_def
	: DEFINE STRING service_list {
	     make_group($2); /* service stack has list of services */
	}
;

network_def
	: NETWORK machine_spec netmask service_list {
	     do_network($2, $3); /* service stack has list of services */
	}
	| NETWORK machine_range service_list {
	     do_network($2, -1); /* service stack has list of services */
	}
;

host_def
        : HOST hostspec service_list {
	     if($2 == 0){
		  printf("Host == 0, %d\n", getlinenum());
	     }
	     else
		  do_host($2); /* service stack has list of services */
	}
;

include
	: INCLUDE STRING {
	     includefile($2);
	}
;

service_list
	: service_entry
	| service_list ',' service_entry
;

service_entry
	: '<' dirspec service_spec inout '>' {
	     push2_service($3, $4, $2, FP_ACC);
	}
        | '<' '!' dirspec service_spec inout '>' {
	     push2_service($4, $5, $3, FP_REJ);
	}
        | STRING {
	     push_group($1);
	     free($1);
	}
;

service_spec
     : service
     | service_range
;

service
     : integer {
	  $$.start = $$.end = $1;
	  $$.protocol = IPPROTO_TCP;
     }
     | integer '/' protocol {
	  $$.start = $$.end = $1;
	  $$.protocol = $3;
     }
     | STRING {
	  struct servent *se;
	  if((se = getservbyname($1, "tcp"))){
	       $$.start = $$.end = htons(se->s_port);
	       $$.protocol = IPPROTO_TCP;
	  }
	  else {
	       fprintf(stderr, "\"%s\", line %d: unknown service: %s/tcp\n", 
		       getfilename(), getlinenum(),$1);
	       parse_error = 1;
	  }
     }
     | STRING '/' protocol {
	  struct servent *se;
	  struct protoent *pe;

	  if((pe = getprotobynumber($3)))
	       if((se = getservbyname($1, pe->p_name))){
		    $$.start = $$.end = htons(se->s_port);
		    $$.protocol = $3;
	       }
	       else {
		    fprintf(stderr, "\"%s\", line %d: unknown service: %s/%s\n",
			    getfilename(), getlinenum(), $1, pe->p_name);
		    parse_error = 1;
	       }
	  else {
	       fprintf(stderr, "\"%s\", line %d: unknown service: %s/%d\n",
		       getfilename(), getlinenum(), $1, $3);
	  }
     }
;

service_range     
     : integer '-' integer {
	  $$.start = $1;
	  $$.end = $3;
	  $$.protocol = IPPROTO_TCP;
     }
     | integer '-' integer '/' protocol {
	  $$.start = $1;
	  $$.end = $3;
	  $$.protocol = $5;
     }
;

dirspec
        : SRC '=' { $$=FP_SRC; }
        | DST '=' { $$=FP_DST; }
        | /* */   { $$=FP_DST; }
;

protocol
     : integer {
	  $$ = $1;
     }
     | STRING {
	  struct protoent *pe;
	  if((pe = getprotobyname($1))){
	       $$ = pe->p_proto;
	  }
	  else {
	       fprintf(stderr, "\"%s\", line %d: unknown protocol: %s\n",
		       getfilename(), getlinenum(), $1);
	       parse_error = 1;
	  }
     }
     | error {
	  fprintf(stderr,
		  "\"%s\", line %d: expecting protocol\n",
		  getfilename(), getlinenum());
	  parse_error = 1;
     }
;

integer
     : INTEGER {
	  $$ = $1;
     }
     | BYTEVAL {
	  $$ = $1;
     }
;

inout
	: IN {
	     $$ = FP_IN+1;
	}
	| OUT {
	     $$ = FP_OUT+1;
	}
	| INOUT {
	     $$ = (FP_IN+1)|(FP_OUT+1);
	}
        | error {
	     fprintf(stderr,
		     "\"%s\", line %d: expecting one of 'in', 'out', or 'in-out'\n",
		     getfilename(), getlinenum());
	     parse_error = 1;
	}
;

hostspec
	: ip_address {
	     $$ = $1;
	}
	| STRING {
	     struct hostent *he;
	     /*
	     struct {
		  int ipaddr:32;
	     } host;
	     */
	     if((he = gethostbyname($1))){
		  /* memcpy(&host, he->h_addr_list[0], 4);
		  $$ = host.ipaddr; 
		  */
		  $$ = htonl(*((int *) he->h_addr_list[0]));
	     }
	     else {
		  fprintf(stderr, "\"%s\", line %d: unknown host: %s\n", 
			  getfilename(), getlinenum(), $1);
		  $$ = 0xFFFFFFFF;
		  parse_error = 1;
	     }
	}
;

machine_spec
	: ip_address {
	     $$.start = $$.end = $1;
	}
;

machine_range
	: ip_address '-' ip_address {
	     $$.start = $1;
	     $$.end = $3;
	}
;

netmask
	: ip_address
;

ip_address
	: BYTEVAL '.' BYTEVAL '.' BYTEVAL '.' BYTEVAL {
	     $$ = $1 << 24 | $3 << 16 | $5 << 8 | $7;
	}
;

