/*
 * Copyright (c) 1993,1994,1995,1997,1998
 *      Texas A&M University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Texas A&M University
 *      and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Developers:
 *     Russell Neeper, David K. Hess, Douglas Lee Schales, David R. Safford
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#ifdef __FreeBSD__
#include <stdlib.h>
#else
#include <malloc.h>
#endif
#include <memory.h>
#include "chario.h"
#include "macros.h"
#include "hosts.h"
#include "services.h"

struct hosttable *hostlist = (struct hosttable *)0;
static int host_count;

static struct accept *acceptlistbuf;
static int accept_count;
static int acceptlistsize;

static struct reject *rejectlistbuf;
static int reject_count;
static int rejectlistsize;

static struct override *overridelistbuf;
static int override_count;
static int overridelistsize;

static int compare_addr(struct accept *, struct accept *);

extern int parse_error;

void
init_hosts(void)
{
     hostlist = (struct hosttable *)0;
     host_count = 0;

     acceptlistsize = sizeof(struct accept);
     acceptlistbuf = (struct accept *)
	  malloc(sizeof(struct accept)*acceptlistsize);
     if(!acceptlistbuf){
	  fprintf(stderr, "%s, line %d: malloc failed\n",
		  __FILE__, __LINE__);
	  exit(1);
     }
     accept_count = 0;

     rejectlistsize = sizeof(struct reject);
     rejectlistbuf = (struct reject *)
	  malloc(sizeof(struct reject)*rejectlistsize);
     if(!rejectlistbuf){
	  fprintf(stderr, "%s, line %d: malloc failed\n",
		  __FILE__, __LINE__);
	  exit(1);
     }
     reject_count = 0;
     
     overridelistsize = sizeof(struct override);
     overridelistbuf = (struct override *)
	  malloc(sizeof(struct override)*overridelistsize);
     if(!overridelistbuf){
	  fprintf(stderr, "%s, line %d: malloc failed\n",
		  __FILE__, __LINE__);
	  exit(1);
     }
     override_count = 0;
}

unsigned int
getnetmask(unsigned int hostaddr)
{
     unsigned int result;
     struct in_addr ha;

     if(IN_CLASSA(hostaddr))
	  result = IN_CLASSA_NET;
     else if(IN_CLASSB(hostaddr))
	  result = IN_CLASSB_NET;
     else if(IN_CLASSC(hostaddr))
	  result = IN_CLASSC_NET;
     else if(IN_CLASSD(hostaddr))
	  result = IN_CLASSD_NET;
     else {
 	  ha.s_addr = htonl(hostaddr);
	  fprintf(stderr, "%s, line %d: Invalid host address '%s'\n",
		  getfilename(), getlinenum(), inet_ntoa(ha));
	  exit(1);
     }
     return result;
}

static int
compare_addr(struct accept *x, struct accept *y)
{
	if (x->flag != y->flag)
		return y->flag - x->flag;
	if (x->netmask != y->netmask)
		return y->netmask - x->netmask;
	return x->ipaddr - y->ipaddr;
}

static int
allow_comp(struct override *x, struct override *y)
{
	if (x->netmask != y->netmask)
		return x->netmask - y->netmask;
	return x->ipaddr - y->ipaddr;
}

void
writetables(char *file)
{
     int fd;
     struct hosttable *rove;
     int i, j, network_count;
     unsigned long network, t, u, f;
     unsigned short p;
     struct {
 	  unsigned int networks_offset;
	  unsigned int classes_offset;
	  unsigned int accept_offset;
 	  unsigned int reject_offset;
	  unsigned int override_offset;
     } header;

     if ((fd = open(file, O_WRONLY|O_CREAT|O_TRUNC, 0600)) == -1) {
	  perror(file);
          return;
     }

     lseek(fd, sizeof(header), SEEK_SET);

     header.networks_offset = htonl(lseek(fd, 0, SEEK_CUR));

     for (network_count=0,rove=hostlist; rove; rove=rove->next)
	  network_count++;
     printf("There %s %d network%s, %d host%s, and ",
	  (network_count == 1) ? "is" : "are", network_count,
	  (network_count == 1) ? "" : "s",
	  host_count, (host_count == 1) ? "" : "s");

     for (rove=hostlist; rove; rove=rove->next) {
	  network = htonl(rove->network);
          write(fd, &network, sizeof(network));
	  write(fd, rove->table, rove->size);
     }
     i = -1;
     write(fd, &i, sizeof(i));

     header.classes_offset = htonl(lseek(fd, 0, SEEK_CUR));

     writeclassestable(fd);

     header.accept_offset = htonl(lseek(fd, 0, SEEK_CUR));

     qsort(acceptlistbuf, accept_count, sizeof(struct accept),
     		(void *) compare_addr);
     i = htonl(accept_count);
     write(fd, &i, sizeof(i));
     for(i=0; i<accept_count; i++) {
          t = htonl(acceptlistbuf[i].ipaddr);
          u = htonl(acceptlistbuf[i].netmask);
          f = htonl(acceptlistbuf[i].flag);
          write(fd, &t, sizeof(t));
          write(fd, &u, sizeof(u));
          write(fd, &f, sizeof(f));
     }

     header.reject_offset = htonl(lseek(fd, 0, SEEK_CUR));

     qsort(rejectlistbuf, reject_count, sizeof(struct reject),
     		(void *) compare_addr);
     i = htonl(reject_count);
     write(fd, &i, sizeof(i));
     for(i=0; i<reject_count; i++) {
          t = htonl(rejectlistbuf[i].ipaddr);
          u = htonl(rejectlistbuf[i].netmask);
          f = htonl(rejectlistbuf[i].flag);
          write(fd, &t, sizeof(t));
          write(fd, &u, sizeof(u));
          write(fd, &f, sizeof(f));
     }

     header.override_offset = htonl(lseek(fd, 0, SEEK_CUR));

     qsort(overridelistbuf, override_count, sizeof(struct override),
     		(void *) allow_comp);
     i = htonl(override_count);
     write(fd, &i, sizeof(i));
     for(i=0; i<override_count; i++) {
          t = htonl(overridelistbuf[i].ipaddr);
          u = htonl(overridelistbuf[i].netmask);
          write(fd, &t, sizeof(t));
          write(fd, &u, sizeof(u));
          for(j=0; j<CLASS_SIZE; j++) {
               p = htons(overridelistbuf[i].ports[j].start);
               write(fd, &p, sizeof(p));
               p = htons(overridelistbuf[i].ports[j].end);
               write(fd, &p, sizeof(p));
          }
     }

     lseek(fd, 0, SEEK_SET);
     write(fd, &header, sizeof(header));

     close(fd);
}
 
void
do_host(unsigned int host)
{
     int class_entry;

     if(host == 0){
	  fprintf(stderr, "%s, line %d: internal error: do_host(host == 0)\n",
		  getfilename(), getlinenum());
	  return;
     }

     if(host != 0xFFFFFFFF){
	  struct hosttable *hrove;
	  unsigned int network;
	  unsigned int netmask;
	  unsigned int hostaddr;

	  netmask = getnetmask(host);
	  network = host & netmask;
	  for(hrove = hostlist;hrove;hrove=hrove->next)
	       if(network == hrove->network)
		    break;
	  if(!hrove){
	       int size;
	       hrove = (struct hosttable *)malloc(sizeof(struct hosttable));
	       if(!hrove){
		    fprintf(stderr, "%s, line %d: malloc failed\n",
			    __FILE__, __LINE__);
		    exit(1);
	       }
	       hrove->network = network;
	       size = (~netmask)+1;
	       hrove->size = size;
	       hrove->table = (unsigned char *)malloc(size);
	       if(!hrove->table){
		    fprintf(stderr, "%s, line %d: malloc failed\n",
			    __FILE__, __LINE__);
		    exit(1);
	       }
	       memset(hrove->table, 0, size);
	       hrove->next = hostlist;
	       hostlist = hrove;
	  }
	  class_entry = make_class();

	  hostaddr = host & (~netmask);
	  hrove->table[hostaddr] = class_entry;

	  host_count++;
     }

     destroy_all_services();
}

void
do_network(struct hostrange hr, unsigned int mask)
{
     struct hosttable *hrove;
     unsigned int network;
     unsigned int netmask;
     unsigned int hostaddr;
     unsigned int start, end;
     int class_entry;

     netmask = getnetmask(hr.start);
     network = hr.start & netmask;

     if (mask < netmask) {
	  fprintf(stderr, "%s, line %d: mask must be equal to or larger than network's class mask\n",
		  getfilename(), getlinenum());
	  parse_error = 1;
	  return;
     }

     for(hrove=hostlist;hrove;hrove=hrove->next)
	  if(network == hrove->network)
	       break;
     if(!hrove){
	  int size;
	  hrove = (struct hosttable *)malloc(sizeof(struct hosttable));
	  if(!hrove){
	       fprintf(stderr, "%s, line %d: malloc failed\n",
		       __FILE__, __LINE__);
	       exit(1);
	  }
	  hrove->network = network;
	  size = (~netmask)+1;
	  hrove->size = size;
	  hrove->table = (unsigned char *)malloc(size);
	  if(!hrove->table){
	       fprintf(stderr, "%s, line %d: malloc failed\n",
		       __FILE__, __LINE__);
	       exit(1);
	  }
	  memset(hrove->table, 0, size);
	  hrove->next = hostlist;
	  hostlist = hrove;
     }
     class_entry = make_class();

     if(hr.end == hr.start){
	  start = hr.start & ~netmask;
	  end = start + ~mask;
     } else {
	  start = hr.start & ~netmask;
	  end = hr.end & ~netmask;
     }

     for(hostaddr=start;hostaddr<=end;hostaddr++){
	  hrove->table[hostaddr] = class_entry;
     }
     destroy_all_services();
}

void
add_accept(unsigned int ipaddr, unsigned int mask, norminv flag)
{
     if(accept_count == acceptlistsize){
	  acceptlistsize *= 2;
	  acceptlistbuf = (struct accept *)
	       realloc(acceptlistbuf,
		       acceptlistsize*sizeof(struct accept));
	  if(!acceptlistbuf){
	       fprintf(stderr, "%s, line %d: realloc failed\n",
		       __FILE__, __LINE__);
	       exit(1);
	  }
     }
     acceptlistbuf[accept_count].ipaddr = ipaddr & mask;
     acceptlistbuf[accept_count].netmask = mask;
     acceptlistbuf[accept_count].flag = flag;
     accept_count++;
     if (accept_count > ACCEPT_TABLE_SIZE+1) {
	       fprintf(stderr, "%s, line %d: accept table overflow\n",
		       __FILE__, __LINE__);
	       exit(1);
     }
}

void
add_reject(unsigned int ipaddr, unsigned int mask, norminv flag)
{
     if(reject_count == rejectlistsize){
	  rejectlistsize *= 2;
	  rejectlistbuf = (struct reject *)
	       realloc(rejectlistbuf,
		       rejectlistsize*sizeof(struct reject));
	  if(!rejectlistbuf){
	       fprintf(stderr, "%s, line %d: realloc failed\n",
		       __FILE__, __LINE__);
	       exit(1);
	  }
     }
     rejectlistbuf[reject_count].ipaddr = ipaddr & mask;
     rejectlistbuf[reject_count].netmask = mask;
     rejectlistbuf[reject_count].flag = flag;
     reject_count++;
     if (reject_count > REJECT_TABLE_SIZE+1) {
	       fprintf(stderr, "%s, line %d: reject table overflow\n",
		       __FILE__, __LINE__);
	       exit(1);
     }
}

void
add_override(unsigned int ipaddr, unsigned int mask)
{
     struct classentry in_src[CLASS_SIZE];
     struct classentry in_dst[CLASS_SIZE];
     struct classentry out_dst[CLASS_SIZE];
     struct classentry udp_in[CLASS_SIZE];
     struct classentry icmp_in[CLASS_SIZE];

     if(override_count == overridelistsize){
	  overridelistsize *= 2;
	  overridelistbuf = (struct override *)
	       realloc(overridelistbuf,
		       overridelistsize*sizeof(struct override));
	  if(!overridelistbuf){
	       fprintf(stderr, "%s, line %d: realloc failed\n",
		       __FILE__, __LINE__);
	       exit(1);
	  }
     }
     overridelistbuf[override_count].ipaddr = ipaddr & mask;
     overridelistbuf[override_count].netmask = mask;
     gen_class(in_src, in_dst, out_dst, udp_in, icmp_in);
     destroy_all_services();
     if(in_src[0].start || in_dst[0].start || udp_in[0].start || icmp_in[0].start)
	  fprintf(stderr, "INFO: \"%s\", line %d: 'override' only supports 'out' TCP ports\n",
		  getfilename(), getlinenum());
     memcpy(overridelistbuf[override_count].ports,
	    out_dst, CLASS_SIZE * sizeof(struct classentry));
     override_count++;
     if (override_count > OVERRIDE_TABLE_SIZE+1) {
	       fprintf(stderr, "%s, line %d: override table overflow\n",
		       __FILE__, __LINE__);
	       exit(1);
     }
}
