/*
 * Copyright (c) 1993,1994,1995,1997,1998
 *      Texas A&M University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Texas A&M University
 *      and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Developers:
 *     Russell Neeper, David K. Hess, Douglas Lee Schales, David R. Safford
 */

#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "drawbrdg.h"
#include "dbmgr.h"
#include "tokens.h"
#include "misc.h"

#define __CLEAR__
#include "text.h"

/*------------------------------------------------------------*/

void
usageClear(int verbose)
{
        fprintf(theOutput, "%s%s",
                (verbose) ? clearText[0] : clearText[0]+1,
                (verbose) ? clearText[1] : "");
}

/*------------------------------------------------------------*/

static int
clearNetworkTables()
{
	networkTableReq u;
	networksTableReq r;
	int i;

	/*
	 * Get a list of the network tables loaded
	 */
	if (ioctl(theSocket, DIOCGNETWORKS, &r)) {
		dfm_perror("DIOCGNETWORKS");
		return ERROR;
	}

	/*
	 * Clear each table in the list
	 */
	for (i = 0; i < MAX_NETWORKS; i++) {
		if (r.table[i].network.s_addr) {
			/* Set which table to clear */
			u.network.s_addr = r.table[i].network.s_addr;
			u.netTableIndex = i;
			/* Send the clear request */
			if (ioctl(theSocket, DIOCRNETWORK, &u)) {
				dfm_perror("DIOCRNETWORK");
				return ERROR;
			}
		}
	}

	return NOERROR;
}

/*------------------------------------------------------------*/

static int
clearClassTables()
{
	register u_short i;
        accessListTableReq table;

	/* XXX
	 * The class table defaults are hard coded here in the clear
	 * routine and also in /sys/net/drawbridge.c.  This needs to be
	 * changed so that the defaults are in one place in a header file.
	 * For now, if these are changed, be sure to also change them in
	 * drawbridge.c.
	 */

	bzero(table.list, sizeof(table.list));
	table.list[0].begin = 25;
	table.list[0].end = 25;
	table.list[1].begin = 53;
	table.list[1].end = 53;

	/*
	 * IN class table
	 */
	for (i=0; i < MAX_ACCESS_LISTS; i++) {
		if (classTableReq(WRITE, IN_CLASS, i, &table))
			return ERROR;
		if (i == 0) {
			table.list[0].begin = table.list[0].end = 0;
			table.list[1].begin = table.list[1].end = 0;
		}
	}

	bzero(table.list, sizeof(table.list));
	table.list[0].begin = 1;
	table.list[0].end = 0xffff;

	/*
	 * OUT class table
	 */
	for (i=0; i < MAX_ACCESS_LISTS; i++) {
		if (classTableReq(WRITE, OUT_CLASS, i, &table))
			return ERROR;
		if (i == 0) {
			table.list[0].begin = table.list[0].end = 0;
			table.list[1].begin = table.list[1].end = 0;
		}
	}

	bzero(table.list, sizeof(table.list));
	table.list[0].begin = 20;
	table.list[0].end = 20;

	/*
	 * SOURCE class table
	 */
	for (i=0; i < MAX_ACCESS_LISTS; i++) {
		if (classTableReq(WRITE, SOURCE_CLASS, i, &table))
			return ERROR;
		if (i == 0) {
			table.list[0].begin = table.list[0].end = 0;
		}
	}

	bzero(table.list, sizeof(table.list));
	table.list[0].begin = 1;
	table.list[0].end = 68;
	table.list[1].begin = 70;
	table.list[1].end = 110;
	table.list[2].begin = 112;
	table.list[2].end = 0xffff;

	/*
	 * UDP class table
	 */
	for (i=0; i < MAX_ACCESS_LISTS; i++) {
		if (classTableReq(WRITE, UDP_CLASS, i, &table))
			return ERROR;
		if (i == 0) {
			table.list[0].begin = table.list[0].end = 0;
			table.list[1].begin = table.list[1].end = 0;
			table.list[2].begin = table.list[2].end = 0;
		}
	}

	return NOERROR;
}

/*------------------------------------------------------------*/

static int
clearAcceptTable()
{
	AcceptTableEntry acceptTable[MAX_ACCEPT_ENTRIES];

	/*
	 * Clear the table and send a write request
	 */
	bzero(acceptTable, sizeof(acceptTable));
	if (ioctl(theSocket, DIOCSACCEPT, acceptTable)) {
		dfm_perror("DIOCSACCEPT");
		return ERROR;
	}

	return NOERROR;
}

/*------------------------------------------------------------*/

static int
clearRejectTable()
{
	RejectTableEntry rejectTable[MAX_REJECT_ENTRIES];

	/*
	 * Clear the table and send a write request
	 */
	bzero(rejectTable, sizeof(rejectTable));
	if (ioctl(theSocket, DIOCSREJECT, rejectTable)) {
		dfm_perror("DIOCSREJECT");
		return ERROR;
	}

	return NOERROR;
}

/*------------------------------------------------------------*/

static int
clearOverrideTable()
{
	OverrideTableEntry overrideTable[MAX_OVERRIDE_ENTRIES];

	/*
	 * Clear the table and send a write request
	 */
	bzero(overrideTable, sizeof(overrideTable));
	if (ioctl(theSocket, DIOCSOVERRIDE, overrideTable)) {
		dfm_perror("DIOCSOVERRIDE");
		return ERROR;
	}

	return NOERROR;
}

/*------------------------------------------------------------*/

void
handleClear()
{
	int result;
	char *currTok;

	currTok = strtok(NULL, DELIMITERS);
	result = findtoken(currTok);

	switch (result) {

	case T_FILTERS:
		/* Clear all tables */
		if (clearNetworkTables() == ERROR)
			return;
		if (clearClassTables() == ERROR)
			return;
		if (clearAcceptTable() == ERROR)
			return;
		if (clearRejectTable() == ERROR)
			return;
		if (clearOverrideTable() == ERROR)
			return;
		break;

	case T_STATS:
		/* Send a clear stats request */
		if (ioctl(theSocket, DIOCCSTATS))
			dfm_perror("DIOCCSTATS");
		break;

	case T_BRIDGE:
		/* Send a clear bridge table request */
		if (ioctl(theSocket, DIOCCBRIDGE))
			dfm_perror("DIOCCBRIDGE");
		break;

	default:
		usageClear(TERSE);
	}
}

/*------------------------------------------------------------*/
