/*
 * Copyright (c) 1993,1994,1995,1997,1998
 *      Texas A&M University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Texas A&M University
 *      and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Developers:
 *     Russell Neeper, David K. Hess, Douglas Lee Schales, David R. Safford
 */

#ifndef _NET_DRAWBRIDGE_H_
#define _NET_DRAWBRIDGE_H_

#include <sys/types.h>
#include <sys/time.h>
#include <syslog.h>

/*==============================================================*/
/*                         Constants                            */
/*==============================================================*/

/* The version string is limited to 32 bytes (including the ending NULL)! */
#define VERSION			"3.0"

enum { NO, YES };
enum { DISCARD_PKT, ALLOW_PKT };
enum { READ, WRITE, REMOVE };
enum { IN_CLASS, OUT_CLASS, SOURCE_CLASS, UDP_CLASS, ICMP_CLASS };
enum { DEFAULT_ACCESS };
enum { NORMAL, INVERT };
enum { DB_DROP_PKT, DB_INSIDE_IF, DB_OUTSIDE_IF,
       DB_BOTH_IF, DB_FORWARD_PKT, DB_PROCESS_PKT }; 

/* -- The maximum number of networks must always be a power of two -- */
/* -- due to the hash function.  In this case 2^5 = 32 entries.    -- */
#define N_H_BITS		5
#define MAX_NETWORKS		(1UL << N_H_BITS)		/* 32 */
#define NET_HASH_SHIFT		(24 - N_H_BITS)			/* 19 */
#define NET_HASH_MASK		((MAX_NETWORKS - 1UL) \
				 << NET_HASH_SHIFT)		/* 0xF80000 */

/* -- The maximum number of bridge table entries must also always -- */
/* -- be a power of two.  In this case 2^13 = 8192 entries.       -- */
#define	A_H_BITS		13
#define MAX_BRIDGE_ADDRESSES	(1 << A_H_BITS)			/* 8192 */
#define ADDRESS_HASH_MASK	(MAX_BRIDGE_ADDRESSES - 1)	/* 0x1FFF */

#define MAX_ACCESS_LISTS	256
#define MAX_ACCESS_RANGES	32
#define MAX_REJECT_ENTRIES	32
#define MAX_ACCEPT_ENTRIES	32
#define MAX_OVERRIDE_ENTRIES	8

/*
 * Network packet defines
 */

#define SW_IP_OFFMASK		0xFF1F	/* byte swapped ip offset mask */

/* -- These should be defined in <net/if_llc.h> but are not -- */
#define LLC_SNAPLEN		8
#define llc_snap		llc_un.type_snap

/*
 * Syslog constants. 
 */

#define DEFAULT_SYSL_MASK	0		/* 0x1FFFF for all messages */
#define DEFAULT_SYSL_FACILITY	LOG_LOCAL0

enum {
	SYSL_UNKNOWN,
	SYSL_COLD_START,	/* cold start */
	SYSL_IN_CLASSD,
	SYSL_OUT_CLASSD,
	SYSL_IN_PORT,
	SYSL_OUT_PORT,		/* only TCP */
	SYSL_IN_TYPE,		/* only srcAddr, dstAddr, and icmp type */
	SYSL_OUT_TYPE,		/* only srcAddr, dstAddr, and icmp type */
	SYSL_IN_REJECT,		/* only protocolNo, srcAddr and dstAddr */
	SYSL_OUT_ACCEPT,	/* only protocolNo, srcAddr and dstAddr */
	SYSL_OUT_OVERRIDE,	/* only TCP */
	SYSL_IN_LENGTH,		/* TCP/UDP-Header maybe not initialized */
	SYSL_OUT_LENGTH,	/* TCP/UDP-Header maybe not initialized */
	SYSL_IN_DOS,		/* denial of service attack packet in */
	SYSL_OUT_DOS,		/* denial of service attack packet out */
	SYSL_IN_PROT,		/* only protocolNo, srcAddr and dstAddr */
	SYSL_OUT_PROT,		/* only protocolNo, srcAddr and dstAddr */
	SYSL_IN_OFFSET,		/* suspect fragment offset */
	SYSL_OUT_OFFSET,	/* suspect fragment offset */
	SYSL_IN_FRAG,		/* fragmented packet */
	SYSL_OUT_FRAG,		/* fragmented packet */
	SYSL_IN_FILTER,		/* no packet information */
	SYSL_OUT_FILTER		/* no packet information */
};

/*
 * Denial of service (DOS) attacks
 */
enum dos_type {
	DOS_SMURF, DOS_PONG
};

/*
 * Error messages.
 */
enum error_msg {
	NO_ERROR=0,
	ERROR_OUT_OF_RANGE=100,	ERROR_UNKNOWN_CLASS,
	ERROR_INVALID_NETWORK,	ERROR_TABLE_FULL,
	ERROR_NET_NOT_FOUND,	ERROR_NO_MEMORY,
	ERROR_INVALID_OP,	ERROR_IF_DOWN,
	ERROR_GET_HW,		ERROR_GET_IP,
	ERROR_INVALID_IF,	ERROR_SAME_IF,
	ERROR_DIFF_IF_TYPE,	ERROR_BAD_IF_TYPE,
	ERROR_INSIDE_DOWN,	ERROR_OUTSIDE_DOWN,
	ERROR_TWO_IP,		ERROR_NO_IP,
	ERROR_INVALID_MODE,	ERROR_NOT_INITILIZED,
	ERROR_RUNNING,		ERROR_NOT_RUNNING,
	MAX_ERROR_NUM  /* <- must be last */
};

/*
 * ioctl definitions
 */
#if (defined(sun) || defined(ibm032)) && !defined(__GNUC__)
/* This is for compatibility with Sun's pcc style header files */
#define DBIOC_GROUP D
#else
/* If you are using gcc this should work */
#define DBIOC_GROUP 'D'
#endif

#define DIOCINIT	_IOW (DBIOC_GROUP,  1, initReq)
#define DIOCSTART	_IO  (DBIOC_GROUP,  2)
#define DIOCSTOP	_IO  (DBIOC_GROUP,  3)
#define DIOCGFLAGS	_IOR (DBIOC_GROUP,  4, flagsReq)
#define DIOCSFLAGS	_IOW (DBIOC_GROUP,  5, flagsReq)
#define DIOCGSTATS	_IOR (DBIOC_GROUP,  6, Statistics)
#define DIOCCSTATS	_IO  (DBIOC_GROUP,  7)
#define DIOCGBRIDGE	_IOWR(DBIOC_GROUP,  8, bridgeReq)
#define DIOCCBRIDGE	_IO  (DBIOC_GROUP,  9)
#define DIOCGCLASS	_IOWR(DBIOC_GROUP, 10, accessListTableReq)
#define DIOCSCLASS	_IOWR(DBIOC_GROUP, 11, accessListTableReq)
#define DIOCGNETWORKS	_IOR (DBIOC_GROUP, 12, networksTableReq)
#define DIOCGNETWORK	_IOWR(DBIOC_GROUP, 13, networkTableReq)
#define DIOCSNETWORK	_IOWR(DBIOC_GROUP, 14, networkTableReq)
#define DIOCRNETWORK	_IOWR(DBIOC_GROUP, 15, networkTableReq)
#define DIOCGOVERRIDE	_IOR (DBIOC_GROUP, 16, overrideTableReq)
#define DIOCSOVERRIDE	_IOW (DBIOC_GROUP, 17, overrideTableReq)
#define DIOCGREJECT	_IOR (DBIOC_GROUP, 18, rejectTableReq)
#define DIOCSREJECT	_IOW (DBIOC_GROUP, 19, rejectTableReq)
#define DIOCGACCEPT	_IOR (DBIOC_GROUP, 20, acceptTableReq)
#define DIOCSACCEPT	_IOW (DBIOC_GROUP, 21, acceptTableReq)
#define DIOCGLOGF	_IOR (DBIOC_GROUP, 22, logFacilityReq)
#define DIOCSLOGF	_IOW (DBIOC_GROUP, 23, logFacilityReq)
#define DIOCGLOGM	_IOR (DBIOC_GROUP, 24, logMaskReq)
#define DIOCSLOGM	_IOW (DBIOC_GROUP, 25, logMaskReq)
#define DIOCGVER	_IOR (DBIOC_GROUP, 26, versionReq)


/*==============================================================*/
/*                           Structures                         */
/*==============================================================*/

typedef union _in_addr {
	u_int32_t s_addr;
	struct {
		u_int8_t s_b[4];
	} s_un_b;
	struct {
		u_int16_t s_w[2];
	} s_un_w;
} in_addr_t;

typedef union _HardwareAddr {
	u_int8_t  bytes[6];
	u_int16_t words[3];
	struct {
		u_int32_t l;
		u_int16_t w;
	} compare;
	struct {
		u_int8_t bytes[6];
	} addr;
} HardwareAddr;

typedef struct _FilterConfig {
	int		logFacility;
	u_int32_t	logMask;
#define CFG_CLR_SIZE	(sizeof(FilterConfig) - sizeof(u_int32_t) - sizeof(int))
#define CFG_CLR_START	in_name[0]
#define MAX_DEV_NAMELEN	16
	char		in_name[MAX_DEV_NAMELEN];
	char		out_name[MAX_DEV_NAMELEN];
	struct ifnet	*in_ifp;
	struct ifnet	*out_ifp;
	struct ifnet	*local_ifp;
	HardwareAddr	in_hwaddr;
	HardwareAddr	out_hwaddr;
	HardwareAddr	local_hwaddr;
	in_addr_t	in_IP;
	in_addr_t	out_IP;
	in_addr_t	local_IP;
	int		in_promiscuous;
	int		out_promiscuous;
	int		media_type;
	int		listenMode;
	int		discardMulticast;
	int		discardNonIp;
	int		discardOtherIp;
	int		discardSuspectOffset;
	int		discardFragmentedICMP;
	int		discardAttackICMP;
} FilterConfig;

typedef struct _Statistics {
	u_short		running;
	struct timeval	startTime;
	u_short		bridgeEntries;
	int		media_type;
#define STATS_CLR_SIZE	( sizeof(Statistics) \
			  - (sizeof(u_short) * 2) \
			  - sizeof(struct timeval) \
			  - sizeof(int) )
#define STATS_CLR_START	droppedIfDown
	u_int32_t	droppedIfDown;
	u_int32_t	droppedIfQfull;
	u_int32_t	droppedNoMbufs;
	u_int64_t	p_insideFiltered;
	u_int64_t	p_outsideFiltered;
	u_int64_t	p_insideRx;
	u_int64_t	p_outsideRx;
	u_int64_t	p_insideTx;
	u_int64_t	p_outsideTx;
	u_int64_t	b_insideFiltered;
	u_int64_t	b_outsideFiltered;
	u_int64_t	b_insideRx;
	u_int64_t	b_outsideRx;
	u_int64_t	b_insideTx;
	u_int64_t	b_outsideTx;

	u_int64_t	f_insideTcpMcast;
	u_int64_t	f_outsideTcpMcast;
	u_int64_t	f_insideUdpMcast;
	u_int64_t	f_outsideUdpMcast;
	u_int64_t	f_insideTcpPort;
	u_int64_t	f_insideTcpPortOverride;
	u_int64_t	f_outsideTcpPort;
	u_int64_t	f_insideUdpPort;	/* not currently used */
	u_int64_t	f_outsideUdpPort;
	u_int64_t	f_insideIcmpType;	/* not currently used */
	u_int64_t	f_outsideIcmpType;

	u_int64_t	f_outsideRejectTable;
	u_int64_t	f_outsideSuspectOffset;
	u_int64_t	f_outsideIcmpFrag;
	u_int64_t	f_outsideShortTcpHdr;
	u_int64_t	f_outsideShortUdpHdr;
	u_int64_t	f_outsideSmurfDos;
	u_int64_t	f_outsidePongDos;
	u_int64_t	f_outsideOtherIp;
	u_int64_t	f_outsideNonIp;

	u_int64_t	f_insideAcceptTable;
	u_int64_t	f_insideSuspectOffset;
	u_int64_t	f_insideIcmpFrag;	/* not currently used */
	u_int64_t	f_insideShortTcpHdr;
	u_int64_t	f_insideShortUdpHdr;
	u_int64_t	f_insideSmurfDos;	/* not currently used */
	u_int64_t	f_insidePongDos;	/* not currently used */
	u_int64_t	f_insideOtherIp;
	u_int64_t	f_insideNonIp;
} Statistics;

typedef struct _SyslogMessageEntry {
	unsigned char *message;
	unsigned char priority;
} SyslogMessageEntry;


/*
 * Structures for tables.
 */

typedef struct _NetTableEntry {
	in_addr_t network;
	u_int8_t *hostTable;
} NetTableEntry;

typedef struct _AccessListTableEntry {
	u_int16_t begin;
	u_int16_t end;
} AccessListTableEntry;

typedef struct _OverrideTableEntry {
	in_addr_t network;
	u_int32_t mask;
	AccessListTableEntry access[MAX_ACCESS_RANGES];
} OverrideTableEntry;

typedef struct _RejectTableEntry {
	in_addr_t network;
	u_int32_t mask;
	u_int32_t flag;
} RejectTableEntry;

typedef struct _AcceptTableEntry {
	in_addr_t network;
	u_int32_t mask;
	u_int32_t flag;
} AcceptTableEntry;

/* Note: it's assumed that this struct will always be 8 bytes long */
typedef struct _HashEntry {
	HardwareAddr address;
	u_int8_t interface;
	u_int8_t valid;
} HashEntry;

/*
 * Dfm request structures for ioctls
 */

typedef struct _initReq {
	char inName[MAX_DEV_NAMELEN];
	char outName[MAX_DEV_NAMELEN];
        int listenMode;
} initReq;

typedef struct _flagsReq {
	int discardMulticast;
	int discardNonIp;
	int discardOtherIp;
	int discardSuspectOffset;
	int discardFragmentedICMP;
	int discardAttackICMP;
} flagsReq;

typedef struct _bridgeReq {
	HashEntry t;
	u_int32_t index;
} bridgeReq;

typedef struct _accessListTableReq {
	u_int8_t  class;
	u_int16_t row;
	AccessListTableEntry list[MAX_ACCESS_RANGES];
} accessListTableReq;   

typedef struct _networksTableReq {
	NetTableEntry table[MAX_NETWORKS];
} networksTableReq;

typedef struct _networkTableReq {
        in_addr_t network;
	u_int16_t netTableIndex;
	u_int32_t offset;
	u_int8_t  table[256];
} networkTableReq;

typedef struct _overrideTableReq {
	OverrideTableEntry table[MAX_OVERRIDE_ENTRIES];
} overrideTableReq;

typedef struct _rejectTableReq {
	RejectTableEntry table[MAX_REJECT_ENTRIES];
} rejectTableReq;

typedef struct _acceptTableReq {
	AcceptTableEntry table[MAX_ACCEPT_ENTRIES];
} acceptTableReq;

typedef struct _logFacilityReq {
	int	facility;
} logFacilityReq;

typedef struct _logMaskReq {
	u_int32_t mask;
} logMaskReq;

typedef struct _versionReq {
	u_int8_t version[32];
} versionReq;

/*==============================================================*/
/*                              Macros                          */
/*==============================================================*/

#define IS_BROADCAST(address) \
	(*((u_int8_t *) (address)) & 0x01)

#define IS_HW_ADDR_EQU(x1, x2) \
	(((x1)->compare.l == (x2).compare.l) && \
	 ((x1)->compare.w == (x2).compare.w))

#define ARP_TARGET_IP_ADDR(x) \
	(*((u_int32_t *) (((struct ether_arp *) (x))->arp_tpa)))

#ifdef DBDEBUG
#define DB_PRINTF(x)   printf x
#else
#define DB_PRINTF(x)
#endif


/*==============================================================*/
/*                            Prototypes                        */
/*==============================================================*/

#ifdef KERNEL
int drawbridge_bridge_lookup __P((HardwareAddr *, struct ifnet *));
int drawbridge_test_pkt __P((struct ifnet **, void *, struct mbuf *));
#endif /* ifdef KERNEL */

#endif /* #ifndef _NET_DRAWBRIDGE_H_ */
