/*
 * Copyright (c) 1993,1994,1995,1997,1998
 *      Texas A&M University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Texas A&M University
 *      and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Developers:
 *     Russell Neeper, David K. Hess, Douglas Lee Schales, David R. Safford
 */

#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "drawbrdg.h"
#include "dbmgr.h"
#include "tokens.h"
#include "misc.h"

#define __LOAD__
#include "text.h"

/*------------------------------------------------------------*/

void
usageLoad(int verbose)
{
	fprintf(theOutput, "%s%s",
		(verbose) ? loadText[0] : loadText[0]+1,
		(verbose) ? loadText[1] : "");
}

/*------------------------------------------------------------*/

static int
loadNetworkTable(char *file, FILE *fp, u_int offset)
{
	u_long size;
	register u_long strip;
	networkTableReq u;
	struct in_addr ha;

	/*
	 * Seek to the networks table
	 */
	fseek(fp, offset, SEEK_SET);

	if (fread(&u.network.s_addr, sizeof(u.network.s_addr), 1, fp) < 1) {
		fprintf(stderr,"   Error reading from %s\n", file);
		return ERROR;
	}

	while (u.network.s_addr != 0xFFFFFFFF) {

		ha.s_addr = u.network.s_addr;
		u.network.s_addr = ntohl(u.network.s_addr);

		if (IN_CLASSB(u.network.s_addr)) {
			u.network.s_addr = u.network.s_addr & IN_CLASSB_NET;
			size = 0x10000UL;
		} else if (IN_CLASSC(u.network.s_addr)) {
			u.network.s_addr = u.network.s_addr & IN_CLASSC_NET;
			size = 0x100UL;
		} else {
			/* We don't handle class A or D addresses. */
			fprintf(theErrors,
				"   %s: Can't handle class A or D addresses\n",
				inet_ntoa(ha));
			return ERROR;
		}
	
		u.offset = 0;
		u.netTableIndex = 0xFFFF;
		
		for (strip = size/sizeof(u.table); strip; strip--) {
			if (fread(u.table, sizeof(u.table), 1, fp) != 1) {
				fprintf(theErrors,
					"   Error reading network %s " \
					"from '%s'\n", inet_ntoa(ha), file);
				return ERROR;
			}
			if (ioctl(theSocket, DIOCSNETWORK, &u)) {
				dfm_perror("DIOCSNETWORK");
				return ERROR;
			}
			u.offset += sizeof(u.table);
		}

		fprintf(theOutput, "   Network %s loaded\n",
			inet_ntoa(ha));

	        if (fread(&u.network.s_addr,sizeof(u.network.s_addr),1,fp)<1) {
			fprintf(stderr,"   Error reading from %s\n", file);
			return ERROR;
		}
	}

	return NOERROR;
}

/*------------------------------------------------------------*/

static int
readClassTableRow(FILE *fd, AccessListTableEntry *p)
{
	register u_short i;

	for (i=0; i < MAX_ACCESS_RANGES; i++) {
		if (fread(p, sizeof(AccessListTableEntry), 1, fd) != 1)
			return ERROR;
		p->begin = ntohs(p->begin);
		p->end = ntohs(p->end);
		p++;
	}
	return NOERROR;
}

/*------------------------------------------------------------*/

static int
loadClassTables(char *file, FILE *fp, u_int offset)
{
	register u_short i;
        accessListTableReq table;

	/*
	 * Seek to the classes table
	 */
	fseek(fp, offset, SEEK_SET);

	/*
	 * IN class table
	 */
	for (i=0; i < MAX_ACCESS_LISTS; i++) {
		if (readClassTableRow(fp, table.list)) {  
			fprintf(theErrors,
				"   Error reading 'in' access table\n");
			return ERROR;
		}
		if (classTableReq(WRITE, IN_CLASS, i, &table))
			return ERROR;
	}

	/*
	 * OUT class table
	 */
	for (i=0; i < MAX_ACCESS_LISTS; i++) {
		if (readClassTableRow(fp, table.list)) {  
			fprintf(theErrors,
				"   Error reading 'out' access table\n");
			return ERROR;
		}
		if (classTableReq(WRITE, OUT_CLASS, i, &table))
			return ERROR;
	}

	/*
	 * SOURCE class table
	 */
	for (i=0; i < MAX_ACCESS_LISTS; i++) {
		if (readClassTableRow(fp, table.list)) {  
			fprintf(theErrors,
				"   Error reading 'source' access table\n");
			return ERROR;
		}
		if (classTableReq(WRITE, SOURCE_CLASS, i, &table))
			return ERROR;
	}

	/*
	 * UDP class table
	 */
	for (i=0; i < MAX_ACCESS_LISTS; i++) {
		if (readClassTableRow(fp, table.list)) {  
			fprintf(theErrors,
				"   Error reading 'udp' access table\n");
			return ERROR;
		}
		if (classTableReq(WRITE, UDP_CLASS, i, &table))
			return ERROR;
	}

	/*
	 * ICMP class table
	 */
	for (i=0; i < MAX_ACCESS_LISTS; i++) {
		if (readClassTableRow(fp, table.list)) {  
			fprintf(theErrors,
				"   Error reading 'icmp' access table\n");
			return ERROR;
		}
		if (classTableReq(WRITE, ICMP_CLASS, i, &table))
			return ERROR;
	}


	printf("   Class tables loaded\n");
	return NOERROR;
}

/*------------------------------------------------------------*/

static int
loadAcceptTable(char *file, FILE *fp, u_int offset)
{
	register u_short i;
	u_int count;
	AcceptTableEntry acceptTable[MAX_ACCEPT_ENTRIES];

	bzero(acceptTable, sizeof(acceptTable));

	/*
	 * Seek to the reject table and read the table size
	 */
	fseek(fp, offset, SEEK_SET);
	fread(&count, sizeof(count), 1, fp);
	count = ntohl(count);

	for (i=0; (i < count) && (i < MAX_ACCEPT_ENTRIES); i++) {

		if (fread(&acceptTable[i], sizeof(AcceptTableEntry),1,fp) < 1) {
			fprintf(stderr,
				"   Error reading from %s\n", file);
			return ERROR;
		}

		acceptTable[i].network.s_addr =
			ntohl(acceptTable[i].network.s_addr);
		acceptTable[i].mask = ntohl(acceptTable[i].mask);
		acceptTable[i].flag = ntohl(acceptTable[i].flag);
	}

	if (ioctl(theSocket, DIOCSACCEPT, acceptTable)) {
		dfm_perror("DIOCSACCEPT");
		return ERROR;
	}

	printf("   Accept table loaded: %d entries\n", i);
	return NOERROR;
}

/*------------------------------------------------------------*/

static int
loadRejectTable(char *file, FILE *fp, u_int offset)
{
	register u_short i;
	u_int count;
	RejectTableEntry rejectTable[MAX_REJECT_ENTRIES];

	bzero(rejectTable, sizeof(rejectTable));

	/*
	 * Seek to the reject table and read the table size
	 */
	fseek(fp, offset, SEEK_SET);
	fread(&count, sizeof(count), 1, fp);
	count = ntohl(count);

	for (i=0; (i < count) && (i < MAX_REJECT_ENTRIES); i++) {

		if (fread(&rejectTable[i],sizeof(RejectTableEntry),1,fp) < 1) {
			fprintf(stderr,
				"   Error reading from %s\n", file);
			return ERROR;
		}

		rejectTable[i].network.s_addr =
			ntohl(rejectTable[i].network.s_addr);
		rejectTable[i].mask = ntohl(rejectTable[i].mask);
		rejectTable[i].flag = ntohl(rejectTable[i].flag);
	}

	if (ioctl(theSocket, DIOCSREJECT, rejectTable)) {
		dfm_perror("DIOCSREJECT");
		return ERROR;
	}

	printf("   Reject table loaded: %d entries\n", i);
	return NOERROR;
}

/*------------------------------------------------------------*/

static int
loadOverrideTable(char *file, FILE *fp, u_int offset)
{
	register u_short i;
	u_int count;
	OverrideTableEntry overrideTable[MAX_OVERRIDE_ENTRIES];

	bzero(overrideTable, sizeof(overrideTable));

	/*
	 * Seek to the override table and read the table size
	 */
	fseek(fp, offset, SEEK_SET);
	fread(&count, sizeof(count), 1, fp);
	count = ntohl(count);

	for (i=0; (i < count) && (i < MAX_OVERRIDE_ENTRIES); i++) {
		register u_short j;

		if (fread(&overrideTable[i],
			  sizeof(OverrideTableEntry), 1, fp) < 1) {
			fprintf(stderr, "   Error reading from %s\n", file);
			return ERROR;
		}

		overrideTable[i].network.s_addr =
			 ntohl(overrideTable[i].network.s_addr);
		overrideTable[i].mask = ntohl(overrideTable[i].mask);

		for (j=0; j < MAX_ACCESS_RANGES; j++) {
			overrideTable[i].access[j].begin =
				ntohs(overrideTable[i].access[j].begin);
			overrideTable[i].access[j].end =
				ntohs(overrideTable[i].access[j].end);
		}
	}
		
	if (ioctl(theSocket, DIOCSOVERRIDE, overrideTable)) {
		dfm_perror("DIOCSOVERRIDE");
		return ERROR;
	}

	printf("   Override table loaded: %d entries\n", i);
	return NOERROR;
}

/*------------------------------------------------------------*/

void
handleLoad()
{
	char *currTok;
	FILE *fp;
	struct {
		unsigned int networks_offset;
		unsigned int classes_offset;
		unsigned int accept_offset;
		unsigned int reject_offset;
		unsigned int override_offset;
	} header;

	/*
	 * Make sure that we have all the parameters
	 */
	currTok = strtok(NULL, DELIMITERS);
	if ((findtoken(currTok) != T_FILTERS) ||
	    !(currTok = strtok(NULL, DELIMITERS)) ) {
		usageLoad(TERSE);
		return;
	}

	/*
	 * Open the specified file
	 */
	if (!(fp = fopen(currTok, "r"))) {
		fprintf(theErrors, "   Can't open file '%s'\n", currTok);
		return;
	}

	/*
	 * Read the table offsets and convert them to host byte order
	 */
	if (fread(&header, sizeof(header), 1, fp) < 1) {
		fprintf(theErrors, "   Error reading from file '%s'\n",
			currTok);
		return;
	}
	header.networks_offset = ntohl(header.networks_offset);
	header.classes_offset = ntohl(header.classes_offset);
	header.accept_offset = ntohl(header.accept_offset);
	header.reject_offset = ntohl(header.reject_offset);
	header.override_offset = ntohl(header.override_offset);

	/*
	 * Load all the tables from the filter table file
	 */
	if (loadNetworkTable(currTok, fp, header.networks_offset) == ERROR)
		return;
	if (loadClassTables(currTok, fp, header.classes_offset) == ERROR)
		return;
	if (loadAcceptTable(currTok, fp, header.accept_offset) == ERROR)
		return;
	if (loadRejectTable(currTok, fp, header.reject_offset) == ERROR)
		return;
	if (loadOverrideTable(currTok, fp, header.override_offset) == ERROR)
		return;
}

/*------------------------------------------------------------*/
