/*
 * Copyright (c) 1993,1994,1995,1997,1998
 *      Texas A&M University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Texas A&M University
 *      and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Developers:
 *     Russell Neeper, David K. Hess, Douglas Lee Schales, David R. Safford
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <ncurses.h>
#include <signal.h>
#include <net/if_types.h>

#include "drawbrdg.h"
#include "dbmgr.h"
#include "tokens.h"
#include "misc.h"

#define __MONITOR__
#include "text.h"

#define PER_SEC(a,b)	(b)?(u_int)(((float)(a)/(float)(b))*(float)1000000):0

static volatile int received_sig;
static firsttime;
static struct sigaction sigint_save;
static struct sigaction sigalrm_save;
static struct sigaction sigwinch_save;
sigset_t sigmask_save;
struct itimerval timer_save;
 
/*------------------------------------------------------------*/

void
usageMonitor(int verbose)
{
	fprintf(theOutput, "%s%s",
		(verbose) ? monitorText[0] : monitorText[0]+1,
		(verbose) ? monitorText[1] : "");
}

/*------------------------------------------------------------*/

static void  
sig_handler(int sig)
{
	received_sig = sig;
}

/*------------------------------------------------------------*/

static void
init_stats_screen()
{
	struct sigaction sa;
	sigset_t sig_mask;
	
	initscr();
	cbreak(); noecho(); timeout(0); clear();

	/* block signals */
	sigemptyset(&sig_mask);
	sigaddset(&sig_mask, SIGINT);
	sigaddset(&sig_mask, SIGALRM);
	sigaddset(&sig_mask, SIGWINCH);
	sigprocmask(SIG_BLOCK, &sig_mask, &sigmask_save);

	/* set up handler for signals */
	sa.sa_handler = sig_handler;
	sigfillset(&sa.sa_mask);
	sa.sa_flags = SA_RESTART;
	sigaction(SIGINT, &sa, &sigint_save);
	sigaction(SIGALRM, &sa, &sigalrm_save);
	sigaction(SIGWINCH, &sa, &sigwinch_save);
}

/*------------------------------------------------------------*/

static void
draw_screen_1(Statistics theStats, Statistics lastStats,
		 u_int p_ag, u_int p_ag_peak,
		 u_int b_ag, u_int b_ag_peak,
		 u_long usecs)
{
	printw("Packets%33s%34s\n", "Inside", "Outside");
	printw("-------%33s%34s\n", "------", "-------");
	printw("Filtered   %25s / %-6s",
		fmt_big(theStats.p_insideFiltered),
		fmt_small(PER_SEC(theStats.p_insideFiltered -
			lastStats.p_insideFiltered, usecs), YES));
	printw("%25s / %-6s\n",
		fmt_big(theStats.p_outsideFiltered),
		fmt_small(PER_SEC(theStats.p_outsideFiltered -
			lastStats.p_outsideFiltered, usecs),YES));
	printw("Received   %25s / %-6s",
		fmt_big(theStats.p_insideRx),
		fmt_small(PER_SEC(theStats.p_insideRx -
			lastStats.p_insideRx, usecs), YES));
	printw("%25s / %-6s\n",
		fmt_big(theStats.p_outsideRx),
		fmt_small(PER_SEC(theStats.p_outsideRx -
			   lastStats.p_outsideRx, usecs), YES));
	printw("Transmitted%25s / %-6s",
		fmt_big(theStats.p_insideTx),
		fmt_small(PER_SEC(theStats.p_insideTx -
			lastStats.p_insideTx, usecs), YES));
	printw("%25s / %-6s\n\n",
		fmt_big(theStats.p_outsideTx),
		fmt_small(PER_SEC(theStats.p_outsideTx -
			lastStats.p_outsideTx, usecs), YES));


	printw("Bytes%35s%34s\n", "Inside", "Outside");
	printw("-----%35s%34s\n", "------", "-------");
	printw("Filtered   %25s / %-6s",
		fmt_big(theStats.b_insideFiltered),
		fmt_abbrv(PER_SEC(theStats.b_insideFiltered -
			lastStats.b_insideFiltered, usecs)));
	printw("%25s / %-6s\n",
		fmt_big(theStats.b_outsideFiltered),
		fmt_abbrv(PER_SEC(theStats.b_outsideFiltered -
			lastStats.b_outsideFiltered, usecs)));
	printw("Received   %25s / %-6s",
		fmt_big(theStats.b_insideRx),
		fmt_abbrv(PER_SEC(theStats.b_insideRx -
			lastStats.b_insideRx, usecs)));
	printw("%25s / %-6s\n",
		fmt_big(theStats.b_outsideRx),
		fmt_abbrv(PER_SEC(theStats.b_outsideRx -
			lastStats.b_outsideRx, usecs)));
	printw("Transmitted%25s / %-6s",
		fmt_big(theStats.b_insideTx),
		fmt_abbrv(PER_SEC(theStats.b_insideTx -
			lastStats.b_insideTx, usecs)));
	printw("%25s / %-6s\n\n",
		fmt_big(theStats.b_outsideTx),
		fmt_abbrv(PER_SEC(theStats.b_outsideTx -
			lastStats.b_outsideTx, usecs)));

	printw("%28s%39s\n", "Aggregate Throughput","Dropped Packets");
	printw("%28s%39s\n", "--------------------","---------------");
	printw("Packets/sec:%11s", fmt_small(p_ag, NO));
	printw("%11s peak       No kernel buffers:   ",
		fmt_small(p_ag_peak, NO));
	printw("%11s\n", fmt_small(theStats.droppedNoMbufs, NO));

	printw("Bits/sec   :%11s", fmt_abbrv(b_ag));
	printw("%11s peak       Interface queue full:",
		fmt_abbrv(b_ag_peak));
	printw("%11s\n\n", fmt_small(theStats.droppedIfQfull, NO));

	printw("Bridge table entries: %s\n\n",
		fmt_small(theStats.bridgeEntries, NO));
}

/*------------------------------------------------------------*/

static void
draw_screen_2(Statistics theStats)
{
	printw("Filter Description%32s%27s\n",
		"Inside Interface", "Outside Interface");
	printw("---------------------   --------------------------" \
	       "   ------------------------\n");
	printw("%-24s", "TCP Override Tbl Denied");
	printw("%26s ", fmt_big(theStats.f_insideTcpPortOverride));
	printw("%26s\n", "n/a");

	printw("%-24s", "TCP Port Denied");
	printw("%26s ", fmt_big(theStats.f_insideTcpPort));
	printw("%26s\n", fmt_big(theStats.f_outsideTcpPort));

	printw("%-24s", "UDP Port Denied");
	printw("%26s ", "n/a");
	printw("%26s\n", fmt_big(theStats.f_outsideUdpPort));

	printw("%-24s", "ICMP Type Denied");
	printw("%26s ", "n/a");
	printw("%26s\n", fmt_big(theStats.f_outsideIcmpType));

	printw("%-24s", "Fragmented ICMP");
	printw("%26s ", "n/a");
	printw("%26s\n", fmt_big(theStats.f_outsideIcmpFrag));

	printw("%-24s", "ICMP \"Smurf\" Attack");
	printw("%26s ", "n/a");
	printw("%26s\n", fmt_big(theStats.f_outsideSmurfDos));

	printw("%-24s", "ICMP \"Pong\" Attack");
	printw("%26s ", "n/a");
	printw("%26s\n", fmt_big(theStats.f_outsidePongDos));

	printw("%-24s", "Accept/Reject Tables");
	printw("%26s ", fmt_big(theStats.f_insideAcceptTable));
	printw("%26s\n", fmt_big(theStats.f_outsideRejectTable));

	printw("%-24s", "Suspect Fragment Offset");
	printw("%26s ", fmt_big(theStats.f_insideSuspectOffset));
	printw("%26s\n", fmt_big(theStats.f_outsideSuspectOffset));

	printw("%-24s", "TCP Header Too Short");
	printw("%26s ", fmt_big(theStats.f_insideShortTcpHdr));
	printw("%26s\n", fmt_big(theStats.f_outsideShortTcpHdr));

	printw("%-24s", "UDP Header Too Short");
	printw("%26s ", fmt_big(theStats.f_insideShortUdpHdr));
	printw("%26s\n", fmt_big(theStats.f_outsideShortUdpHdr));

	printw("%-24s", "TCP Multicast");
	printw("%26s ", fmt_big(theStats.f_insideTcpMcast));
	printw("%26s\n", fmt_big(theStats.f_outsideTcpMcast));

	printw("%-24s", "UDP Multicast");
	printw("%26s ", fmt_big(theStats.f_insideUdpMcast));
	printw("%26s\n", fmt_big(theStats.f_outsideUdpMcast));

	printw("%-24s", "Other IP Protocol");
	printw("%26s ", fmt_big(theStats.f_insideOtherIp));
	printw("%26s\n", fmt_big(theStats.f_outsideOtherIp));

	printw("%-24s", "Non IP Protocol");
	printw("%26s ", fmt_big(theStats.f_insideNonIp));
	printw("%26s\n", fmt_big(theStats.f_outsideNonIp));

	printw("\n\n");
}

/*------------------------------------------------------------*/

static int
display_stats(int screen, int update)
{
	static Statistics theStats;
	static Statistics lastStats;
	static u_int p_ag_peak = 0;
	static u_int b_ag_peak = 0;
	static u_int p_ag, b_ag;
	static struct timeval thisTime, lastTime;
	static u_long usecs = 0;
	u_long upTime, upDays, upHours, upMinutes, upSeconds;

	if (!update || firsttime) {

		lastStats = theStats;
		lastTime = thisTime;

   		if (ioctl(theSocket, DIOCGSTATS, &theStats)) {
			dfm_perror("DIOCGFLAGS");
			return ERROR;
		}
		gettimeofday(&thisTime, NULL);

		if (firsttime) {
			lastStats = theStats;
			lastTime = thisTime;
			firsttime = 0;
		}

		usecs = ((thisTime.tv_sec * 1000000) + thisTime.tv_usec) -
			   ((lastTime.tv_sec * 1000000) + lastTime.tv_usec);

		p_ag = (theStats.p_insideTx - lastStats.p_insideTx) +
		       (theStats.p_outsideTx - lastStats.p_outsideTx);

		p_ag = PER_SEC(p_ag, usecs); /* packets per second */
		if (p_ag > p_ag_peak)
			p_ag_peak = p_ag;
	
		b_ag = (theStats.b_insideTx - lastStats.b_insideTx) +
		       (theStats.b_outsideTx - lastStats.b_outsideTx);
		b_ag = PER_SEC(b_ag, usecs); /* bytes per second */

		if (theStats.media_type == IFT_ETHER)
			/* Calculate bits per second:
				9.6 microsecond interpacket gap = 96 bits
				preamble + start + frame check = 96 bits
			 */
			/*b_ag = (p_ag * (96 + 100)) + (b_ag * 8); */
			b_ag = (p_ag * (96 + 96)) + (b_ag * 8);
		else if (theStats.media_type == IFT_FDDI)
			/* Calculate bits per second:
				16 symbol interpacket gap = 64 bits
				start + frame ctl + frame check + end +
					status = 16 symbols = 64 bits
			*/
			b_ag = (p_ag * (64 + 64)) + (b_ag * 8);
		else b_ag = 0;
			
		if (b_ag > b_ag_peak)
			b_ag_peak = b_ag;
	}

	move(0,0);
	printw("\n%48s\n\n","--- DRAWBRIDGE STATS ---");

	if (screen == 1) {
		draw_screen_1(theStats, lastStats,
				 p_ag, p_ag_peak,
				 b_ag, b_ag_peak,
				 usecs);
			
	} else if (screen == 2) {
		draw_screen_2(theStats);
	}

	if (theStats.running) {
		upDays = upHours = upMinutes = upSeconds = 0;
		upTime = time(NULL) - theStats.startTime.tv_sec;
		if (upTime) {
			upDays = upTime / (24UL * 60UL * 60UL);
			upTime -= upDays * 24UL * 60UL * 60UL;
			upHours = upTime / (60 * 60);
			upTime -= upHours * 60 * 60;
			upMinutes = upTime / 60;
			upSeconds = upTime - upMinutes * 60;
		}
		printw("Uptime: %lu days %lu hours %lu minutes %lu " \
			"seconds\n", upDays, upHours, upMinutes,
			 upSeconds);
	} else
		printw("Drawbridge is not running.\n");

	printw("\t    (Press <space bar> to toggle screens; <q> to quit)");
	refresh();
	return NOERROR;
}

/*------------------------------------------------------------*/

void
handleMonitor()
{
	u_int seconds = 1;
	u_int screen = 1;
	struct itimerval timer;
	sigset_t sig_mask;
	int c, stop, ret;
	char *currTok;

	currTok = strtok(NULL, DELIMITERS);
	if (currTok) {
		if ((findtoken(currTok) != T_INTERVAL) ||
		    ((currTok = strtok(NULL, DELIMITERS)) == NULL)) {
			usageMonitor(TERSE);
			return;
		}
		seconds = atoi(currTok);
		if ((seconds == 0) || (seconds > 60))
			seconds = 1;
	}

	init_stats_screen();

	/* set up sigsuspend sig mask */
	sigfillset(&sig_mask);
	sigdelset(&sig_mask, SIGINT);
	sigdelset(&sig_mask, SIGALRM);
	sigdelset(&sig_mask, SIGWINCH);

	/* set up interval timer */
	timer.it_value.tv_sec = 0;
	timer.it_value.tv_usec = 10000;
	timer.it_interval.tv_sec = seconds;
	timer.it_interval.tv_usec = 0;
	if (setitimer(ITIMER_REAL, &timer, &timer_save) != 0) {
		fprintf(stderr, "%s, %d, setitimer failed: %s",
			__FILE__, __LINE__, strerror(errno));
		exit(1);
	}

	firsttime = 1;
	for(ret=NOERROR,stop=0; !stop;) {

		received_sig = -1;
		sigsuspend(&sig_mask);
		switch (received_sig) {

		case SIGINT:
			stop = 1;
			break;

		case SIGWINCH:
			endwin();
			init_stats_screen();
			if (display_stats(screen, 1) == ERROR) {
				stop = 1;
				ret = ERROR;
			}
			break;

		case SIGALRM:
			c = getch();
			switch (c) {
			case ' ':
				screen = (screen == 1) ? 2 : 1;
				break;
			case 'q':
				stop = 1;
				break;
			}

			if (!stop &&
			    display_stats(screen, 0) == ERROR) {
				stop = 1;
				ret = ERROR;
			}
			break;
		}
	}

	printw("\r%65s", " ");
	refresh();
	endwin();

	/* disable timer and restore signal handlers */
	setitimer(ITIMER_REAL, &timer_save, NULL);
	sigaction(SIGINT, &sigint_save, NULL);
	sigaction(SIGALRM, &sigalrm_save, NULL);
	sigaction(SIGWINCH, &sigwinch_save, NULL);
	sigprocmask(SIG_SETMASK, &sigmask_save, NULL);
}

/*------------------------------------------------------------*/
