/*
 * Copyright (c) 1993,1994,1995,1997,1998
 *      Texas A&M University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Texas A&M University
 *      and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Developers:
 *     Russell Neeper, David K. Hess, Douglas Lee Schales, David R. Safford
 */

#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>
#include <netdb.h>
#include <ctype.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdlib.h>

#include "drawbrdg.h"
#include "dbmgr.h"
#include "tokens.h"
#include "misc.h"

#define __SHOW__
#include "text.h"

/*------------------------------------------------------------*/

void
usageShow(int verbose)
{
	fprintf(theOutput, "%s%s",
		(verbose) ? showText[0] : showText[0]+1,
		(verbose) ? showText[1] : "");
}

/*------------------------------------------------------------*/

static int
queryAcceptTable()
{
	register u_short i;
	AcceptTableEntry acceptTable[MAX_ACCEPT_ENTRIES];

	if (ioctl(theSocket, DIOCGACCEPT, acceptTable)) {
		dfm_perror("DIOCGACCEPT");
		return ERROR;
	}

	for (i=0; i < MAX_ACCEPT_ENTRIES; i++) {
		struct in_addr a,m;

		if ((acceptTable[i].network.s_addr == 0) &&
		    (acceptTable[i].mask == 0))
			break;

		if (i == 0)
			fprintf(theOutput, "   Packets from:\n");

		a.s_addr = htonl(acceptTable[i].network.s_addr);
		m.s_addr = htonl(acceptTable[i].mask);
		fprintf(theOutput, "      inside address %-15s ", inet_ntoa(a));
		fprintf(theOutput, "mask %-15s  %s\n", inet_ntoa(m),
			acceptTable[i].flag ? "discard" : "process");
	}

	if (i == 0) {
		fprintf(theOutput, "   Accept address table is empty\n");
	} else
		fprintf(theOutput, "   All other IP addresses from the " \
			"inside will be discarded.\n");

	return NOERROR;
}

/*------------------------------------------------------------*/

static int
queryRejectTable()
{
	register u_short i;
	RejectTableEntry rejectTable[MAX_REJECT_ENTRIES];

	if (ioctl(theSocket, DIOCGREJECT, rejectTable)) {
		dfm_perror("DIOCGREJECT");
		return ERROR;
	}

	for (i=0; i < MAX_REJECT_ENTRIES; i++) {
		struct in_addr a,m;

		if ((rejectTable[i].network.s_addr == 0) &&
		    (rejectTable[i].mask == 0))
			break;

		if (i == 0)
			fprintf(theOutput, "   Packets from:\n");
		a.s_addr = htonl(rejectTable[i].network.s_addr);
		m.s_addr = htonl(rejectTable[i].mask);
		fprintf(theOutput, "      outside address %-15s ",inet_ntoa(a));
		fprintf(theOutput, "mask %-15s  %s\n", inet_ntoa(m),
			rejectTable[i].flag ? "process" : "discard");
	}

	if (i == 0)
		fprintf(theOutput, "   Reject address table is empty\n");
	else
		fprintf(theOutput, "   All other IP addresses from the " \
			"outside will be processed.\n");

	return NOERROR;
}


/*------------------------------------------------------------*/

int
queryOverrideTable()
{
	register u_short i;
	OverrideTableEntry overrideTable[MAX_OVERRIDE_ENTRIES];

	if (ioctl(theSocket, DIOCGOVERRIDE, overrideTable)) {
		dfm_perror("DIOCGOVERRIDE");
		return ERROR;
	}

	for (i=0; i < MAX_OVERRIDE_ENTRIES; i++) {
		register u_short j;
		struct in_addr a,m;

		if ((overrideTable[i].network.s_addr == 0) &&
		    (overrideTable[i].mask == 0))
			break;

		if (i == 0)
			fprintf(theOutput, "   Allow all outgoing " \
						"packets to:\n");

		a.s_addr = htonl(overrideTable[i].network.s_addr);
		m.s_addr = htonl(overrideTable[i].mask);
		fprintf(theOutput, "      address %-15s ", inet_ntoa(a));
		fprintf(theOutput, "mask %-15s", inet_ntoa(m));
		fprintf(theOutput, "  ports ");
		for (j=0; j < MAX_ACCESS_RANGES; j++) {
			if (overrideTable[i].access[j].begin == 0)
				break;
			if (j) fprintf(theOutput, "%55s", " ");
			fprintf(theOutput, "%5d-%-5d\n",
				overrideTable[i].access[j].begin,
				overrideTable[i].access[j].end);
		}
	}

	if (i == 0)
		fprintf(theOutput, "   Override table is empty\n");

	return NOERROR;
}

/*------------------------------------------------------------*/

static int
queryClassTable(u_short row)
{
	u_short i;
	char flag;

	accessListTableReq in_class;
	accessListTableReq out_class;
	accessListTableReq source_class;
	accessListTableReq udp_class;
	accessListTableReq icmp_class;

	if (row > MAX_ACCESS_LISTS) {
		fprintf(theErrors, "Class number out of range\n");
		return ERROR;
	}

	if (classTableReq(READ, IN_CLASS, row, &in_class))
		return ERROR;
	if (classTableReq(READ, OUT_CLASS, row, &out_class))
		return ERROR;
	if (classTableReq(READ, SOURCE_CLASS, row, &source_class))
		return ERROR;
	if (classTableReq(READ, UDP_CLASS, row, &udp_class))
		return ERROR;
	if (classTableReq(READ, ICMP_CLASS, row, &icmp_class))
		return ERROR;


	fprintf(theOutput, "     TCP dst       TCP dst       TCP src" \
			   "       UDP dst        ICMP\n");
	fprintf(theOutput, "     port in       port out      port in" \
			   "       port in       type in\n");
	fprintf(theOutput, "   -----------   -----------   -----------" \
			   "   -----------   -----------\n");

	for (i=0,flag=1; i<MAX_ACCESS_RANGES && flag; i++) {

		flag = 0;
		if (in_class.list[i].begin) {
			fprintf(theOutput, "   %5d-%-5d",
			  in_class.list[i].begin, in_class.list[i].end);
			flag = 1;
		} else
			fprintf(theOutput, "%14s"," ");

		if (out_class.list[i].begin) {
			fprintf(theOutput, "   %5d-%-5d",
			  out_class.list[i].begin, out_class.list[i].end);
			flag = 1;
		} else
			fprintf(theOutput, "%14s"," ");

		if (source_class.list[i].begin) {
			fprintf(theOutput, "   %5d-%-5d",
			  source_class.list[i].begin, source_class.list[i].end);
			flag = 1;
		} else
			fprintf(theOutput, "%14s"," ");

		if (udp_class.list[i].begin) {
			fprintf(theOutput, "   %5d-%-5d",
			  udp_class.list[i].begin, udp_class.list[i].end);
			flag = 1;
		} else
			fprintf(theOutput, "%14s"," ");

		if (icmp_class.list[i].begin) {
			fprintf(theOutput, "   %5d-%-5d\n",
			  icmp_class.list[i].begin-1, icmp_class.list[i].end-1);
			flag = 1;
		} else
			fprintf(theOutput, "\n");
	}

	return NOERROR;
}

/*------------------------------------------------------------*/

static int
queryNetworkTable(char *hoststr, char cflag)
{
	u_short stripIndex;
	struct in_addr hostaddr;
	networkTableReq u;
	u_long host;

	if (isalpha(hoststr[0])) {
		struct hostent *h;

		if ((h = gethostbyname(hoststr))) {
			host =  htonl(*((long *) h->h_addr_list[0]));
		} else {
			fprintf(theErrors, "   Unknown host '%s'\n", hoststr);
			return ERROR;
		}
	} else
		host = inet_network(hoststr);

	if (IN_CLASSB(host)) {
		u.network.s_addr = host & IN_CLASSB_NET;
		u.offset = ((host & 0x0000ff00) >> 8) * sizeof(u.table);
	} else if (IN_CLASSC(host)) {
		u.network.s_addr = host & IN_CLASSC_NET;
		u.offset = 0;
	} else {
		/* We don't handle class A or D addresses. */
		fprintf(theErrors,
			"   Can not handle class A or D addresses\n");
		return ERROR;
	}

	u.netTableIndex = 0xFFFF;
	stripIndex = (host & 0x000000ff);

	if (ioctl(theSocket, DIOCGNETWORK, &u)) {
		dfm_perror("DIOCGNETWORK");
		return ERROR;
	}

	hostaddr.s_addr = htonl(host);
	fprintf(theOutput, "   Host %s is using class %d\n",
		inet_ntoa(hostaddr), u.table[stripIndex]);

	if (cflag) {
		fprintf(theOutput, "\n");
		queryClassTable(u.table[stripIndex]);
	}

	return NOERROR;
}

/*------------------------------------------------------------*/

static int
showNetworks()
{
	networksTableReq r;
	struct in_addr a;
	int i;

	if (ioctl(theSocket, DIOCGNETWORKS, &r)) {
		dfm_perror("DIOCGNETWORKS");
		return ERROR;
	}

	for (i = 0; i < MAX_NETWORKS; i++) {
		if (r.table[i].network.s_addr)
			break;
	}
	
	if (i < MAX_NETWORKS) {
		fprintf(theOutput,
			"   The following networks are loaded:\n");

		for (i = 0; i < MAX_NETWORKS; i++) {
			if (r.table[i].network.s_addr) {
				a.s_addr = htonl(r.table[i].network.s_addr);
				fprintf(theOutput,
					"      network %s\n", inet_ntoa(a));
			}
		}
	} else 
		fprintf(theOutput, "   No networks are loaded\n");

	return NOERROR;
}

/*------------------------------------------------------------*/

static void
showStats()
{
	Statistics theStats;
	u_long upTime, upDays, upHours, upMinutes, upSeconds;

	if (ioctl(theSocket, DIOCGSTATS, &theStats)) {
		dfm_perror("DIOCGFLAGS");
		return;
	}

	fprintf(theOutput, "\nPackets%33s%34s\n", "Inside", "Outside");
	fprintf(theOutput, "-------%33s%34s\n", "------", "-------");
	fprintf(theOutput, "Filtered %31s", fmt_big(theStats.p_insideFiltered));
	fprintf(theOutput, "%34s\n", fmt_big(theStats.p_outsideFiltered));
	fprintf(theOutput, "Received    %28s", fmt_big(theStats.p_insideRx));
	fprintf(theOutput, "%34s\n", fmt_big(theStats.p_outsideRx));
	fprintf(theOutput, "Transmitted %28s", fmt_big(theStats.p_insideTx));
	fprintf(theOutput, "%34s\n\n", fmt_big(theStats.p_outsideTx));

	fprintf(theOutput, "Bytes%35s%34s\n", "Inside", "Outside");
	fprintf(theOutput, "-----%35s%34s\n", "------", "-------");
	fprintf(theOutput, "Filtered %31s", fmt_big(theStats.b_insideFiltered));
	fprintf(theOutput, "%34s\n", fmt_big(theStats.b_outsideFiltered));
	fprintf(theOutput, "Received    %28s", fmt_big(theStats.b_insideRx));
	fprintf(theOutput, "%34s\n", fmt_big(theStats.b_outsideRx));
	fprintf(theOutput, "Transmitted %28s", fmt_big(theStats.b_insideTx));
	fprintf(theOutput, "%34s\n\n", fmt_big(theStats.b_outsideTx));

	fprintf(theOutput, "      Dropped Packets\n");
	fprintf(theOutput, "      ---------------\n");
	fprintf(theOutput, "No kernel buffers:   %11s\n",
		fmt_small(theStats.droppedNoMbufs, NO));

	fprintf(theOutput, "Interface queue full:%11s\n\n",
		fmt_small(theStats.droppedIfQfull, NO));

	fprintf(theOutput, "Bridge table entries:%11s\n\n",
		fmt_small(theStats.bridgeEntries, NO));

	if (theStats.running) {
		upDays = upHours = upMinutes = upSeconds = 0;
		upTime = time(NULL) - theStats.startTime.tv_sec;
		if (upTime) {
			upDays = upTime / (24UL * 60UL * 60UL);
			upTime -= upDays * 24UL * 60UL * 60UL;
			upHours = upTime / (60 * 60);
			upTime -= upHours * 60 * 60;
			upMinutes = upTime / 60;
			upSeconds = upTime - upMinutes * 60;
		}
		fprintf(theOutput, "Uptime: %lu days %lu hours %lu minutes %lu "
			"seconds\n\n", upDays, upHours, upMinutes, upSeconds);
	} else
		fprintf(theOutput, "Drawbridge is not running.\n\n");
}

/*------------------------------------------------------------*/

void
handleShow()
{
	int result;
	char *currTok;


	currTok = strtok(NULL, DELIMITERS);
	result = findtoken(currTok);

	switch (result) {

	case T_HOST:
		currTok = strtok(NULL, DELIMITERS);
		if (!currTok) {
			fprintf(theErrors, "   Missing host address/name\n");
			break;
		}
		queryNetworkTable(currTok, NO);
		break;

	case T_CLASS:
		currTok = strtok(NULL, DELIMITERS);
		if (!currTok) {
			fprintf(theErrors, "   Missing class number\n");
			break;
		}
		if (strIsNumeric(currTok))
			queryClassTable(atoi(currTok));
		else
			queryNetworkTable(currTok, YES);
		break;

	case T_NETWORK:
		showNetworks();
		break;

	case T_ACCEPT:
		queryAcceptTable();
		break;

	case T_REJECT:
		queryRejectTable();
		break;

	case T_OVERRIDE:
		queryOverrideTable();
		break;

	case T_FLAGS: {
		flagsReq flags;

		if (ioctl(theSocket, DIOCGFLAGS, &flags)) {
			dfm_perror("DIOCGFLAGS");
			break;
		}

		fprintf(theOutput, "   Multicast      = %s\n",
			flags.discardMulticast ? "discard" : "forward");
		fprintf(theOutput, "   NonIp          = %s\n",
			flags.discardNonIp ? "discard" : "forward");
		fprintf(theOutput, "   OtherIp        = %s\n",
			flags.discardOtherIp ? "discard" : "forward");
		fprintf(theOutput, "   SuspectOffset  = %s\n",
			flags.discardSuspectOffset ? "discard" : "forward");
		fprintf(theOutput, "   FragmentedICMP = %s\n",
			flags.discardFragmentedICMP ? "discard" : "forward");
		fprintf(theOutput, "   AttackICMP     = %s\n",
			flags.discardAttackICMP ? "discard" : "forward");
		break;
	}

	case T_STATS:
		showStats();
		break;

	case T_LOGFACILITY: {
		logFacilityReq freq;

		if (ioctl(theSocket, DIOCGLOGF, &freq)) {
			dfm_perror("DIOCGLOGF");
			break;
		}
		if ((freq.facility >= 0) && (freq.facility <= LOG_NFACILITIES))
			fprintf(theOutput, "   log facility = %s\n",
				facilities[freq.facility]);
		else
			fprintf(theOutput,"   log facility = %d (invalid)\n",
				freq.facility);
		break;
	}

	case T_LOGMASK: {
		logMaskReq mreq;

		if (ioctl(theSocket, DIOCGLOGM, &mreq)) {
			dfm_perror("DIOCGLOGM");
			break;
		}

		fprintf(theOutput,
			"   log mask = 0x%lX\n", (u_long)mreq.mask);
		break;
	}

	case T_BRIDGE: {
		int i;
		bridgeReq breq;

		for (i=0; i<MAX_BRIDGE_ADDRESSES; i++) {
			breq.t.valid = NO;
			breq.index = i;
			if (ioctl(theSocket, DIOCGBRIDGE, &breq)) {
				dfm_perror("DIOCGBRIDGE");
				break;
			}
			if (breq.t.valid == YES) {
				fprintf(theOutput, "   " \
					"%02x:%02x:%02x:%02x:%02x:%02x   %s\n",
					breq.t.address.bytes[0],
					breq.t.address.bytes[1],
					breq.t.address.bytes[2],
					breq.t.address.bytes[3],
					breq.t.address.bytes[4],
					breq.t.address.bytes[5],
					(breq.t.interface == DB_INSIDE_IF) ?
						"inside" : "outside");
			}
		}
		break;
	}

	default:
		usageShow(TERSE);
	}
}

/*------------------------------------------------------------*/
