/*
 * Copyright (c) 1993,1994,1995,1997,1998
 *      Texas A&M University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Texas A&M University
 *      and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Developers:
 *     Russell Neeper, David K. Hess, Douglas Lee Schales, David R. Safford
 */

#include "drawbridge.h"

#if NDRAWBRIDGE > 0

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/conf.h>
#include <sys/mbuf.h>
#include <sys/ioctl.h>
#include <sys/fcntl.h>
#include <sys/socket.h>
#include <sys/errno.h>
#include <sys/syslog.h>
#include <sys/kernel.h>
#include <machine/stdarg.h>

#include <net/if.h>
#include <net/if_types.h>
#include <net/if_llc.h>

#include <netinet/in.h>
#include <netinet/in_var.h>
#include <netinet/if_ether.h>
#include <netinet/if_fddi.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <netinet/ip_icmp.h>

/* #define DBDEBUG 1 */

#include <net/drawbrdg.h>

/* --- Globals --- */

int initialized = NO;
int rdonly = NO;

Statistics DBstats = { NO }; 			/* def running state */
FilterConfig DBcfg = { DEFAULT_SYSL_FACILITY,	/* def log facility  */
		       DEFAULT_SYSL_MASK };	/* def log mask      */

/* MAC address hash table for bridge code     */
/* Addresses are stored in network byte order */
HashEntry hashTable [MAX_BRIDGE_ADDRESSES];

/* IP network hash table */
NetTableEntry netTable[MAX_NETWORKS];

/* class tables */
AccessListTableEntry in     [MAX_ACCESS_LISTS] [MAX_ACCESS_RANGES];
AccessListTableEntry out    [MAX_ACCESS_LISTS] [MAX_ACCESS_RANGES];
AccessListTableEntry source [MAX_ACCESS_LISTS] [MAX_ACCESS_RANGES];
AccessListTableEntry udp    [MAX_ACCESS_LISTS] [MAX_ACCESS_RANGES];
AccessListTableEntry icmp   [MAX_ACCESS_LISTS] [MAX_ACCESS_RANGES];

/* IP address access tables */
AcceptTableEntry   acceptTable   [MAX_ACCEPT_ENTRIES];
RejectTableEntry   rejectTable   [MAX_REJECT_ENTRIES];
OverrideTableEntry overrideTable [MAX_OVERRIDE_ENTRIES];

/* --- Prototypes --- */

static void syslogMessage __P((u_int,...));
static void initTables __P((void));
static void copyAccessList __P((int, AccessListTableEntry *,
				AccessListTableEntry *));
static int accessListRequest __P((char, accessListTableReq *));
static int networkTableRequest __P((char, networkTableReq *));
static int generalTableRequest __P((int, u_int8_t *, u_int8_t *, u_int));
static int bridgeRequest __P((int, bridgeReq *));
static int networkLookup __P((u_int32_t));
static int checkRejectTable __P((struct ip *ipHeader, u_int32_t srcAddr,
				u_int32_t dstAddr));
static int checkAcceptTable __P((struct ip *ipHeader, u_int32_t srcAddr,
				u_int32_t dstAddr));
static int checkIncomingTcp __P((u_int32_t, u_int32_t, u_int16_t, u_int16_t));
static int checkOutgoingTcp __P((u_int32_t, u_int32_t, u_int16_t, u_int16_t));
static int checkIncomingUdp __P((u_int32_t, u_int32_t, u_int16_t, u_int16_t));
static int checkOutgoingUdp __P((u_int32_t, u_int32_t, u_int16_t, u_int16_t));
static int checkIncomingIcmp __P((u_int32_t, u_int32_t, u_int8_t));
static int checkIncomingPacket __P((u_int16_t, u_int8_t *, u_int, int));
static int checkOutgoingPacket __P((u_int16_t, u_int8_t *, u_int));
static int bridge __P((int, HardwareAddr *, HardwareAddr *));
static int db_open __P((dev_t, int, int, struct proc *));
static int db_close __P((dev_t, int, int, struct proc *));
static int db_ifpromisc __P((struct ifnet *, int));
static int db_gethwaddr __P((struct ifnet *, HardwareAddr *));
static int db_getIPaddr __P((struct ifnet *, in_addr_t *));
static int db_ioctl __P((dev_t dev, int cmd, caddr_t addr, int flags,
			struct proc *p));
static int initRequest __P((initReq *));

#ifdef DEVFS
#include <sys/devfsext.h>
void	*devfs_token;
#endif

#define CDEV_MAJOR 20		/* 20 is for locally defined char devices */
static struct cdevsw db_cdevsw =
	{ db_open,  db_close, noread,    nowrite,
	  db_ioctl, nostop,   nullreset, nodevtotty,
	  noselect, nommap,   NULL,     "drawbridge",  NULL,  -1 };

/* ------------------------------------------------------------------------ */
/*
 * Thanks to Klaus-Peter Kossakowski and Uwe Ellermann at DFN-CERT
 * for contributing much of the syslog implementation.
 *
 * All parameters are in host byte ordering.
 */

/*
 * Note:  If this table is modified, the corresponding
 * enum struct in drawbrdg.h must also be modified.
 */
static SyslogMessageEntry syslogMessages[] = {
	{ "unknown event",				LOG_WARNING },
	{ "cold start",					LOG_INFO    },
	{ "incoming class D:",				LOG_WARNING },
	{ "outgoing class D:",				LOG_WARNING },
	{ "incoming port:",				LOG_WARNING },
	{ "outgoing port:",				LOG_WARNING },
	{ "incoming type:",				LOG_WARNING },
	{ "outgoing type:",				LOG_WARNING },
	{ "incoming via reject table:",			LOG_WARNING },
	{ "outgoing via accept table:",			LOG_WARNING },
	{ "outgoing via override table:",		LOG_WARNING },
	{ "incoming header too short:",			LOG_WARNING },
	{ "outgoing header too short:",			LOG_WARNING },
	{ "incoming D-O-S attack:",			LOG_WARNING },
	{ "outgoing D-O-S attack:",			LOG_WARNING },
	{ "incoming IP:",				LOG_NOTICE  },
	{ "outgoing IP:",				LOG_WARNING },
	{ "incoming fragment with IP offset == 1:",	LOG_WARNING },
	{ "outgoing fragment with IP offset == 1:",	LOG_WARNING },
	{ "incoming fragment:",				LOG_WARNING },
	{ "outgoing fragment:",				LOG_WARNING },
	{ "incoming MAC layer protocol:",		LOG_INFO    },
	{ "outgoing MAC layer protocol:",		LOG_WARNING },
	{ NULL } /* Marks the end of the messages. */
};

static char *syslogDosTypes[] = { "\"Smurf\"", "\"Pong\"" };

static void
syslogMessage(u_int eventNo,...)
{
        va_list ap;
	int protocolNo;
	in_addr_t srcAddr;
	in_addr_t dstAddr;
	int srcPort;
	int dstPort;
	int type;
	int protocol;

	/*
	 * If this message is disabled then just return.
	 */
	if ( !((1UL << eventNo) & DBcfg.logMask) )
		return;

	switch ( eventNo ) {
		case SYSL_IN_OFFSET:
		case SYSL_OUT_OFFSET:
		case SYSL_IN_FRAG:
		case SYSL_OUT_FRAG:
		case SYSL_IN_REJECT:
		case SYSL_OUT_ACCEPT:
		case SYSL_IN_PROT:
		case SYSL_OUT_PROT: {	/* IP - general */
			va_start(ap,eventNo);

			protocolNo = va_arg(ap,int);
			srcAddr    = va_arg(ap,in_addr_t);
			dstAddr    = va_arg(ap,in_addr_t);

			va_end(ap);

			log(	(DBcfg.logFacility |
				 syslogMessages[eventNo].priority),
				"drawbridge: %s protocol %d " \
				"from %u.%u.%u.%u to %u.%u.%u.%u\n",
				syslogMessages[eventNo].message, protocolNo,
				srcAddr.s_un_b.s_b[3], srcAddr.s_un_b.s_b[2],
				srcAddr.s_un_b.s_b[1], srcAddr.s_un_b.s_b[0],
				dstAddr.s_un_b.s_b[3], dstAddr.s_un_b.s_b[2],
				dstAddr.s_un_b.s_b[1], dstAddr.s_un_b.s_b[0]
			);
			break;
		}

		case SYSL_IN_CLASSD:
		case SYSL_OUT_CLASSD:
		case SYSL_IN_PORT:
		case SYSL_OUT_PORT:
		case SYSL_OUT_OVERRIDE:
		case SYSL_IN_LENGTH:
		case SYSL_OUT_LENGTH: {	/* IP - TCP/UDP */
			va_start(ap,eventNo);

			protocolNo = va_arg(ap,int);
			srcAddr    = va_arg(ap,in_addr_t);
			dstAddr    = va_arg(ap,in_addr_t);
			srcPort    = va_arg(ap,int);
			dstPort    = va_arg(ap,int);

			va_end(ap);

			log(	(DBcfg.logFacility |
				 syslogMessages[eventNo].priority),
				"drawbridge: %s %s from %u.%u.%u.%u " \
				"port %u to %u.%u.%u.%u port %u\n",
				(protocolNo == IPPROTO_UDP) ? "UDP" : "TCP",
				syslogMessages[eventNo].message,
				srcAddr.s_un_b.s_b[3], srcAddr.s_un_b.s_b[2],
				srcAddr.s_un_b.s_b[1], srcAddr.s_un_b.s_b[0],
				srcPort,
				dstAddr.s_un_b.s_b[3], dstAddr.s_un_b.s_b[2],
				dstAddr.s_un_b.s_b[1], dstAddr.s_un_b.s_b[0],
				dstPort
			);
			break;
		}

		case SYSL_IN_TYPE: {	/* IP - ICMP */
			va_start(ap,eventNo);

			srcAddr = va_arg(ap,in_addr_t);
			dstAddr = va_arg(ap,in_addr_t);
			type    = va_arg(ap,int);

			va_end(ap);

			log(	(DBcfg.logFacility |
				 syslogMessages[eventNo].priority),
				"drawbridge: ICMP %s from " \
				"%u.%u.%u.%u type %u to %u.%u.%u.%u\n",
				syslogMessages[eventNo].message,
				srcAddr.s_un_b.s_b[3], srcAddr.s_un_b.s_b[2],
				srcAddr.s_un_b.s_b[1], srcAddr.s_un_b.s_b[0],
				type,
				dstAddr.s_un_b.s_b[3], dstAddr.s_un_b.s_b[2],
				dstAddr.s_un_b.s_b[1], dstAddr.s_un_b.s_b[0]
			);
			break;
		}

		case SYSL_IN_DOS: {	/* Denial of service attack */
			va_start(ap,eventNo);

			srcAddr = va_arg(ap,in_addr_t);
			dstAddr = va_arg(ap,in_addr_t);
			type    = va_arg(ap,int);

			va_end(ap);

			log(	(DBcfg.logFacility |
				syslogMessages[eventNo].priority),
				"drawbridge: %s %s from " \
				"%u.%u.%u.%u to %u.%u.%u.%u\n",
				syslogDosTypes[type],
				syslogMessages[eventNo].message,
				srcAddr.s_un_b.s_b[3], srcAddr.s_un_b.s_b[2],
				srcAddr.s_un_b.s_b[1], srcAddr.s_un_b.s_b[0],
				dstAddr.s_un_b.s_b[3], dstAddr.s_un_b.s_b[2],
				dstAddr.s_un_b.s_b[1], dstAddr.s_un_b.s_b[0]
			);
			break;
		}

		case SYSL_IN_FILTER:
		case SYSL_OUT_FILTER: {	/* MAC layer protocol */
			va_start(ap,eventNo);
			protocol = va_arg(ap,int);
			va_end(ap);

			log(	(DBcfg.logFacility |
				 syslogMessages[eventNo].priority),
				"drawbridge: %s 0x%04X\n",
				syslogMessages[eventNo].message, protocol
			);
			break;
		}

		case SYSL_COLD_START:	/* Startup message */
			log(	(DBcfg.logFacility |
				 syslogMessages[eventNo].priority),
				"drawbridge: %s\n",
				syslogMessages[eventNo].message
			);
			break;

		default:		/* Should never get here */
			log(	(DBcfg.logFacility | LOG_WARNING),
				"drawbridge: unknown event\n"
			);
			break;
	}
}

/* ------------------------------------------------------------------------ */

static void
initTables(void)
{
	/*
	 * Clear all the tables and access lists
	 */
	bzero(netTable, sizeof(netTable));
	bzero(hashTable, sizeof(hashTable));
	bzero(acceptTable, sizeof(acceptTable));
	bzero(rejectTable, sizeof(rejectTable));
	bzero(overrideTable, sizeof(overrideTable));

	bzero(in, sizeof(in));
	bzero(out, sizeof(out));
	bzero(source, sizeof(source));
	bzero(udp, sizeof(udp));
	bzero(icmp, sizeof(icmp));

	/* XXX
	 * The class table defaults are hard coded here in the init
	 * routine and also in clear.c of the manager.  This needs
	 * to be changed so that the defaults are in one place in a
	 * header file.  For now, if these are changed, be sure to
	 * also change them in clear.c.
	 */
        in[0][0].begin = 25;     /* TCP - Mail in */
        in[0][0].end = 25;
        in[0][1].begin = 53;     /* TCP - Name service in */
        in[0][1].end = 53;
        out[0][0].begin = 1;     /* TCP - Everything out */
        out[0][0].end = 0xFFFF;
        source[0][0].begin = 20; /* TCP - FTP data connections in */
        source[0][0].end = 20;
        udp[0][0].begin = 1;     /* UDP - Disallow TFTP and Portmapper in */
        udp[0][0].end = 68;
        udp[0][1].begin = 70;
        udp[0][1].end = 110;
        udp[0][2].begin = 112;
        udp[0][2].end = 0xFFFF;
	icmp[0][0].begin = 1;	 /* ICMP - Everything in         */
	icmp[0][0].end = 0x100;  /* (0-255 stored offset by one) */
}

/* ------------------------------------------------------------------------ */

static void
copyAccessList(int req, AccessListTableEntry *x, AccessListTableEntry *y)
{
	register int i;

	if (req == WRITE) {
		for (i=0; i<MAX_ACCESS_RANGES; i++)
			y[i] = x[i];
	} else {
		for (i=0; i<MAX_ACCESS_RANGES; i++)
			x[i] = y[i];
	}
}

/* ------------------------------------------------------------------------ */

static int
accessListRequest(char req, accessListTableReq *u)
{
	if (u->row > MAX_ACCESS_LISTS)
		return ERROR_OUT_OF_RANGE;

	switch (u->class) {
	case IN_CLASS:
		copyAccessList(req, u->list, in[u->row]);
		break;

	case OUT_CLASS:
		copyAccessList(req, u->list, out[u->row]);
		break;

	case SOURCE_CLASS:
		copyAccessList(req, u->list, source[u->row]);
		break;

	case UDP_CLASS:
		copyAccessList(req, u->list, udp[u->row]);
		break;

	case ICMP_CLASS:
		copyAccessList(req, u->list, icmp[u->row]);
		break;

	default:
		return ERROR_UNKNOWN_CLASS;
	}

	return NO_ERROR;
}

/* ------------------------------------------------------------------------ */

static int
networkTableRequest(char req, networkTableReq *u)
{
	u_int32_t size;
	u_int16_t hash;
	register u_int16_t curr;

	/*
	 * If called with a valid index and network, the caller already knows
	 * the correct index into the network table.
	 */
	if ((u->netTableIndex != 0xFFFF) && (u->network.s_addr != 0) &&
	    (netTable[u->netTableIndex].network.s_addr == u->network.s_addr)) {

		curr = u->netTableIndex;

	/*
	 * Otherwise, we need to find the index into the table based
	 * on the network provided.
	 */
	} else {
		/*
		 * Do some quick checks on the specified network
		 */
		if (IN_CLASSB(u->network.s_addr)) {
			size = 0x10000UL;
			if (u->network.s_addr & IN_CLASSB_HOST)
				return ERROR_INVALID_NETWORK;
		} else if (IN_CLASSC(u->network.s_addr)) {
			size = 0x100UL;
			if (u->network.s_addr & IN_CLASSC_HOST)
				return ERROR_INVALID_NETWORK;
		} else {
			/* We do not support A or D networks. */
			return ERROR_INVALID_NETWORK;
		}

		/*
		 * Hash into the table to see if the network has been defined
		 */
		hash = (u_short)((u->network.s_addr & NET_HASH_MASK)
				 >> NET_HASH_SHIFT);

		curr = hash;
		while ((netTable[curr].network.s_addr != 0) &&
		       (netTable[curr].network.s_addr != u->network.s_addr)) {
			/*
			 * Increment 'curr' and check to see if we have wrapped.
			 * If so, the table is full and the address was not
			 * found so return an error code.
			 */
			if ((curr = (curr+1) % MAX_NETWORKS) == hash) {
				return (req == WRITE) ? ERROR_TABLE_FULL :
							ERROR_NET_NOT_FOUND;
			}
		}

		/*
		 * If the network is not in the table and this is a WRITE
		 * request, then add it to the table.
		 */
		if (netTable[curr].network.s_addr == 0) {

			if (req == WRITE) {
				/*
				 * add the network to the table
			 	 */
				netTable[curr].hostTable = (u_char *)
					malloc(size,M_DEVBUF,M_NOWAIT);

				if (netTable[curr].hostTable == NULL)
					return ERROR_NO_MEMORY;

				bzero(netTable[curr].hostTable, size);

				netTable[curr].network.s_addr =
					 u->network.s_addr;
			} else {
				/*
				 * The requested network was not found
				 */
				return ERROR_NET_NOT_FOUND;
			}
		}

		/*
		 * We now have the index and can pass it back to the caller.
		 */
		u->netTableIndex = curr;
	}

	/*
	 * Do the request
	 */
	if (req == READ)
		bcopy(netTable[curr].hostTable+u->offset, u->table,
		       sizeof(u->table)); 
	else if (req == WRITE)
		bcopy(u->table, netTable[curr].hostTable+u->offset, 
		       sizeof(u->table)); 
	else if (req == REMOVE) {
		netTable[curr].network.s_addr = 0L;
		free(netTable[curr].hostTable, M_DEVBUF);
	}

	return NO_ERROR;
}

/* ------------------------------------------------------------------------ */

static int
generalTableRequest(int rw, u_char *table, u_char *r, u_int size)
{
	if (rw == READ)
		bcopy(table, r, size);
	else if (rw == WRITE)
		bcopy(r, table, size);
	else
		return ERROR_INVALID_OP;

	return NO_ERROR;
}

/* ------------------------------------------------------------------------ */

static int
bridgeRequest(int rw, bridgeReq *r)
{
	int s;

	if (rw != READ)
		return ERROR_INVALID_OP;

	if (r->index >= MAX_BRIDGE_ADDRESSES)
		return ERROR_OUT_OF_RANGE;

	s = splimp();
	bcopy(&hashTable[r->index], &r->t, sizeof(r->t));
	splx(s);

	return NO_ERROR;
}

/* ------------------------------------------------------------------------ */

static int
networkLookup(u_int32_t host)
{
	u_int16_t hash;
	in_addr_t networkPart;
	in_addr_t hostPart;
	register u_int16_t curr;

	/*
	 * Get the network and host from the address.
	 */
	if (IN_CLASSB(host)) {

		/* Class B address. */
		networkPart.s_addr = host & IN_CLASSB_NET;
		hostPart.s_addr = host & IN_CLASSB_HOST;
	}
	else if (IN_CLASSC(host)) {

		/* Class C address. */
		networkPart.s_addr = host & IN_CLASSC_NET;
		hostPart.s_addr = host & IN_CLASSC_HOST;
	}
	else {
		/* We don't handle class A's yet. They get the default index */
		return DEFAULT_ACCESS;
	}

	/*
	 * Hash into the table and see if the network has been defined.
	 */
	hash = (u_short) ((networkPart.s_addr & NET_HASH_MASK)
			  >> NET_HASH_SHIFT);

	for (curr=hash; netTable[curr].network.s_addr!=networkPart.s_addr;) {

		if (netTable[curr].network.s_addr == 0)
			/* not found - entry is unused */
			return DEFAULT_ACCESS;

		if ((curr = (curr+1) % MAX_NETWORKS) == hash)
			/* not found - searched the entire table */
			return DEFAULT_ACCESS;
	}

	/*
	 * found - return the access list index
	 */
	return netTable[curr].hostTable[hostPart.s_addr];
}

/* ------------------------------------------------------------------------ */
/*
 * Parameters are expected to be in host byte order
 */

static int
checkIncomingTcp(u_int32_t srcAddr, u_int32_t dstAddr,
		 u_int16_t srcPort, u_int16_t dstPort)
{
	register int i;
	int accessIndex;
	AccessListTableEntry *accessList;

	/*
	 * Check for multicast
	 */
	if (IN_CLASSD(dstAddr)) {
		if (DBcfg.discardMulticast) {
			DBstats.f_outsideTcpMcast++;
			syslogMessage(SYSL_IN_CLASSD, IPPROTO_TCP,
			      	srcAddr, dstAddr, (int)srcPort, (int)dstPort);
			return DISCARD_PKT;
		}
		return ALLOW_PKT;
	}

	/*
	 * Do the lookup to get the index.
	 */
	accessIndex = networkLookup(dstAddr);
	accessList = in[accessIndex];

	/*
	 * See if the destination port is allowed by the 'in' access list
	 */
	for (i=0; accessList[i].begin; i++) {
		if ((dstPort >= accessList[i].begin) &&
		    (dstPort <= accessList[i].end))
	    		return ALLOW_PKT;
	}

	/*
	 * If the destination port is not allowed but is over 900, then
	 * check the source port.  NOTE: This is 900 instead of 1024 because
	 * some broken FTP implementations use ports below 1024.  If you want
	 * to enforce the use of non-privileged ports, change this to 1024.
	 */
	if (dstPort > 900) {
		accessList = source[accessIndex];

		for (i=0; accessList[i].begin; i++) {
			if ((srcPort >= accessList[i].begin) &&
			    (srcPort <= accessList[i].end))
				return ALLOW_PKT;
		}
	}

	DBstats.f_outsideTcpPort++;
	syslogMessage(SYSL_IN_PORT, IPPROTO_TCP,
		      srcAddr, dstAddr, (int)srcPort, (int)dstPort);

	return DISCARD_PKT;
}

/* ------------------------------------------------------------------------ */
/*
 * Parameters are expected to be in host byte order
 */

static int
checkOutgoingTcp(u_int32_t srcAddr, u_int32_t dstAddr,
		 u_int16_t srcPort, u_int16_t dstPort)
{
	register int i, j;
	AccessListTableEntry *accessList;

	/*
	 * Check for multicast
	 */
	if (IN_CLASSD(dstAddr)) {
		if (DBcfg.discardMulticast) {
			DBstats.f_insideTcpMcast++;
			syslogMessage(SYSL_OUT_CLASSD,IPPROTO_TCP,
				srcAddr, dstAddr, (int)srcPort, (int)dstPort);
			return DISCARD_PKT;
		}
		return ALLOW_PKT;
	}

	accessList = out[ networkLookup(srcAddr) ];

	/*
	 * See if the destination port is allowed by the 'out' access list
	 */
	for (i=0; accessList[i].begin; i++) {
		if ((dstPort >= accessList[i].begin) &&
		    (dstPort <= accessList[i].end))
			return ALLOW_PKT;
	}

	/*
	 * Still not allowed so check the outgoing packet to see
	 * if the destination is on the override allow list.
	 */
	for (i=0; ; i++) {
		register u_int32_t addr, mask;

		addr = overrideTable[i].network.s_addr;
		mask = overrideTable[i].mask;

		if ((i >= MAX_OVERRIDE_ENTRIES) || (!addr && !mask)) {
    			/* not in the list */
			DBstats.f_insideTcpPort++;
			syslogMessage(SYSL_OUT_PORT, IPPROTO_TCP,
				srcAddr, dstAddr, (int)srcPort, (int)dstPort);
			return DISCARD_PKT;
		}
		if ((dstAddr & mask) == addr)
			/* found */
		   	break;
	}

	/*
	 * It's in the list so check if the destination port is allowed.
	 */
	for (j=0; overrideTable[i].access[j].begin; j++) {
		if ((dstPort >= overrideTable[i].access[j].begin) &&
		    (dstPort <= overrideTable[i].access[j].end))
	    		return ALLOW_PKT;
	}

	/*
	 * If we made it this far, the destination port is not allowed
	 */
	DBstats.f_insideTcpPortOverride++;
	syslogMessage(SYSL_OUT_OVERRIDE, IPPROTO_TCP,
		      srcAddr, dstAddr, (int)srcPort, (int)dstPort);

	return DISCARD_PKT;
}
	
/* ------------------------------------------------------------------------ */
/*
 * Parameters are expected to be in host byte order
 */

static int
checkIncomingUdp(u_int32_t srcAddr, u_int32_t dstAddr,
		 u_int16_t srcPort, u_int16_t dstPort)
{
	register int i;
	AccessListTableEntry *accessList;

	/*
	 * Check for multicast
	 */
	if (IN_CLASSD(dstAddr)) {
		if (DBcfg.discardMulticast) {
			DBstats.f_outsideUdpMcast++;
			syslogMessage(SYSL_IN_CLASSD,IPPROTO_UDP,
				srcAddr, dstAddr, (int)srcPort, (int)dstPort);
			return DISCARD_PKT;
		}
		return ALLOW_PKT;
	}

	accessList = udp[ networkLookup(dstAddr) ];

	/*
	 * See if the destination port is allowed by the 'udp' access list.
	 */
	for (i=0; accessList[i].begin; i++) {
		if ((dstPort >= accessList[i].begin) &&
		    (dstPort <= accessList[i].end))
			return ALLOW_PKT;
	}

	/*
	 * If we made it this far, the destination port is not allowed
	 */
	DBstats.f_outsideUdpPort++;
	syslogMessage(SYSL_IN_PORT, IPPROTO_UDP,
		      srcAddr, dstAddr, (int)srcPort, (int)dstPort);

	return DISCARD_PKT;
}

/* ------------------------------------------------------------------------ */
/*
 * Parameters are expected to be in host byte order
 */

static int
checkOutgoingUdp(u_int32_t srcAddr, u_int32_t dstAddr,
		 u_int16_t srcPort, u_int16_t dstPort)
{
	/*
	 * Check for multicast
	 */
	if (IN_CLASSD(dstAddr)) {
		if (DBcfg.discardMulticast) {
			DBstats.f_insideUdpMcast++;
			syslogMessage(SYSL_OUT_CLASSD,IPPROTO_UDP,
				srcAddr, dstAddr, (int)srcPort, (int)dstPort);
			return DISCARD_PKT;
		}
		return ALLOW_PKT;
	}

	/* Allow all UDP out for now. */
	return ALLOW_PKT;
}

/* ------------------------------------------------------------------------ */
/*
 * Parameters are expected to be in host byte order
 */

static int
checkIncomingIcmp(u_int32_t srcAddr, u_int32_t dstAddr, u_int8_t type)
{
	register int i;
	AccessListTableEntry *accessList;

	accessList = icmp[ networkLookup(dstAddr) ];

	/* Type is stored offset by one in the access list */
	type++;

	/*
	 * See if the ICMP type is allowed by the 'icmp' access list.
	 */
	for (i=0; accessList[i].begin; i++) {
		if ((type >= accessList[i].begin) &&
		    (type <= accessList[i].end))
			return ALLOW_PKT;
	}

	/*
	 * If we made it here, the ICMP type is not allowed
	 */
	DBstats.f_outsideIcmpType++;
	syslogMessage(SYSL_IN_TYPE, srcAddr, dstAddr, (int)type-1);

	return DISCARD_PKT;
}

/* ------------------------------------------------------------------------ */

static int
checkRejectTable(struct ip *ipHeader, u_int32_t srcAddr, u_int32_t dstAddr)
{
	register int i;
	register u_int32_t addr, mask;

	/*
	 * Make sure the IP source address in this packet
	 * from the outside is not on the reject list.
	 */
	for (i=0; ; i++) {
		register u_int32_t addr;
		addr = rejectTable[i].network.s_addr;
		mask = rejectTable[i].mask;

		if ((i >= MAX_REJECT_ENTRIES) || (!addr && !mask))
			/*
			 * Not found so accept the packet
			 */
			break;
		
		if ((srcAddr & mask) == addr) {
			/*
			 * Found - if this is an inverted rule,
			 * then accept, otherwise, discard.
			 */
			if (rejectTable[i].flag == INVERT) {
				break;
			} else {
				DBstats.f_outsideRejectTable++;
				syslogMessage(SYSL_IN_REJECT,
					(int)ipHeader->ip_p,
					srcAddr, dstAddr);
				return DISCARD_PKT;
			}
		}
	}

	return ALLOW_PKT;
}

/* ------------------------------------------------------------------------ */
/*
 * Parameters 'protocol' and 'protolen' are expected to be in host byte order
 * The packet pointed to by 'pkt' is, of course, in network btye order
 */

static int
checkIncomingPacket(u_int16_t protocol, u_int8_t *pkt,
		    u_int protolen, int checkAccept)
{
	struct ip *ipHeader;
	struct tcphdr *tcpHeader;
	struct udphdr *udpHeader;
	struct icmp *icmpHeader;
	u_int32_t srcAddr, dstAddr;
	u_int16_t ip_len;

	switch (protocol) {

	case ETHERTYPE_IP:
		ipHeader = (struct ip *)pkt;
		srcAddr = ntohl(ipHeader->ip_src.s_addr);
		dstAddr = ntohl(ipHeader->ip_dst.s_addr);

		if (checkAccept) {
			if (checkAcceptTable(ipHeader, srcAddr, dstAddr)
			    == DISCARD_PKT)
				return DISCARD_PKT;
		} else {
			if (checkRejectTable(ipHeader, srcAddr, dstAddr)
			    == DISCARD_PKT)
				return DISCARD_PKT;
		}

		/*
		 * Pass all IP fragments that do not have offset 0 (beginning
		 * of the packet) without checking since the TCP/UDP
		 * headers are not in this packet.
		 */
		if ((ipHeader->ip_off & SW_IP_OFFMASK) != 0) {
			/*
			 * If option is set, then discard pkts with offset of 1
			 */
			if (DBcfg.discardSuspectOffset &&
			    (ntohs(ipHeader->ip_off & SW_IP_OFFMASK) == 1) &&
			    (ipHeader->ip_p == IPPROTO_TCP)) {

				DBstats.f_outsideSuspectOffset++;
				syslogMessage(SYSL_IN_OFFSET, IPPROTO_TCP,
					srcAddr, dstAddr);
				return DISCARD_PKT;
			}

			/*
			 * If option is set, then discard fragmented ICMP pkts
			 */
			if (DBcfg.discardFragmentedICMP &&
			    (ipHeader->ip_p == IPPROTO_ICMP)) {

				DBstats.f_outsideIcmpFrag++;
			    	syslogMessage(SYSL_IN_FRAG, IPPROTO_ICMP,
					srcAddr, dstAddr);
				return DISCARD_PKT;
			}

			return ALLOW_PKT;
		}

		switch (ipHeader->ip_p) {

		case IPPROTO_TCP:
			tcpHeader = (struct tcphdr *)(pkt + 
						(ipHeader->ip_hl << 2)); 
			/*
			 * Make sure this packet (fragment) includes enough of
			 * the TCP header before making the other checks.
			 * Otherwise a SYN can sneak through since we might be
			 * trying to test something which is actually in the
			 * next fragment.
			 *
			 * NOTE: we drop all of these since we do not keep any
			 *       state between fragments.
			 *
			 * Many thanks to Uwe Ellermann at DFN-CERT for
			 * reporting this problem.
			 */
			ip_len = ntohs(ipHeader->ip_len);
			if ((ip_len > protolen) ||
			    ((ip_len - (ipHeader->ip_hl << 2)) < 14)) {
				DBstats.f_outsideShortTcpHdr++;
				syslogMessage(SYSL_IN_LENGTH, IPPROTO_TCP,
					srcAddr, dstAddr,
					ntohs(tcpHeader->th_sport),
					ntohs(tcpHeader->th_dport));

				return DISCARD_PKT;
			}

			/* Check for "ACKless SYN" */
			if ((tcpHeader->th_flags & (TH_SYN|TH_ACK)) == TH_SYN) {
				return checkIncomingTcp(srcAddr, dstAddr,
						ntohs(tcpHeader->th_sport),
						ntohs(tcpHeader->th_dport));
			}
			break;

		case IPPROTO_UDP:
			udpHeader = (struct udphdr *)(pkt + 
						(ipHeader->ip_hl << 2));
			/*
			 * Make sure this packet (fragment) includes enough
			 * of the UDP header before making the other checks.
			 */
			ip_len = ntohs(ipHeader->ip_len);
			if ((ip_len > protolen) ||
			    ((ip_len - (ipHeader->ip_hl << 2)) < 4)) {
				DBstats.f_outsideShortUdpHdr++;
				syslogMessage(SYSL_IN_LENGTH, IPPROTO_UDP,
					srcAddr, dstAddr,
					ntohs(udpHeader->uh_sport),
					ntohs(udpHeader->uh_dport));

				return DISCARD_PKT;
			}

			/* Check UDP protocols. */

			return checkIncomingUdp(srcAddr, dstAddr,
						ntohs(udpHeader->uh_sport),
						ntohs(udpHeader->uh_dport));
			break;

		case IPPROTO_ICMP:
			icmpHeader = (struct icmp *)
				     (pkt + (ipHeader->ip_hl << 2));

			/*
			 * If option is set, then discard fragmented ICMP pkts
			 */
			if (DBcfg.discardFragmentedICMP &&
			    (ipHeader->ip_off & IP_MF) != 0) {
				DBstats.f_outsideIcmpFrag++;
			    	syslogMessage(SYSL_IN_FRAG, IPPROTO_ICMP,
					srcAddr, dstAddr);

				return DISCARD_PKT;
			}

			/*
			 * If option is set, discard attack ICMP pkts.
			 *
			 * If the ICMP type, code, identifier, and seq num all
			 * equal 0, this may be an ICMP echo reply attack pkt.
			 */
			if (DBcfg.discardAttackICMP &&
			    (*((u_int16_t *)icmpHeader) == 0) && /*type & code*/
			    (icmpHeader->icmp_void == 0)) {      /*id & seq*/

				if (icmpHeader->icmp_cksum == 0xFFFF) {
				
					/* This is a "smurf" attack */
					DBstats.f_outsideSmurfDos++;
			    		syslogMessage(SYSL_IN_DOS,
						      srcAddr, dstAddr,
						      DOS_SMURF);
					return DISCARD_PKT;

				} else if (*((u_int16_t *)icmpHeader->icmp_data)
					   == htons(0x4500)) {
					/* This is a "pong" attack */
					DBstats.f_outsidePongDos++;
			    		syslogMessage(SYSL_IN_DOS,
						      srcAddr, dstAddr,
						      DOS_PONG);
					return DISCARD_PKT;
				}
			}

			return checkIncomingIcmp(srcAddr, dstAddr,
						 icmpHeader->icmp_type);
			break;

		default:
			if (DBcfg.discardOtherIp) {

				DBstats.f_outsideOtherIp++;
				syslogMessage(SYSL_IN_PROT, (int)ipHeader->ip_p,
					srcAddr, dstAddr);

				return DISCARD_PKT;
			}

			/* Everything is allowed so far. */
			break;
		}

		break;

	case ETHERTYPE_ARP:
	case ETHERTYPE_REVARP:
		/* pass these guys through no matter what */
		break;

	default:
		/* Only pass the rest if told to do so. */
		if (DBcfg.discardNonIp) {

			/* Boy will this get ugly if the syslog mask is not */
			/* configured properly..... */

			DBstats.f_outsideNonIp++;
			syslogMessage(SYSL_IN_FILTER, protocol);
			return DISCARD_PKT;
		}
	}

	return ALLOW_PKT;
}

/* ------------------------------------------------------------------------ */

static int
checkAcceptTable(struct ip *ipHeader, u_int32_t srcAddr, u_int32_t dstAddr)
{
	register int i;
	register u_int32_t addr, mask;
	
	/*
	 * Make sure the IP src address in this packet is in the accept table.
	 */
	for (i=0; ; i++) {
		addr = acceptTable[i].network.s_addr;
		mask = acceptTable[i].mask;

		if ((i >= MAX_ACCEPT_ENTRIES) || (!addr && !mask)) {
			/*
			 * If the table is empty, then default to forward all
			 * packets.  If it's not empty and the address was not
			 * found, then discard.
			 */
			if (i == 0)
				break;
			DBstats.f_insideAcceptTable++;
			syslogMessage(SYSL_OUT_ACCEPT, (int)ipHeader->ip_p,
				srcAddr, dstAddr);
			return DISCARD_PKT;
		}
		if ((srcAddr & mask) == addr) {
			/*
			 * Found - if this is an inverted rule, then discard,
			 * otherwise, accept.
			 */
			if (acceptTable[i].flag == INVERT) {
				DBstats.f_insideAcceptTable++;
				syslogMessage(SYSL_OUT_ACCEPT,
					(int)ipHeader->ip_p,
					srcAddr, dstAddr);
				return DISCARD_PKT;
			}
			break;
		}
	}

	return ALLOW_PKT;
}

/* ------------------------------------------------------------------------ */
/*
 * Parameters 'protocol' and 'protolen' are expected to be in host byte order
 * The packet pointed to by 'pkt' is, of course, in network btye order
 */

static int
checkOutgoingPacket(u_int16_t protocol, u_int8_t *pkt, u_int protolen)
{
	struct ip *ipHeader;
	struct tcphdr *tcpHeader;
	struct udphdr *udpHeader;
	u_int16_t ip_len;
	u_int32_t srcAddr, dstAddr;

	switch (protocol) {

	case ETHERTYPE_IP:
		ipHeader = (struct ip *)pkt;
		srcAddr = ntohl(ipHeader->ip_src.s_addr);
		dstAddr = ntohl(ipHeader->ip_dst.s_addr);

		if (checkAcceptTable(ipHeader,srcAddr,dstAddr) == DISCARD_PKT)
			return DISCARD_PKT;

		/*
		 * Pass all IP fragments that do not have offset 0 (beginning
		 * of the packet) without checking since the TCP/UDP
		 * headers are not in this packet.  An area for improvement
		 * would be to cache session info so we could drop all
		 * disallowed fragments also instead of just the first one.
		 */
		if ((ipHeader->ip_off & SW_IP_OFFMASK) != 0) {

			if (DBcfg.discardSuspectOffset &&
			    (ntohs(ipHeader->ip_off & SW_IP_OFFMASK) == 1) &&
			    (ipHeader->ip_p == IPPROTO_TCP)) {

				DBstats.f_insideSuspectOffset++;
				syslogMessage(SYSL_OUT_OFFSET, IPPROTO_TCP,
					srcAddr, dstAddr);

				return DISCARD_PKT;
			}

			return ALLOW_PKT;
		}

		switch (ipHeader->ip_p) {

		case IPPROTO_TCP:
			tcpHeader = (struct tcphdr *)(pkt + 
						(ipHeader->ip_hl << 2)); 
			/*
			 * Make sure this packet (fragment) includes enough
			 * of the TCP header before making the other checks.
			 */
			ip_len = ntohs(ipHeader->ip_len);
			if ((ip_len > protolen) ||
			    ((ip_len - (ipHeader->ip_hl << 2)) < 14)) {
				DBstats.f_insideShortTcpHdr++;
				syslogMessage(SYSL_OUT_LENGTH, IPPROTO_TCP,
					srcAddr, dstAddr,
					ntohs(tcpHeader->th_sport),
					ntohs(tcpHeader->th_dport));

				return DISCARD_PKT;
			}

			/* Check for "ACKless SYN" */
			if ((tcpHeader->th_flags & (TH_SYN|TH_ACK)) == TH_SYN) {
				return checkOutgoingTcp(srcAddr, dstAddr,
						ntohs(tcpHeader->th_sport),
						ntohs(tcpHeader->th_dport));
			}
			break;

		case IPPROTO_UDP:
			udpHeader = (struct udphdr *)(pkt + 
						(ipHeader->ip_hl << 2));
			/*
			 * Make sure this packet (fragment) includes enough
			 * of the UDP header before making the other checks.
			 */
			ip_len = ntohs(ipHeader->ip_len);
			if ((ip_len > protolen) ||
			    ((ip_len - (ipHeader->ip_hl << 2)) < 4)) {
				DBstats.f_insideShortUdpHdr++;
				syslogMessage(SYSL_OUT_LENGTH, IPPROTO_UDP,
					srcAddr, dstAddr,
					ntohs(udpHeader->uh_sport),
					ntohs(udpHeader->uh_dport));

				return DISCARD_PKT;
			}

			return  checkOutgoingUdp( srcAddr, dstAddr,
					ntohs(udpHeader->uh_sport),
					ntohs(udpHeader->uh_dport));
			break;

		case IPPROTO_ICMP:
			/* always pass ICMP out */
			break;

		default:
			if (DBcfg.discardOtherIp) {

				DBstats.f_insideOtherIp++;
				syslogMessage(SYSL_OUT_PROT,(int)ipHeader->ip_p,
					srcAddr, dstAddr);

				return DISCARD_PKT;
			}

			/* Everything is allowed so far. */
			break;
		}

		break;

	case ETHERTYPE_ARP:
	case ETHERTYPE_REVARP:
		/* pass these guys through no matter what */
		break;

	default:
		/* Only pass the rest if told to do so. */
		if (DBcfg.discardNonIp) {

			DBstats.f_insideNonIp++;
			syslogMessage(SYSL_OUT_FILTER, (int)protocol);
			return DISCARD_PKT;
		}
	}

	return ALLOW_PKT;
}

/* ------------------------------------------------------------------------ */
/*
 * Parameter 'dst' is expected to be in network byte order
 */

int
drawbridge_bridge_lookup(HardwareAddr *dst, struct ifnet *ifp)
{
	register int hash;
	register int curr;

	if (!DBstats.running ||
	    ((ifp != DBcfg.in_ifp) && (ifp != DBcfg.out_ifp)))
	    	/*
		 * Drawbridge is not running or this is not one of
		 * our interfaces so just process the pkt normally.
		 */
		return DB_PROCESS_PKT;

	/*
	 * If broadcast, then the packet needs to go to any interface
	 * that is setup for 'listen'.  If neither is enabled for
	 * 'listen', then drop the packet.
	 */
	if (IS_BROADCAST(dst))
		return DBcfg.listenMode;

	/*
	 * Check the bridge table for the destination address.
	 */
	hash = (dst->words[0] ^ dst->words[1] ^ dst->words[2])
		& ADDRESS_HASH_MASK;

	for (curr = hash; hashTable[curr].valid; ) {

		if (IS_HW_ADDR_EQU(dst, hashTable[curr].address)) {
		    	/*
			 * We found the address and know the side it's on.
			 * If that interface is enabled for 'listen', then
			 * send the packet.  Otherwise, drop it.
			 */
			return (hashTable[curr].interface & DBcfg.listenMode);
		}

		/*
		 * Increment 'curr' and check to see if we have wrapped.
		 * If so, the table is full and the address was not found.
		 */
		if ((curr = (curr+1) & ADDRESS_HASH_MASK) == hash)
			break;
	}

	/*
	 * Not found.  We don't know which side the destination is on so
	 * send the packet out the interface(s) that have 'listen' enabled.
	 */
	return DBcfg.listenMode;
} 

/* ------------------------------------------------------------------------ */
/*
 * Parameters 'src' and 'dst' are expected to be in network byte order
 */

static int
bridge(int interface, HardwareAddr *src, HardwareAddr *dst)
{
	int pass = YES;

	/*
	 * Check if the destination address is a broadcast or
	 * multicast address and always forward if so.
	 */
	if (!IS_BROADCAST(dst)) {
		register int curr;
		register int hash;

		/*
		 * Check the bridge table for the destination address.
		 */
		hash = (dst->words[0] ^ dst->words[1] ^ dst->words[2])
			& ADDRESS_HASH_MASK;

		for (curr = hash; hashTable[curr].valid; ) {

			if (IS_HW_ADDR_EQU(dst, hashTable[curr].address)) {
			    	/*
				 * We found the address and know which side of
				 * the bridge it lives on.  Only forward if
				 * this packet is for a host on the other side.
				 */
				if (interface == hashTable[curr].interface)
					pass = NO;
				break;
			}

			/*
			 * Increment 'curr' and check to see if we have wrapped.
			 * If so, the table is full and the address was not
			 * found so forward the packet regardless.
			 */
			if ((curr = (curr+1) & ADDRESS_HASH_MASK) == hash)
				break;
		}
	}

	/*
	 * Now check the source address and make sure it's in the bridge
	 * table.  Checking for broadcast is somewhat superfluous but if
	 * some broken implementation sources a broadcast or multicast
	 * address, we sure don't want to add it to the bridge table!
	 */
	if (!IS_BROADCAST(src)) {
		register int curr;
		register int hash;

		/*
		 * Check the bridge table for the source address.
		 */
		hash = (src->words[0] ^ src->words[1] ^ src->words[2])
			& ADDRESS_HASH_MASK;

		for (curr = hash; ; ) {

			if (IS_HW_ADDR_EQU(src, hashTable[curr].address)) {
				int s;
				
			    	/*
				 * It's already in the table so just make
				 * sure that the side of the bridge it's on
				 * stays up to date.
				 */
				s = splimp();
				
				hashTable[curr].interface = interface;
				
				splx(s);
			    	break;
			}

			/*
			 * If the above compare was false, and this slot is
			 * open, then the address is not in the table and
			 * we can just instert it here.
			 */
			if (hashTable[curr].valid == NO) {
				int s;

				/*
				 * Wrap this due to reentrancy problems.
				 */
				s = splimp();
				
                                hashTable[curr].address.addr = src->addr;
                                hashTable[curr].interface = interface;
                                hashTable[curr].valid = YES;

				splx(s);

				DBstats.bridgeEntries++;
				break;
			}

			/*
			 * Increment 'curr' and check to see if we have wrapped.
			 * If so, then the table is already full.
			 */
			if ((curr = (curr+1) & ADDRESS_HASH_MASK) == hash)
				break;
		}
	}

	return pass;
}

/* ------------------------------------------------------------------------ */
/*
 * 'ifp' is a pointer to the interface pointer
 * 'h' is a pointer to the packet header
 * 'm' is a pointer to the mbuf containing the packet
 */

int
drawbridge_test_pkt(struct ifnet **ifp, void *h, struct mbuf *m)
{
	u_int16_t protocol;
	u_int8_t *pkt;
	u_int protolen, pktlen;
	HardwareAddr *dhost;
	HardwareAddr *shost;

	if (!DBstats.running)
		return DB_PROCESS_PKT;

	switch (DBcfg.media_type) {
	case IFT_ETHER:
		dhost = (HardwareAddr *)((struct ether_header *)h)->ether_dhost;
		shost = (HardwareAddr *)((struct ether_header *)h)->ether_shost;
		protocol = ntohs(((struct ether_header *)h)->ether_type);
		pkt = m->m_data;
		protolen = m->m_pkthdr.len;
		pktlen = m->m_pkthdr.len + sizeof(struct ether_header);
		break;

	case IFT_FDDI:
		dhost = (HardwareAddr *)((struct fddi_header *)h)->fddi_dhost;
		shost = (HardwareAddr *)((struct fddi_header *)h)->fddi_shost;
		if (*((u_char *)(m->m_data)) == LLC_SNAP_LSAP) {
			protocol = ntohs(mtod(m, struct llc *)->
					 llc_snap.ether_type);
                	pkt = m->m_data + LLC_SNAPLEN;
			protolen = m->m_pkthdr.len - LLC_SNAPLEN;
		} else {
			protocol = 0;
                	pkt = m->m_data;
			protolen = m->m_pkthdr.len;
		}
		pktlen = m->m_pkthdr.len + sizeof(struct fddi_header);
		break;

	default:
		printf("unsupported media - disabling drawbridge\n");
		DBstats.running = NO;
		return DB_PROCESS_PKT;
	}

	if (*ifp == DBcfg.in_ifp) {   /* pkt from the inside interface */

		DBstats.p_insideRx++;
		DBstats.b_insideRx += pktlen;

		/*
		 * Check the bridge table to see if the destination host is
		 * on the same side as the source host.  If so, drop the pkt.
		 */
		if (bridge(DB_INSIDE_IF, shost, dhost) == NO)
			return DB_DROP_PKT;

		/*
		 * The packet should be delivered locally if it is directed
		 * to us or if it is an ARP for us.  If it is for us and
		 * listenMode is disabled for the interface, drop the pakcet.
		 */
		if (IS_HW_ADDR_EQU(dhost, DBcfg.local_hwaddr) ||
		    (IS_BROADCAST(dhost) && (protocol == ETHERTYPE_ARP) &&
		    (DBcfg.local_IP.s_addr == ARP_TARGET_IP_ADDR(pkt)))) {
			if (DBcfg.listenMode & DB_INSIDE_IF) {
				/* Check the incoming packet filters */
				if (checkIncomingPacket(protocol, pkt,
				    protolen, 1) == DISCARD_PKT) {
					DBstats.p_outsideFiltered++;
					DBstats.b_outsideFiltered += pktlen;
					return DB_DROP_PKT;
				}
				*ifp = DBcfg.local_ifp;
				m->m_pkthdr.rcvif = DBcfg.local_ifp;
				return DB_PROCESS_PKT;
			} else
				return DB_DROP_PKT;
		}

		/*
		 * Check to see if the packet should be allowed through.
		 */
		if (checkOutgoingPacket(protocol,pkt,protolen) == DISCARD_PKT) {
			DBstats.p_insideFiltered++;
			DBstats.b_insideFiltered += pktlen;
			return DB_DROP_PKT;
		}

		/*
		 * Forward the pkt to the outside interface
		 */
		DBstats.p_outsideTx++;
		DBstats.b_outsideTx += pktlen;
		*ifp = DBcfg.out_ifp;
		return DB_FORWARD_PKT;

	} else if (*ifp == DBcfg.out_ifp) {  /* from the outside interface */

		DBstats.p_outsideRx++;
		DBstats.b_outsideRx += pktlen;

		/*
		 * Check the bridge table to see if the destination host is
		 * on the same side as the source host.  If so, drop the pkt.
		 */
		if (bridge(DB_OUTSIDE_IF, shost, dhost) == NO)
			return DB_DROP_PKT;

		/*
		 * The packet should be delivered locally if it is directed
		 * to us or if it is an ARP for us.  If it is for us and
		 * listenMode is disabled for the interface, drop the pakcet.
		 */
		if (IS_HW_ADDR_EQU(dhost, DBcfg.local_hwaddr) ||
		    (IS_BROADCAST(dhost) && (protocol == ETHERTYPE_ARP) &&
		    (DBcfg.local_IP.s_addr == ARP_TARGET_IP_ADDR(pkt)))) {
			if (DBcfg.listenMode & DB_OUTSIDE_IF) {
				*ifp = DBcfg.local_ifp;
				m->m_pkthdr.rcvif = DBcfg.local_ifp;

				/* Check the incoming packet filters */
				if (checkIncomingPacket(protocol, pkt,
				    protolen, 0) == DISCARD_PKT) {
					DBstats.p_outsideFiltered++;
					DBstats.b_outsideFiltered += pktlen;
					return DB_DROP_PKT;
				}
				return DB_PROCESS_PKT;
			} else
				return DB_DROP_PKT;
		}

		/*
		 * Check to see if the packet should be allowed through.
		 */
		if (checkIncomingPacket(protocol,pkt,protolen, 0)==DISCARD_PKT){
			DBstats.p_outsideFiltered++;
			DBstats.b_outsideFiltered += pktlen;
			return DB_DROP_PKT;
		}

		/*
		 * forward the pkt to the inside interface
		 */
		DBstats.p_insideTx++;
		DBstats.b_insideTx += pktlen;
		*ifp = DBcfg.in_ifp;
		return DB_FORWARD_PKT;
	}

	/*
	 * The pkt is not from either filter interface so process it normally
	 */
	return DB_PROCESS_PKT;
}

/* ------------------------------------------------------------------------ */

static int
db_open(dev_t dev, int flags, int fmt, struct proc *p)
{
	if (minor(dev) >= 1)
		return ENXIO;

	return NO_ERROR;
}

/* ------------------------------------------------------------------------ */

static int
db_close(dev_t dev, int flags, int fmt, struct proc *p)
{
	if (minor(dev) >= 1)
		return ENXIO;

	return NO_ERROR;
}

/* ------------------------------------------------------------------------ */

static int
db_ifpromisc(struct ifnet *ifp, int pswitch)
{
        struct ifreq ifr;
	int s,r;

	s = splimp();
        if (pswitch) {
                if ((ifp->if_flags & IFF_UP) == 0)
                        return ERROR_IF_DOWN;
                if (ifp->if_pcount++ != 0)
                        return NO_ERROR;
                ifp->if_flags |= IFF_PROMISC;
        } else {
                if (--ifp->if_pcount > 0)
                        return NO_ERROR;
                ifp->if_flags &= ~IFF_PROMISC;
        }
        ifr.ifr_flags = ifp->if_flags;
        r = ((*ifp->if_ioctl)(ifp, SIOCSIFFLAGS, (caddr_t)&ifr));
	splx(s);

	return r;
}

/* ------------------------------------------------------------------------ */

static int
db_gethwaddr(struct ifnet *ifp, HardwareAddr *addr)
{
	/* Assume that ifp is actually pointer to arpcom structure */
	register struct arpcom *ac = (struct arpcom *)ifp;

	if (ac == NULL)
		return ERROR_GET_HW;

	bcopy(ac->ac_enaddr, (caddr_t)addr, sizeof(*addr));

	return NO_ERROR;
}

/* ------------------------------------------------------------------------ */

static int
db_getIPaddr(struct ifnet *ifp, in_addr_t *addr)
{
	struct in_ifaddr *ia;

	IFP_TO_IA(ifp, ia);

	if (ia == NULL)
		return ERROR_GET_IP;

	bcopy(&(ia->ia_addr.sin_addr), (caddr_t)addr, sizeof(*addr));

	return NO_ERROR;
}

/* ------------------------------------------------------------------------ */

static int
initRequest(initReq *req)
{
	int error;

	/*
	 * Clear all config info and stats (except log info)
	 */
	initialized = NO;
	bzero(&DBcfg.CFG_CLR_START, CFG_CLR_SIZE);
	bzero(&DBstats, sizeof(DBstats));

	/*
	 * Map the interface names to interface pointers.  Also
	 * make sure the interfaces are valid and that the user
	 * didn't specify the same interface for both sides. 
	 */
	if ((DBcfg.in_ifp = ifunit(req->inName)) == NULL) {
		DB_PRINTF(("DIOCINIT: invalid interface (inside)\n"));
		return ERROR_INVALID_IF; /* invalid interface */
	}
	if ((DBcfg.out_ifp = ifunit(req->outName)) == NULL) {
		DB_PRINTF(("DIOCINIT: invalid interface (outside)\n"));
		return ERROR_INVALID_IF; /* invalid interface */
	}
	if (DBcfg.in_ifp == DBcfg.out_ifp) {
		DB_PRINTF(("DIOCINIT: can't use same interface\n"));
		return ERROR_SAME_IF;
	}

	/*
	 * check the media type of the interfaces
	 */
	DBcfg.media_type = DBstats.media_type = DBcfg.in_ifp->if_type;
	if (DBcfg.media_type != DBcfg.out_ifp->if_type) {
		DB_PRINTF(("DIOCINIT: can't use different media types\n"));
		return ERROR_DIFF_IF_TYPE;  /* must be same media*/
	}
	if ((DBcfg.media_type != IFT_ETHER) && (DBcfg.media_type != IFT_FDDI)) {
		DB_PRINTF(("DIOCINIT: unsupported media type\n"));
		return ERROR_BAD_IF_TYPE;  /* media not supported */
	}

	/*
	 * make sure both interfaces are up and running
	 */
	if ((DBcfg.in_ifp->if_flags & (IFF_UP|IFF_RUNNING)) !=
	    (IFF_UP|IFF_RUNNING)) {
	    	DB_PRINTF(("DIOCINIT: inside interface down or not running\n"));
		return ERROR_INSIDE_DOWN;
	}
	if ((DBcfg.out_ifp->if_flags & (IFF_UP|IFF_RUNNING)) !=
	    (IFF_UP|IFF_RUNNING)) {
	    	DB_PRINTF(("DIOCINIT:outside interface down or not running\n"));
		return ERROR_OUTSIDE_DOWN;
	}

	/*
	 * get hw addr of interfaces
	 */
	if ((error = db_gethwaddr(DBcfg.in_ifp, &DBcfg.in_hwaddr))) {
		DB_PRINTF(("DIOCINIT: could not get hw addr (inside)\n"));
		return error;
	}
	if ((error = db_gethwaddr(DBcfg.out_ifp, &DBcfg.out_hwaddr))) {
		DB_PRINTF(("DIOCINIT: could not get hw addr (outside)\n"));
		return error;
	}

	/*
	 * get IP addr of interfaces
	 */
	if ((error = db_getIPaddr(DBcfg.in_ifp, &DBcfg.in_IP))) {
		DB_PRINTF(("DIOCINIT: could not get IP addr (inside)\n"));
		return error;
	}
	if ((error = db_getIPaddr(DBcfg.out_ifp, &DBcfg.out_IP))) {
		DB_PRINTF(("DIOCINIT: could not get IP addr (outside)\n"));
		return error;
	}

	/*
	 * Init other stuff
	 */
	initTables();

	strncpy(DBcfg.in_name,req->inName, MAX_DEV_NAMELEN);
	strncpy(DBcfg.out_name,req->outName, MAX_DEV_NAMELEN);

	DBcfg.listenMode = req->listenMode;

	if (DBcfg.listenMode) {
		if ((DBcfg.in_IP.s_addr != 0L) &&
		    (DBcfg.out_IP.s_addr != 0L)) {
			DB_PRINTF(("DIOCINIT: Only one interface can be " \
				   "configured with an IP number\n"));
			return ERROR_TWO_IP;
		} else if ((DBcfg.in_IP.s_addr == 0L) &&
			   (DBcfg.out_IP.s_addr == 0L)) {
			DB_PRINTF(("DIOCINIT: One interface must be " \
				   "configured with an IP number\n"));
			return ERROR_NO_IP;
		}

		if (DBcfg.in_IP.s_addr != 0L) {
			DBcfg.local_IP = DBcfg.in_IP;
			DBcfg.local_hwaddr = DBcfg.in_hwaddr;
			DBcfg.local_ifp = DBcfg.in_ifp;
		} else {
			DBcfg.local_IP = DBcfg.out_IP;
			DBcfg.local_hwaddr = DBcfg.out_hwaddr;
			DBcfg.local_ifp = DBcfg.out_ifp;
		}

		if ((DBcfg.listenMode != DB_INSIDE_IF) &&
		    (DBcfg.listenMode != DB_OUTSIDE_IF) &&
		    (DBcfg.listenMode != DB_BOTH_IF))
			return ERROR_INVALID_MODE;
	}

	DB_PRINTF(("DIOCINIT: inside  hw addr " \
		   "%02x:%02x:%02x:%02x:%02x:%02x\n",
		   DBcfg.in_hwaddr.bytes[0], DBcfg.in_hwaddr.bytes[1],
		   DBcfg.in_hwaddr.bytes[2], DBcfg.in_hwaddr.bytes[3],
		   DBcfg.in_hwaddr.bytes[4], DBcfg.in_hwaddr.bytes[5]));
	DB_PRINTF(("DIOCINIT: inside  ip addr " \
		   "%d.%d.%d.%d\n",
		   DBcfg.in_IP.s_un_b.s_b[0],
		   DBcfg.in_IP.s_un_b.s_b[1],
		   DBcfg.in_IP.s_un_b.s_b[2],
		   DBcfg.in_IP.s_un_b.s_b[3]));
	DB_PRINTF(("DIOCINIT: outside hw addr " \
		   "%02x:%02x:%02x:%02x:%02x:%02x\n",
		   DBcfg.out_hwaddr.bytes[0], DBcfg.out_hwaddr.bytes[1],
		   DBcfg.out_hwaddr.bytes[2], DBcfg.out_hwaddr.bytes[3],
		   DBcfg.out_hwaddr.bytes[4], DBcfg.out_hwaddr.bytes[5]));
	DB_PRINTF(("DIOCINIT: outside ip addr " \
		   "%d.%d.%d.%d\n",
		   DBcfg.out_IP.s_un_b.s_b[0],
		   DBcfg.out_IP.s_un_b.s_b[1],
		   DBcfg.out_IP.s_un_b.s_b[2],
		   DBcfg.out_IP.s_un_b.s_b[3]));
	DB_PRINTF(("DIOCINIT: local ip addr " \
		   "%d.%d.%d.%d\n",
		   DBcfg.local_IP.s_un_b.s_b[0],
		   DBcfg.local_IP.s_un_b.s_b[1],
		   DBcfg.local_IP.s_un_b.s_b[2],
		   DBcfg.local_IP.s_un_b.s_b[3]));

	/*
	 * All went well so set initialized to yes
	 */
	initialized = YES;
	syslogMessage(SYSL_COLD_START);

	return NO_ERROR;
}

/* ------------------------------------------------------------------------ */

static int
db_ioctl(dev_t dev, int cmd, caddr_t addr, int flags, struct proc *p)
{
	int s, error = 0;
	static struct timeval stopTime;

	if (initialized == 0) {
		if ((cmd != DIOCINIT)  && (cmd != DIOCGLOGF) &&
	    	    (cmd != DIOCSLOGF) && (cmd != DIOCGLOGM) &&
	    	    (cmd != DIOCSLOGM) && (cmd != DIOCGVER)) {
			DB_PRINTF(("DBIOCTL: not initialized\n"));
			return ERROR_NOT_INITILIZED; /* not initialized */
		}
	}

	switch (cmd) {

	case DIOCINIT:		/* Init */
		/*
		 * If drawbridge is running then don't try to reinit
		 */
		if ((flags & FWRITE) == 0)
			return EACCES;
		if (DBstats.running) {
			DB_PRINTF(("DBIOCTL: can't init while drawbridge " \
				   "is running\n"));
			error = ERROR_RUNNING;
			break;
		}
		error = initRequest((initReq *)addr);
		break;

	case DIOCSTART:		/* Start */
		if ((flags & FWRITE) == 0)
			return EACCES;
		if (DBstats.running) {
			DB_PRINTF(("DBIOCTL: drawbridge is already running\n"));
			error = ERROR_RUNNING;
			break;
		}
		/*
                 * put both interfaces into promiscuous mode
                 */
		if (!DBcfg.in_promiscuous) {
			if ((error = db_ifpromisc(DBcfg.in_ifp, 1))) {
				printf("Drawbridge error: couldn't enable " \
					" promiscuous mode for '%s'.\n",
					DBcfg.in_name);
				break;
			}
			DBcfg.in_promiscuous = YES;
		}
		if (!DBcfg.out_promiscuous) {
			if ((error = db_ifpromisc(DBcfg.out_ifp, 1))) {
				printf("Drawbridge error: couldn't enable " \
					" promiscuous mode for '%s'.\n",
					DBcfg.out_name);
				db_ifpromisc(DBcfg.in_ifp, 0);
				DBcfg.in_promiscuous = NO;
				break;
			}
			DBcfg.out_promiscuous = YES;
		}

		if (DBstats.startTime.tv_sec == 0)
			DBstats.startTime = time;
		else
			DBstats.startTime.tv_sec += (time.tv_sec -
						     stopTime.tv_sec);

		DBstats.running = YES;

		DB_PRINTF(("Drawbridge running...\n"));

		break;

	case DIOCSTOP:		/* Stop */
		if ((flags & FWRITE) == 0)
			return EACCES;
		if (!DBstats.running) {
			DB_PRINTF(("DBIOCTL: drawbridge is not running\n"));
			error = ERROR_NOT_RUNNING; /* not running */
			break;
		}
		DBstats.running = NO;
		stopTime = time;

		/*
                 * take both interfaces out of promiscuous mode
                 */
		if (DBcfg.in_promiscuous) {
			if ((error = db_ifpromisc(DBcfg.in_ifp, 0))) {
				printf("Drawbridge error: couldn't disable " \
					" promiscuous mode for '%s'.\n",
					DBcfg.in_name);
				break;
			}
			DBcfg.in_promiscuous = NO;
		}
		if (DBcfg.out_promiscuous) {
			if ((error = db_ifpromisc(DBcfg.out_ifp, 0))) {
				printf("Drawbridge error: couldn't disable " \
					" promiscuous mode for '%s'.\n",
					DBcfg.out_name);
				break;
			}
			DBcfg.out_promiscuous = NO;
		}

		DB_PRINTF(("Drawbridge stopped\n"));

		break;

	case DIOCGFLAGS: {	/* Read the flags */
		flagsReq *req = (flagsReq *)addr;

		req->discardMulticast = DBcfg.discardMulticast;
        	req->discardNonIp = DBcfg.discardNonIp;
		req->discardOtherIp = DBcfg.discardOtherIp;
		req->discardSuspectOffset = DBcfg.discardSuspectOffset;
		req->discardFragmentedICMP = DBcfg.discardFragmentedICMP;
		req->discardAttackICMP = DBcfg.discardAttackICMP;
		break;
	}

	case DIOCSFLAGS: {	/* Write the flags */
		flagsReq *req = (flagsReq *)addr;

		if ((flags & FWRITE) == 0)
			return EACCES;
        	DBcfg.discardMulticast = req->discardMulticast;
        	DBcfg.discardNonIp = req->discardNonIp;
		DBcfg.discardOtherIp = req->discardOtherIp;
		DBcfg.discardSuspectOffset = req->discardSuspectOffset;
		DBcfg.discardFragmentedICMP = req->discardFragmentedICMP;
		DBcfg.discardAttackICMP = req->discardAttackICMP;
		break;
	}

	case DIOCGSTATS:	/* Get the stats */
		*((Statistics *)addr) = DBstats;
		break;

	case DIOCCSTATS:	/* Clear the stats */
		if ((flags & FWRITE) == 0)
			return EACCES;
		s = splimp();
		bzero(&DBstats.STATS_CLR_START, STATS_CLR_SIZE);
		splx(s);
		break;

	case DIOCCBRIDGE:	/* Clear the bridge table */
		if ((flags & FWRITE) == 0)
			return EACCES;
		s = splimp();
		bzero(hashTable, sizeof(hashTable));
		DBstats.bridgeEntries = 0;
		splx(s);
		break;

	case DIOCGBRIDGE:	/* Read the bridge table */
		error = bridgeRequest(READ, (bridgeReq *)addr);
		break;

	case DIOCGCLASS:	/* Read from a class table */
		error = accessListRequest(READ,(accessListTableReq *)addr);
		break;

	case DIOCSCLASS:	/* Write to a class table */
		if ((flags & FWRITE) == 0)
			return EACCES;
		error = accessListRequest(WRITE,(accessListTableReq *)addr);
		break;

	case DIOCGNETWORK:	/* Read from a network table */
		error = networkTableRequest(READ, (networkTableReq *)addr);
		break;

	case DIOCSNETWORK:	/* Write to a network table */
		if ((flags & FWRITE) == 0)
			return EACCES;
		error = networkTableRequest(WRITE, (networkTableReq *)addr);
		break;

	case DIOCRNETWORK:	/* Remove a network table */
		if ((flags & FWRITE) == 0)
			return EACCES;
		error = networkTableRequest(REMOVE, (networkTableReq *)addr);
		break;

	case DIOCGNETWORKS:	/* Read the list of networks */
		bcopy((caddr_t)netTable, (caddr_t)addr, sizeof(netTable));
		break;

	case DIOCGOVERRIDE:	/* Read the override table */
		error = generalTableRequest(READ, (u_char *)overrideTable, addr,
					    sizeof(overrideTable));
		break;

	case DIOCSOVERRIDE:	/* Write the override table */
		if ((flags & FWRITE) == 0)
			return EACCES;
		error = generalTableRequest(WRITE,(u_char *)overrideTable, addr,
					    sizeof(overrideTable));
		break;

	case DIOCGREJECT:	/* Read the reject table */
		error = generalTableRequest(READ, (u_char *)rejectTable, addr,
					    sizeof(rejectTable));
		break;

	case DIOCSREJECT:	/* Write the reject table */
		if ((flags & FWRITE) == 0)
			return EACCES;
		error = generalTableRequest(WRITE, (u_char *)rejectTable, addr,
					    sizeof(rejectTable));
		break;

	case DIOCGACCEPT:	/* Read the accept address table */
		error = generalTableRequest(READ, (u_char *)acceptTable, addr,
					    sizeof(acceptTable));
		break;

	case DIOCSACCEPT:	/* Write the accept address table */
		if ((flags & FWRITE) == 0)
			return EACCES;
		error = generalTableRequest(WRITE, (u_char *)acceptTable, addr,
					    sizeof(acceptTable));
		break;

	case DIOCGLOGF:		/* Read the log facility */
		((logFacilityReq *)addr)->facility = DBcfg.logFacility >> 3;
		break;

	case DIOCSLOGF:		/* Write the log facility */
		if ((flags & FWRITE) == 0)
			return EACCES;
		DBcfg.logFacility = ((logFacilityReq *)addr)->facility << 3;
		break;

	case DIOCGLOGM:		/* Read the log mask */
		((logMaskReq *)addr)->mask = DBcfg.logMask;
		break;

	case DIOCSLOGM:		/* Write the log mask */
		if ((flags & FWRITE) == 0)
			return EACCES;
		DBcfg.logMask = ((logMaskReq *)addr)->mask;
		break;

	case DIOCGVER:		/* Read the version of Drawbridge */
		strncpy( ((versionReq *)addr)->version, VERSION, 32 );
		break;

	default:
		DB_PRINTF(("Invalid operation\n"));
		error = ERROR_INVALID_OP;
		break;

	}
	return error;
}

/* ------------------------------------------------------------------------ */

static db_drvsw_installed = 0;

static
void db_drvinit(void *unused)
{
	dev_t dev;

	if ( ! db_drvsw_installed ) {
		dev = makedev(CDEV_MAJOR, 0);
		cdevsw_add(&dev, &db_cdevsw, NULL);
		db_drvsw_installed = 1;
#ifdef DEVFS
		devfs_token = devfs_add_devswf(&db_cdevsw, 0, DV_CHR,
					UID_ROOT, GID_WHEEL, 0600,
					"drawbridge");
#endif
	}
}

SYSINIT(dbdev,SI_SUB_DRIVERS,SI_ORDER_MIDDLE+CDEV_MAJOR,db_drvinit,NULL)

#endif /* NDRAWBRIDGE > 0 */

/* ------------------------------------------------------------------------ */
