/* Gyve.m --- The definition for application object

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "Gyve.h"
#include "GyveView.h"
#include "GyveCanvas.h"
#include "GyveNil.h"
#include "gconsole.h"
#include "GyveDrawingEngine.h"
#include "GyveSelectionsLayer.h"
#include "GyveWindow.h"
#include "GyveBuffer.h"
#include "GuileConsole.h"
#include "PSText.h"
#include "PSTextStyle.h"
#include "PSPaintStyle.h"
#include "toolbox.h"
#include "PSFigObj.h"
#include "public.h"
#include "PSFigObjGroup.h"
#include "LayersPallet.h"
#include "ToolBox.h"
#include "PSMaskedFigObjGroup.h"
#include <assert.h>
#include <stdlib.h>
#include <Foundation/NSBundle.h>
#include <Foundation/NSString.h>
#include <Foundation/NSProcessInfo.h>
#include <Foundation/NSException.h>
#include <Foundation/NSAutoreleasePool.h>
#include <Foundation/NSNotification.h>
#include <Foundation/NSArray.h>
#include <Foundation/NSSet.h>
#include <gdk/gdkx.h>

#include <DPS/dpsfriends.h>
#include <DPS/dpsXuserpath.h>   
#include <DPS/dpsXshare.h>
#include <DPS/dpsclient.h>
#include <DPS/dpsXclient.h>
#ifdef HAVE_DPS_NXAGENT    
#include <DPS/dpsNXargs.h>
#endif

#include <ScriptKit/Guile.h>
#include <ScriptKit/GuileInterpreter.h>

#include "utilities.h"

/*
 * Notification
 */
NSString * GyveDocumentWindowWillChangeNotification = 
@"GyveDocumentWindowWillChangeNotification";
NSString * GyveDocumentWindowDidChangeNotification = 
@"GyveDocumentWindowDidChangeNotification";

NSString * GyveDocumentWindowWillBeAddedNotification =
@" GyveDocumentWindowWillBeAddedNotification";
NSString * GyveDocumentWindowWasAddedNotification =
@"GyveDocumentWindowWasAddedNotification";

/*
 * Private functions
 */
static void GyveDPSTextBackstop (DPSContext ctxt, char *buf,
				   long unsigned int count);
static void GyveDPSErrorProc (DPSContext ctxt,
			      DPSErrorCode errorCode,
			     long unsigned int arg1, long unsigned int arg2);

static GtkWidget * boot_progress_bar_new();
static void boot_progress_bar_update(GtkWidget * pbar,
				      gfloat percentage,
				      char * message);
static void boot_progress_bar_destroy(GtkWidget * pbar);

/*
 * Private categories
 */ 
@interface Gyve(Init)
- (BOOL)dpsInit;
- (BOOL)bundleInit;
- (BOOL)guileInit;
@end

@implementation Gyve(Init)
- (BOOL)dpsInit
{
  BOOL result_of_init;
  /* 
   * Check availability of DPS extension.
   */
#if HAVE_DPS_NXAGENT    
  boot_progress_bar_update(pbar, 0.5, "Set DPS NXAGENT to lauch automatically...");
  XDPSNXSetClientArg(XDPSNX_AUTO_LAUNCH, (void *)True);
  boot_progress_bar_update(pbar, 0.6, "Set DPS NXAGENT to lauch automatically...done");
  result_of_init = YES;
#else    
  boot_progress_bar_update(pbar, 0.5, "Check X server supports DPS extension...");
  if (XDPSExtensionPresent(GDK_DISPLAY()))
    {
      boot_progress_bar_update(pbar, 0.6,
			       "Check X server supports DPS extension...YES");
      result_of_init = YES;
    }
  else 
    {
      boot_progress_bar_update(pbar, 0.6,
			       "Check X server supports DPS extension...NO");
      result_of_init = NO;
    }
#endif  /* HAVE_DPS_NXAGENT */  

  /*
   * Construct shared dps context
   */
  if (result_of_init)
    {
       boot_progress_bar_update(pbar, 0.65,
			       "Build a shared DPS context...done");
       shared_dps_context = XDPSGetSharedContext(GDK_DISPLAY());

      boot_progress_bar_update(pbar, 0.7,
			       "Build a shared DPS context...done");
    }
  return result_of_init;
}
- (BOOL)bundleInit
{
  extern char *objc_find_executable(const char *name);
  NSString * process_name = [[NSProcessInfo processInfo] processName];
  NSString * path_for_gyve = nil, * path_for_gyvebundle = nil;
  char * cpath_for_gyve = NULL;

  boot_progress_bar_update(pbar, 0.75,
			   "Search system bundle...");
  cpath_for_gyve = objc_find_executable([process_name cString]);
  assert(cpath_for_gyve);
  path_for_gyve = [NSString stringWithCString: cpath_for_gyve];
  OBJC_FREE(cpath_for_gyve);
  
  /* /foo/bar/baz/gyve => /foo/bar/baz/gyvebundle */
  path_for_gyvebundle = [[path_for_gyve stringByDeletingLastPathComponent]
			  stringByAppendingString: @"/gyvebundle"];

  system_root_bundle = [[NSBundle bundleWithPath: path_for_gyvebundle] 
			 retain];
  if (system_root_bundle)
    {
      boot_progress_bar_update(pbar, 0.8,
			       "Search system bundle...found");
      fprintf(stderr, "found\n");
      fprintf(stderr, "bundle> %s\n", [[system_root_bundle bundlePath] 
					cString]);
      fprintf(stderr, "resources> %s\n", [[system_root_bundle resourcePath] 
				  cString]);
      return YES;
    }
  else
    {
      boot_progress_bar_update(pbar, 0.8,
			       "Search system bundle...cannot find");
      return NO;
    }
}
- (BOOL)guileInit
{
  boot_progress_bar_update(pbar, 0.2, "Bootstrapping guile...");
  [GuileInterpreter initializeInterpreter];
  [GuileSCM description];
  [GuileScript description];
  boot_progress_bar_update(pbar, 0.25, "Bootstrapping guile...");
  guile_interpreter = [[GuileInterpreter alloc] init];
  boot_progress_bar_update(pbar, 0.4, "Bootstrapping guile...");
  [guile_interpreter eval: @"(use-modules (languages gstep-guile))"];
  // [guile_interpreter eval: @"(use-modules (gyve2 gyve))"];
  boot_progress_bar_update(pbar, 0.45, "Bootstrapping guile...done");
  return YES;
}
@end

/*
 * Gyve public categories
 */ 
static Gyve * gyve = nil;
@implementation Gyve
- (id)initWithArguments:(char **)_argv
  count:(int)_argc
  environment:(char **)_env
{
  gtk_init (&_argc, &_argv);
  
  pbar = boot_progress_bar_new();
  gtk_window_set_wmclass (GTK_WINDOW(pbar), "gyve_startup", "Gyve");
  gtk_window_position(GTK_WINDOW(pbar),
		      GTK_WIN_POS_CENTER);
  gtk_widget_show(pbar);
  gtk_widget_map (pbar);
  gtk_window_set_policy (GTK_WINDOW (pbar), FALSE, TRUE, FALSE);
  while (gtk_events_pending ())
    gtk_main_iteration_do (TRUE);
  
  mode = batch_mode;
  
  [self guileInit];
  [self dpsInit];
  [self bundleInit];
  figures_register    = [[NSMutableDictionary alloc] initWithCapacity: 2];

  // FOR DEBUG 
  if (_argc == 3)
    {
      if (streq(_argv[1], "--eval") || streq(_argv[1], "-e"))
	{
	  fprintf(stderr, "- eval script -\n");
	  gh_eval_str (_argv[2]);
	  fprintf(stderr, "- eval script -\n");
	}
      else if  (streq(_argv[1], "-l") || streq(_argv[1], "--load"))
	{
	  FILE * fp = fopen(_argv[2],
			    "r");
	  if (fp)
	    {
	      fclose(fp);
	      gh_eval_file (_argv[2]);
	    }
	  else
	    fprintf(stderr, "*** Loading failed: %s\n", _argv[2]);
	}
    }
  // END FOR DEBUG
  gyve = self; 
  return self ;
}
-(void)dealloc
{
  /* DPSSpace space = DPSSpaceFromContext(shared_dps_context); */
  [system_root_bundle release], system_root_bundle = nil;
  [guile_interpreter release], guile_interpreter = nil;
  /* DPSDestroySpace(space); 
  space = NULL; */
  shared_dps_context = NULL;
  [super dealloc];
}
- (DPSContext)sharedDPSContext
{
  return shared_dps_context;
}
- (NSMutableDictionary *) figureRegister
{
  return figures_register;
}

- (void)run
{
  BOOL isTerminated = NO;

  NSAutoreleasePool * pool = [NSAutoreleasePool new];
  boot_progress_bar_update(pbar, 0.9,
			   "Create the tool pallet...");
  toolbox_show();
  boot_progress_bar_update(pbar, 1.0,
			   "Create the tool pallet...done");
  boot_progress_bar_destroy(pbar); 
  [self loadInitScriptFile];
  [pool release];
  
  while (isTerminated == NO) {
    NSAutoreleasePool *localPool = [NSAutoreleasePool new];
    gint returnValue;
    returnValue = gtk_main_iteration();
    if (returnValue == TRUE) isTerminated = YES;
    [localPool release];
    localPool = nil;
  }
}
+ (Gyve *)application
{
  return gyve;
}
- (enum GUI_mode)mode
{
  return mode;
}
- (void)setMode:(enum GUI_mode)m
{
  mode = m;
}

@end

@implementation Gyve(GuileService)
- (GuileInterpreter *)guileInterpreter
{
  return guile_interpreter;
}
- (BOOL)loadScriptFile: (NSString *)str
{
  [guile_interpreter loadFile: str];
  return YES;
}
- (BOOL)loadInitScriptFile
{
  FILE * fp;
  char * home = "HOME";
  char * home_c_path;
  NSString * home_objc_path;
  home_c_path = getenv(home);
  if (NULL == home_c_path)
    {
      return NO;
    }
  home_objc_path = [NSString stringWithCString: home_c_path];
  home_objc_path = [home_objc_path stringByAppendingString: @"/.gyve"];
  
  home_c_path = (char *)[home_objc_path cString];
  fp = fopen(home_c_path, "r");
  if (fp == NULL)
    {
      return NO;
    }
  fclose(fp);
  return [self loadScriptFile: home_objc_path];
}
@end

@implementation  Gyve(BundleService)
- (NSBundle *)systemRootBundle
{
  return system_root_bundle;
}
@end

/*
 * Private functions
 */

static void
GyveDPSTextBackstop (DPSContext ctxt, char *buf, long unsigned int count)
{
  fprintf(stderr, "-- GyveDPSTextBackstop --\n");
  DPSDefaultTextBackstop (ctxt, buf, count);
}

static void 
GyveDPSErrorProc (DPSContext ctxt,
		  DPSErrorCode errorCode,
		  long unsigned int arg1, long unsigned int arg2)
{
  fprintf(stderr, "-- GyveDPSErrorBackstop --\n");
  DPSDefaultErrorProc(ctxt, errorCode, arg1, arg2);
}


static GtkWidget * label = NULL;
static   GtkWidget * progress_bar;

static GtkWidget *
boot_progress_bar_new()
{
  GtkWidget * vbox;
  GtkWidget * window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), "GYVE Startup");
  gtk_container_border_width (GTK_CONTAINER (window), 3);
  
  vbox = gtk_vbox_new (FALSE, 0);  
  gtk_container_add (GTK_CONTAINER (window), vbox);
  gtk_widget_show (vbox);
  
  label = gtk_label_new("Gtk init...done");
  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);

  progress_bar = gtk_progress_bar_new();
  gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar), 0.1);
  gtk_box_pack_start (GTK_BOX (vbox), progress_bar, TRUE, TRUE, 0);  
  gtk_widget_show(progress_bar);
  return window;
}
static void
boot_progress_bar_update(GtkWidget * pbar,
			 gfloat percentage,
			 char * message)
{
  gtk_label_set(GTK_LABEL(label), message);
  gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar), percentage);
  
}

static void
boot_progress_bar_destroy(GtkWidget * pbar)
{
  gtk_widget_destroy(pbar);
}


void
gyve_copy_for_key(NSString * key)
{
  int i, max;
  GyveBuffer * document_buffer = [GyveBuffer currentBuffer];
  NSMutableDictionary * figure_register = [[Gyve application] figureRegister];
  PSFigObjSelectedProxy * figobj_proxy;
  PSTextSelectedProxy * text_proxy;
  id <PSFigObj> figobj;
  NSString * string;
  if (nil == document_buffer) return ;
  max 	= [document_buffer countSelectedFigObjs];
  if (max < 1)
    {
      gyve_message("No figure object to cut.");
      return ;
    }
  else if (max == 1)
    {
      //
      figobj_proxy = [document_buffer selectedFigObjAtIndex: 0];
      if ((YES 	  == [figobj_proxy isKindOfClass: [PSTextSelectedProxy class]])
	  && (YES == [(PSTextSelectedProxy *)figobj_proxy hasSelectionRange]))
	{
	  text_proxy = (PSTextSelectedProxy *)figobj_proxy;
	  string = [text_proxy substring];
	  [figure_register setObject: [[string copy] autorelease] 
			   forKey: key];
	}
      else
	{
	  figobj = [[[figobj_proxy targetForFigObjProxy] copy] autorelease];
	  [figure_register setObject: figobj forKey: key];
	}
    }
  else
    {
      NSMutableArray * figures = [NSMutableArray array];
      for (i = 0; i < max; i++)
	{
	  figobj_proxy = [document_buffer selectedFigObjAtIndex: i]; 
	  figobj = [[[figobj_proxy targetForFigObjProxy] copy] autorelease];
	  [figures addObject: figobj];
	}
      [figure_register setObject: figures forKey: key];
    }
}

void
gyve_cut_for_key(NSString * key)
{
  int i, max;
  GyveBuffer * document_buffer = [GyveBuffer currentBuffer];
  GyveWindow * document_window = [GyveWindow currentWindow];
  NSMutableDictionary * figure_register = [[Gyve application] figureRegister];
  PSFigObjSelectedProxy * figobj_proxy;
  PSTextSelectedProxy * text_proxy;

  id <PSFigObj> figobj;
  id <PSText> text;
  NSString * string;
  
  if (nil == document_buffer)
    {
      return ;
    }
  
  max 	= [document_buffer countSelectedFigObjs];
  if (max < 1)
    {
      gyve_message("No figure object to cut.");
      return ;
    }
  else if (max == 1)
    {
      // 
      figobj_proxy = [document_buffer selectedFigObjAtIndex: 0];
      if ((YES 	  == [figobj_proxy isKindOfClass: [PSTextSelectedProxy class]])
	  && (YES == [(PSTextSelectedProxy *)figobj_proxy hasSelectionRange]))
	{
	  text_proxy = (PSTextSelectedProxy *)figobj_proxy;
	  string = [text_proxy substring];
	  [figure_register setObject: [[string copy] autorelease] 
			   forKey: key];
	  [document_window redrawLockRetain];
	  text = [text_proxy targetForTextProxy];
	  [document_window redrawFigObj: text];
	  text 	= [text_proxy beginModificationByCopy];
	  [text_proxy removeTextElements];
	  [text_proxy endModification];
	  [document_window redrawFigObj: text];
	  [document_window redrawLockRelease];      
	}
      else
	{
	  figobj = [[[figobj_proxy targetForFigObjProxy] copy] autorelease];
	  [figure_register setObject: figobj forKey: key];
	  [document_window redrawLockRetain];
	  [document_window redrawFigObj: figobj];
	  [figobj_proxy delete];
	  [document_window redrawLockRelease];      
	}
    }
  else
    {
      const int current_index = 0;
      NSMutableArray * figures = [NSMutableArray array];
      [document_window redrawLockRetain];
      for (i = 0; i < max; i++)
	{
	  figobj_proxy =  [document_buffer selectedFigObjAtIndex: current_index];
	  figobj = [[[figobj_proxy targetForFigObjProxy] copy] autorelease];
	  [figures addObject: figobj];
	  [document_window redrawFigObj: figobj];
	  [figobj_proxy delete];
	}
      [document_window redrawLockRelease];
      [figure_register setObject: figures forKey: key];
    }
}

void
gyve_paste_for_key(NSString * key)
{
  GyveBuffer * document_buffer = [GyveBuffer currentBuffer];
  GyveWindow * document_window = [GyveWindow currentWindow];
  NSMutableDictionary * figure_register = [[Gyve application] figureRegister];
  id value = [figure_register objectForKey: key];
  NSArray * array_value;
  id<PSFigObj> figobj_value;
  id<PSText> text_value;

  PSTextSelectedProxy * text_proxy;
  id <PSText> text;

  [document_window redrawLockRetain];
  if (value == nil)
    {
      gyve_message("No figure object to paste.");
    }
  else if (YES == [value isKindOfClass: [NSArray class]])
    {
      int max, i;
      array_value = (NSArray *)value;
      max 	  = [array_value count];
      [document_buffer unSelectAll];
      for (i = 0; i < max; i++)
	{
	  figobj_value = [[[array_value objectAtIndex: i] copy] 
			   autorelease];
	  [document_buffer putFigObj: figobj_value];
	  [document_window redrawFigObj: figobj_value];
	}
    }
  else if (YES == [value conformsToProtocol: @protocol (PSFigObj)])
    {
      figobj_value = [[(id<PSFigObj>)value copy] autorelease];
      [document_buffer unSelectAll];
      [document_buffer putFigObj: figobj_value];
      [document_window redrawFigObj: figobj_value];
    }
  else if (YES == [value isKindOfClass: [NSString class]])
    {
      if ((1 == [document_buffer countSelectedFigObjs])
	  && (YES == [[document_buffer selectedFigObjAtIndex: 0] 
		       isKindOfClass: [PSTextSelectedProxy class]])
	  && (YES == [(PSTextSelectedProxy *)[document_buffer selectedFigObjAtIndex: 0]
					     hasInsertionIndex]))
	       
	{
	  text_proxy = (PSTextSelectedProxy *)[document_buffer selectedFigObjAtIndex: 0];
	  text = [text_proxy targetForTextProxy];
	  [document_window redrawFigObj: text];
	  text = [text_proxy beginModificationByCopy];
	  [text_proxy insertString: (NSString *)value];
	  [document_window redrawFigObj: text];
	  [text_proxy endModification];
	}
      else
	{
	  [document_buffer unSelectAll];
	  text_value = [[[PSTextAtPoint alloc] initWithString: (NSString *)value
					       point: &NSZeroPoint] autorelease];
	  [document_buffer putFigObj: text_value];
	  [document_window redrawFigObj: text_value];
	}
    }
  else
    {
      // abort
    }
  [document_window redrawLockRelease];  
}  
