/* GyveBuffer.m --- The definition for layers

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

/* Writing methods of GyveBuffer 
   1. 褬ɬפʥ᥽åɤ, 
   [self changedSinceLastDrawing: YES];
   Ȥ뤳. 

   If the redrawing without the image cache is required, add following
   an expressoin:
   [self changedSinceLastDrawing: YES];
   
   2. */

#include "GyveBuffer.h"
#include "GyveNil.h"

#include "GyveLayer.h"
#include "GyveSelectionsLayer.h"
#include "utilities.h"

#include <Foundation/NSException.h>
#include <Foundation/NSSet.h>
#include <Foundation/NSEnumerator.h>

NSString * GyveBufferWrongSelectionException = @"GyveBufferWrongSelectionException";

@implementation GyveBuffer
/*
 * Buffer ID 
 */
static NSMutableArray * buffers_array 	= nil;
static NSMutableSet * buffers_observers = nil;
static int current_buffer_ID 	      = 0;
+ (void)addBufferObserver: (NSObject<GyveBuffersObserver>*)observer
{
  if (nil == buffers_observers)
    buffers_observers = [[NSMutableSet set] retain];
  [buffers_observers addObject: observer];
}
+ (void)removeBufferObserver: (NSObject<GyveBuffersObserver>*)observer
{
  if (nil == buffers_observers) return;
  else [buffers_observers removeObject: observer];
}
+ (int)issueBufferIDFor: (GyveBuffer *)buffer
{
  if (nil == buffers_array)
    {
      buffers_array  = [[NSMutableArray array] retain];
      [buffers_array addObject: [GyveNil gyvenil]];
    }
  [buffers_array addObject: buffer];

  if (buffers_observers)
    [buffers_observers makeObjectsPerform: @selector(addedBuffer:)
		       withObject: buffer];

  return [buffers_array indexOfObject: buffer];
}
+ (GyveBuffer *)bufferFromID:(int)bufferID
{
  GyveBuffer * result;
  if (nil == buffers_array) return nil;	// Exception?
  if ([buffers_array count] > bufferID)
    {
      result = [buffers_array objectAtIndex: bufferID];
      if (result == [GyveNil gyvenil]) return nil;
      else return result;
    }
  else
    return nil;
}
+ (void)setCurrentBuffer:(GyveBuffer *)buffer
{
  GyveBuffer * old_buffer = [self currentBuffer];
  current_buffer_ID = [buffer bufferID];
  if (nil != buffers_observers)
    {
      id<GyveBuffersObserver> observer;
      NSEnumerator * e = [buffers_observers objectEnumerator];
      while (observer = [e nextObject], observer)
	[observer changeBufferTo: buffer from: old_buffer];
    }
}
+ (GyveBuffer * )currentBuffer
{
  GyveBuffer * result = [buffers_array objectAtIndex: current_buffer_ID];
  if (result == [GyveNil gyvenil])
    return nil;
  else
    return result;
}
- (int)bufferID
{
  return buffer_ID;
}
+ (GyveBuffer * )lookUpBufferByName: (NSString *)_name
{
  id result = nil;
  int max, i;
  max = [buffers_array count];
  for (i = 0; i < max ; i++)
    {
      result = [buffers_array objectAtIndex: i];
      if (result == [GyveNil gyvenil]) 
	continue ;
      else if ([[(GyveBuffer *)result bufferName] isEqual: _name])
	return (GyveBuffer *)result;
      else 
	continue ;
    }
  return nil;
}
/*
 * Create and destory
 */
- initWithWidth: (int)w height: (int)h name: (NSString *)n
{
  [super init];
  width = w;
  height = h;
  file_path = nil;
  name = [n copy];
  selections_layer = [[GyveSelectionsLayer alloc] init];
  [self addNewLayerOnTop];
  [self changedSinceLastDrawing: YES];
  buffer_ID = 0;
  buffer_ID = [GyveBuffer issueBufferIDFor: self];
  return self ;
}
- initWithWidth: (int)w height: (int)h name: (NSString *)n layer: (GyveLayer *)layer
{
  [self initWithWidth: w height: h name: n];
  [self insertLayer: layer atIndex: 0];
  return self ;
}
- (void)dealloc
{
  [selections_layer release], selections_layer = nil;
  if (name)[name release], name = nil;
  buffer_ID = 0;
  [super dealloc];
}

/* 
 * Properties
 */
- (NSString *)bufferName
{
  return name;
}
- (int)bufferHeight
{
  return height;
}
- (int)bufferWidth
{
  return width;
}

/*
 * Operations on Figures
 */
- (int)beginOperation
{
  return 0;
}
- (void)endOperation: (int) op
{
  [self changedSinceLastDrawing: YES];
}
- (GyveLayer *) layerContainsFigObj: (id<PSFigObj>)figobj
{
  GyveLayer * contains_layer = nil;
  GyveLayer * layer;
  int i, max;
  max = [self countLayers];
  for (i = 0; i < max; i++)
    {
      if (nil != contains_layer) break;
      layer = [self layerAtIndex: i];
      if([layer containsFigObj: figobj])
	contains_layer = layer;
    }
  return contains_layer;
}
- (BOOL) containsFigObj: (id<PSFigObj>)figobj
{
  if (nil == [self layerContainsFigObj: figobj])
    return NO;
  else
    return YES;
}

- (PSFigObjSelectedProxy*)putFigObj: (id<PSFigObj>)figobj
{
  GyveLayer * layer = [self topLayerToWrite];
  if  (nil != layer)
    {
      [layer addFigObjOnTop:figobj];
      [self changedSinceLastDrawing: YES];
      return [self selectFigObj: figobj];
    }
  else
    return nil;
}
- (void)deleteFigObj: (id<PSFigObj>)figobj
{
  [self changedSinceLastDrawing: YES];
  if (YES == [(NSObject *)figobj isKindOfClass: [PSFigObjSelectedProxy class]])
    [(PSFigObjSelectedProxy* )figobj delete];
  else
    [[self selectFigObj: figobj] delete];
}
- (PSFigObjSelectedProxy*)selectFigObj: (id<PSFigObj>)figobj
{
  /*
    1. figobj  buffer˴ޤޤƤ뤫ɤǧ. 
    2. ǧβ, figobjselections layerβܤ٤Ĵ٤. 
    3. 
   */
  int index_to_insert 	     = 0;
  GyveLayer * layer_to_insert= nil;
  GyveLayer * layer;

  int i, max;
  if (YES == [figobj isSelected])
    {
      [NSException raise: GyveBufferWrongSelectionException
		   format: @"The selection target is marked as selection"];
      return nil;
    }

  max = [self countLayers];
  for (i = 0; i < max; i++)
    {
      if (nil != layer_to_insert) break;
      layer = [self layerAtIndex: i];
      if (YES == [layer containsObject: figobj])
	{
	  index_to_insert += [layer countSelectedFigObjsBefore: figobj];
	  layer_to_insert = layer;
	}
      else
	index_to_insert += [layer countSelectedFigObjs];
    }
  
  if (layer_to_insert == nil) return nil; // Should throw an exception
  
  return [selections_layer putFigObj: figobj
			  fromLayer: layer_to_insert
			   ofBuffer: self
			   atIndex: index_to_insert];
}
- (void)unSelectAll
{
  [selections_layer unSelectAll]; 
}

- (void)unSelect: (id<PSFigObj>)figobj
{
  PSFigObjSelectedProxy * proxy;
  if (YES == [(NSObject *)figobj isKindOfClass: [PSFigObjSelectedProxy class]])
    [(PSFigObjSelectedProxy* )figobj unSelect];
  else
    {
      proxy = [selections_layer lookUpProxyForFigObj: figobj];
      if (proxy) [proxy unSelect];
    }
}

- (const NSPoint *)firstPoint 
{GYVE_SHOULD_NOT_IMPLEMENT(return NULL);}
- (const NSPointValue *)firstPointValue
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (const NSPoint *)lastPoint
{GYVE_SHOULD_NOT_IMPLEMENT(return NULL);}
- (const NSPointValue *)lastPointValue
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (id<PointsEnumerating>)pointsEnumerator
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (id<PointsEnumerating>)reversePointsEnumerator
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (PSSegment *)firstSegment
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (PSSegment *)lastSegment
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (id<SegmentsEnumerating>)segmentsEnumerator
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (id<SegmentsEnumerating>)reverseSegmentsEnumerator
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (id<PSFigObj>)firstPrimitiveFigObj
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (id<PSFigObj>)lastPrimitiveFigObj
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (id<PSFigObj>)firstFigObj
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (id<PSFigObj>)lastFigObj
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (id<FigObjsPrimitiveEnumerating>)primitiveFigObjsEnumerator
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (id<FigObjsPrimitiveEnumerating>)reversePrimitiveFigObjsEnumerator
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (id<FigObjsEnumerating>)figObjsEnumerator
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (id<FigObjsEnumerating>)reverseFigObjsEnumerator
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (int)indexOfFigObj: (id<PSFigObj>)figobj
{GYVE_SHOULD_NOT_IMPLEMENT(return 0);}
- (id<LayerEnumerating>)layerEnumerator
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (id<LayerEnumerating>)reverseLayerEnumerator
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (int)countSegments {GYVE_SHOULD_NOT_IMPLEMENT(return 0);}
- (int)countPoints {GYVE_SHOULD_NOT_IMPLEMENT(return 0);}
- (int)countFigObjs
{
  GYVE_SHOULD_NOT_IMPLEMENT(return 0);
}
- (id<PSFigObj>)figObjAtIndex: (int)n
{
  GYVE_SHOULD_NOT_IMPLEMENT(return nil);
}


/*
 * Operations on Layers 
 */
- (int)countLayers
{
  return [self count];
}
- (BOOL)containsLayer: (GyveLayer *)layer
{
  return [self containsObject: layer];
}  
- (int)indexOfLayer: (GyveLayer *)layer
{
  return [self indexOfObject: layer];
}
- (GyveLayer *)layerAtIndex: (int)n
{
  return (GyveLayer *)[self objectAtIndex: n];
}
- (void)removeLayer: (GyveLayer *)layer
{
  [self removeObject: layer];
  [self changedSinceLastDrawing: YES];
}
- (void)removeLayerAtIndex: (int)i
{
  [self removeObjectAtIndex: i];
  [self changedSinceLastDrawing: YES];
}
- (void)insertLayer: (GyveLayer *)layer atIndex: (int)n
{
  [self insertObject: layer atIndex: n];
  [self changedSinceLastDrawing: YES];
}
- (void)addNewLayerOnTop
{
  GyveLayer * layer = [[[GyveLayer alloc] initWithName: new_layer_name] autorelease];
  [self appendObject: layer];
}
- (BOOL)duplicateLayerAtIndex:  (int)n
{
  GyveLayer * new_layer = [self layerAtIndex: n];
  if (nil == new_layer) return NO;
  [self insertLayer: [[new_layer copy] autorelease] atIndex: n];
  return YES;
}
- (BOOL)moveToTopLayerAtIndex: (int)n
{
  GyveLayer * layer = [[self layerAtIndex: n] shallowretain];
  if (nil == layer) return NO;
  [self removeLayerAtIndex: n];
  [self appendObject: layer];
  [self changedSinceLastDrawing: YES];
  return YES;
}
- (BOOL)moveUpLayerAtIndex: (int)n
{
  GyveLayer * layer;
  if ([self countLayers] < n) return NO;
  if (n < 0) return NO;

  layer = [[self layerAtIndex: n] shallowretain];
  if (nil == layer) return NO;

  [self removeLayerAtIndex: n];
  [self insertLayer: layer atIndex: n+1];
  [self changedSinceLastDrawing: YES];
  
  return YES;
}
- (BOOL)moveDownLayerAtIndex:  (int)n
{
  GyveLayer * layer;
  if (n < 1) return NO;
  if ([self countLayers] <= n) return NO;

  layer = [[self layerAtIndex: n] shallowretain];
  if (nil == layer) return NO;
  
  [self removeLayerAtIndex: n];
  [self insertLayer: layer atIndex: n-1];
  [self changedSinceLastDrawing: YES];
  return YES;
}
- (BOOL)moveToBottomLayerAtIndex: (int)n
{
  GyveLayer * layer = [[self layerAtIndex: n] shallowretain];
  if (nil == layer) return NO;
  [self removeLayerAtIndex: n];
  [self prependObject: layer];
  [self changedSinceLastDrawing: YES];
  return YES;
}
- (void)falltenLayers
{
  {GYVE_SHOULD_NOT_IMPLEMENT(return);}  
}
- (void)mergeVisibleLayers
{
  {GYVE_SHOULD_NOT_IMPLEMENT(return);} 
}
- (GyveLayer *)firstLayer
{
  return [self layerAtIndex: 0];
}
- (GyveLayer *)lastLayer
{
  return [self layerAtIndex: [self countLayers] - 1];
}

/*
 * Operations on Selections Layer 
 */
- (GyveLayer *)selectionsLayerSnapShot
{
  return [[self selectionsLayer] mutableCopy];
}
- (GyveSelectionsLayer *)selectionsLayer
{
  return selections_layer;
}
- (int)countSelectedFigObjs
{
  return [selections_layer countFigObjs];
}
- (PSFigObjSelectedProxy *)selectedFigObjAtIndex: (int)n
{
  return (PSFigObjSelectedProxy *)[selections_layer figObjAtIndex: n];
}
- (BOOL)beginModificationByCopyOfAll
{
  return [selections_layer beginModificationByCopyOfAll];
}
- (BOOL)endModificationOfAll
{
  return [selections_layer endModificationOfAll];
}
/*
 * Help for Drawing
 */
- (void)changedSinceLastDrawing: (BOOL)flag
{
  changed_since_last_drawing = flag;
}
- (BOOL)isChangedSinceLastDrawing
{
  int max;
  int i;
  if (YES ==  changed_since_last_drawing) return YES;

  max = [self countLayers];
  for (i = 0; i < max ; i++)
    {
      if (YES == [[self layerAtIndex: i] isChangedSinceLastDrawing])
	return YES;
    }
  return NO;  
}

/*
 * TMP
 */
- (BOOL)_expandFigObj: (PSFigObjGroup *)figobj_group
{
  int i, j, max;
  GyveLayer * layer;
  NSRange range;
  BOOL result = NO;
  max = [self countLayers];
  for (i = 0; i < max; i++)
    {
      layer = [self layerAtIndex: i];
      if (YES == [layer containsObject: figobj_group])
	{
	  range = [layer expandFigObj: figobj_group];
	  [self changedSinceLastDrawing: YES];
	  result = YES;
	  break;
	}
    }

  if (YES == result)
    {
      for (j = 0; j < range.length; j++)
	[self selectFigObj:
		[layer figObjAtIndex: range.location + j]];
    }

  return result;
}
- (BOOL)_expandText: (id<PSText>)text inRange: (NSRange *)range
{
  // TODO
  return NO;
}
- (void)_deleteFigObjInternal:  (id<PSFigObj>)figobj
{
  GyveLayer * layer;
  int i, max;
  max = [self countLayers];
  for (i = 0; i < max; i++)
    {
      layer = [self layerAtIndex: i];
      if (YES == [layer containsObject: figobj])
	{
	  [layer removeObject: figobj];
	  [self changedSinceLastDrawing: YES];
	  break;
	}
    }
}
- (void)_replaceFigObj:  (id<PSFigObj>)old 
	    withFigObj: (id<PSFigObj>)new
	       atLayer: (GyveLayer *)layer0
{
  GyveLayer * layer;
  int i, max;
  
  layer = layer0;
  if ((nil != layer) && YES == [layer containsObject: old])
    {
      [layer replaceObject: old withObject: new];
      [self changedSinceLastDrawing: YES];
      return ;
    }
  
  max = [self countLayers];
  for (i = 0; i < max; i++)
    {
      layer = [self layerAtIndex: i];
      if (YES == [layer containsObject: old])
	{
	  [layer replaceObject: old withObject: new];
	  [self changedSinceLastDrawing: YES];
	  break;
	}
    }
}
- (GyveLayer *)topLayerToWrite
{
  id <Enumerating> e = [self reverseObjectEnumerator];
  GyveLayer * writable_layer = nil;
  GyveLayer * insepected_layer = nil;
  while (insepected_layer = (GyveLayer *)[e nextObject], insepected_layer)
    {
      if ([insepected_layer isWritable])
	{
	  writable_layer = insepected_layer;
	  break;
	}
    }
  e = nil;
  insepected_layer = nil;
  return writable_layer;
}
@end

@implementation BuffersMenu
+ (GyveGUI *)sharedObject
{
  return nil;
}
- initWithHandler: (void *)_handler data: (void *)_data
{
  GtkWidget * menu_item;
  int i, max = [buffers_array count];
  id buffer;
  [super init];

  handler = _handler;
  data 	  = _data;

  widget = gtk_menu_new();
  for (i = 0; i < max; i++)
    {
      buffer = [buffers_array objectAtIndex: i];
      if (buffer == [GyveNil gyvenil]) continue ;
      menu_item = gtk_menu_item_new_with_label([[(GyveBuffer *)buffer bufferName]
						 cString]);
      if (handler) 
	gtk_signal_connect(GTK_OBJECT(menu_item), "select",
			   GTK_SIGNAL_FUNC(handler), data);
      gtk_menu_append (GTK_MENU (widget), menu_item);
      gtk_widget_show (menu_item);
    }
  [GyveBuffer addBufferObserver: self];
  return self ;
}
- (void)addedBuffer: (GyveBuffer *)buffer
{
  GtkWidget * menu_item = gtk_menu_item_new_with_label([[buffer bufferName] 
							 cString]);
  if (handler) 
    gtk_signal_connect(GTK_OBJECT(menu_item), "select",
		       GTK_SIGNAL_FUNC(handler), data);
  gtk_menu_append (GTK_MENU (widget), menu_item);
  gtk_widget_show (menu_item);
}
- (void)removedBuffer: (GyveBuffer *)buffer
{
  // TODO
}
- (void)changeBufferTo: (GyveBuffer *)buffer1 from: (GyveBuffer *)buffer2
{
  
}
@end
