/* GyveCreatingTool.m --- Creating tools

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "GyveCreatingTool.h"
#include "GyveBuffer.h"
#include "GyveCanvas.h"
#include "PSPath.h"
#include "PSSegment.h"
#include "GyveSelectionsLayer.h"
#include "draw.h"

#include <math.h>
#include <DPS/psops.h>

#include <gdk/gdkkeysyms.h>


@implementation GyveSimpleCreatingTool
- (void)buttonPressEvent: (GdkEventButton * )event
		onCanvas: (GyveCanvas *)canvas
		 atPoint: (NSPoint *)point 
	      withBuffer: (GyveBuffer *)buffer
{
  [super buttonPressEvent: event
	 onCanvas: canvas 
	 atPoint: point
	 withBuffer: buffer];
  [self setLayer: [buffer topLayerToWrite]];
  
  if (nil == [self layer])
    {
      return;
    }
  [canvas redrawLockRetain];
  [canvas redrawBBox: [buffer selectionsLayer] expandBy: 2.0];
  [buffer unSelectAll];
  [canvas redrawLockRelease];
}
- (void)buttonReleaseEvent: (GdkEventButton *)event 
		  onCanvas: (GyveCanvas *)canvas
		   atPoint: (NSPoint *)point 
		withBuffer: (GyveBuffer *)buffer
{
  NSRect rect;
  PSPath * path;
  BOOL local_after_dragging = after_dragging;
  rect = [self calcRect];
  [super buttonReleaseEvent: event
	 onCanvas: canvas 
	 atPoint: point
	 withBuffer: buffer];
  
  if (nil == [self layer]
       || (NO == local_after_dragging))
    {
      return;
    }

  [canvas redrawLockRetain];
  {
    [canvas redrawRect: &rect expandBy: 2.0];
    rect = [self calcRect];
    path = [self createFigObjInRect:&rect];
    [buffer putFigObj:  path];
    [canvas redrawFigObj: path];
  }
  [canvas redrawLockRelease];
  
}
- (void)dragNotifyEvent: (GdkEventMotion *)event 
	       onCanvas: (GyveCanvas *)canvas
		atPoint: (NSPoint *)point 
	     withBuffer: (GyveBuffer *)buffer
{
  NSRect rect;

  rect = [self calcRect];
  [super dragNotifyEvent: event
	 onCanvas: canvas 
	 atPoint: point
	 withBuffer: buffer];

  if (nil == [self layer])
    {
      return;
    }
  [canvas redrawRect: &rect expandBy: 2.0];
  rect = [self calcRect];
  [self drawTemporaryFigObjInRect: &rect];
}
- (id<PSFigObj>)createFigObjInRect: (NSRect *)rect
{
  [self subclassResponsibility:_cmd];
  return nil;
}
- (void)drawTemporaryFigObjInRect: (NSRect *)rect
{
  [self subclassResponsibility:_cmd];
}
- (NSRect)calcRect
{
  [self subclassResponsibility:_cmd];
  return NSZeroRect;
}
@end

@implementation  GyveRectangleTool
- (id<PSFigObj>)createFigObjInRect: (NSRect *)rect
{
  return [PSPath rectPathFromRect: rect];
}
- (void)drawTemporaryFigObjInRect: (NSRect *)rect
{
  [self drawRect: rect];
}
- (void)drawRect: (NSRect *)rect
{
  id<PSColor> color;
  if (nil == [self layer])
    {
      return ;
    }
  color = [self layerColor];
  PSsetrgbcolor ([color red],
		 [color green],
		 [color blue]);  
  
  PSsetlinewidth(0.0);
  PSrectstroke (rect->origin.x, rect->origin.y, 
		rect->size.width, rect->size.height);
}
- (NSRect)calcRect
{
  return [self pointsRectFromStart];
}
@end

@implementation GyveCenteredRectangleTool: GyveRectangleTool
- (NSRect)calcRect
{
  return [self pointsCenteredRectFromStart];
}
@end

@implementation  GyveOvalTool
- (id<PSFigObj>)createFigObjInRect: (NSRect *)rect
{
  return [PSPath ovalPathFromRect: rect];
}
- (void)drawTemporaryFigObjInRect: (NSRect *)rect
{
  [self drawOval: rect];
}
- (void)drawOval: (NSRect *)rect
{
  NSPoint center;
  id<PSColor> color;
  if (nil == [self layer])
    {
      return ;
    }
  color = [self layerColor];
  PSsetrgbcolor ([color red],
		 [color green],
		 [color blue]);  
  
  PSsetlinewidth(0.0);
  center = NSRect_center(rect);
  psw_ellipse_stroke(center.x, center.y,
		     (rect->size.width)/2.0, (rect->size.height)/2.0);
}
- (NSRect)calcRect
{
  return [self pointsRectFromStart];
}
@end

@implementation GyveCenteredOvalTool
- (NSRect)calcRect
{
  return [self pointsCenteredRectFromStart];
}
@end

@implementation GyvePenTool
- init
{
  [super init];
  path 	       = nil;
  path_proxy   = nil;
  last_segment = nil;
  new 	       = NO;
  return self ;
}
- (void)finishSessionOnCanvas: (GyveCanvas *)canvas
		  withBuffer: (GyveBuffer *)buffer
{
  [path_proxy release], path_proxy     = nil;
  new 				       = NO;
  [last_segment release], last_segment = nil;
  path 				       = nil;
}
- (void)endDragNotifyEvent: (GdkEventButton *)event
	       onCanvas: (GyveCanvas *)canvas
		atPoint: (NSPoint *)point 
	     withBuffer: (GyveBuffer *)buffer
{
  [super endDragNotifyEvent: event
	 onCanvas: canvas
	 atPoint: point
	 withBuffer: buffer];
  if (nil == last_segment)
    {
      fprintf(stderr, "error \n");
    }
  if (YES == new)
    {
      [path_proxy endModification];
      new = NO;
    }
  else
    {
      [path addCurvetoPoints: [last_segment pointAtIndex: 0]];
      [path_proxy endModification];
      path 	       = nil;
      [last_segment release], last_segment = nil;
    }
  [canvas redrawAll];
}

- (void)draggingNotifyEvent: (GdkEventMotion *)event 
		  onCanvas: (GyveCanvas *)canvas
		   atPoint: (NSPoint *)point 
	     withBuffer: (GyveBuffer *)buffer
{
  NSPoint tmp;
  [super draggingNotifyEvent: event
	 onCanvas: canvas
	 atPoint: point
	 withBuffer: buffer];

  [canvas redrawAll];
  if (path_proxy == nil)
    {
      fprintf(stderr, "error \n");
    }
  else if (last_segment && (dps_curveto == [last_segment dpsOperation]))
    {
      if (YES == new)
	{
	  [last_segment setPoint: point atIndex: 0];
	  psw_draw_handle(point->x, point->y);
	}
      else
	{
	  tmp.x = (2*base_point.x) - (point->x);
	  tmp.y = (2*base_point.y) - (point->y);
	  psw_draw_handle(base_point.x, base_point.y);
	  psw_draw_handle(tmp.x, tmp.y);
	  psw_draw_handle(point->x, point->y);
	  psw_draw_line(tmp.x, tmp.y,
			base_point.x, base_point.y);
	  psw_draw_line(point->x, point->y,
			base_point.x, base_point.y);
	  [last_segment setPoint: &tmp atIndex: 1];
	}
    }
}
- (void)clickNotifyEvent: (GdkEventButton *)event 
		onCanvas: (GyveCanvas *)canvas
		 atPoint: (NSPoint *)point 
	      withBuffer: (GyveBuffer *)buffer
{
  [super clickNotifyEvent: event
	 onCanvas: canvas
	 atPoint: point
	 withBuffer: buffer];
	 
  if (path_proxy == nil)
    {
      path 	 = [[[PSPath alloc] initWithMovetoPoint: &start_point]
		     autorelease];
      path_proxy = [[buffer putFigObj: path] retain];
      base_point = start_point;
    }
  else if (last_segment  && (dps_curveto == [last_segment dpsOperation]))
    {
      [last_segment setPoint: point atIndex: 1];
      [last_segment setPoint: point atIndex: 2];
      [path addCurvetoPoints: [last_segment pointAtIndex: 0]];
      [last_segment release], last_segment = nil;
    }
  else if (last_segment == nil)
    {
      path 	 = (PSPath *)[path_proxy beginModificationByCopy];
      [path addLinetoPoint: point];
      [path_proxy endModification];
      path = nil;
    }
  else
    {
      fprintf(stderr, "error \n");
    }
  [canvas redrawAll];
}

- (void)firstDragNotifyEvent: (GdkEventMotion *)event 
		    onCanvas: (GyveCanvas *)canvas
		     atPoint: (NSPoint *)point 
		  withBuffer: (GyveBuffer *)buffer
{
  NSPoint points[3];
  [super firstDragNotifyEvent: event
	 onCanvas: canvas
	 atPoint: point
	 withBuffer: buffer];
  if (path_proxy == nil)
    {
      path 	 = [[[PSPath alloc] initWithMovetoPoint: &start_point]
		     autorelease];
      path_proxy = [[buffer putFigObj: path] retain];
      new 	 = YES;
    }
  path 	 = (PSPath *)[path_proxy beginModificationByCopy];
  base_point = start_point;
  [canvas redrawAll];
  
  if (last_segment && (dps_curveto == [last_segment dpsOperation]))
    {
      [last_segment setPoint: &start_point atIndex: 2];
      points[1].x = 2*base_point.x - point->x;
      points[1].y = 2*base_point.y - point->y;
      psw_draw_line(points[1].x, points[1].y,
		    start_point.x, start_point.y);
      psw_draw_handle(points[1].x, points[1].y);
      psw_draw_handle(start_point.x, start_point.y);
      [last_segment setPoint: &points[1] atIndex: 1];
    }
  else if (last_segment 
	   &&((dps_lineto == [last_segment dpsOperation])
	      || (dps_moveto == [last_segment dpsOperation])))
    {
      fprintf(stderr, "error\n");
    }
  else if (new == YES)
    {
      points[0] = *point;
      points[1] = NSZeroPoint;
      points[2] = NSZeroPoint;
      psw_draw_handle(points[0].x, points[0].y);
      last_segment = [[PSSegment alloc] initWithDPSOperation: dps_curveto
					points: points];      
    }
  else if (new == NO)
    {
      PSSegment * prev_segment = [path segmentAtIndex: [path countSegments] - 1];
      if (dps_curveto == [prev_segment dpsOperation])
	{
	  NSPoint o, p;
	  o = *[prev_segment lastPoint];
	  p = *[prev_segment pointAtIndex: 1];
	  points[0].x = 2*(o.x) - p.x;
	  points[0].y = 2*(o.y) - p.y;
	}
      else
	{
	  points[0] = *[prev_segment lastPoint];
	}
      psw_draw_handle(points[0].x, points[0].y);
      points[1] = *point;
      points[2] = start_point;
      psw_draw_line(points[1].x, points[1].y,
		    points[2].x, points[2].y);
      last_segment = [[PSSegment alloc] initWithDPSOperation: dps_curveto
					points: points];      
    }
}
@end
