/* GyveDrawingEngin.h --- 

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "GyveDrawingEngine.h"
#include "PSColor.h"
#include "GyveBuffer.h"
#include "GyveLayer.h"
#include "PSFigObj.h"
#include "PSPath.h"
#include "PSSegment.h"
#include "GyveSelectionsLayer.h"
#include "PSImage.h"
#include "constant.h"
#include "draw.h"
#include "text.h"
#include "PSPaintStyle.h"
#include "PSMaskedFigObjGroup.h"
#include "PSCompoundPaths.h"
#include "PSText.h"
#include "PSTextStyle.h"
#include "PSTextElement.h"
#include "ToolBox.h"

#include <math.h>
#include <string.h>		// 
#include <DPS/psops.h>
#include <Foundation/NSObjCRuntime.h>

PSTextStyle * default_text_style = nil;

@interface DrawingEnginePaintStyle: PSPaintStyle
@end

@interface DrawingEngineCache: NSObject
{
  struct GtkDPSImageCache image_cache;
  struct coord_tr coord_tr;
  BOOL cache_dirtiness;
}
- initWithDPSarea: (GtkDPSArea * )area coordTr: (struct coord_tr*) _coord_tr;
- (BOOL)isCacheDirty;
- (void)shouldUpdate;
- (void)tryToUpdate;
- (void)forceToUpdate;
- (void)mapOnRect: (NSRect *)rect_dps;
- (void)coordTrChangesTo: (struct coord_tr*) _coord_tr;
@end

@implementation GyveDrawingEngine
- init
{
  [self shouldNotImplement: _cmd];
  return nil;
}
- initWithBuffer:(GyveBuffer *)b
{
  [super init];
  if (nil == b)
    {
      [self release];
      return nil;
    }
  buffer = [b retain];
  absolute_scale = 1.0;
  drawing_phase = DRAWING_NOTHING_NOW;
  drawing_mode 	= ARTWORK_MODE;
  paint_style 	= [[DrawingEnginePaintStyle alloc] init];
  text_style 	= nil;
  if (default_text_style == nil)
    default_text_style = [[PSTextStyle alloc]
			   initWithFontName: @"Times-Roman" size: 36.0];
  image_cache 	= nil;
  document_size = NSZeroSize;
  return self;
}
- (void)setDocumentSize: (NSSize *) size
{
  document_size = *size;
}
- (NSSize)documentSize
{
  return document_size;
}
- (void)setAbsoluteScale:(float) scale
{
  PSscale(scale/absolute_scale, scale/absolute_scale);
  absolute_scale = scale;
}
- (float)absoluteScale
{
  return absolute_scale;
}
- (void)redrawRect: (NSRect *)rect
{
  struct bbox rect_as_bbox;
  GyveLayer * layer;
  int i, max;
  NSRect document_rect;

  if (buffer == [GyveBuffer currentBuffer])
    {
      tool_draw 	= YES;
      tool_draw_request = [[ToolBox tool] requestForDrawingEngine];
    }
  else
    {
      tool_draw = NO;
    }

  NSRect_to_bbox(rect, &rect_as_bbox);
  if (image_cache)
    {
      if (YES == [buffer isChangedSinceLastDrawing])
	{
	  [image_cache shouldUpdate];
	}
      else if (NO == [image_cache isCacheDirty])
	{
	  [image_cache mapOnRect: rect];
	  goto drawing_selection;
	}
    }
       
  /* [0], [1], pswrap ؿǰĤˤޤȤ뤳. 
     ¿®. */ 

  // [0] Setup view clipping...
  /* ΰζ˾ */  
  PSrectviewclip(rect->origin.x, rect->origin.y,
		 rect->size.width, rect->size.height);
  // [1] Clear out -> gray
  /* Ȥɤ٤ο */
  PSsetrgbcolor(0.85, 0.85, 0.85); 
  
  /* Ȥɤ٤ */
  NSRect_from_NSSize(&document_rect, &document_size);
  if (NO == NSIntersectsRect(document_rect, *rect))
    {
      /* ɥΰȺΰδ֤˽Ťʤ꤬ʤ. 
	 => ɥΰγ¦˺ΰ褬 
	 => ΰ򳥿ɤ٤ɤ */
      PSrectfill(rect->origin.x, rect->origin.y,
		 rect->size.width, rect->size.height);
    }
  else if (YES == NSContainsRect(document_rect, *rect))
    {
      /* ɥΰȺΰδ֤˽ŤʤäƤ뤦, 
	 ΰ褬˥ɥΰ˴ޤޤƤ. 
	 => ʤˤ⤷ʤɤ. */
    }
  else if (YES == NSContainsRect(*rect, document_rect))
    {
      /* ɥΰȺΰδ֤˽ŤʤäƤ뤦, 
	 ɥΰ褬ΰ˴˴ޤޤƤ. 
	 => ŤʤäƤʤʬ, ʤ, ΰ¦
	 ɥΰγ¦ɤ٤ */
      psw_rim_fill(rect->origin.x, rect->origin.y, // outer
		   rect->size.width, rect->size.height,
		   0.0, 0.0,	//  inner
		   document_size.width, document_size.height);
    }
  else
    {
      /* ɥΰȺΰδ֤˽ŤʤäƤ뤬, 
	 ޴طϤʤ.
	 => ɥΰ賰Ǥ, ΰ¦
	 ɤ٤ */
      NSRect  inner;
      inner = NSIntersectionRect(*rect, document_rect);
      psw_rim_fill(rect->origin.x, rect->origin.y,
		   rect->size.width, rect->size.height,
		   inner.origin.x, inner.origin.y,
		   inner.size.width, inner.size.height);
      
    }
  
  PSgsave();			// DocumentFrame
  {
    drawing_phase = DRAWING_GRIDS_LAYER;
    if (YES != NSSize_is_zero_size(&document_size))
      {
	PSrectclip(0.0, 0.0,
		   document_size.width, document_size.height);
      }
    // [1] Clear out -> white
    PSsetrgbcolor(1.0, 1.0, 1.0);	
    PSrectfill(rect->origin.x, rect->origin.y,
	       rect->size.width, rect->size.height);    

    // [2] Bottom grid layer...
    [self drawDocumentFrame];

    // [3] Document layers
    [self setUpDocumentLayersDrawing];
    max = [buffer countLayers];
    for (i = 0; i < max; i++)
      {
	layer = [buffer layerAtIndex: i];
	/* Buffer->Layer->FigObj
	   ȺƵŪ˲ߤʤ褷ƤȤˤ, 
	   Υ٥Ƚꤹ뤳Ȥ§Ȥ. */
	if ([layer isVisible])
	  {
	    [self redrawBBox: &rect_as_bbox ofLayer: layer];
	  }
      }
    [buffer changedSinceLastDrawing: NO];
  }
  PSgrestore();			// DocumentFrame

  // [4] Create cache...if you need
  if (image_cache) 
    [image_cache tryToUpdate];

  drawing_selection:
  // [5] Selection layer...
  PSgsave();
  [self setUpSelectionsLayerDrawing];
  layer = [buffer selectionsLayer];
  [self redrawBBox: &rect_as_bbox ofLayer: layer];
  layer = nil;
  PSgrestore();

  // [6] 
  if (YES == tool_draw)
    {
      [[ToolBox tool] drawAfterAll];
    }
  // 
  drawing_phase = DRAWING_NOTHING_NOW;
  text_style  = nil;
  PSinitviewclip();

  return;
}
- (BOOL)calcBBoxOfFigObj: (NSObject<PSFigObj>*)figobj
{
  if (YES == [figobj isFigObjProxy])
    figobj = [figobj targetForFigObjProxy];

  if ([figobj conformsToProtocol: @protocol(PSText)])
    return [self calcBBoxOfText: (id<PSText>)figobj];
  else
    return NO;
}
- (void)dealloc
{
  [buffer release], buffer 	     = nil;
  [paint_style release], paint_style = nil;
  if (image_cache) [image_cache release], image_cache = nil;
  [super dealloc] ;
}
- (enum drawing_mode)drawingMode
{
  return drawing_mode;
}
- (void)setDrawingMode: (enum drawing_mode)mode
{
  drawing_mode = mode;
  if (image_cache) [image_cache shouldUpdate];
}
- (void)imageCacheRequestWithArea: (GtkDPSArea * )area coordTr: (struct coord_tr*) _coord_tr
{
  if (image_cache) [image_cache release], image_cache = nil;
  if (area != NULL)
    image_cache = [[DrawingEngineCache alloc] initWithDPSarea: area
						coordTr: _coord_tr];
}
- (void)coordTrChangesTo: (struct coord_tr *)coord_tr
{
  if (image_cache == nil) return;
  if (coord_tr == NULL) [image_cache shouldUpdate];
  else [image_cache coordTrChangesTo: coord_tr];
}
@end
@implementation GyveDrawingEngine(Protected)
- (void)redrawBBox: (struct bbox*)bbox ofLayer: (GyveLayer *)layer
{
  int i, max;
  id<PSFigObj> figobj;
  max = [layer countFigObjs];
  for (i = 0; i < max; i++)
    {
      figobj = [layer figObjAtIndex: i];
      
      /* figobj褹٤ɤȽ */
      if (bbox_is_zero_bbox([figobj bboxCStructure])
	  || // Text¸Τɲ
	  bbox_intersects([figobj bboxCStructure], bbox))
	{
	  if ((drawing_phase == DRAWING_SELECTIONS_LAYER) &&
	      (YES == [figobj isFigObjProxy]))
	    {
	      GyveLayer * base_layer;
	      id<PSColor> color;
	      current_figobj_proxy = (PSFigObjSelectedProxy *)figobj;
	      base_layer = [current_figobj_proxy layer];
	      color 	 = [base_layer layerColor];
	      PSsetrgbcolor([color red],
			    [color green],
			    [color blue]);
	      current_figobj_root = [current_figobj_proxy targetForFigObjProxy];
	    }
	  else
	    {
	      current_figobj_root = figobj;
	    }
	  [self drawFigObj: current_figobj_root];
	  current_figobj_proxy = nil;
	  current_figobj_root = nil;
	}
    }
  [layer changedSinceLastDrawing: NO];
  return ;
}
- (void)drawDocumentFrame
{

}
- (void)drawFigObj: (NSObject<PSFigObj>*)figobj
{
  if ([figobj isMemberOfClass: [PSPath class]])
    [self drawPath: (PSPath *)figobj];
  else if ([figobj conformsToProtocol: @protocol(PSText)])
    [self drawText: (id<PSText>)figobj];
  else if ([figobj isMemberOfClass: [PSImage class]])
    [self drawImage: (PSImage *)figobj];
  else if ([figobj isMemberOfClass: [PSMaskedFigObjGroup class]])
    [self drawMaskedFigObjGroup: (PSMaskedFigObjGroup *)figobj];
  else if ([figobj isMemberOfClass: [PSFigObjGroup class]])
    [self drawFigObjGroup: (PSFigObjGroup *)figobj];
  else if ([figobj isMemberOfClass: [PSCompoundPaths class]])
    [self drawCompoundPaths: (PSCompoundPaths *)figobj];
  else
    ;				// Wrong type
  // TEST
  if ([figobj isZeroBBox])
    {
      [figobj calcBBox];
    }
}
- (void)drawPath: (PSPath *)path
{
  /*
   * This is a sample code.
   */
  drawing_engine_send_path (path, YES);
  if (drawing_phase == DRAWING_DOCUMENT_LAYERS)
    {
      if (drawing_mode != ARTWORK_MODE)
	{
	  [self setPaintStyle: [path paintStyle]];
	  PSgsave();
	  {
	    // PSsetrgbcolor(0.5, 0.5, 0.5);
	    if ([[path paintStyle] filling])
	      PSsetrgbcolor([(id<PSColor>)[[path paintStyle] filling] red],
			    [(id<PSColor>)[[path paintStyle] filling] green],
			    [(id<PSColor>)[[path paintStyle] filling] blue]);
	    PSfill();
	  }
	  PSgrestore();
	  if ([[path paintStyle] stroking])
	    PSsetrgbcolor([(id<PSColor>)[[path paintStyle] stroking] red],
			  [(id<PSColor>)[[path paintStyle] stroking] green],
			  [(id<PSColor>)[[path paintStyle] stroking] blue]);
	  PSstroke();
	}
      else
	{
	  PSstroke();
	}
    }
  else
    {
      PSstroke();
      if (tool_draw == YES)
	{
	  [self drawToolRequest: tool_draw_request
		forPath: (PSPath *)path];
	}
    }
}
- (void)drawToolRequest: (int)request forPath: (PSPath *)path
{
  int i, max;
  NSPoint p;
  struct bbox * bbox;
  PSSegment * seg;
  if (request & tool_drawing_by_self_before)
    {
      
    }
  if (request & tool_drawing_anchor_point)
    {
      max = [path countSegments];
      for (i = 0; i < max; i++)
	{
	  seg = [path segmentAtIndex: i];
	  switch ([seg dpsOperation])
	    {
	    case dps_moveto:
	    case dps_lineto:
	      p = *[seg pointAtIndex: 0];
	      psw_draw_handle(p.x, p.y);
	      break;
	    case dps_curveto:
	      p = *[seg pointAtIndex: 2];
	      psw_draw_handle(p.x, p.y);
	      break;
	    }
	}
    }
  if (request & tool_drawing_direction_point)
    {
      max = [path countSegments];
      for (i = 0; i < max; i++)
	{
	  seg = [path segmentAtIndex: i];
	  switch ([seg dpsOperation])
	    {
	    case dps_curveto:
	      p = *[seg pointAtIndex: 0];
	      psw_draw_handle(p.x, p.y);
	      p = *[seg pointAtIndex: 1];
	      psw_draw_handle(p.x, p.y);
	      break;
	    }
	}
    }
  if (request & tool_drawing_direction_line)
    {
      NSPoint last_point;
      max = [path countSegments];
      for (i = 0; i < max; i++)
	{
	  seg = [path segmentAtIndex: i];
	  switch ([seg dpsOperation])
	    {
	    case dps_moveto:
	      last_point = *[seg pointAtIndex: 0];
	      break;
	    case dps_lineto:
	      last_point = *[seg pointAtIndex: 0];
	      break;
	    case dps_curveto:
	      p = *[seg pointAtIndex: 0];
	      psw_draw_line(last_point.x, last_point.y,
			    p.x, p.y);
	      p = *[seg pointAtIndex: 1];
	      last_point = *[seg pointAtIndex: 2];
	      psw_draw_line(last_point.x, last_point.y,
			    p.x, p.y);
	      break;
	    }
	}
    }
  if (request & tool_drawing_bbox_center)
    {
      bbox = [path bboxCStructure];
      p = bbox_center(bbox);
      psw_draw_handle(p.x, p.y);
    }
  if (request & tool_drawing_bbox_rect)
    {
      bbox = [path bboxCStructure];
      psw_draw_bbox_rect(bbox_element(bbox, BBOX_LLX),
			 bbox_element(bbox, BBOX_LLY),
			 bbox_element(bbox, BBOX_URX),
			 bbox_element(bbox, BBOX_URY));
    }
  if (request & tool_drawing_bbox_point)
    {
      bbox = [path bboxCStructure];
      psw_draw_bbox_handles(bbox_element(bbox, BBOX_LLX),
			    bbox_element(bbox, BBOX_LLY),
			    bbox_element(bbox, BBOX_URX),
			    bbox_element(bbox, BBOX_URY));
    }
}

- (void)drawText: (NSObject<PSText>*)text
{
  if ([text isKindOfClass: [PSTextAtPoint class]])
    [self drawTextAtPoint: (PSTextAtPoint *)text];
  else
    {
    }
}
- (BOOL)calcBBoxOfText: (NSObject<PSText>*)text
{
  if ([text isKindOfClass: [PSTextAtPoint class]])
    {
      return [self calcBBoxOfTextAtPoint: (PSTextAtPoint *)text];
    }
  else
    {
      return NO;
    }
}
- (void)drawTextAtPoint: (PSTextAtPoint *)text
{
  int i, max;
  const NSPoint point     = {0.0, 0.0};
  NSPoint new_point 	    = point;
  NSPoint beginning_of_line = point;
  char cstr[2];
  PSTextElement * elt;
  BOOL new_line  	  	  = NO;
  BOOL has_text_selections 	  = NO;
  NSRange * text_selections_range = NULL;
  cstr[1] =  '\0';
  max 	  = [text countTextElements];
  
  if ((YES == [text isZeroBBox]) 
      || YES == [text hasZeroBBoxTextElement])
    {
      [self calcBBoxOfTextAtPoint: text];
    }

  if (drawing_phase != DRAWING_DOCUMENT_LAYERS)
    {
      struct bbox * text_bbox;
      text_bbox = [text bboxCStructure];
      PSrectstroke(bbox_element(text_bbox, BBOX_LLX),
		   bbox_element(text_bbox, BBOX_LLY),
		   bbox_element(text_bbox, BBOX_URX)- bbox_element(text_bbox, BBOX_LLX),
		   bbox_element(text_bbox, BBOX_URY)- bbox_element(text_bbox, BBOX_LLY));
      if ((YES == [current_figobj_proxy isKindOfClass: 
					  [PSTextSelectedProxy class]]) 
	  && (YES == [(PSTextSelectedProxy *)current_figobj_proxy hasSelectionRange])
	  && (YES == bbox_equal([text bboxCStructure],	// Not Draggging now
				[[current_figobj_proxy originalTargetForFigObjProxy]
				  bboxCStructure])))
	{
	  has_text_selections 	= YES;
	  text_selections_range = [(PSTextSelectedProxy *)current_figobj_proxy
							  selectionRange];
	}
      else
	{
	  return;
	}
    }
  
  PSgsave();
  PSconcat([text mappingMatrix]);
  for (i = 0; i < max ; i++)
    {
      elt      = [text textElementAtIndex: i];
      new_line = [elt isNewLine];
      if ((drawing_phase == DRAWING_DOCUMENT_LAYERS) &&
	  (drawing_mode != ARTWORK_MODE))
	{
	  id<PSColor>c;
	  [self setPaintStyle: [elt paintStyle]];
	  c = (id<PSColor>)[[elt paintStyle] stroking];
	  if (nil != c)
	    {
	      PSsetrgbcolor([c red], [c green], [c blue]);
	    }
	}

      [self setTextStyle: [elt textStyle]];
      cstr[0] = (char)[elt character];
      // New pointη׻(Ȱ)
      if (YES == new_line)
	{
	  /*
	  float font_bbox[4];
	  float font_matrix[6];
	  NSPoint delta;
	  psw_get_font_bbox(font_bbox);
	  delta.y = font_bbox[BBOX_URY] - font_bbox[BBOX_LLY];
	  delta.x = 0;
	  psw_get_font_matrix(font_matrix);
	  matrix_apply_on_point(font_matrix, &delta); */
	  NSPoint delta;
	  delta.x = 0;
	  delta.y = [text_style fontSize];
	  beginning_of_line.y -= delta.y;
	  new_point = beginning_of_line;
	  [elt zeroBBox];
	}
      else if (drawing_phase == DRAWING_DOCUMENT_LAYERS)
	{
	  psw_draw_text_element(new_point.x, new_point.y, cstr,
				&(new_point.x), &(new_point.y));
	}
      else if (has_text_selections
	       && (YES == NSLocationInRange(i, *text_selections_range)))
	{
	  int range_max;
	  psw_draw_text_element_with_highlight(new_point.x, new_point.y, 
					       [text_style fontSize],
					       cstr,
					       &(new_point.x), 
					       &(new_point.y));
	  // Checking range over
	  range_max = text_selections_range->location + text_selections_range->length;
	  if (range_max < i)
	    {
	      break;
	    }
	}
      else
	{
	  psw_calc_text_element_width(new_point.x, new_point.y, cstr,
				      &(new_point.x), &(new_point.y));
	}
    }
  PSgrestore();
}
- (BOOL)calcBBoxOfTextAtPoint: (PSTextAtPoint *)text
{
  int i, max;
  const NSPoint point     = {0.0, 0.0};
  NSPoint new_point 	    = point;
  NSPoint beginning_of_line = point;
  char cstr[2];
  PSTextElement * elt;
  float * mapping_matrix  = [text mappingMatrix];

  cstr[1] =  '\0';
  max 	  = [text countTextElements];

  if (NO == matrix_invert(mapping_matrix, NULL))
    {
      fprintf(stderr, "No inverse matrix for The text\n");
    }

  if (0 == max)
    {
      [text calcBBox];
      return NO;
    }

  PSgsave();
  PSconcat(mapping_matrix);
  for (i = 0; i < max ; i++)
    {
      elt = [text textElementAtIndex: i];
      [self setTextStyle: [elt textStyle]];

      if (YES == [elt isNewLine])
	{
	  /* 1. ߥեȤBBoxեȥǥʥ꤫
	        . matrix Ѵ */
	  /* 2. 뤿ξmoveto */
	  /* 3. new_point⹹ */
	  /*
	  float font_bbox[4];
	  float font_matrix[6];
	  NSPoint delta;
	  psw_get_font_bbox(font_bbox);
	  delta.y = font_bbox[BBOX_URY] - font_bbox[BBOX_LLY];
	  delta.x = 0;
	  psw_get_font_matrix(font_matrix);
	  matrix_apply_on_point(font_matrix, &delta); */
	  NSPoint delta;
	  delta.x = 0;
	  delta.y = [text_style fontSize];
	  beginning_of_line.y -= delta.y;
	  new_point = beginning_of_line;
	  [elt zeroBBox];
	}
      else
	{
	  cstr[0] = (char)[elt character];
	  {
	    int j;
	    float bbox[4];
	    struct bbox * elt_bbox = [elt bboxCStructure];
	    psw_get_text_element_bbox(new_point.x, new_point.y, cstr,
				      bbox);
	    for (j = 0; j < 4; j++)
	      bbox_set(elt_bbox, j, bbox[j]);

	    /* bboxκɸϤ, textmappint matrix
	       ΤǤ. 
	       κɸϤˤ, bbox׻ɬפ??? */
	    matrix_apply_on_bbox(mapping_matrix, elt_bbox); 
	    psw_calc_text_element_width(new_point.x, new_point.y, cstr,
					&(new_point.x), &(new_point.y));
	  }
	}
    }
  PSgrestore();
  [text calcBBox];
  if (YES == [text isZeroBBox])
    {
      return NO;
    }
  else
    {  
      return YES;
    }
}
- (void)drawImage: (PSImage *)image
{
  float * mapping_matrix   = [image mappingMatrix];
	
  if ((drawing_phase == DRAWING_DOCUMENT_LAYERS) &&
      (drawing_mode != ARTWORK_MODE))
    {
      int w 				 = [image imageWidth];
      int h 				 = [image imageHeight];
      const int depth 			 = 8;
      const char * gyveimgbuf 		 = "gyveimgbuf";
      const float base_mapping_matrix[6] = {w, 0.0, 0.0, -h, 0.0, h};


      unsigned char * line_data 	 = NULL;
      int number_of_sent_line;

      int number_of_sent_pixels_in_line;
      char * line_string = malloc(2 * 3 * w + 1);
      
      line_string[2 * 3 * w] = '\0';
      
      PSgsave();
      {
	PSconcat(mapping_matrix);
	psw_alloc_string_buffer(gyveimgbuf, w);
    
	PSsendint(w);
	PSsendint(h);
	PSsendint(depth);
	PSsendfloatarray(base_mapping_matrix, 6);
	psw_put_img_source_proc (gyveimgbuf);
	PSfalse();
	PSsendint(3);
	PScolorimage();
	/* Send data */
	for (number_of_sent_line = 0; 
	     number_of_sent_line < h;
	     number_of_sent_line++)
	  {
	    line_data = [image mutablePixelsOfLine: number_of_sent_line];
	    for (number_of_sent_pixels_in_line = 0;
		 number_of_sent_pixels_in_line < w;
		 number_of_sent_pixels_in_line++)
	      {
		sprintf(&(line_string[6 * number_of_sent_pixels_in_line]), 
			"%02X%02X%02X", 
			line_data[3*number_of_sent_pixels_in_line+RED_INDEX], 
			line_data[3*number_of_sent_pixels_in_line+GREEN_INDEX],
			line_data[3*number_of_sent_pixels_in_line+BLUE_INDEX]);
		
	      }
	    PSsendstring(line_string);
	  }
      psw_free_string_buffer(gyveimgbuf);
      }
      PSgrestore();
      free(line_string);
      
    }
  else
    {
      /* 
        3
	 +---+ 2
	 |   |
       0 +---+
             1	 
       */
      NSPoint point0 = {0.0, 0.0};
      NSPoint point1 = {1.0, 0.0};
      NSPoint point2 = {1.0, 1.0};
      NSPoint point3 = {0.0, 1.0};
      matrix_apply_on_point(mapping_matrix, &point0);
      matrix_apply_on_point(mapping_matrix, &point1);
      matrix_apply_on_point(mapping_matrix, &point2);
      matrix_apply_on_point(mapping_matrix, &point3);
      PSmoveto(point0.x, point0.y);
      PSlineto(point1.x, point1.y);
      PSlineto(point2.x, point2.y);
      PSlineto(point3.x, point3.y);
      PSlineto(point0.x, point0.y);
      PSclosepath();
      PSstroke();
      PSmoveto(point0.x, point0.y);
      PSlineto(point2.x, point2.y);
      PSstroke();
      PSmoveto(point1.x, point1.y);
      PSlineto(point3.x, point3.y);
      PSstroke();
    }
}
- (void)drawFigObjGroup: (PSFigObjGroup *)group
{
  NSObject <PSFigObj> * figobj;
  int i, max;
  max = [group countFigObjs];
  for (i = 0; i < max; i++)
    {
      figobj = [group figObjAtIndex: i];
      [self drawFigObj: figobj];
    }
}
- (void)drawMaskedFigObjGroup: (PSMaskedFigObjGroup *)masked_group
{
  PSgsave();
  {
    if (drawing_phase == DRAWING_DOCUMENT_LAYERS)
      {
	drawing_engine_send_path([masked_group maskPath], YES);
	PSclip();
	PSnewpath() ;
      }
    else if (drawing_phase == DRAWING_SELECTIONS_LAYER)
      {
	[self drawPath: [masked_group maskPath]];
      }
    [self drawFigObjGroup: masked_group];
  }
  PSgrestore();
}
- (void)drawCompoundPaths: (PSCompoundPaths *)compound_paths
{
  if (drawing_phase == DRAWING_DOCUMENT_LAYERS)
    {
      drawing_engine_send_compound_paths(compound_paths);
      if (drawing_mode != ARTWORK_MODE)
	{
	  [self setPaintStyle: [compound_paths paintStyle]];
	  PSgsave();
	  {
	    // PSsetrgbcolor(0.5, 0.5, 0.5);
	    if ([[compound_paths paintStyle] filling])
	      PSsetrgbcolor([(id<PSColor>)[[compound_paths paintStyle] filling] red],
			    [(id<PSColor>)[[compound_paths paintStyle] filling] green],
			    [(id<PSColor>)[[compound_paths paintStyle] filling] blue]);
	    PSeofill();
	  }
	  PSgrestore();
	  if ([[compound_paths paintStyle] stroking])
	    PSsetrgbcolor([(id<PSColor>)[[compound_paths paintStyle] stroking] red],
			  [(id<PSColor>)[[compound_paths paintStyle] stroking] green],
			  [(id<PSColor>)[[compound_paths paintStyle] stroking] blue]);
	  PSstroke();
	}
      else
	{
	  PSstroke();
	}
    }
  else
    {
      // 
      [self drawFigObjGroup: (PSFigObjGroup *)compound_paths];
    }
}
- (void)setTextStyle: (PSTextStyle *)new_style
{
  BOOL text_style_is_changed;
  if (nil != new_style)
    {
      text_style = new_style;
      text_style_is_changed = YES;
    }
  else if (nil == text_style)
    {
      text_style 	    = default_text_style;
      text_style_is_changed = YES;
    }
  else
    {
      text_style_is_changed = NO;
    }
  
  if (YES == text_style_is_changed)
    {
      psw_set_text_style([[text_style fontName] cString],
			 [text_style fontSize]);
    }
}
- (void)setPaintStyle: (PSPaintStyle *)new_style
{
  if ((paint_style == new_style) || (nil == new_style)) return;
  [paint_style installPaintStyle: new_style];
}
- (void)setUpSelectionsLayerDrawing
{
  // still under work.
  drawing_phase = DRAWING_SELECTIONS_LAYER;
  // still under work.
  PSsetlinewidth(0.0);
}
- (void)setUpDocumentLayersDrawing
{
  PSsetrgbcolor(0.0, 0.0, 0.0);
  [paint_style installPaintStyle: [[[PSPaintStyle alloc] init] autorelease]];
  drawing_phase = DRAWING_DOCUMENT_LAYERS;
}
@end

void 
drawing_engine_send_path (PSPath *path, BOOL newpath)
{
  const NSPoint * point[3];
  PSSegment * seg;
  int point_index;
  
  int max, i;
  max = [path countSegments];
  for (i = 0; i < max; i++)
    {
      seg = [path segmentAtIndex: i];
      switch ([seg dpsOperation])
	{
	case dps_moveto:
	  point[0] = [seg pointAtIndex: 0];
	  if (YES == newpath) PSnewpath();
	  PSmoveto(point[0]->x, point[0]->y);
	  break;
	case dps_lineto:
	  point[0] = [seg pointAtIndex: 0];
	  PSlineto(point[0]->x, point[0]->y);
	  break;
	case dps_curveto:
	  for (point_index = 0; point_index < 3; point_index++)
	    point[point_index] = [seg pointAtIndex: point_index];
	  PScurveto(point[0]->x, point[0]->y,
		    point[1]->x, point[1]->y,
		    point[2]->x, point[2]->y);
	  break;
	}
    }
  
  if ([path isClosedPath])
    PSclosepath();
}

void
drawing_engine_send_compound_paths (PSCompoundPaths *compound_paths)
{
  NSObject <PSFigObj> *figobj;
  int i, max;
  max = [compound_paths countFigObjs];

  PSnewpath();

  for (i = 0; i < max; i++)
    {
      figobj = [compound_paths figObjAtIndex: i];
      if ([figobj isKindOfClass: [PSFigObjGroup class]])
	drawing_engine_send_compound_paths((PSCompoundPaths *)figobj);
      else if ([figobj isKindOfClass: [PSPath class]])
	drawing_engine_send_path((PSPath *)figobj, NO);
      else
	;
    }
  if ([compound_paths isClosedPath])
    PSclosepath();
}

@implementation DrawingEnginePaintStyle
- (void)setLineJoin: (enum ps_line_join)lj
{
  if (line_join != lj)
    {
      line_join = lj;
      PSsetlinejoin(line_join);
    }
}
- (void)setLineCap: (enum ps_line_cap)lc
{
  if (line_cap != lc)
    {
      line_cap = lc;
      PSsetlinecap(line_cap);
    }
}
- (void)setMiterLimit: (float)ml
{
  if (miter_limit != ml)
    {
      miter_limit = ml;
      PSsetmiterlimit(miter_limit);
    }
}
- (void)setLineWidth:(float)width
{
  if (line_width != width)
    {
      line_width = width;
      PSsetlinewidth(line_width);
    }
}
@end


@implementation DrawingEngineCache: NSObject
- initWithDPSarea: (GtkDPSArea * )area coordTr: (struct coord_tr*) _coord_tr
{
  [super init];
  image_cache.target_area = area;
  image_cache.image 	   = NULL;
  [self coordTrChangesTo: _coord_tr];
  [self shouldUpdate];
  return self; 
}
- (BOOL)isCacheDirty
{
  return cache_dirtiness;
}
- (void)shouldUpdate
{
  cache_dirtiness = YES;
}
- (void)tryToUpdate
{
  if (cache_dirtiness == YES) [self forceToUpdate];
}

- (void)forceToUpdate
{
  GtkDPSArea * area = image_cache.target_area;
  if (image_cache.image != NULL)
    gtk_dps_area_destroy_image_cache(&image_cache);
  gtk_dps_area_create_image_cache(area, &image_cache, 
				  0, 0,
				  GTK_WIDGET(area)->allocation.width,
				  GTK_WIDGET(area)->allocation.height);
  cache_dirtiness = NO;
}
- (void)mapOnRect: (NSRect *)rect_dps
{
  NSRect rect_x;
  if (image_cache.image != NULL)
    {
      coord_tr_dpsRect2xRect(&coord_tr, rect_dps, &rect_x);
      gtk_dps_area_map_image_cache (&image_cache,
				    (int)(rect_x.origin.x), (int)(rect_x.origin.y),
				    (int)(rect_x.size.width), (int)(rect_x.size.height));
    }
  else
    fprintf(stderr, "No image to map\n");
				
}
- (void)coordTrChangesTo: (struct coord_tr*) _coord_tr
{
  coord_tr_copy (_coord_tr, &coord_tr);
  [self shouldUpdate];
}
- (void)dealloc
{
  if (image_cache.image != NULL)
    gtk_dps_area_destroy_image_cache(&image_cache);
  [super dealloc];
}

@end
