/* GyveEditingTool.m --- Editing tool protocol 

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "GyveEditingTool.h"
#include "GyveCanvas.h"
#include "GyveBuffer.h"
#include "GyveSelectionsLayer.h"
#include "PSPath.h"
#include "public.h"
#include "draw.h"
#include "geometry.h"
#include "constant.h"
#include "PSMatrix.h"
#include "GyveHitDetector.h"
#include "PaintStylePallet.h"
#include "GyveScrolledCanvas.h"
#include "TextStylePallet.h"
#include "geometry.h"
#include "PSTextElement.h"
#include "PSText.h"
#include "draw.h"
#include "utilities.h"
#include "Gyve.h"
#include "PSSegment.h"

#include <gdk/gdkkeysyms.h>
#include <DPS/psops.h>
#include <Foundation/NSString.h>

static float scatter_radian(float radian, float scatter_unit);

static NSString * register_key = @"tmp_dup";

/* GyveEditingTool */
@implementation GyveEditingTool
- init
{
  [super init];
  init_bbox = zero_bbox;
  return self ;
}
- (void)duplicateSelectedFigObjsInBuffer: (GyveBuffer *)buffer
{
  // 1. 򤵤Ƥ޷򥳥ԡ, 
  // 2. ٤Ƥο޷
  // 3. ڡȤ. 
  // TODO: 򤵤Ƥ޷¸ߤƤ쥤˥ڡ
  // 褦ˤ. 
  gyve_copy_for_key(register_key);
  [buffer unSelectAll];
  gyve_paste_for_key(register_key);
}
@end

/* GyveReshapeTool */
@implementation  GyveReshapeTool
- init
{
  [super init];
  hit_detector = [[GyveHitDetector alloc] initWithTarget: nil];
  target_point = NULL;
  target_proxy = nil;
  neighbor_points[0] = NULL;
  neighbor_points[1] = NULL;
  other_side_target_point = NULL;
  return self;
}
- (int)requestForDrawingEngine
{
  return tool_drawing_anchor_point;
}
- (void)dealloc
{
  [hit_detector release], hit_detector = nil;
  [super dealloc];
}
- (void)endDragNotifyEvent: (GdkEventButton * )event
		  onCanvas: (GyveCanvas *)canvas
		   atPoint: (NSPoint *)point 
		withBuffer: (GyveBuffer *)buffer
{
  if (nil == target_proxy)
    {
      return ;
    }
  [canvas redrawLockRetain];
  [canvas redrawBBoxCStructure: &init_bbox expandBy: 2.0];
  [canvas redrawFigObj: [target_proxy targetForFigObjProxy]];
  [target_proxy endModification];
  [canvas redrawLockRelease];
  [target_proxy release], target_proxy = nil;
  target_point 			       = NULL;
  other_side_target_point = NULL;
  neighbor_points[0] = NULL;
  neighbor_points[1] = NULL;
}
- (void)draggingNotifyEvent: (GdkEventMotion *)event 
		  onCanvas: (GyveCanvas *)canvas
		   atPoint: (NSPoint *)point 
	     withBuffer: (GyveBuffer *)buffer
{
  NSPoint diff;
  int count;
  PSPath * path;
  [super draggingNotifyEvent: event
	 onCanvas: canvas
	 atPoint: point
	 withBuffer: buffer];

  if (nil == target_proxy)
    {
      return ;
    }
  count = [hit_detector count];
  [canvas redrawLockRetain];
  /* point->segment->path */
  path = [hit_detector objectAtIndex: count - 1 -1 -1];
  [canvas redrawFigObj: path];
  diff.x = point->x - target_point->x;
  diff.y = point->y - target_point->y;
  *target_point = *point;
  if (NULL != other_side_target_point)
    {
      *other_side_target_point = *point;
    }
  if ((YES == shift) && (NULL != neighbor_points[0]))
    {
      NSPoint_add(neighbor_points[0], &diff, neighbor_points[0]);
    }
  if ((YES == shift) && (NULL != neighbor_points[1]))
    {
      NSPoint_add(neighbor_points[1], &diff, neighbor_points[1]);
    }
  [path calcBBox];
  [canvas redrawFigObj: path];
  [target_proxy zeroBBox];
  [canvas redrawLockRelease];
}
- (void)firstDragNotifyEvent: (GdkEventMotion *)event 
	       onCanvas: (GyveCanvas *)canvas
		atPoint: (NSPoint *)point 
	     withBuffer: (GyveBuffer *)buffer
{
  id<PSFigObj> figobj;
  PSPath * path;
  NSValue * point_value;
  NSPoint p;
  int hit_count, segment_count;
  int point_index, segment_index;
  PSSegment * segment, * other_side_segment, * neighbor_segment;

  [super firstDragNotifyEvent: event
	 onCanvas: canvas
	 atPoint: point
	 withBuffer: buffer];

  if (nil != target_proxy)
    {
      [target_proxy release], target_proxy = nil;
    }

  [hit_detector setTargetForHitDetectrion: [buffer selectionsLayer]];
  if (YES == [hit_detector detectHitUnderPoint: &start_point withDepth: 
			     HIT_DETECT_DEPTH_ANCHOR_POINT])
    {
      /* 0: selections layer, 1: proxy */
      target_proxy = [hit_detector objectAtIndex: 1];
      if (control == YES)
	{
	  figobj = [[target_proxy targetForFigObjProxy] copy];
	  [target_proxy unSelect];
	  target_proxy = [buffer putFigObj: figobj];
	}
      [target_proxy retain];

      figobj = [target_proxy beginModificationByCopy];
      [hit_detector setTargetForHitDetectrion: figobj];
      [hit_detector detectHitUnderPoint: &start_point withDepth: 
		      HIT_DETECT_DEPTH_ANCHOR_POINT];
      hit_count     = [hit_detector count];
      path 	    = [hit_detector objectAtIndex: hit_count - 1 -1 -1];
      init_bbox     = *[path bboxCStructure];
      segment 	    = [hit_detector objectAtIndex: hit_count - 1 -1];
      point_value   = [hit_detector objectAtIndex: hit_count - 1];
      p             = [point_value pointValue];
      point_index   = [segment indexOfPoint: &p];
      target_point  = [segment pointAtIndex: point_index];
      segment_index = [path indexOfSegment: segment];
      segment_count  = [path countSegments];
      
      if (segment_index == 0)
	{
	  neighbor_points[0] = NULL;
	  if (segment_count > 1)
	    {
	      neighbor_segment = [path segmentAtIndex: 1];
	      if (dps_curveto == [neighbor_segment dpsOperation])
		neighbor_points[1] = [neighbor_segment pointAtIndex: 0];
	      else
		neighbor_points[1] = NULL;
	    }
	  else
	    {
	      neighbor_points[1] = NULL;
	    }
	}
      else if (segment_index + 1 == segment_count)
	{
	  neighbor_points[1] = NULL;
	  if (dps_curveto == [segment dpsOperation])
	    neighbor_points[0] = [segment pointAtIndex: 1];
	  else
	    neighbor_points[0] = NULL;
	}
      else
	{
	  // neighbor_points[0]
	  if (dps_curveto == [segment dpsOperation])
	    {
	      neighbor_points[0] = [segment pointAtIndex: 1];
	    }
	  else
	    {
	      neighbor_segment = [path segmentAtIndex: segment_index - 1];
	      if (dps_curveto == [neighbor_segment dpsOperation])
		neighbor_points[0] = [neighbor_segment pointAtIndex: 1];
	      else
		neighbor_points[0] = NULL;
	    }

	  neighbor_segment = [path segmentAtIndex: segment_index + 1];
	  if (dps_curveto == [neighbor_segment dpsOperation])
	    neighbor_points[1] = [neighbor_segment pointAtIndex: 0];
	  else
	    neighbor_points[1] = NULL;
	}
      
      /* Other side */
      if (((segment_index + 1) == segment_count)
	  && (YES == [path isClosedPath])
	  && ((point_index == 2)
	      || ((point_index == 0) 
		  && (dps_curveto != [segment dpsOperation]))))
	{
	  other_side_segment = [path segmentAtIndex: 0];
	  other_side_target_point = [other_side_segment
				      firstPoint];
	  if (NO == NSPoint_equal(other_side_target_point,
				   target_point))
	    other_side_target_point = NULL;
	  else
	    {
	      other_side_segment = [path segmentAtIndex: 1];
	      if (dps_curveto == [other_side_segment dpsOperation])
		{
		  neighbor_points[1] = [other_side_segment
					 pointAtIndex: 0];
		}
	    }
	}
      else if ((segment_index  == 0)
	       && (YES == [path isClosedPath]))
	{
	  other_side_segment = [path segmentAtIndex:
				       segment_count - 1];
	  other_side_target_point = [other_side_segment
				      lastPoint];
	  if (NO == NSPoint_equal(other_side_target_point,
				  target_point))
	    other_side_target_point = NULL;
	  else if (dps_curveto == [other_side_segment dpsOperation])
	    {
	      neighbor_points[0] = [other_side_segment
				     pointAtIndex: 1];
	    }
	}
      else
	{
	  other_side_target_point = NULL;
	}
    }
}

@end 


/* GyveDirectionTool */
@implementation GyveDirectionTool
- init
{
  [super init];
  init_bbox    	   = zero_bbox;
  hit_detector 	   = [[GyveHitDetector alloc] initWithTarget: nil];
  target_points[0] = NULL;
  target_points[1] = NULL;
  return self ;
}

- (int)requestForDrawingEngine
{
  return tool_drawing_anchor_point
    | tool_drawing_direction_point
    | tool_drawing_direction_line;
}
- (void)endDragNotifyEvent: (GdkEventButton * )event
		  onCanvas: (GyveCanvas *)canvas
		   atPoint: (NSPoint *)point 
		withBuffer: (GyveBuffer *)buffer
{
  if (nil == target_proxy)
    {
      return ;
    }
  [canvas redrawLockRetain];
  [canvas redrawBBoxCStructure: &init_bbox expandBy: 2.0];
  [canvas redrawFigObj: [target_proxy targetForFigObjProxy]];
  [target_proxy endModification];
  [canvas redrawLockRelease];
  [target_proxy release], target_proxy = nil;
  target_points[0] = NULL;
  target_points[1] = NULL;
}
- (void)draggingNotifyEvent: (GdkEventMotion *)event 
		  onCanvas: (GyveCanvas *)canvas
		   atPoint: (NSPoint *)point 
	     withBuffer: (GyveBuffer *)buffer
{
  int count;
  PSPath * path;
  [super draggingNotifyEvent: event
	 onCanvas: canvas
	 atPoint: point
	 withBuffer: buffer];

  if (nil == target_proxy)
    {
      return ;
    }
  count = [hit_detector count];
  [canvas redrawLockRetain];
  /* point->segment->path */
  path = [hit_detector objectAtIndex: count - 1 -1 -1];
  [canvas redrawFigObj: path];
  if ((target_points[1] != NULL)
      && (target_points[0] != NULL))
    {
      *target_points[0] = *point;
      target_points[1]->x = 2*(base_point.x) - target_points[0]->x;
      target_points[1]->y = 2*(base_point.y) - target_points[0]->y;
    }
  else if (target_points[0] != NULL)
    {
      *(target_points[0]) = *point;
    }
  else if (target_points[1] != NULL)
    {
      *(target_points[1]) = *point;
    }
  [path calcBBox];
  [canvas redrawFigObj: path];
  [target_proxy zeroBBox];
  [canvas redrawLockRelease];
}
- (void)firstDragNotifyEvent: (GdkEventMotion *)event 
	       onCanvas: (GyveCanvas *)canvas
		atPoint: (NSPoint *)point 
	     withBuffer: (GyveBuffer *)buffer
{
  id<PSFigObj> figobj;
  PSPath * path;
  NSValue * point_value;
  NSPoint p;
  int hit_count;
  int point_index, segment_index, segment_count;
  PSSegment * segment, * segment_next, * segment_prev;

  [super firstDragNotifyEvent: event
	 onCanvas: canvas
	 atPoint: point
	 withBuffer: buffer];

  if (nil != target_proxy)
    {
      [target_proxy release], target_proxy = nil;
    }

  [hit_detector setTargetForHitDetectrion: [buffer selectionsLayer]];
  if (YES == [hit_detector detectHitUnderPoint: &start_point withDepth: 
				  HIT_DETECT_DEPTH_ANCHOR_POINT])
    {
      /* 0: selections layer, 1: proxy */
      target_proxy = [hit_detector objectAtIndex: 1];
      if (control == YES)
	{
	  figobj = [[target_proxy targetForFigObjProxy] copy];
	  [target_proxy unSelect];
	  target_proxy = [buffer putFigObj: figobj];
	}
      [target_proxy retain];
      figobj = [target_proxy beginModificationByCopy];

      [hit_detector setTargetForHitDetectrion: figobj];
      [hit_detector detectHitUnderPoint: &start_point withDepth: 
		      HIT_DETECT_DEPTH_ANCHOR_POINT];
      hit_count        = [hit_detector count];
      path 	       = [hit_detector objectAtIndex: hit_count - 1 -1 -1];
      init_bbox        = *[path bboxCStructure];
      segment 	       = [hit_detector objectAtIndex: hit_count - 1 -1];
      segment_index    = [path indexOfSegment: segment];
      segment_count    = [path countSegments];
      base_point = [(NSValue *)[hit_detector objectAtIndex:
							hit_count - 1]
					pointValue];
      /* target_points[0] */
      if (dps_curveto == [segment dpsOperation])
	{
	  target_points[0]    = [segment pointAtIndex: 1];
	  *(target_points[0]) = base_point;
	}
      else if (dps_lineto == [segment dpsOperation])
	{
	  segment = [path curveSegmentAtIndex: segment_index];
	  target_points[0] = [segment pointAtIndex: 1];
	  *(target_points[0]) = base_point;
	}
      else if (dps_moveto == [segment dpsOperation])
	{	
	  if ((YES == [path isClosedPath])
	      && (YES  == NSPoint_equal([[path segmentAtIndex:
						 segment_count - 1]
					  lastPoint],
					[segment firstPoint])))
	    {
	      segment_prev = [path curveSegmentAtIndex:
				     segment_count - 1];
	      target_points[0] = [segment_prev pointAtIndex: 2];
	      *(target_points[0]) = base_point;
	    }
	  else
	    {
	      target_points[0] = NULL;
	    }
	}
      
      /* target_points[1] */
      if (dps_curveto == [segment dpsOperation]
	  || dps_lineto == [segment dpsOperation])
	{
	  if (segment_index + 1 < segment_count)
	    {
	      segment_next = [path curveSegmentAtIndex: segment_index+1];
	      target_points[1] = [segment_next pointAtIndex: 0];
	      *(target_points[1]) = base_point;
	    }
	  else if ((YES    == [path isClosedPath])
		   && (YES == NSPoint_equal([[path segmentAtIndex: 0]
					      firstPoint],
					[segment lastPoint])))
	    {
	      segment_next = [path curveSegmentAtIndex: 1];
	      target_points[1] = [segment_next pointAtIndex: 0];
	      *(target_points[1]) = base_point;
	    }
	  else
	    {
	      target_points[1] = NULL;
	    }
	}
      else if (dps_moveto == [segment dpsOperation])
	{
	  segment = [path curveSegmentAtIndex: segment_index+1];
	  target_points[1] = [segment pointAtIndex: 0];
	  *(target_points[1]) = base_point;
	}
    }
  else if (YES == [hit_detector detectHitUnderPoint: &start_point withDepth: 
			     HIT_DETECT_DEPTH_DIRECTRION_POINT])
    {
      /* 0: selections layer, 1: proxy */
      target_proxy = [hit_detector objectAtIndex: 1];
      if (control == YES)
	{
	  figobj = [[target_proxy targetForFigObjProxy] copy];
	  [target_proxy unSelect];
	  target_proxy = [buffer putFigObj: figobj];
	}
      [target_proxy retain];

      figobj = [target_proxy beginModificationByCopy];
      [hit_detector setTargetForHitDetectrion: figobj];
      [hit_detector detectHitUnderPoint: &start_point withDepth: 
		      HIT_DETECT_DEPTH_DIRECTRION_POINT];
      hit_count     = [hit_detector count];
      path 	    = [hit_detector objectAtIndex: hit_count - 1 -1 -1];
      init_bbox     = *[path bboxCStructure];
      segment 	    = [hit_detector objectAtIndex: hit_count - 1 -1];
      point_value   = [hit_detector objectAtIndex: hit_count - 1];
      p             = [point_value pointValue];
      point_index   = [segment indexOfPoint: &p];
      target_points[0]  = [segment pointAtIndex: point_index];
      target_points[1]  = NULL;
    }
}
- (void)startSessionOnCanvas: (GyveCanvas *)canvas 
		  withBuffer: (GyveBuffer *)buffer
{
  if ([buffer countSelectedFigObjs] > 0)
    {
      [canvas redrawBBox: [buffer selectionsLayer] expandBy: 2.0];
    }
}
@end

/* GyveMatrixTool */
@implementation GyveMatrixTool
- init
{
  [super init];
  matrix = nil;
  return self;
}
- (void)dealloc
{
  if (matrix != nil)
    {
      [matrix release], matrix = nil;
    }
  [super dealloc];
}
- (int)requestForDrawingEngine
{
  return tool_drawing_anchor_point;
}
- (void)buttonPressEvent: (GdkEventButton * )event
		onCanvas: (GyveCanvas *)canvas
		 atPoint: (NSPoint *)point 
	      withBuffer: (GyveBuffer *)buffer
{
  [super buttonPressEvent: event
	 onCanvas: canvas
	 atPoint: point
	 withBuffer: buffer];
  if (matrix != nil)
    {
      [matrix release];
    }
  matrix = [[PSMatrixApplier alloc] initWithVisitingTarget: 
					[buffer selectionsLayer]];
  op = [buffer beginOperation];
}
- (void)buttonReleaseEvent: (GdkEventButton * )event
		onCanvas: (GyveCanvas *)canvas
		 atPoint: (NSPoint *)point 
	      withBuffer: (GyveBuffer *)buffer
{
  [canvas redrawLockRetain];
  [super buttonReleaseEvent: event
	 onCanvas: canvas
	 atPoint: point
	 withBuffer: buffer];
  [canvas redrawBBoxCStructure: &init_bbox expandBy: 2.0];
  [canvas redrawBBox: [buffer selectionsLayer] expandBy: 2.0];
  [canvas redrawLockRelease];
  [buffer endOperation: op];
  
}

- (void)endDragNotifyEvent: (GdkEventButton * )event
		  onCanvas: (GyveCanvas *)canvas
		   atPoint: (NSPoint *)point 
		withBuffer: (GyveBuffer *)buffer
{
  if ([buffer countSelectedFigObjs] > 0)
    {
      [canvas redrawBBox: [buffer selectionsLayer] expandBy: 2.0];
      [buffer endModificationOfAll];
    }
}
- (void)dragNotifyEvent: (GdkEventMotion *)event 
	       onCanvas: (GyveCanvas *)canvas
		atPoint: (NSPoint *)point 
	     withBuffer: (GyveBuffer *)buffer
{
  if ([buffer countSelectedFigObjs] > 0)
    {
      [self calcMatrixWithPoint: point];
      [super dragNotifyEvent: event
	     onCanvas: canvas
	     atPoint: point
	     withBuffer: buffer];
      [self applyMatrix: matrix onCanvas: canvas];
    }
  else
    {
      [super dragNotifyEvent: event
	     onCanvas: canvas
	     atPoint: point
	     withBuffer: buffer];
    }
}

- (void)applyMatrix: (PSMatrixApplier *)m onCanvas: (GyveCanvas *)canvas
{
  [canvas redrawLockRetain];
  [canvas redrawBBox: (id<PSBBox>)[m visitingTarget] expandBy: 2.0];
  [m visitUpward];	// 
  [canvas redrawBBox: (id<PSBBox>)[m visitingTarget] expandBy: 2.0];
  [canvas redrawLockRelease];
}
- (void)startSessionOnCanvas: (GyveCanvas *)canvas 
		  withBuffer: (GyveBuffer *)buffer
{
  struct bbox bbox;
  NSPoint p;
  if ([buffer countSelectedFigObjs] > 0)
    {
      [self setLayer: [buffer topLayerToWrite]];
      bbox =  *[[buffer selectionsLayer] bboxCStructure];
      p    = bbox_center(&bbox);
      [self setClickPoint: &p onCanvas: canvas];
    }
}
- (void)finishSessionOnCanvas: (GyveCanvas *)canvas 
		   withBuffer: (GyveBuffer *)buffer
{
  if ([buffer countSelectedFigObjs] > 0)
    {
      [self eraseClickPoint: [self clickPoint] onCanvas: canvas];
    }
}
- (void)drawClickPoint: (NSPoint *)p onCanvas: (GyveCanvas *)canvas
{
  psw_draw_center(p->x, p->y, 3.0);
}
- (void)drawAfterAll
{
  [self drawClickPoint: [self clickPoint] onCanvas: NULL];
}
- (void)eraseClickPoint: (NSPoint *)p onCanvas: (GyveCanvas *)canvas
{
  NSRect rect;
  NSRect_from_center(&rect, p, 7.0, 7.0);
  [canvas  redrawRect: &rect];
}
- (void)calcMatrixWithPoint: (NSPoint *)p
{
  [self subclassResponsibility:_cmd];
}
- (void)firstDragNotifyEvent: (GdkEventMotion *)event 
		    onCanvas: (GyveCanvas *)canvas
		     atPoint: (NSPoint *)point 
		  withBuffer: (GyveBuffer *)buffer
{
  if ([buffer countSelectedFigObjs] > 0)
    {
      init_bbox = *[[buffer selectionsLayer] bboxCStructure];
      if (YES == control)
	{
	  [self duplicateSelectedFigObjsInBuffer: buffer];
	}
      [buffer beginModificationByCopyOfAll];
    }
}
@end

@implementation GyveSelectionTool
- init
{
  [super init];
  hit_detector = [[GyveHitDetector alloc] initWithTarget: nil];
  return self ;
}
- (void)dealloc
{
  [hit_detector release], hit_detector = nil;
  [super dealloc];
}
- (int)requestForDrawingEngine
{
  return tool_drawing_anchor_point ;
}
- (void)clickNotifyEvent: (GdkEventButton *)event 
		onCanvas: (GyveCanvas *)canvas
		 atPoint: (NSPoint *)point 
	      withBuffer: (GyveBuffer *)buffer
{
  [self trySelectFigObjsOnCanvas: canvas atPoint: point withBuffer: buffer];
}
- (void)trySelectFigObjsOnCanvas: (GyveCanvas *)canvas
			 atPoint: (NSPoint *)point
		      withBuffer: (GyveBuffer *)buffer
{
  GyveLayer * local_layer;
  NSObject<PSFigObj>*figobj;
  NSObject<PSFigObj>* primitive_figobj;
  PSPaintStyle * paint_style;
  
  // 
  [hit_detector setTargetForHitDetectrion: [buffer selectionsLayer]];
  if (YES == [hit_detector detectHitUnderPoint: point])
    {
      return;
    }

  [hit_detector setTargetForHitDetectrion: buffer];
  if (YES == [hit_detector detectHitUnderPoint: point])
    {
      [canvas redrawLockRetain];
      if (NO == shift)
	{
	  [canvas redrawBBox: [buffer selectionsLayer] expandBy: 2.0];
	  [buffer unSelectAll];
	}

      /* 1st object -> buffer
	 2nd object -> layer
	 3rd object -> THE TOP LEVEL FIGOBJ AROUND THE POINT */
      figobj = [hit_detector thirdObject];
      [buffer selectFigObj: figobj];

      /* Picking the paint style of the primitive figobj
	 and updating paint style pallet.
	 TODO:
	 Text, text elemetn, text style... */
      if (YES == [figobj conformsToProtocol:@protocol(FigObjsContaining)])
	{
	  primitive_figobj = [(id<FigObjsContaining>)figobj 
						     lastPrimitiveFigObj];
	}
      else
	{
	  primitive_figobj = figobj;
	}
      if ([primitive_figobj respondsToSelector: @selector(paintStyle)])
	{
	  paint_style = [primitive_figobj paintStyle];
	  [[PaintStylePallet paintStylePallet] installPaintStyle: paint_style];
	}

      [canvas redrawFigObj: figobj];
      [canvas redrawLockRelease];
      return;
    } 
  
  [hit_detector setTargetForHitDetectrion: [buffer selectionsLayer]];
  if (YES == [hit_detector detectHitAroundPoint: point])
    {
      return;
    }

  [hit_detector setTargetForHitDetectrion: buffer];
  if (YES == [hit_detector detectHitAroundPoint: point])
    {
      local_layer = (GyveLayer *)[hit_detector secondObject];
      if ([local_layer isKindOfClass: [GyveSelectionsLayer class]])
	{
	  if (NO == shift)
	    {
	      [canvas redrawLockRetain];
	      [canvas redrawBBox: (GyveSelectionsLayer *)local_layer
		      expandBy: 2.0];
	      [buffer unSelectAll];
	      [canvas redrawLockRelease];
	    }
	}
      else // nomarl layer
	{
	  figobj = [hit_detector thirdObject];
	  [canvas redrawLockRetain];
	  if (NO == shift)
	    {
	      [canvas redrawBBox: [buffer selectionsLayer] expandBy: 2.0];
	      [buffer unSelectAll];
	    }
	  [buffer selectFigObj: figobj];
	  [canvas redrawFigObj: figobj];

	  if (YES == [[hit_detector thirdObject] 
		       conformsToProtocol:@protocol(FigObjsContaining)])
	    {
	      primitive_figobj = [figobj lastPrimitiveFigObj];
	    }
	  else
	    {
	      primitive_figobj = figobj;
	    }
	  if (YES == [primitive_figobj respondsToSelector: @selector(paintStyle)])
	    {
	      paint_style = [primitive_figobj paintStyle];
	      [[PaintStylePallet paintStylePallet] installPaintStyle: paint_style];
	    }
	  [canvas redrawLockRelease];
	}
    }
  else if ((NO == shift)
	   && ([buffer countSelectedFigObjs] > 0))
    {
      [canvas redrawLockRetain];
      [canvas redrawBBox: [buffer selectionsLayer] expandBy: 2.0];
      [buffer unSelectAll];
      [canvas redrawLockRelease];
    } 
}

- (void)calcMatrixWithPoint: (NSPoint *)p
{
  NSPoint diff;
  NSPoint_diff(p, [self lastPoint], &diff);

  if (YES == shift)
    {
      if (YES == direction_x)
	{
	  diff.y = 0.0;
	}
      else
	{
	  diff.x = 0.0;
	}
    }
  [matrix setMatrixElement: diff.x atIndex: MATRIX_TX];
  [matrix setMatrixElement: diff.y atIndex: MATRIX_TY];
}
- (void)firstDragNotifyEvent: (GdkEventMotion *)event 
		    onCanvas: (GyveCanvas *)canvas
		     atPoint: (NSPoint *)point 
		  withBuffer: (GyveBuffer *)buffer
{
  NSPoint p;
  [self trySelectFigObjsOnCanvas: canvas atPoint: point withBuffer: buffer];
  [super firstDragNotifyEvent: event
	 onCanvas: canvas
	 atPoint: point
	 withBuffer: buffer];

  NSPoint_diff(point, [self startPoint], &p);
  if (fabs(p.x) > fabs(p.y))
    {
      direction_x = YES;
    }
  else
    {
      direction_x = NO;
    }
}
- (void)drawClickPoint: (NSPoint *)p onCanvas: (GyveCanvas *)canvas
{
}
- (void)eraseClickPoint: (NSPoint *)p onCanvas: (GyveCanvas *)canvas
{
}
@end

@implementation GyveRotateTool: GyveMatrixTool
- (void)firstDragNotifyEvent: (GdkEventMotion *)event 
		    onCanvas: (GyveCanvas *)canvas
		     atPoint: (NSPoint *)point 
		  withBuffer: (GyveBuffer *)buffer
{
  radian_sum = 0.0;
  [super firstDragNotifyEvent: event
	 onCanvas: canvas
	 atPoint: point
	 withBuffer: buffer];
}
- (void)calcMatrixWithPoint: (NSPoint *)p
{
  NSPoint center = *[self clickPoint];
  float sin0;
  float cos0;
  float unit_radian = 0.52359877559829882;// Pi/6
  float radian;

  radian  = [self pointsAngleForPoint: p];
  if (YES == shift)
    {
      radian_sum += radian;
      if (fabs(radian_sum) > unit_radian)
	{
	  radian     = (radian_sum > 0.0)?unit_radian: -unit_radian;
	  radian_sum = 0.0;
	}
      else
	{
	  radian = 0.0;
	}
    }

  sin0 = (float)sin(radian);
  cos0 = (float)cos(radian);
  [matrix setMatrixElement: cos0 atIndex: MATRIX_A];
  [matrix setMatrixElement: sin0 atIndex: MATRIX_B];  
  [matrix setMatrixElement: -sin0 atIndex: MATRIX_C];
  [matrix setMatrixElement: cos0 atIndex: MATRIX_D];  
  [matrix setDeltaFromOriginX: center.x Y: center.y];
}
@end


@implementation GyveScaleTool
- (void)firstDragNotifyEvent: (GdkEventMotion *)event 
		    onCanvas: (GyveCanvas *)canvas
		     atPoint: (NSPoint *)point 
		  withBuffer: (GyveBuffer *)buffer
{
  NSPoint p;
  [super firstDragNotifyEvent: event
	 onCanvas: canvas
	 atPoint: point
	 withBuffer: buffer];
  NSPoint_diff(point, [self startPoint], &p);
  if (fabs(p.x) > fabs(p.y))
    {
      direction_x = YES;
    }
  else
    {
      direction_x = NO;
    }
}
- (void)calcMatrixWithPoint: (NSPoint *)p
{
  float scale_x, scale_y;
  NSPoint center = *[self clickPoint];  
  scale_x 	 = [self pointsXScaleForPoint: p];
  scale_y 	 = [self pointsYScaleForPoint: p];
  if (YES == shift)
    {
      if (YES ==direction_x)
	{
	  scale_y = 1.0;
	}
      else
	{
	  scale_x = 1.0;
	}
    }
  if (YES == float_zero(scale_x))
    {
      scale_x = 1.0;
    }
  if (YES == float_zero(scale_y))
    {
      scale_y = 1.0;
    }
  [matrix setMatrixElement: scale_x atIndex: MATRIX_A];
  [matrix setMatrixElement: scale_y atIndex: MATRIX_D];  
  [matrix setDeltaFromOriginX: center.x Y: center.y];
}
@end

static float
scatter_radian(float radian, float scatter_unit)
{
  int i;
  BOOL plus = (radian > 0.0)? YES: NO;
  radian    = fabs(radian);
  for (i = -1; radian > 0.0; i++)
    {
      radian -= scatter_unit;
    }
  radian = scatter_unit * i;
  if (YES == plus)
    {
      radian = fabs(radian);
    }
  else
    {
      radian = -fabs(radian);
    }
  return radian;
      
}
