 /* GyveHitDetector.m --- class wrapping hit detecting functions
   
   Copyright (C) 1998 Free Software Foundation, Inc.
   
   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#define CONTROL_BOX_SIZE 14.0

#include "GyveHitDetector.h"
#include "geometry.h"
#include <Foundation/NSException.h>
#include "GyveLayer.h"
#include "GyveBuffer.h"
#include "PSImage.h"
#include "public.h"
#include "PSMaskedFigObjGroup.h"
#include "PSSegment.h"
#include "PSFigObj.h"
#include "GyveSelectionsLayer.h"
#include "PSTextElement.h"

static BOOL detect_hit_on_xxx (id self, 
			       id target, 
			       id_predicator_t predicator, 
			       SEL sel,
			       enum hit_detect_method method, 
			       void * argument);

/* predicating, bbox checkingstacking̥᥽åɤǤ. */

@implementation GyveHitDetector
- initWithTarget: t
{
  [super init];
  target = t;
  scale  = 0.0;
  radius = 1.5;
  floats_data_init(&coords); 
  DPSUserPathOps_data_init(&ops);
  return self ;
}
- (void)dealloc
{
  floats_data_free(&coords);
  DPSUserPathOps_data_free(&ops);
  [super dealloc];
}
- (void)setTargetForHitDetectrion: t
{
  target = t;
}
- (void)setRadius: (float) r
{
  radius = r;
}
- (void)setAbsoluteScale:(float) s
{
  scale = s;
}
- secondObject
{
  return [self objectAtIndex: 1];
}
- thirdObject
{
  return [self objectAtIndex: 2];
}
- (BOOL)detectHitUnderPoint: (NSPoint *) point
{
  return [self detectHitUnderPoint: point 
	       withPredicator: id_predicator_yes];
}
- (BOOL)detectHitUnderPoint: (NSPoint *) point
		  withDepth: (enum hit_detect_depth)d
{
  return [self detectHitUnderPoint: point 
	       withPredicator: id_predicator_yes
	       depth: d];
}
- (BOOL)detectHitUnderPoint: (NSPoint *) point 
	 withPredicator: (id_predicator_t)p
{
  return [self detectHitUnderPoint: point
	       withPredicator: p
	       depth: HIT_DETECT_DEPTH_DEFAULT];
}
- (BOOL)detectHitUnderPoint: (NSPoint *) point 
	 withPredicator: (id_predicator_t)p
		      depth: (enum hit_detect_depth)d
{
  [self empty];
  depth = d;
  return [self detectHitByMethod: HIT_DETECT_UNDER_POINT
	       withArgument: point
	       predicator: p];    
}
- (BOOL)detectHitAroundPoint: (NSPoint *) point
{
  [self empty];
  return [self detectHitAroundPoint: point
	       withPredicator: id_predicator_yes];
}
- (BOOL)detectHitAroundPoint: (NSPoint *) point 
	  withPredicator: (id_predicator_t)p
{
  [self empty];
  return [self detectHitByMethod: HIT_DETECT_AROUND_POINT
	       withArgument: point
	       predicator: p];
}
- (BOOL)detectHitAcrossRect: (NSRect *) rect
{
  [self empty];
  return [self detectHitAcrossRect: rect
	       withPredicator: id_predicator_yes];
}
- (BOOL)detectHitAcrossRect: (NSRect *) rect
	 withPredicator: (id_predicator_t)p
{
  return [self detectHitByMethod: HIT_DETECT_ACROSS_RECT
	       withArgument: rect
	       predicator: p];
}
@end

		  
@implementation GyveHitDetector(Protected)
- (BOOL)detectHitByMethod: (enum hit_detect_method)m
	     withArgument: (void *)arg
	       predicator: (id_predicator_t)p
{
  BOOL result;
  BOOL should_push_and_pop;
  method      = m;
  argument    = arg;
  predicator      = p;

  should_push_and_pop = (method == HIT_DETECT_UNDER_POINT 
			 || method == HIT_DETECT_AROUND_POINT);
  if (should_push_and_pop) [self pushObject: target];
  
  if (YES == predicator(target))
    {
      if ((NO == [target conformsToProtocol: @protocol(PSBBox)]) ||
	  (YES == detect_hit_on_bbox ([(id<PSBBox>)target bboxCStructure], 
				     method, argument)))
	result = [self detectHitOnTarget: target];
      else
	result = NO;
    }
  else
    result = NO;
  
  if (should_push_and_pop && (NO == result))
    [self popObject];
  
  return result;  
}

- (BOOL)detectHitOnTarget: _target
{
  if (nil == _target)
    return NO;
  else if ([_target isKindOfClass: [GyveBuffer class]])
    return [self detectHitOnBuffer: (GyveBuffer *)_target];
  else if ([_target isKindOfClass: [GyveLayer class]])
    return [self detectHitOnLayer: (GyveLayer *)_target];
  else if ([_target conformsToProtocol: @protocol(PSFigObj)])
    return [self detectHitOnFigObj: (id<PSFigObj>)_target];
  else 
    return NO;
}
- (BOOL)detectHitOnBuffer: (GyveBuffer *)buffer
{
  BOOL result = NO;
  BOOL should_push_and_pop = (method == HIT_DETECT_UNDER_POINT
			      || method == HIT_DETECT_AROUND_POINT);
  GyveLayer * layer;
  
  if (should_push_and_pop)
    {
      layer = (GyveLayer *)[buffer selectionsLayer];
      [self pushObject: layer];
      result = 
	detect_hit_on_xxx(self, layer, predicator, @selector(detectHitOnLayer:),
			  method, argument) ;      
      if (NO == result)
	[self popObject];      
      else
	return YES;
      
      FOR_ARRAY_REVERSE(buffer,layer)
	{
	  if (NO == result)
	    {
	      [self pushObject: layer];
	      result = 
		detect_hit_on_xxx(self, layer, predicator, @selector(detectHitOnLayer:),
				  method, argument) ;
	      if (NO == result) [self popObject];
	    }
	}
      END_FOR_ARRAY(buffer);
	
    }
  else
    {
      layer  = (GyveLayer *)[buffer selectionsLayer];
      result = 
	detect_hit_on_xxx(self, layer, predicator, @selector(detectHitOnLayer:),
			  method, argument) 
	|| result;
      FOR_ARRAY_REVERSE(buffer,layer)
	{
      	  result = 
	    detect_hit_on_xxx(self, layer, predicator, 
			      @selector(detectHitOnLayer:), method, argument)
	    || result;
	}
      END_FOR_ARRAY(buffer);
    }
  return result;
}
- (BOOL)detectHitOnLayer: (GyveLayer *)layer
{
  BOOL result = NO;
  BOOL should_push_and_pop;
  id<PSFigObj> figobj;

  if (NO == [layer isWritable]) return NO;
  should_push_and_pop = (method == HIT_DETECT_UNDER_POINT 
			 || method == HIT_DETECT_AROUND_POINT);
  if (should_push_and_pop)
    {
      FOR_ARRAY_REVERSE(layer, figobj)
	{
	  if (NO == result)
	    {
	      [self pushObject: figobj];
	      result = detect_hit_on_xxx (self, 
					  figobj, 
					  predicator, 
					  @selector(detectHitOnFigObj:),
					  method,
					  argument);
	      if (NO == result) [self popObject];
	    }
	}
      END_FOR_ARRAY(layer);
    }
  else
    {
      FOR_ARRAY_REVERSE(layer, figobj)
	{
	  result = detect_hit_on_xxx (self, 
				      figobj, 
				      predicator, 
				      @selector(detectHitOnFigObj:),
				      method,
				      argument) ||
	    result;
	}
      END_FOR_ARRAY(layer);
    }
  return result;
}

- (BOOL)detectHitOnFigObj: (id<PSFigObj>)figobj
{
  if ([(NSObject *)figobj isKindOfClass: [PSFigObjSelectedProxy class]])
    return [self detectHitOnFigObjProxy: (PSFigObjSelectedProxy *)figobj];
  else if ([(NSObject *)figobj isKindOfClass: [PSPath class]])
    return [self detectHitOnPath: (PSPath *)figobj];
  else if ([(NSObject *)figobj conformsToProtocol: @protocol(PSText)])
    return [self detectHitOnText: (id<PSText>)figobj];
  else if ([(NSObject *)figobj isKindOfClass: [PSImage class]])
    return [self detectHitOnImage: (PSImage *)figobj];
  else if ([(NSObject *)figobj isMemberOfClass: [PSFigObjGroup class]])
    return [self detectHitOnFigObjGroup: (PSFigObjGroup *)figobj];
  else if ([(NSObject *)figobj isMemberOfClass: [PSMaskedFigObjGroup class]])
    return [self detectHitOnMaskedFigObjGroup: (PSMaskedFigObjGroup *)figobj];
  else if ([(NSObject *)figobj isMemberOfClass: [PSCompoundPaths class]])
    return [self detectHitOnCompoundPaths: (PSCompoundPaths *)figobj];
  else
    return NO;
}
- (BOOL)detectHitOnFigObjProxy: (PSFigObjSelectedProxy *)figobj_proxy
{
  id<PSFigObj> figobj = [figobj_proxy targetForFigObjProxy];
  [self pushObject: figobj];
  if (YES == [self detectHitOnFigObj: figobj])
    {
      return YES;
    }
  else
   {
     [self popObject];
     return NO;
   }
}
- (BOOL)detectHitOnPath: (PSPath *)path
{
  Bool result;
  float bbox[4];
  DPSUserPathAction action;
  struct bbox * bbox_cstruct = [path bboxCStructure];

  int coords_length;
  int ops_length;
  if (method == HIT_DETECT_ACROSS_RECT)
    {
      return YES;
    }

  if (depth == HIT_DETECT_DEPTH_DEFAULT)
    {
      hit_detector_pack_path (path, &coords, &coords_length, 
			      &ops, &ops_length);

      bbox[BBOX_LLX] = bbox_element(bbox_cstruct, BBOX_LLX);
      bbox[BBOX_LLY] = bbox_element(bbox_cstruct, BBOX_LLY);
      bbox[BBOX_URX] = bbox_element(bbox_cstruct, BBOX_URX);
      bbox[BBOX_URY] = bbox_element(bbox_cstruct, BBOX_URY);

      if (method == HIT_DETECT_UNDER_POINT)
	{
	  action = dps_inustroke;
	}
      else
	{
	  if (YES == [path isKindOfClass: [PSPath class]])
	    action = dps_inufill;	// IN
	  else
	    action = dps_inueofill;	// For PSCompoundPaths
	}


      result = PSHitUserPath (((NSPoint *)argument)->x,
			      ((NSPoint *)argument)->y,
			      radius,
			      (DPSPointer)coords.data,
			      coords_length,
			      dps_float,
			      ops.data,
			      ops_length,
			      (DPSPointer)bbox,
			      action);
      if (result)
	return YES;
      else
	return NO;
    }
  else if (depth == HIT_DETECT_DEPTH_SEGMENT)
    {
      /* TODO: Should implement */
    }
  else if (depth == HIT_DETECT_DEPTH_ANCHOR_POINT)
    {
      int i, max;
      PSSegment * seg;
      NSPoint p;
      BOOL local_result;
      max = [path countSegments];
      for (i = max; 0 < i; i--)
	{
	  seg = [path segmentAtIndex: i - 1];
	  [self pushObject: seg];
	  switch ([seg dpsOperation])
	    {
	    case dps_moveto:
	    case dps_lineto:
	      p = *[seg pointAtIndex: 0];
	      break;
	    case dps_curveto:
	      p = *[seg pointAtIndex: 2];
	      break;
	    }
	  [self pushObject: [NSValue valueWithPoint: p]] ;
	  local_result = detect_hit_on_point_rect(&p, CONTROL_BOX_SIZE, 
						  ((NSPoint *)argument));
	  if (YES == local_result)
	    {
	      return YES;
	    }
	  else
	    {
	      [self popObject];
	    }
	  [self popObject];
	}
      return NO;
    }
  else if (depth == HIT_DETECT_DEPTH_DIRECTRION_POINT)
    {
      int i, max;
      PSSegment * seg;
      NSPoint p;
      BOOL local_result;
      max = [path countSegments];
      for (i = max; 0 < i; i--)
	{
	  seg = [path segmentAtIndex: i - 1];
	  [self pushObject: seg];
	  
	  if (dps_curveto == [seg dpsOperation])
	    {
	      p = *[seg pointAtIndex: 0];
	      [self pushObject: [NSValue valueWithPoint: p]] ;
	      local_result = detect_hit_on_point_rect(&p, 
						      CONTROL_BOX_SIZE, 
						      ((NSPoint *)argument));
	      if (local_result)
		{
		  return YES;
		}
	      else
		{
		  [self popObject];
		}

	      p = *[seg pointAtIndex: 1];
	      [self pushObject: [NSValue valueWithPoint: p]] ;
	      local_result = detect_hit_on_point_rect(&p, CONTROL_BOX_SIZE, 
						      ((NSPoint *)argument));
	      if (local_result)
		{
		  return YES;
		}
	      else
		{
		  [self popObject];
		}
	    }
	  [self popObject];
	}
      return NO;
    }
  else if (depth == HIT_DETECT_DEPTH_DIRECTRION_LINE)
    {
      /* TODO: Should implement */
      return NO;		
    }
  return NO;			// ???
}
- (BOOL)detectHitOnText: (id<PSText>)text
{
  if ([(NSObject *)text isKindOfClass: [PSTextAtPoint class]])
    return [self detectHitOnTextAtPoint: (PSTextAtPoint *)text];
  else
    return NO;
}
- (BOOL)detectHitOnTextAtPoint: (PSTextAtPoint *)text
{
  PSTextElement * elt;
  if (method == HIT_DETECT_ACROSS_RECT)
    {
      return YES;
    }
  else if ((method   == HIT_DETECT_UNDER_POINT)
	   && (depth != HIT_DETECT_DEPTH_DEFAULT))
    {
      return NO;
    }
  else
    {
      if (YES == [text isPointInBBox: (NSPoint *)argument])
	{
	  int i, max = [text countTextElements];
	  for (i = 0; i < max; i++)
	    {
	      elt = [text textElementAtIndex: i];
	      if (YES == [elt isPointInBBox: (NSPoint *)argument])
		{
		  [self pushObject: elt];
		  return YES;
		}
	    }
	  return NO;
	}
      else
	{
	  return NO;
	}
    }
}
- (BOOL)detectHitOnImage: (PSImage *)image
{
  /* TODO */
  float * mapping_matrix;
  PSPath * path;
  NSPoint point0 = {0.0, 0.0};
  NSPoint point1 = {1.0, 0.0};
  NSPoint point2 = {1.0, 1.0};
  NSPoint point3 = {0.0, 1.0};

  if (method == HIT_DETECT_ACROSS_RECT)
    {
      return YES;
    }
  else if ((method   == HIT_DETECT_UNDER_POINT)
	   && (depth != HIT_DETECT_DEPTH_DEFAULT))
    {
      return NO;
    }
  
  mapping_matrix =  [image mappingMatrix];
  matrix_apply_on_point(mapping_matrix, &point0);
  matrix_apply_on_point(mapping_matrix, &point1);
  matrix_apply_on_point(mapping_matrix, &point2);
  matrix_apply_on_point(mapping_matrix, &point3);
  path = [[PSPath alloc] initWithMovetoPoint: &point0];
  [path autorelease];
  [path addLinetoPoint: &point1];
  [path addLinetoPoint: &point2];
  [path addLinetoPoint: &point3];
  [path addLinetoPoint: &point0];
  [path closePath];
  return [self detectHitOnPath: path];
}
- (BOOL)detectHitOnFigObjGroup: (PSFigObjGroup *)group
{
  id<PSFigObj> figobj;
  BOOL result = NO;
  if (method == HIT_DETECT_ACROSS_RECT) return YES;

  FOR_ARRAY_REVERSE(group, figobj)
    {
      if (NO == result)
	{
	  [self pushObject: figobj];
	  result = detect_hit_on_xxx (self, 
				      figobj, 
				      predicator, 
				      @selector(detectHitOnFigObj:),
				      method,
				      argument);
	  if (NO == result) [self popObject];
	}
    }
  END_FOR_ARRAY(group);
  return result;
}
- (BOOL)detectHitOnMaskedFigObjGroup: (PSMaskedFigObjGroup *)masked_group
{
  PSPath * mask = [masked_group maskPath];
  BOOL result = NO;
  if (method == HIT_DETECT_ACROSS_RECT)
    {
      return YES;
    }
  
  [self pushObject: mask];
  result = detect_hit_on_xxx (self,
			      mask,
			      predicator,
			      @selector(detectHitOnPath:),
			      method, 
			      argument);
  if (NO == result)
    [self popObject];
  else
    return YES;

  return [self detectHitOnFigObjGroup: (PSFigObjGroup *)masked_group];
}
- (BOOL)detectHitOnCompoundPaths: (PSCompoundPaths *)compound_paths
{
  if (method == HIT_DETECT_ACROSS_RECT)
    {
      return YES;
    }
  return [self detectHitOnFigObjGroup: (PSFigObjGroup *)compound_paths];
}
@end

static BOOL
detect_hit_on_xxx (id self, id target, id_predicator_t predicator, SEL sel,
		   enum hit_detect_method method, 
		   void * argument)
{
  id result;
  if (YES == predicator(target))
    {
      if ((NO == [target conformsToProtocol: @protocol(PSBBox)]) ||
	  YES == detect_hit_on_bbox ([(id<PSBBox>)target bboxCStructure], 
				     method, 
				     argument))
	{
	  result = [self performSelector: sel withObject: target];
	  if (result == nil)
	    return NO;
	  else
	    return YES;
	}
      else
	return NO;
    }
  else 
    return NO;
}


BOOL
detect_hit_on_bbox(struct bbox * bbox, 
		   enum hit_detect_method method, 
		   void * arg)
{
  NSPoint * point = NULL;
  NSRect * rect   = NULL;
  NSRect rect_from_bbox;
  
  if (HIT_DETECT_UNDER_POINT  == method 
      || HIT_DETECT_AROUND_POINT == method)
    point = arg;
  else if (HIT_DETECT_ACROSS_RECT == method)
    rect = arg;
  else
    NSCAssert (0, @"Wrong detect hit method.");
  
  if (point)
    {
      return bbox_contains_point(bbox, point);
    }
  else
    {
      bbox_to_rect(bbox, &rect_from_bbox);
      return NSIntersectsRect(*rect, rect_from_bbox);
    }
    
}

BOOL
detect_hit_on_point_rect(NSPoint * rectcenter,
			 float size,
			 NSPoint * hitpoint)
{
  BOOL result;
  NSRect rect;
  NSRect_from_center(&rect, rectcenter, size, size);  
  result = NSPointInRect(*hitpoint, rect);
  return result;
}


void
hit_detector_pack_path (id<PSPath> path, 
			struct floats_data * coords,
			int * coords_length,
			struct DPSUserPathOps_data * ops,
			int * ops_length)
{
  hit_detector_pack_coords(path, coords, coords_length);
  hit_detector_pack_ops(path, ops, ops_length);
}
void
hit_detector_pack_coords (id<PSPath> path, 
			  struct floats_data * coords,
			  int * coords_length)
{
  int index = 0;
  
  const NSPoint * point;
  id<PointsEnumerating> points_enum = [path pointsEnumerator];
  int num_of_points  = [path countPoints];
  
  floats_data_expand (coords, 2*num_of_points);
  
  *coords_length = 2*num_of_points;
  while (point = [points_enum nextPoint], point)
    {
      (coords->data)[index++] = point->x;
      (coords->data)[index++] = point->y;
    }
}
void
hit_detector_pack_ops (id<PSPath> path, 
		       struct DPSUserPathOps_data * ops,
		       int * ops_length)
{
  int index = 0;

  PSSegment * seg;
  id<SegmentsEnumerating> seg_enum = [path segmentsEnumerator];
  int num_of_segments = [path countSegments];
  BOOL is_closed     = [path isClosedPath];
  if (YES == is_closed) num_of_segments++;
  
  *ops_length = num_of_segments;
  DPSUserPathOps_data_expand (ops, num_of_segments);
  while (seg = [seg_enum nextSegment], seg)
    (ops->data)[index++] = [seg dpsOperation];
  if (YES == is_closed) (ops->data)[index] = dps_closepath;
}
