/* GyveLayer.m --- The definition for layers

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "GyveLayer.h"
#include "GyveNil.h"
#include <Foundation/NSString.h>
#include <Foundation/NSException.h>
#include "PSFigObjGroup.h"
#include "PSMaskedFigObjGroup.h"
#include "PSProcessColor.h"
#include "PSPath.h"

NSString * GyveLayerNotWritableException = @"GyveLayerNotWritableException";

@implementation GyveLayer
/*
 * Layer ID 
 */
static NSMutableArray * layers_array = nil;
+ (int)issueLayerIDFor: (GyveLayer *)layer
{
  if (nil == layers_array)
    {
      layers_array  = [[NSMutableArray array] retain];
      [layers_array addObject: [GyveNil gyvenil]];
    }
  [layers_array addObject: layer];
  return [layers_array indexOfObject: layer];
}
+ (GyveLayer *)layerFromID:(int)layerID
{
  GyveLayer * result;
  if (nil == layers_array) return nil;	// Exception?
  if ([layers_array count] > layerID)
    {
      result = [layers_array objectAtIndex: layerID];
      if (result == [GyveNil gyvenil]) return nil;
      else return result;
    }
  else
    return nil;
}
- (int)layerID
{
  return layer_ID;
}

/*
 * Create and destory
 */
- initWithName: (NSString *)_name
{
  [super  init];
  printable = writable = visible = YES;
  name 				 = [_name copy];
  changed_since_last_drawing 	 = YES;
  layer_ID 			 = [GyveLayer issueLayerIDFor: self];

#define make_color(r,g,b) \
      ([[PSProcessColor alloc] initWithRed: color_element_from_char(r)\
			       green: color_element_from_char(g)\
			       blue: color_element_from_char(b)])
  switch (layer_ID%3)
    {
    case 0:
      color = make_color(79, 145, 255);
      break;
    case 1: 
      color = make_color(255, 79, 145);
      break;
    case 2:
      color = make_color(145, 255, 79);
      break;
    default:
      assert (0);
    }
  return self ;
}
- copyWithZone: (NSZone *)zone
{
  GyveLayer * layer = [[[self class] alloc] initWithName: name];
  [layer setPrintable: [self isPrintable]];
  [layer setWritable: [self isWritable]];
  [layer setVisible: [self isVisible]];
  [layer setLayerColor: [self layerColor]];
  return layer;
}
- mutableCopyWithZone: (NSZone *)zone
{
  GyveLayer * copy 		   = [super shallowCopy];
  copy->name 			   = [name retain];
  copy->writable 		   = writable;
  copy->visible 		   = visible;
  copy->printable 		   = printable;
  copy->color 			   = [color retain];
  copy->changed_since_last_drawing = YES;
  copy->layer_ID 		   = 0;
  return copy;
}
- (void)dealloc
{
  [name release], name 	 = nil;
  [color release], color = nil;
  layer_ID = 0;
  [super dealloc];
}

/*
 * Properties
 */
- (NSString *)layerName
{
  return name;
}
- (void)setLayerName: (NSString *)n
{
  [name release];
  name = [n copy];
}
- (id<PSColor>)layerColor
{
  return (id<PSColor>)color;
}
- (void)setLayerColor: (id<PSColor>)c
{
  if (c != color)
    {
      [color release];
      color = [c copy];
      [self changedSinceLastDrawing: YES];
    }
}
- setPrintable: (BOOL)b
{
  printable = b;
  return self ;
}
- (BOOL)isPrintable
{
  return printable;
}

- setWritable: (BOOL)b
{
  writable = b;
  return self ;
}
- (BOOL)isWritable
{
  return writable;
}
- setVisible: (BOOL)b
{
  if (b != visible) [self changedSinceLastDrawing: YES];
  visible = b;
  return self ;
}
- (BOOL)isVisible
{
  return visible;
}

/*
 * Operations on figobjs
 */
- (int)countFigObjs
{
  return [self count];
}
- (BOOL)containsFigObj: (id<PSFigObj>)figobj
{
  return [self containsObject: figobj];
}
- (int)indexOfFigObj: (id<PSFigObj>)figobj
{
  return [self indexOfObject: figobj];
}
- (id<PSFigObj>)figObjAtIndex: (int)n
{
  return (id<PSFigObj>)[self objectAtIndex: n];
}
- (void)removeFigObj: (id<PSFigObj>)figobj
{
  [self removeObject: figobj];
  [self changedSinceLastDrawing: YES];

}
- (void)removeFigObjAtIndex: (int)n
{
  [self removeObjectAtIndex: n];
  [self changedSinceLastDrawing: YES];
}
- (void)insertFigObj: (id<PSFigObj>)newfig atIndex: (int)n
{
  [self insertObject: newfig atIndex: n];
}
- (void)addFigObjOnTop: (id<PSFigObj>)newfig
{
  [self appendObject: newfig];
  [self changedSinceLastDrawing: YES];
}
- (int)countSelectedFigObjs
{
  int num_of_selections = 0;
  id<PSFigObj> target;
  int i, max;
  max = [self countFigObjs];
  for (i = 0; i < max ; i++)
    {
      target = [self figObjAtIndex: i];
      if ([target isSelected])
	num_of_selections++;
    }
  return num_of_selections;
}
- (int)countSelectedFigObjsBefore: (id<PSFigObj>)figobj
{
  BOOL found = NO;
  int num_of_selections = 0;
  id<PSFigObj> target;
  int i, max;
  max = [self countFigObjs];
  for (i = 0; i < max; i++)
    {
      target = [self figObjAtIndex: i];
      if (target == figobj)
	found = YES;
      if ((found == NO) && [target isSelected])
	num_of_selections++;
    }
  return num_of_selections;
}
- (id<FigObjsEnumerating>)figObjsEnumerator
{
  GYVE_SHOULD_NOT_IMPLEMENT(return nil);
}
- (id<FigObjsEnumerating>)reverseFigObjsEnumerator
{
  GYVE_SHOULD_NOT_IMPLEMENT(return nil);
}

- (id<FigObjsPrimitiveEnumerating>)reversePrimitiveFigObjsEnumerator
{
  GYVE_SHOULD_NOT_IMPLEMENT(return nil);
}
- (id<FigObjsPrimitiveEnumerating>)primitiveFigObjsEnumerator
{
  GYVE_SHOULD_NOT_IMPLEMENT(return nil);
}
- (id<PSFigObj>)firstFigObj
{
  return (id<PSFigObj>)[self firstObject];
}
- (id<PSFigObj>)lastFigObj
{
  return (id<PSFigObj>)[self lastObject];
}
- (id<PSFigObj>)firstPrimitiveFigObj
{
  id<PSFigObj>result = [self firstFigObj];

  if (result == nil) return nil;

  if (YES == [result isFigObjProxy])
    result = [result targetForFigObjProxy];

  while (NO == id_predicator_primitive_figobj(result))
    result = [result firstFigObj];
  return result;
}
- (id<PSFigObj>)lastPrimitiveFigObj
{
  id<PSFigObj>result = [self lastFigObj];

  if (result == nil) return nil;

  if (YES == [result isFigObjProxy])
    result = [result targetForFigObjProxy];

  while (NO == id_predicator_primitive_figobj(result))
    result = [result lastFigObj];
  return result;
}
- (NSRange)expandFigObj: (PSFigObjGroup *)figobj_group
{
  int i, max;
  int index = [self indexOfFigObj: [figobj_group shallowretain]];
  id<PSFigObj> figobj;
  NSRange result;
  [self removeObjectAtIndex: index];
  max = [figobj_group countFigObjs];
  for (i = 0; i < max; i++)
    {
      figobj = [figobj_group figObjAtIndex: i];
      [self insertObject: figobj atIndex: index + i];
    }

  if (YES == [figobj_group isKindOfClass: [PSMaskedFigObjGroup class]])
    {
      figobj = [(PSMaskedFigObjGroup *)figobj_group maskPath];
      [self insertObject: figobj atIndex: index + i];
      i++;
    }
  result.location = index;
  result.length   = i;
  [self changedSinceLastDrawing: YES];
  return result;
}
/*
 * Help for Drawing
 */
- (BOOL)isChangedSinceLastDrawing
{
  return changed_since_last_drawing;
}
- (void)changedSinceLastDrawing: (BOOL)flag
{
  changed_since_last_drawing = flag;
}

@end
