/* GyvePrinter.m --- 

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "GyvePrinter.h"
#include "GyveBuffer.h"
#include "GyveVisitor.h"
#include "PSPath.h"
#include "PSMaskedFigObjGroup.h"
#include "PSCompoundPaths.h"
#include "containing.h"
#include "PSSegment.h"

/*
 * Functions
 */
static void Ypushindent(FILE * fp, int * indent_offset);
static void Ystroke(FILE * fp, int * indent_offset);
static void Yfill(FILE * fp, int * indent_offset);
static void Yclip(FILE * fp, int * indent_offset);

static void Ycomment(FILE * fp, int * indent_offset, char * comment);
static void Ysetrgbcolor(FILE * fp, int * indent_offset, float r, float g, float b);
static void Ygsave(FILE * fp, int * indent_offset);
static void Ygrestore(FILE * fp, int * indent_offset);
static void Ypushpath(FILE * fp, int * indent_offset, id<PSPath>path);

@implementation GyveEPSPrinter
- initWithFilePath: (NSString *)path buffer: (GyveBuffer *)b
{
  [super initWithVisitingTarget: b];
  fp 	 	= fopen([path cString], "w");
  indent_offset = 0;
  if (fp == NULL)
    {
      DESTROY(self);
   }
  return self;
}
- (void)dealloc
{
  fclose(fp), fp = NULL;
  [super dealloc];
}
- (void)print
{
  [self printHeader];
  [self visitUpward];
  [self printFooter];
}
- (void)printHeader
{
  fprintf(fp, "%s\n", "%!PS-Adobe-3.0");
  fprintf(fp, "%s %s/%s-%s\n", "%%Creator:", 
	  [[[self class] description] cString],
	  PACKAGE, 
	  VERSION);
  fprintf(fp, "%s %d\n", "%%Pages:", 1);
  fprintf(fp, "%s\n", "%%EndComments");
  fprintf(fp, "%s\n", "%%BeginProlog");
  fprintf(fp, "%s\n", "%%EndProlog");
  fprintf(fp, "%s\n", "%%BeginSetup");
  fprintf(fp, "%s\n", "%%EndSetup");
  fprintf(fp, "%s %d %d\n", "%%Page:", 1, 1);
  fprintf(fp, "%s\n", "%%BeginPageSetup");
  fprintf(fp, "%s\n", "%%EndPageSetup");
  Ygsave(fp, &indent_offset);
  Ysetrgbcolor(fp, &indent_offset, 0.5, 0.5, 0.5);
}
- (void)printFooter
{
  Ygrestore(fp, &indent_offset);
  fprintf(fp, "showpage\n");
  fprintf(fp, "%s\n", "%%PageTrailer");
  fprintf(fp, "%s\n", "%%Trailer");
  fprintf(fp, "%s\n", "%%EOF");
}
- (enum gyve_visitor_request)visitAtProxy: (PSFigObjSelectedProxy *)proxy
{
  return visitor_stop;
}
- (enum gyve_visitor_request)visitAtBuffer:(GyveBuffer *)buffer
{
  Ycomment(fp, &indent_offset, [[buffer bufferName] cString]);
  return visitor_detail;
}
/*
 * Layer
 */ 
- (enum gyve_visitor_request)visitAtLayer:(GyveLayer *)layer
{
  if (YES == [layer isPrintable])
    {
      Ycomment(fp, &indent_offset, "Begin Layer");
      indent_offset += 2;
      return visitor_detail;
    }
  else
    {
      return visitor_continue;
    }
}
- (void)leaveFromLayer:(GyveLayer *)layer
{
  if (YES == [layer isPrintable])
    {
      indent_offset -= 2;
      Ycomment(fp, &indent_offset, "End Layer");
    }
}
/*
 * Path
 */
- (enum gyve_visitor_request)visitAtPath:(PSPath *)path
{
  Ycomment(fp, &indent_offset, "Begin Path");
  indent_offset += 2;
  Ypushpath(fp, &indent_offset, path);
  Ystroke(fp, &indent_offset);
  
  return visitor_continue;
}
- (void)leaveFromPath:(PSPath *)path
{
  indent_offset -= 2;
  Ycomment(fp, &indent_offset, "End Path");
}
/*
 * PSText
 */ 
- (enum gyve_visitor_request)visitAtText:(NSObject<PSText>*)text
{
  Ycomment(fp, &indent_offset, "Begin Text");
  return visitor_continue;
}
- (void)leaveFromText:(NSObject<PSText>*)text
{
  Ycomment(fp, &indent_offset, "End Text");
}
/*
 * Image
 */
- (enum gyve_visitor_request)visitAtImage:(PSImage *)image
{
  Ycomment(fp, &indent_offset, "Begin Image");
  return visitor_continue;
}
- (void)leaveFromImage:(PSImage *)image
{
  Ycomment(fp, &indent_offset, "End Image");
}
/*
 * Group
 */ 
- (enum gyve_visitor_request)visitAtFigObjGroup:(PSFigObjGroup *)group
{
  Ycomment(fp, &indent_offset, "Begin Group");
  indent_offset += 2;
  return visitor_detail; 
}
- (void)leaveFromFigObjGroup:(PSFigObjGroup *)group
{
 indent_offset -= 2;
  Ycomment(fp, &indent_offset, "End Group");
}
/* 
 * Masked Group->OK
 */
- (enum gyve_visitor_request)visitAtMaskedFigObjGroup:(PSMaskedFigObjGroup *)masked_group
{
  Ycomment(fp, &indent_offset, "Begin Masked Group");
  indent_offset += 2;
  Ygsave(fp, &indent_offset);
  Ypushpath(fp, &indent_offset, [masked_group maskPath]);
  Yclip(fp, &indent_offset);
  return visitor_detail;
}

- (void)leaveFromMaskedFigObjGroup:(PSMaskedFigObjGroup *)masked_group
{
  Ygrestore(fp, &indent_offset);
  indent_offset -= 2;
  Ycomment(fp, &indent_offset, "End Masked Group");
}

/*
 * Compound paths->OK
 */
-(enum gyve_visitor_request)visitAtCompoundPaths: (PSCompoundPaths *)compound_path
{
  Ycomment(fp, &indent_offset, "Begin Compound Paths");
  indent_offset += 2;
  Ypushpath(fp, &indent_offset, compound_path);
  Yfill(fp, &indent_offset);
  return visitor_continue;
}
- (void)leaveFromCompoundPaths: (PSCompoundPaths *)compound_paths
{
  indent_offset -= 2;
  Ycomment(fp, &indent_offset, "End Compound Paths");
}

- (void)leaveFromProxy: (PSFigObjSelectedProxy *)proxy
{
}
- (void)leaveFromBuffer:(GyveBuffer *)buffer
{
}
@end

static void
Ypushindent(FILE * fp, int * indent_offset)
{
  int i;
  for (i = 0; i < *indent_offset; i++)
    {
      fprintf(fp, " ");
    }
}
static void
Ygsave(FILE * fp, int * indent_offset)
{
  Ypushindent(fp, indent_offset);
  fprintf(fp, "%s\n", "gsave");
  *indent_offset += 2;
}

static void
Ygrestore(FILE * fp, int * indent_offset)
{
  *indent_offset -= 2;  
  Ypushindent(fp, indent_offset);
  fprintf(fp, "%s\n", "gsave");
}

static void
Ycomment(FILE * fp, int * indent_offset, char * comment)
{
  Ypushindent(fp, indent_offset);
  fprintf(fp, "%s --- %s ---\n", "%", comment);
}

static void
Ystroke(FILE * fp, int * indent_offset)
{
  Ypushindent(fp, indent_offset);
  fprintf(fp, "%s\n", "stroke");
}
static void
Yfill(FILE * fp, int * indent_offset)
{
  Ypushindent(fp, indent_offset);
  fprintf(fp, "%s\n", "eofill");
}

static void
Yclip(FILE * fp, int * indent_offset)
{
  Ypushindent(fp, indent_offset);
  fprintf(fp, "%s newpath\n", "clip");
}

static void
Ysetrgbcolor(FILE * fp, int * indent_offset, float r, float g, float b)
{
  Ypushindent(fp, indent_offset);
  fprintf(fp, "%f %f %f %s\n", r, g, b,  "setrgbcolor");
}

static void
Ypushpath(FILE * fp, int * indent_offset, id<PSPath>path)
{
  NSPoint * p;
  id<SegmentsEnumerating> e = [path segmentsEnumerator];
  PSSegment * seg;
  while (seg = [e nextSegment], seg)
    {
      Ypushindent(fp, indent_offset);
      switch ([seg dpsOperation])
	{
	case dps_moveto:
	  p = [seg pointAtIndex: 0];
	  fprintf(fp, "%f %f ", p->x, p->y);
	  fprintf(fp, "%s\n", "moveto");
	  break;
	case dps_lineto:
	  p = [seg pointAtIndex: 0];
	  fprintf(fp, "%f %f ", p->x, p->y);
	  fprintf(fp, "%s\n", "lineto");
	  break;
	case dps_curveto:
	  p = [seg pointAtIndex: 0];
	  fprintf(fp, "%f %f ", p->x, p->y);
	  p = [seg pointAtIndex: 1];
	  fprintf(fp, "%f %f ", p->x, p->y);
	  p = [seg pointAtIndex: 2];
	  fprintf(fp, "%f %f ", p->x, p->y);
	  fprintf(fp, "%s\n", "curveto");
	  break;
	}
    }
  if (YES == [path isClosedPath])
    {
      Ypushindent(fp, indent_offset);
      fprintf(fp, "%s\n", "closepath");
    }
}
