/* GyveSelectionsLayer.m --- The definition for selction special layer

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "GyveSelectionsLayer.h"
#include "GyveBuffer.h"
#include "utilities.h"
#include <stdio.h>

#include <Foundation/NSException.h>

#include <Foundation/NSString.h>

static NSString * PSFigObjSelectedProxyMultiModification
= @"PSFigObjSelectedProxyMultiModification";


@implementation  PSFigObjSelectedProxy
+ proxyForFigObj: (NSObject<PSFigObj> *)figobj 
       fromLayer: (GyveLayer *)layer
	ofBuffer: (GyveBuffer *)_buffer
{
  Class c = [figobj selectedProxyClass];
  if (Nil != c)
    {
      return [[[c alloc] initForFigObj: figobj 
			fromLayer: layer 
			ofBuffer: _buffer] 
	       autorelease];
    }
  else
    {
      return [[[self alloc] initForFigObj: figobj 
			    fromLayer: layer 
			    ofBuffer: _buffer]
	       autorelease];
    }
}
- (NSObject<PSFigObj> *)originalTargetForFigObjProxy
{
  if (backup)
    return backup;
  else
    return [self targetForFigObjProxy];
}
- (BOOL)isTargetFigObjIsChanged
{
  if (backup)
    return YES;
  else
    return NO;
}
- (GyveLayer *)layer
{
  return  layer_from;
}
- (GyveBuffer *)buffer;
{
  return buffer;
}
- (void)dealloc
{
  [layer_from release], layer_from = nil;
  [backup release], backup 	   = nil;
  buffer 			   = nil;
  [super dealloc];
}

- (void)reinit
{
  if (backup) [backup release], backup = nil;
}
- (NSObject<PSFigObj> *)beginModificationByCopy
{
  if (backup)
    {
      if ([backup isKindOfClass: [GyveLayer class]])
	[NSException raise: PSFigObjSelectedProxyMultiModification
		     format: @"Layer of modification target is already changed."];
      else
	[NSException raise: PSFigObjSelectedProxyMultiModification
		     format: @"Modification is already started."];
    }
  backup = target;
  target = [backup copy];
  [(id<PSFigObj>)target markAsSelected];
  return target;
}
- (void)beginModificationWithNewTarget: (BYREF NSObject<PSFigObj> *)new_figobj
{
  if (backup)
    {
      if ([backup isKindOfClass: [GyveLayer class]])
	[NSException raise: PSFigObjSelectedProxyMultiModification
		     format: @"Layer of modification target is already changed."];
      else
	[NSException raise: PSFigObjSelectedProxyMultiModification
		     format: @"Modification is already started."];
    }
  backup = target;
  target = [new_figobj retain];
  [target markAsSelected];
  return ;
}
- (void)endModification
{
  if (backup == nil || [backup isKindOfClass: [GyveLayer class]])
    [NSException raise: PSFigObjSelectedProxyMultiModification
		 format: @"Unbalanced modification block."];
  [buffer _replaceFigObj: (id<PSFigObj>)backup 
	  withFigObj: (id<PSFigObj>)target
	  atLayer: layer_from];
  [(id<PSFigObj>)target markAsSelected];
  [self reinit];
}
- (void)unSelect
{
  [self shallowretain];
  // calc bbox by yourself
  [target unMarkAsSelected];
  
  if (backup != nil && ![backup isKindOfClass: [GyveLayer class]])
    {
      [(id<PSFigObj>)backup unMarkAsSelected];
    }
  [[buffer selectionsLayer] removeFigObj: self];
  // memory management condes are requird here.
}
- (void)expand
{
  NSObject <PSFigObj> * local_target = [target shallowretain];
  GyveBuffer * local_buffer 	     = buffer;
  if (YES == [target isKindOfClass: [PSFigObjGroup class]])
    {
      [self unSelect];
      [local_buffer _expandFigObj: (PSFigObjGroup *)local_target];
    }
}
- (void)delete
{
  [[self retain] autorelease];
  [self unSelect];
  [buffer _deleteFigObjInternal: [(id<PSFigObj>)self targetForFigObjProxy]];
}
@end

@implementation PSFigObjSelectedProxy(Protected)
- initForFigObj:  (NSObject<PSFigObj> *)figobj 
      fromLayer:  (GyveLayer *)layer
       ofBuffer: (GyveBuffer *)_buffer
{
  [super initForFigObj: [figobj markAsSelected]];
  layer_from 	      = [layer retain];
  buffer 	      = _buffer;
  backup 	      = nil;
  return self ;
}
@end

@implementation GyveSelectionsLayer
- (PSFigObjSelectedProxy*)lookUpProxyForFigObj:(NSObject<PSFigObj> *)figobj
{
  PSFigObjSelectedProxy * figobj_proxy = nil;
  PSFigObjSelectedProxy * selected_figobj;

  if ([figobj isKindOfClass: [PSFigObjSelectedProxy class]])
    {
      if ([self containsObject: figobj])
	figobj_proxy = (PSFigObjSelectedProxy *)figobj;
      else
	figobj_proxy = [self lookUpProxyForFigObj: [figobj targetForFigObjProxy]];
    }
  else
    {
      int i, max;
      max = [self countFigObjs];
      for (i = 0; i < max; i++)
	{
	  selected_figobj = [self figObjAtIndex: i];
	  if (nil == figobj_proxy)
	    {
	      if ((figobj == [selected_figobj originalTargetForFigObjProxy])||
		  (figobj == [selected_figobj targetForFigObjProxy]))
		figobj_proxy = selected_figobj;
	    }
	}
    }
  return figobj_proxy;
}
- (void)calcBBox
{
  float delta;
  PSFigObjSelectedProxy * figobj_proxy = nil;
  id<PSFigObj> figobj;
  int i, max;
  [self zeroBBox];
  max = [self countFigObjs];
  for (i = 0; i < max; i++)
    {
      figobj_proxy = (PSFigObjSelectedProxy *)[self figObjAtIndex: i];
      figobj       =  [figobj_proxy targetForFigObjProxy];
      if (NO == bbox_is_zero_bbox([figobj bboxCStructure]))
	{
	  bbox_expand_by_bbox(&bbox, [figobj bboxCStructure]);
	  delta = [figobj deltaForExpandingBBox];
	  bbox_expand_by_delta(&bbox, delta);
	}
    }
}
- (struct bbox *)bboxCStructure
{
  [self calcBBox];
  return &bbox;
}
- (void)zeroBBox
{
  bbox = zero_bbox;
}
- (BOOL)isZeroBBox
{
  return bbox_is_zero_bbox([self bboxCStructure]);
}
- (BOOL)expandBBoxByBBox: (id<PSBBox>)b
{
  // TODO
  fprintf(stderr, 
	  "Don't use this method: [GyveSelectionsLayer expandBBoxByBBox]\n");
  return bbox_expand_by_bbox(&bbox, [b bboxCStructure]);
}
- (void)reportBBoxTo: (id<PSBBox>)b
{
  [b expandBBoxByBBox: self];
}
- (BOOL)isPointInBBox: (NSPoint *)p
{
  return bbox_contains_point([self bboxCStructure], p);
}
- (BOOL)intersectsWithBBox: (id<PSBBox>)b
{
  return bbox_intersects([self bboxCStructure], [b bboxCStructure]);
}
- (BOOL)intersectsWithRect: (NSRect *)r 
{
  struct bbox b;
  NSRect_to_bbox (r, &b);
  return bbox_intersects([self bboxCStructure], &b);
}
@end

@implementation GyveSelectionsLayer(GyveBuffer)
- init;
{
  [super initWithName: @"Selections Layer"];
  bbox = zero_bbox;
  printable = NO;
  writable  = YES;
  visible   = YES;
  return self ;
}
- (PSFigObjSelectedProxy*)putFigObj: (id<PSFigObj>)figobj
			  fromLayer: (GyveLayer *)layer
			   ofBuffer: (GyveBuffer *)buffer
			    atIndex: (int)index 
{
  id<PSFigObj> fig_proxy;
  fig_proxy = [PSFigObjSelectedProxy proxyForFigObj: figobj
				     fromLayer: layer
				     ofBuffer: buffer];
  [self insertFigObj: fig_proxy atIndex: index];
  return fig_proxy;
}
- (BOOL)beginModificationByCopyOfAll
{
  BOOL result = NO;
  PSFigObjSelectedProxy * selected_figobj;
  int i, max;
  max = [self countFigObjs];
  for (i = 0; i < max; i++)
    {
      selected_figobj = [self figObjAtIndex: i];
      if ([selected_figobj isTargetFigObjIsChanged] == NO)
	{
	  [selected_figobj beginModificationByCopy];
	  result = YES;
	}
    }
  return result;
}
- (BOOL)endModificationOfAll
{
  BOOL result = NO;
  PSFigObjSelectedProxy * selected_figobj;
  GyveLayer * copy_layer = [self mutableCopy];
  int i, max;
  max = [copy_layer countFigObjs];
  for (i = 0; i < max; i++)
    {
      selected_figobj = [copy_layer figObjAtIndex: i];
      if ([selected_figobj isTargetFigObjIsChanged] == YES)
	{
	  [selected_figobj endModification];
	  result = YES;
	}
    }
  [copy_layer release];
  return result;
}
- (void)unSelectAll
{
  PSFigObjSelectedProxy * selected_figobj;
  GyveLayer * copy = [self mutableCopy];
  int i, max;
  max = [copy countFigObjs];
  for (i = 0; i < max; i++)
    {
      selected_figobj = [copy figObjAtIndex: i];
      [selected_figobj unSelect];
    }
  [copy release];
}
@end
