/* GyveTextTool.m --- Text editing tool protocol 

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "GyveTextTool.h"
#include "GyveSelectionsLayer.h"
#include "PSText.h"
#include "PSTextElement.h"
#include "GyveHitDetector.h"
#include "GyveCanvas.h"
#include "GyveBuffer.h"
#include "TextStylePallet.h"
#include "GyveWindow.h"

#include <DPS/psops.h>
#include <gdk/gdkkeysyms.h>

static BOOL key_check(GdkEventKey *event);

@implementation  GyveTextTool
- init
{
  [super init];
  cursor_point 	     = NSZeroPoint;
  text_proxy 	     = nil;
  erasing_cursor_now = NO;
  return self ;
}
- (int)requestForDrawingEngine
{
  return 0;
}
- (void)buttonPressEvent: (GdkEventButton * )event
		onCanvas: (GyveCanvas *)canvas
		 atPoint: (NSPoint *)point 
	      withBuffer: (GyveBuffer *)buffer
{
  PSTextAtPoint * text;
  PSTextElement * elt;
  GyveHitDetector * hit_detector = [[[GyveHitDetector alloc]
				      initWithTarget: buffer] 
				     autorelease];

  dragged = NO;
  [canvas redrawLockRetain];
  [canvas redrawBBox: [buffer selectionsLayer] expandBy: 2.0];
  [buffer unSelectAll];
  
  if (YES == [hit_detector detectHitAroundPoint: point
			   withPredicator: id_predicator_text])
    {
      // Highlight range and insertion point setting up.
      text       = [hit_detector objectAtIndex: [hit_detector count] - 2];
      text_proxy = (PSTextSelectedProxy *)[buffer selectFigObj: text];
      elt 	 = [hit_detector objectAtIndex: [hit_detector count] - 1];
      base_location = [text indexOfTextElement: elt];
      [text_proxy setInsertionIndex: base_location];
    }
  else
    { 
      // New text
      if (nil != text_proxy)
	{
	  [self finishSessionOnCanvas: canvas withBuffer: buffer];
	}
      else
	{
	  [self eraseTextCursor];
	}
      
      if (nil != [buffer topLayerToWrite])
	{
	  cursor_point = *point;
	  [self drawTextCursor]; 
	}      
    }
  [canvas redrawLockRelease];
}
- (void)buttonReleaseEvent:(GdkEventButton *)event 
			onCanvas: (GyveCanvas *)canvas
			atPoint: (NSPoint *)point 
		withBuffer: (GyveBuffer *)buffer
{
  PSTextAtPoint * text;
  PSTextElement * elt;
  NSRange range;
  int release_location;
  GyveHitDetector * hit_detector;

  if (nil == text_proxy)
    {
      return;
    }

  text = (PSTextAtPoint*)[text_proxy targetForFigObjProxy];
  hit_detector  = [[[GyveHitDetector alloc]
		     initWithTarget: text] autorelease];
  if (NO == [hit_detector detectHitAroundPoint: point
			  withPredicator: id_predicator_text])
    {
      [text_proxy unSelect];
      text_proxy = nil;
      text = nil;
    }
  else if (YES == dragged)
    {
      elt = [hit_detector objectAtIndex: [hit_detector count] - 1];
      release_location = [text indexOfTextElement: elt];
      range 	       = NSRange_from_locations(release_location,
						base_location);
      [text_proxy setSelectionRange: &range];
      [canvas redrawFigObj: text]; 
    }
  else
    {
      [canvas redrawFigObj: text]; 
    }
  dragged = NO;
}
- (void)keyPressEvent:  (GdkEventKey *)event
	  ofCharacter: (int)character
	     onCanvas:  (GyveCanvas *)canvas
	   withBuffer: (GyveBuffer *)buffer
{
  PSTextAtPoint * text;
  int index;

  if (NO == key_check(event))
    {
      return;
    }

  // Create a new text object
  if (nil == text_proxy)
    {
      if (YES == NSPoint_is_zero_point(&cursor_point))
	{
	  return ;
	}
      
      text 	 = [[[PSTextAtPoint alloc] initWithPoint: &cursor_point]
		     autorelease];
      text_proxy = (PSTextSelectedProxy *)[buffer putFigObj: text];
      [text_proxy setInsertionIndex: 0];
    }
  else 
    {
      text = (PSTextAtPoint *)[text_proxy targetForFigObjProxy];
    }

  [canvas redrawLockRetain];
  if (event->keyval == GDK_BackSpace)
    {
      [canvas redrawFigObj: text]; 
      text = (PSTextAtPoint *)[text_proxy beginModificationByCopy];
      [text_proxy removeTextElements];
      [canvas redrawFigObj: text]; 
      [text_proxy endModification];
    }
  else if (event->keyval == GDK_rightarrow 
	   || event->keyval == GDK_leftarrow
	   || event->keyval == GDK_Left
	   || event->keyval == GDK_Right)
    {
      if (YES == [text_proxy hasInsertionIndex])
	{
	  [text_proxy clearSelectionRange];
	  index = [text_proxy insertionIndex];
	  if (event->keyval 	== GDK_leftarrow
	      || (event->keyval == GDK_Left))

	    {
	      index--;
	    }
	  else
	    {
	      index++;
	    }
	  if (index < 0)
	    {
	      index = 0;
	    }
	  else if (index > [text countTextElements])
	    {
	      index = [text countTextElements];
	    }
	  [text_proxy setInsertionIndex: index];
	  [self eraseTextCursor];
	  [self calcTextCursorPosition];
	  [self drawTextCursor];
	}
    }
  else
    {
      if ([text countTextElements] > 0)
	{
	  [canvas redrawFigObj: text];
	}
      text = (PSTextAtPoint *)[text_proxy beginModificationByCopy];
      [text_proxy insertTextElement: 
		    [PSTextElement textElementWithCharacter: character]];
      [canvas redrawFigObj: text];    
      [text_proxy endModification];
    }
  [self eraseTextCursor];

  [canvas redrawLockRelease];
}
- (void)startSessionOnCanvas: (GyveCanvas *)canvas 
		  withBuffer: (GyveBuffer *)buffer
{
}
- (void)finishSessionOnCanvas: (GyveCanvas *)canvas
		   withBuffer: (GyveBuffer *)buffer
{
  text_proxy 	   = nil;
  [self eraseTextCursor];
  cursor_point 	   = NSZeroPoint;
}
- (void)drawAfterAll
{
  if (YES == erasing_cursor_now || nil == text_proxy)
    {
      return ;
    }
  else
    {
      [self calcTextCursorPosition];
      [self drawTextCursor];
    }
}
- (void)calcTextCursorPosition
{
  PSTextAtPoint * text;
  PSTextElement * elt;
  struct  bbox * bbox;
  int i;
  text = (PSTextAtPoint *)[text_proxy targetForFigObjProxy];

  if (YES == [text_proxy hasInsertionIndex])
    {
      i = [text_proxy insertionIndex];
      if (i == 0)
	{
	  elt = nil;
	}
      else
	{
	  elt = [text textElementAtIndex: i - 1];
	}
    }
  else
    {
      elt = [text lastTextElement];
    }

  if (nil != elt)
    {
      bbox 	     = [elt bboxCStructure];
      if (NO == bbox_is_zero_bbox(bbox))
	{
	  cursor_point.x = bbox_element(bbox, BBOX_URX);
	  cursor_point.y = bbox_element(bbox, BBOX_LLY);
	}
    }
  else
    {
      NSPoint point = NSZeroPoint;
      matrix_apply_on_point([text mappingMatrix], &point);
      cursor_point.x = point.x;
      cursor_point.y = point.y;
    }
}
- (void)drawTextCursor
{
  float s = [[TextStylePallet textStylePallet] fontSize];
  if (YES == NSPoint_is_zero_point(&cursor_point))
    {
      return ;
    }
  PSgsave();
  PSsetlinewidth(2.0);
  PSmoveto(cursor_point.x, cursor_point.y);
  PSlineto(cursor_point.x, cursor_point.y+s);
  PSstroke();
  PSgrestore();  
}
- (void)eraseTextCursor
{
  NSRect rect;
  float s = [[TextStylePallet textStylePallet] fontSize];

  erasing_cursor_now = YES;
  rect.origin.x = cursor_point.x - 2.0;
  rect.origin.y = cursor_point.y - 1.0;
  rect.size.height = s + 1.0;
  rect.size.width  = 4.0;
  [[GyveWindow currentWindow] redrawRect: &rect];
  // cursor_point = NSZeroPoint;
  erasing_cursor_now = NO;
}

- (void)motionNotifyEvent: (GdkEventMotion *)event 
		 onCanvas: (GyveCanvas *)canvas
		  atPoint: (NSPoint *)point 
	       withBuffer: (GyveBuffer *)buffer
{
}
- (void)dragNotifyEvent: (GdkEventMotion *)event 
	       onCanvas: (GyveCanvas *)canvas
		atPoint: (NSPoint *)point 
	     withBuffer: (GyveBuffer *)buffer
{
  PSTextAtPoint * text;
  PSTextElement * elt;
  NSRange range;
  int release_location;
  GyveHitDetector * hit_detector;

  dragged = YES;

  if (nil == text_proxy)
    {
      return;
    }

  text = (PSTextAtPoint*)[text_proxy targetForFigObjProxy];
  hit_detector  = [[[GyveHitDetector alloc]
		     initWithTarget: text] autorelease];
  if (NO == [hit_detector detectHitAroundPoint: point
			  withPredicator: id_predicator_text])
    {
      return ;
    }
  else
    {
      elt = [hit_detector objectAtIndex: [hit_detector count] - 1];
      release_location = [text indexOfTextElement: elt];
      range = NSRange_from_locations(release_location,
				     base_location);
      [text_proxy setSelectionRange: &range];
      [canvas redrawFigObj: text]; 
    }
}
- (void)eraseClickPoint: (NSPoint *)p onCanvas: (GyveCanvas *)canvas{}
- (void)drawClickPoint: (NSPoint *)p onCanvas: (GyveCanvas *)canvas{}
@end

static
BOOL key_check(GdkEventKey *event)
{
  if (((event->length == 0) && (event->keyval == 65505))
      || ((event->length == 0) && (event->keyval == GDK_Control_L))
      || (event->state & GDK_CONTROL_MASK)
      || (event->state & GDK_MOD1_MASK))
    {
      return NO;
    }
  else
    {
      return YES;
    }
}
