/* GyveTool.m --- tool protocol 

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "GyveTool.h"
#include "geometry.h"
#include <math.h>
#include <gdk/gdkkeysyms.h>

@implementation GyveTool
- init
{
  [super init];
  layer       	 = nil;
  start_point 	 = NSZeroPoint;
  last_point  	 = NSZeroPoint;
  click_point 	 = NSZeroPoint;
  after_dragging = NO;
  shift   = NO;
  control = NO;
  mod1 	  = NO;
  return self ;
}
- (int)requestForDrawingEngine
{
  return tool_drawing_anchor_point;
}
- (void)buttonPressEvent: (GdkEventButton * )event
		onCanvas: (GyveCanvas *)canvas
		 atPoint: (NSPoint *)point 
	      withBuffer: (GyveBuffer *)buffer
{
  [self recordStartPoint: point];
  after_dragging = NO;
  shift   = ((event->state) & (GDK_SHIFT_MASK))?YES:NO;
  control = ((event->state) & (GDK_CONTROL_MASK))?YES:NO;
  mod1 = ((event->state) & (GDK_MOD1_MASK))?YES:NO;
}
- (void)buttonReleaseEvent:(GdkEventButton *)event 
			onCanvas: (GyveCanvas *)canvas
			atPoint: (NSPoint *)point 
		withBuffer: (GyveBuffer *)buffer
{
  control = ((event->state) & (GDK_CONTROL_MASK))?YES:NO;;
  shift = ((event->state) & (GDK_SHIFT_MASK))?YES:NO;;
  mod1 = ((event->state) & (GDK_MOD1_MASK))?YES:NO;
  if (YES  == after_dragging)
    {
      [self endDragNotifyEvent: event
	    onCanvas: canvas
	    atPoint: point
	    withBuffer: buffer];
    }
  else
    {
      [self clickNotifyEvent: event
	    onCanvas: canvas
	    atPoint: point
	    withBuffer: buffer];
      [self setClickPoint: point onCanvas: canvas];
    }
  [self updateLastPoint: point];
  after_dragging = NO;
}
- (void)motionNotifyEvent: (GdkEventMotion *)event 
		 onCanvas: (GyveCanvas *)canvas
		  atPoint: (NSPoint *)point 
	       withBuffer: (GyveBuffer *)buffer
{
  shift = ((event->state) & (GDK_SHIFT_MASK))?YES:NO;;
  control = ((event->state) & (GDK_CONTROL_MASK))?YES:NO;;
  mod1 = ((event->state) & (GDK_MOD1_MASK))?YES:NO;
  [self updateLastPoint: point];
}
- (void)dragNotifyEvent: (GdkEventMotion *)event 
	       onCanvas: (GyveCanvas *)canvas
		atPoint: (NSPoint *)point 
	     withBuffer: (GyveBuffer *)buffer
{
  shift = ((event->state) & (GDK_SHIFT_MASK))?YES:NO;;
  control = ((event->state) & (GDK_CONTROL_MASK))?YES:NO;;
  mod1 = ((event->state) & (GDK_MOD1_MASK))?YES:NO;
  if (YES == after_dragging)
    {
      [self draggingNotifyEvent:event 
		  onCanvas: canvas
		   atPoint: point 
	     withBuffer: buffer];
    }
  else
    {
      [self firstDragNotifyEvent: event
	    onCanvas: canvas
	    atPoint: point
	    withBuffer: buffer];
    }
  [self updateLastPoint: point];
  after_dragging = YES;
}
- (void)startSessionOnCanvas:(GyveCanvas *)canvas 
		  withBuffer: (GyveBuffer *)buffer
{
}
- (void)finishSessionOnCanvas:(GyveCanvas *)canvas 
		  withBuffer: (GyveBuffer *)buffer
{
}
- (void)keyPressEvent:  (GdkEventKey *)event
	  ofCharacter: (int)character
	     onCanvas:  (GyveCanvas *)canvas
	   withBuffer: (GyveBuffer *)buffer
{
}
- (void)drawAfterAll
{
  
}
- (void)recordStartPoint: (NSPoint *)point
{
  start_point = *point;
  [self updateLastPoint: point];
}
- (void)updateLastPoint:  (NSPoint *)point
{
  last_point = *point;
}
- (void)setClickPoint:   (NSPoint *)point onCanvas: (GyveCanvas *)canvas
{
  if ((NULL == point) && (nil != canvas))
    {
      [self eraseClickPoint: &click_point onCanvas: canvas];
      click_point = *point;
    }
  else if ((NULL != point) && (nil != canvas))
    {
      [self eraseClickPoint: &click_point onCanvas: canvas];
      click_point = *point;
      [self drawClickPoint: &click_point onCanvas: canvas];
    }
  else
    {
      click_point = *point;
    }
}
- (void)setLayer: (GyveLayer *)_layer
{
  layer = _layer;		// retain?
}
- (GyveLayer *)layer
{
  return layer;
}
- (id<PSColor>)layerColor
{
  if (nil != layer)
    {
      return [layer layerColor];
    }
  else
    {
      return nil;
    }
}
// Last
- (NSPoint *)lastPoint
{
  return &last_point;
}
- (NSRect)pointsRectFromPoint: (NSPoint *)p
{
  NSRect rect;
  NSPoint tmp;
  if (YES == shift)
    {
      tmp = [self pointsDiffFromPoint: p];
      NSPoint_add(&tmp, p, &tmp);
      NSRect_from_points(&rect, &tmp, p);
    }
  else
    {
      NSRect_from_points(&rect, p, [self lastPoint]);
    }
  return rect;
}
- (NSRect)pointsCenteredRectFromPoint: (NSPoint *)p
{
  NSPoint vector     = [self pointsDiffFromPoint: p];
  NSPoint base_point = *p;

  NSRect new_rect;
  vector.x = fabs(vector.x);
  vector.y = fabs(vector.y);

  new_rect.origin.x   = base_point.x - vector.x;
  new_rect.origin.y   = base_point.y - vector.y;
  
  new_rect.size.width = 2 * vector.x;
  new_rect.size.height = 2 * vector.y;
  return new_rect;
}
- (NSPoint)pointsDiffFromPoint: (NSPoint *)p
{
  NSPoint result;
  float distance;
  NSPoint tmp;
  if (YES == shift)
    {
      distance = NSPoint_distance([self lastPoint], p);
      distance *= cos(M_PI_4);	// Defined in math.h
      NSPoint_diff([self lastPoint], p, &tmp);
      result.x = (tmp.x < 0.0)? -distance:distance;
      result.y = (tmp.y < 0.0)? -distance:distance;
    }
  else
    {
      NSPoint_diff([self lastPoint], p, &result);
    }
  return result;
}
// Start
- (NSPoint *)startPoint
{
  return &start_point;
}
- (NSRect)pointsRectFromStart
{
  return [self pointsRectFromPoint: [self startPoint]];
}
- (NSRect)pointsCenteredRectFromStart
{
  return [self pointsCenteredRectFromPoint: [self startPoint]];
}
- (NSPoint)pointsDiffFromStart
{
  return [self pointsDiffFromPoint: [self startPoint]];
}
// Click
- (NSPoint *)clickPoint
{
  return &click_point;
}
- (NSRect)pointsRectFromClick
{
  return [self pointsRectFromPoint: [self clickPoint]];
}
- (NSRect)pointsCenteredRectFromClick
{
  return [self pointsCenteredRectFromPoint: [self clickPoint]];
}
- (NSPoint)pointsDiffFromClick
{
  return [self pointsDiffFromPoint: [self clickPoint]];
}
- (void)drawClickPoint: (NSPoint *)p onCanvas: (GyveCanvas *)canvas
{
}
- (void)eraseClickPoint: (NSPoint *)p onCanvas: (GyveCanvas *)canvas
{
}
- (float)pointsAngleForPoint:  (NSPoint *)p
{
  float dx1, dx2, dy1, dy2, radian;
  dx1 = last_point.x - click_point.x;
  dx2 = p->x - click_point.x;
  dy1 = last_point.y - click_point.y;
  dy2 = p->y - click_point.y;
  {
    float dx, dy, dx0, dy0, COS, SIN;
    dx = dx1;
    dy = dy1;
    dx0 = dx2;
    dy0 = dy2;
    COS = (dx*dx0+dy*dy0)/(sqrt(dx*dx + dy*dy)*sqrt(dx0*dx0 + dy0*dy0));
    SIN = (dx*dy0)-(dx0*dy);
    radian = acos(COS);
    if (SIN < 0.0) radian = - radian;
  }
  return radian;
}
  /*  @ click
      * last
      o current

            @ 
            |                 
            |
            |
            |      o 
            |      |          
            |   *  |
            |   |  |
      ------+---+--+----> X
	    d1 d2           

	    (d1+d2)/d1 (X)Ψ(mag) */

- (float)pointsXScaleForPoint:  (NSPoint *)p
{
  float dx1, dx2, mag_x;
  dx1 = last_point.x - click_point.x;
  dx2 = p->x - last_point.x;
  if (YES == float_zero(dx1))
    {
      return 0.0;
    }
  mag_x = (dx1 + dx2)/dx1; 
  if (YES == float_zero(mag_x))
    {
      return 0.0;
    }
  else
    {
      return mag_x;
    }
}
- (float)pointsYScaleForPoint:  (NSPoint *)p
{
  float dy1, dy2, mag_y;
  dy1 = last_point.y - click_point.y;
  dy2 = p->y - last_point.y;
  if (YES == float_zero(dy1))
    {
      return 0.0;
    }
  mag_y = (dy1 + dy2)/dy1; 
  if (YES == float_zero(mag_y))
    {
      return 0.0;
    }
  else
    {
      return mag_y;
    }
}

- (void)firstDragNotifyEvent: (GdkEventMotion *)event 
	       onCanvas: (GyveCanvas *)canvas
		atPoint: (NSPoint *)point 
		  withBuffer: (GyveBuffer *)buffer {}
- (void)draggingNotifyEvent: (GdkEventMotion *)event 
		  onCanvas: (GyveCanvas *)canvas
		   atPoint: (NSPoint *)point 
		withBuffer: (GyveBuffer *)buffer {}
- (void)endDragNotifyEvent: (GdkEventButton *)event
	       onCanvas: (GyveCanvas *)canvas
		atPoint: (NSPoint *)point 
		withBuffer: (GyveBuffer *)buffer {}
- (void)clickNotifyEvent: (GdkEventButton *)event 
		onCanvas: (GyveCanvas *)canvas
		 atPoint: (NSPoint *)point 
	      withBuffer: (GyveBuffer *)buffer {}

@end
