/* GyveView.m --- The definition for gyve view

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "GyveView.h"
#include "geometry.h"
#include "PSPath.h"
#include "PSPaintStyle.h"
#include "init.h"

#include <Foundation/NSGeometry.h>
#include <Foundation/NSException.h>

#include <DPS/dpsops.h>
#include <DPS/psops.h>

#include <math.h>

#include <gtkDPS/gtkDPS.h>
#include <gtk/gtk.h>
#include <gdk/gdktypes.h>
/*
 * Coord translation related functions
 */ 
#include "coord_tr.h"	// generated from coord_tr.psw
#define DGS_BUGGY 1

void
coord_tr_get(struct coord_tr * tr, DPSContext ctxt)
{
  psw_get_transform(ctxt, tr->ctm, tr->invctm,
		    &(tr->x_offset), &(tr->y_offset));
}

void
coord_tr_copy(struct coord_tr * src, struct coord_tr * dist)
{
  int matrix_elt_index;

  dist->updated = src->updated;
  for (matrix_elt_index = 0; matrix_elt_index < 6; matrix_elt_index++)
    {
      dist->ctm[matrix_elt_index] = src->ctm[matrix_elt_index];
      dist->invctm[matrix_elt_index] = src->invctm[matrix_elt_index];
    }
  dist->x_offset = src->x_offset;
  dist->y_offset = src->y_offset;
}

void
coord_tr_x2dps(struct coord_tr * tr,
	    int in_x, int in_y, float * out_x, float * out_y)
{
  in_x -= tr->x_offset;
  in_y -= tr->y_offset;

  *out_x =
    tr->invctm[MATRIX_A] * (float)in_x +
    tr->invctm[MATRIX_C] * (float)in_y +
#ifdef DGS_BUGGY
    0.0;
#else 
    tr->invctm[MATRIX_TX];
#endif /* Def: DGS_BUGGY */

  *out_y =
    tr->invctm[MATRIX_B] * (float)in_x +
    tr->invctm[MATRIX_D] * (float)in_y +
#ifdef DGS_BUGGY
    0.0;
#else 
    tr->invctm[MATRIX_TY];
#endif  /* DGS_BUGGY */
}

void
coord_tr_dps2x(struct coord_tr * tr, 
	       float in_x, float in_y, 
	       int * out_x, int * out_y)
{
  *out_x =
    tr->ctm[MATRIX_A] * in_x +
    tr->ctm[MATRIX_C] * in_y +
#ifdef DGS_BUGGY
    0.0 +
#else 
    tr->ctm[MATRIX_TX] + 
#endif  /* DGS_BUGGY */
    tr->x_offset;
  *out_x = (*out_x < 0)? floor(*out_x): *out_x;	// 0?

  *out_y =
    tr->ctm[MATRIX_B] * in_x +
    tr->ctm[MATRIX_D] * in_y +
#ifdef DGS_BUGGY
    0.0 +
#else 
    tr->ctm[MATRIX_TY] + 
#endif /* Def: DGS_BUGGY */
    tr->y_offset;
  *out_y = (*out_y < 0)? floor(*out_y): *out_y;	// 0?
}

void
coord_tr_dpsRect2xRect(struct coord_tr * tr, 
		       NSRect * in_dps_rect, 
		       NSRect * out_x_rect)
{
  /*      
    a_dps     a_x 
     *---+      *---+
     |DPS|  =>  | X |  
     +---*      +---*
        b_dps     b_x */    
  NSPoint a_dps, b_dps;
  NSPoint a_x_float, b_x_float;
  int a_x[2], b_x[2];
  int x = 0;
  int y = 1;

  if ((tr == NULL) || (in_dps_rect == NULL) ||  (out_x_rect == NULL))
    {
      fprintf(stderr, "Wrong args (coord_tr_dpsRect2xRect)\n");
      return ;
    }
  a_dps.x = in_dps_rect->origin.x;
  a_dps.y = in_dps_rect->origin.y + in_dps_rect->size.height;

  b_dps.x = in_dps_rect->origin.x + in_dps_rect->size.width;
  b_dps.y = in_dps_rect->origin.y;

  coord_tr_dps2x(tr, 
		 a_dps.x, a_dps.y,
		 &(a_x[x]), &(a_x[y]));
  coord_tr_dps2x(tr, 
		 b_dps.x, b_dps.y,
		 &(b_x[x]), &(b_x[y]));
  
  a_x_float.x = a_x[x];
  a_x_float.y = a_x[y];
  b_x_float.x = b_x[x];
  b_x_float.y = b_x[y];
  
  NSRect_from_points(out_x_rect, &a_x_float, &b_x_float);
}

/*
 * GyveView
 */

NSString * GyveViewWrongGdkEventToTranslate = @"GyveViewWrongGdkEventToTranslate";

@implementation GyveView
- initWithViewWidth: (int)w_x height: (int)h_x
{
  [super init];
  document_size_x.width = w_x;
  document_size_x.height = h_x;
  widget = (GtkWidget * )gtk_dps_area_new();
  gtk_dps_area_size(GTK_DPS_AREA(widget), w_x, h_x);
  drawing_engine     = nil;
  coord_tr.updated = NO;
  total_redraw_rect = NSZeroRect;
  [self setupSignalHandler];

  redraw_lock_depth = 0;
  receiving_expose_events_sequence = NO;

  return self ;
}
- setDrawingEngine: (NSObject<GyveDrawingEngine>*)e
{
  float scale = 1.0;
  if (e && (e != drawing_engine))
    {
      if (drawing_engine)
	{
	  scale = [self absoluteScale];
	  [drawing_engine release], drawing_engine = nil;
	}
      drawing_engine = [e retain];
      if ([self dpsContextBegin])
	{
	  //
	  [drawing_engine setAbsoluteScale: scale];
	  // 
	  // [drawing_engine installPSIntializeCode];
	}
      [self dpsContextEnd];
    }
  return self ;
}
- (void)dealloc
{
  if (drawing_engine)[drawing_engine release], drawing_engine = nil;
  gtk_widget_unref(GTK_WIDGET(widget));
  [super dealloc];
}
- (DPSContext)dpsContextBegin
{
  /* TODO: recursive entriesػߤ뤳. */
  DPSContext current_ctxt = NULL;
  if (GTK_WIDGET_REALIZED(widget))
    {
      gtk_dps_area_begin(GTK_DPS_AREA (widget));
      current_ctxt =  gtk_dps_get_current_raw_context();
    }
  return current_ctxt;
}
- (void)dpsContextEnd
{
  if (GTK_WIDGET_REALIZED(widget))
    gtk_dps_area_end(GTK_DPS_AREA (widget));
}
- (void)setAbsoluteScale: (float)scale
{
  NSSize size 	  = [self documentSizeInX];
  float old_scale = [self absoluteScale];
  size.width 	  /= old_scale;
  size.height 	  /= old_scale;
  gtk_widget_set_usize(widget,
		       size.width * scale + (40 * scale),
		       size.height * scale + (40 * scale)); 
  if (drawing_engine && [self dpsContextBegin])
        [drawing_engine setAbsoluteScale: scale];
  [self dpsContextEnd];
  
  coord_tr.updated = NO;
  [self updateCoordTr];
}
- (float)absoluteScale
{
    return [drawing_engine absoluteScale];
}
- (NSObject<GyveDrawingEngine>*)drawingEngine
{
    return drawing_engine;
}
- (void)redrawAll
{
  gint w_x, h_x;
  float X0, Y0, X1, Y1;
  GdkDrawable * drawable;
  NSRect redraw_rect;

  if (nil == drawing_engine) return ;

  drawable =GTK_DPS_AREA(widget)->dps_widget.widget.window;
  if (drawable)
    {
      gdk_window_get_size(drawable, &w_x, &h_x);
      if (NO == coord_tr.updated) [self updateCoordTr];
      coord_tr_x2dps(&coord_tr, 0, 0, &X0, &Y0) ;
      coord_tr_x2dps(&coord_tr, 0+w_x, 0+h_x, &X1, &Y1);      
      redraw_rect = NSMakeRect(X0, Y1, (X1 - X0), (Y0 - Y1));
      [self redrawRect: &redraw_rect];
    }
}
- (void)redrawRect: (const NSRect *)rect
{
  if (nil == drawing_engine) return ;
  if (rect)
    {
      if (NSRect_is_zero_rect(rect)) return ;
	
      /* rect total_redraw_rect  */
      if (NSRect_is_zero_rect(&total_redraw_rect))
	{
	  total_redraw_rect = *rect;
	}
      else
	{
	  total_redraw_rect = NSUnionRect(*rect,
					  total_redraw_rect);
	}
    }

  if (redraw_lock_depth == 0)
    {
      if ([self dpsContextBegin])
	{
	  /* total_redraw_rect  */
	  [drawing_engine redrawRect: &total_redraw_rect];
	  /* ΰ򥯥ꥢ. */
	  total_redraw_rect = NSZeroRect;
	}
      [self dpsContextEnd];
    }
}
- (void)redrawRect: (const NSRect *)rect expandBy: (float)delta
{
  if (YES == NSRect_is_zero_rect(rect)) return ;
  if (delta == 0.0) 
    [self redrawRect: rect];
  else
    {
      NSRect expanded_rect;
      NSRect_copy(rect, &expanded_rect);
      NSRect_expand_by_delta (&expanded_rect, delta);
      [self redrawRect: &expanded_rect];
    }
}
- (void)redrawBBox: (id<PSBBox>)bbox
{
  NSRect rect;
  bbox_to_rect([bbox bboxCStructure], &rect);
  if (YES == NSRect_is_zero_rect(&rect)) return ;  

  [self redrawRect: &rect];
}
- (void)redrawBBox: (id<PSBBox>)bbox  expandBy: (float)delta
{
  NSRect rect;
  bbox_to_rect([bbox bboxCStructure], &rect);

  if (YES == NSRect_is_zero_rect(&rect)) return ;    

  [self redrawRect: &rect expandBy: delta];
}
- (void)redrawBBoxCStructure: (struct bbox *)bbox 
{
  NSRect rect;
  bbox_to_rect(bbox, &rect);
  if (YES == NSRect_is_zero_rect(&rect)) return ;  
  [self redrawRect: &rect];
}
- (void)redrawBBoxCStructure: (struct bbox *)bbox expandBy: (float)delta
{
  NSRect rect;
  bbox_to_rect(bbox, &rect);
  if (YES == NSRect_is_zero_rect(&rect)) return ;  
  [self redrawRect: &rect expandBy: delta];
}
- (void)redrawFigObj: (id<PSFigObj>)figobj
{
  if (bbox_is_zero_bbox([figobj bboxCStructure]))
    {
      /* bboxzeroξ, 
	 figobjbbox׻ǤƤʤ, ޷Ǥ뤳Ȥ̣Ƥ. */
      // calc bbox
      if (YES == [self calcBBoxOfFigObj: figobj])
	[self redrawFigObj: figobj];
      else
	return ;
    }
  else if ([drawing_engine drawingMode] == PREVIEW_MODE)
    {
      float delta = 0.0;
      delta = [figobj deltaForExpandingBBox];
      [self redrawBBox: figobj expandBy: delta];
    }
  else
    {
      [self redrawBBox: figobj expandBy: 2.0];
    }
}

- redrawLockRetain
{
  redraw_lock_depth++;
  return self ;
}
- (void)redrawLockRelease
{
  redraw_lock_depth--;
  [self redrawRect: NULL];
}
- (int)redrawLockCount
{
  return redraw_lock_depth;
}
- (void)enableImageCache
{
  if (drawing_engine)
    [drawing_engine imageCacheRequestWithArea: GTK_DPS_AREA(widget)
		    coordTr: &coord_tr];
}
- (void)disableImageCache
{
  if (drawing_engine)
    [drawing_engine imageCacheRequestWithArea: NULL
		    coordTr: &coord_tr];
}

- (NSPoint)translatePointFromXToDPS: (NSPoint *)xpoint
{
  NSPoint dps_point;
  int x_x = (int)(xpoint->x), y_x = (int)(xpoint->y);
  
  coord_tr_x2dps(&coord_tr, 
		 x_x, y_x,
		 &(dps_point.x), &(dps_point.y));
  
  return dps_point;
}
- (NSPoint)translatePointFromGdkEventToDPS: (GdkEvent *)event
{
  NSPoint xpoint;
  switch(event->type)
    {
    case GDK_MOTION_NOTIFY:
      xpoint.x = event->motion.x ;
      xpoint.y = event->motion.y ;
      break;
    case GDK_BUTTON_PRESS:
    case GDK_2BUTTON_PRESS:
    case GDK_3BUTTON_PRESS:
    case GDK_BUTTON_RELEASE:
      xpoint.x = event->button.x ;
      xpoint.y = event->button.y ;
      break;
    default:
      [NSException raise: GyveViewWrongGdkEventToTranslate
		   format: @"Wrong type of gdkEvent: %d", event->type];
    }
  return [self translatePointFromXToDPS: &xpoint];
}
- (NSRect)translateRectFromXToDPS: (NSRect *)rect_x
{
  NSRect rect_dps;
  NSPoint start_x, end_x;
  NSPoint start_dps, end_dps;

  start_x = rect_x->origin;
  end_x   = NSRect_end(rect_x);

  start_dps = [self translatePointFromXToDPS: &start_x];
  end_dps   = [self translatePointFromXToDPS: &end_x];

  NSRect_from_points(&rect_dps, &start_dps, &end_dps);
  return rect_dps;
}
- (NSSize)translateSizeFromXToDPS: (NSSize *)size_x
{
  NSRect rect_x;
  NSRect_from_NSSize(&rect_x, size_x);
  return [self translateRectFromXToDPS: &rect_x].size;
}

- (NSSize)dpsAreaSize
{
  NSRect rect_x;
  NSRect rect_dps;

  NSRect_from_gtkAllocation(&rect_x, &(widget->allocation));
  rect_dps = [self translateRectFromXToDPS: &rect_x];
  return rect_dps.size;
}

- (NSSize)documentSize
{
  if (drawing_engine)
    return [drawing_engine documentSize];
  else
    return NSZeroSize;
}
- (NSSize)dpsAreaSizeInX
{
  NSSize size;
  NSSize_from_gtkAllocation(&size, &(widget->allocation));
  return size;
}
- (NSSize)documentSizeInX
{
  NSSize result;
  float scale = [drawing_engine absoluteScale];
  result.width = document_size_x.width * scale;
  result.height = document_size_x.height * scale;
  return result;
}
@end

static void expose_handler(GtkWidget *widget, 
			   GdkEventExpose *event, 
			   gpointer data);
static void install_expose_handler(GtkWidget *widget, 
				   GdkEventExpose *event, 
				   gpointer data);


static void realize_handler(GtkWidget *widget, gpointer data);

static void size_allocate_handler(GtkWidget      *widget,
				  GtkAllocation  *allocation, 
				  gpointer data);

static void enter_notify_handler(GtkDPSContext *gtk_dps_context,
				 GdkDrawable *gdk_drawable,
				 gpointer data);

static void leave_notify_handler(GtkDPSContext *gtk_dps_context,
				 GdkDrawable *gdk_drawable,
				 gpointer data);

/*
 * Cateogry for subclass
 */
@implementation GyveView(Protected)
- (void)changeAreaSizeWidth: (int)w_x height: (int)h_x
{
  NSPoint offset;
  NSSize tmp;
  NSSize area_size, doc_size;
  coord_tr.updated = NO;
  [self updateCoordTr];
  area_size = [self dpsAreaSizeInX];
  doc_size  = [self documentSizeInX];
  NSSize_diff(&area_size, &doc_size,  &tmp);
  offset.x = tmp.width/2.0;
  offset.y = tmp.height/2.0;
  offset.y = area_size.height - offset.y;
  [self setXoffsetX: (int)offset.x Y: (int)offset.y];
}
- (void)setXoffsetX: (int)x_x Y: (int)y_x
{
  if ([self dpsContextBegin])
    PSsetXoffset(x_x, y_x);
  [self dpsContextEnd];

  coord_tr.updated = NO;
  [self updateCoordTr];
}
- (void)updateCoordTr
{
  DPSContext ctxt = [self dpsContextBegin];
  if (ctxt)
    {
      coord_tr_get(&coord_tr, ctxt);
      if (drawing_engine) [drawing_engine coordTrChangesTo: &coord_tr];
      coord_tr.updated = YES;
      [self calcDocumentSize];
    }
  else
    {
      coord_tr.updated = NO;
    }
  [self dpsContextEnd];
}
- (void)setupSignalHandler
{
  gint old_mask = gtk_widget_get_events(GTK_WIDGET(widget));
  gint new_mask = GDK_EXPOSURE_MASK;
  gtk_widget_set_events(GTK_WIDGET(widget), old_mask| new_mask);
  gtk_signal_connect(GTK_OBJECT(widget), "expose_event",
		     GTK_SIGNAL_FUNC (install_expose_handler),
		     self);

  gtk_signal_connect(GTK_OBJECT(widget), "realize",
		     GTK_SIGNAL_FUNC (realize_handler),
		     self);
}
- (void)exposeEvent: (GdkEventExpose*)e
{
  NSPoint a, b, xpoint;
  NSRect redraw_rect;

  if (nil == drawing_engine) return ;
  if (NO == coord_tr.updated) [self updateCoordTr];

  /* GdkEventExpose e褹ΰDPSκɸϤǷ׻. */ 
  xpoint.x = (float)(e->area.x);
  xpoint.y = (float)(e->area.y);
  a = [self translatePointFromXToDPS: &xpoint];

  xpoint.x = (float)(e->area.x + e->area.width);
  xpoint.y = (float)(e->area.y + e->area.height);
  b = [self translatePointFromXToDPS: &xpoint];

  NSRect_from_points(&redraw_rect, &a, &b);
  
  if (e->count > 0)
    {
      /* eΤȤ, ޤ EventExpose ³, ΰ򵭲, 
	 褷ʤ. */
      if (NO == receiving_expose_events_sequence)
	{
	  /* Ϣ³expoise٥ȤκǽǤ, 
	     redrawå, expoise٥νǤ
	     Ȥ򼫳Ф. */
	  [self redrawLockRetain];
	  receiving_expose_events_sequence = YES;
	}
      [self redrawRect: &redraw_rect];
    }
  else
    {
      if (YES == receiving_expose_events_sequence)
	{
	  /* 1İʾevent exposeκǸΥ٥. 
	     ׻ΰä, ޤޤǷ׻ΰ
	     ̿ȯԤ. */
	  [self redrawRect: &redraw_rect];
	  [self redrawLockRelease];
	  receiving_expose_events_sequence = NO;
	}
      else
	{
	  /* total_redraw_rectzeroΤȤ, e EventExpose
	     ǤϤʤ. ʤñȤEventExposeǤ, 
	     ׻ΰ褹ɬפ. */
	  [self redrawRect: &redraw_rect];
	}
    }
}
- (void)enterContextNotify: (DPSContext)ctxt
{
  if (coord_tr.updated == NO)
    {
      coord_tr_get(&coord_tr, ctxt);
      coord_tr.updated = YES;      
    }
  if (ctxt)
    {
      float scale;
      NSPoint offset;
      NSSize tmp;
      NSSize area_size, doc_size;
      /* 
       * Q.ؿƤӽФν(offset->scale)ϤǤ?
       * Q. scaleΰ, absolute_magnitude򻲾ȤʤƤ
       *    c_trФϤ. 
       */
      scale = [self absoluteScale];
      area_size = [self dpsAreaSizeInX];
      doc_size  = [self documentSizeInX];
      NSSize_diff(&area_size, &doc_size,  &tmp);
      offset.x = tmp.width/2.0;
      offset.y = tmp.height/2.0;
      offset.y = area_size.height - offset.y;
      DPSsetXoffset(ctxt, (int)offset.x, (int)offset.y);
      DPSscale(ctxt, scale, scale);
    }
}
- (GtkWidget *)dpsAreaWidget
{
  return widget;
}
- (void)calcDocumentSize
{
  NSSize tmp;
  NSSize size = [self documentSizeInX];
  if (drawing_engine)
    {
      tmp = [self translateSizeFromXToDPS: &size];
      [drawing_engine setDocumentSize: &tmp];
    }
}
- (BOOL)calcBBoxOfFigObj: (id<PSFigObj>)figobj
{
  if (drawing_engine)
    return [drawing_engine calcBBoxOfFigObj: figobj];
  else
    return NO;
}
@end

/*
 * Signal Handler
 */
static void
install_expose_handler (GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
  void * this_func = install_expose_handler;
  gtk_signal_disconnect_by_func(GTK_OBJECT(widget),
				GTK_SIGNAL_FUNC(this_func), 
				data);
  gtk_signal_connect(GTK_OBJECT(widget), "expose_event",
		     GTK_SIGNAL_FUNC (expose_handler),
		     data);
  {
    GtkDPSArea * gtkObject;
    GtkDPSWidget dps_widget;
    GtkObject * dps_ctxt_obj;
    float scale;
    NSPoint offset;
    NSSize tmp;
    NSSize area_size, doc_size;
    DPSContext ctxt;

    gtkObject = GTK_DPS_AREA(widget);
    dps_widget = *GTK_DPS_WIDGET(&(gtkObject->dps_widget));
    dps_ctxt_obj = GTK_OBJECT(dps_widget.gtk_dps_context);
    ctxt = *(((GtkDPSContext*)dps_ctxt_obj)->
			gdk_dps_context->
			dps_context);
    
    /* 
     * Q.ؿƤӽФν(offset->scale)ϤǤ?
     * Q. scaleΰ, absolute_magnitude򻲾ȤʤƤ
     *    c_trФϤ. 
     */
    scale = [(GyveView *)data absoluteScale];
    area_size = [(GyveView *)data dpsAreaSizeInX];
    doc_size  = [(GyveView *)data documentSizeInX];
    NSSize_diff(&area_size, &doc_size,  &tmp);
    offset.x = tmp.width/2.0;
    offset.y = tmp.height/2.0;
    offset.y = area_size.height - offset.y;
    DPSsetXoffset(ctxt, (int)offset.x, (int)offset.y);
    DPSscale(ctxt, scale, scale);
    DPSWinstall_procs(ctxt);
    gtk_signal_connect(dps_ctxt_obj,
		       "enter_notify",
		       GTK_SIGNAL_FUNC (enter_notify_handler),
		       data);
    gtk_signal_connect(dps_ctxt_obj,
		       "re_enter_notify",
		       GTK_SIGNAL_FUNC (enter_notify_handler),
		       data);
    gtk_signal_connect(dps_ctxt_obj,
		       "leave_notify",
		       GTK_SIGNAL_FUNC (leave_notify_handler),
		       data);
    gtk_signal_connect(GTK_OBJECT(widget), "size_allocate",
		       GTK_SIGNAL_FUNC (size_allocate_handler),
		       data);
  }
  [(GyveView *)data updateCoordTr];
  [(GyveView *)data redrawAll];
}

static void
expose_handler (GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
  [(GyveView *)data exposeEvent: event];
}

static void
realize_handler(GtkWidget *widget, gpointer data)
{
  [(GyveView *)data updateCoordTr];
}

static void
size_allocate_handler(GtkWidget      *widget,
		      GtkAllocation  *allocation, 
		      gpointer data)
{
  [(GyveView *)data changeAreaSizeWidth: allocation->width
	       height: allocation->height];
}

static void
enter_notify_handler(GtkDPSContext *gtk_dps_context,
		     GdkDrawable *gdk_drawable,
		     gpointer data)
{
  DPSContext ctxt = *(((GtkDPSContext*)gtk_dps_context)->
		      gdk_dps_context->
		      dps_context);
  [(GyveView *)data enterContextNotify: ctxt]; 
}

static void
leave_notify_handler(GtkDPSContext *gtk_dps_context,
		     GdkDrawable *gdk_drawable,
		     gpointer data)
{
  DPSContext ctxt = *(((GtkDPSContext*)gtk_dps_context)->
		      gdk_dps_context->
		      dps_context);
  [(GyveView *)data enterContextNotify: ctxt];   
}
