/* GyveVisitor.m --- Automatic function applier to the figobj in any container

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include <stdio.h>
#include "GyveVisitor.h"
#include "PSFigObj.h"
#include "GyveSelectionsLayer.h"
#include "PSMaskedFigObjGroup.h"
#include "PSPath.h"
#include <Foundation/NSException.h>

@implementation GyveVisitor
- (enum gyve_visitor_request)visitAtBuffer:(GyveBuffer *)buffer
{GYVE_SUBCLASS_RESPONSIBILITY(return visitor_stop);}
- (enum gyve_visitor_request)visitAtLayer:(GyveLayer *)layer
{GYVE_SUBCLASS_RESPONSIBILITY(return visitor_stop);}
- (enum gyve_visitor_request)visitAtPath:(PSPath *)path
{GYVE_SUBCLASS_RESPONSIBILITY(return visitor_stop);}
- (enum gyve_visitor_request)visitAtText:(NSObject<PSText>*)text
{GYVE_SUBCLASS_RESPONSIBILITY(return visitor_stop);}
- (enum gyve_visitor_request)visitAtImage:(PSImage *)image
{GYVE_SUBCLASS_RESPONSIBILITY(return visitor_stop);}
- (enum gyve_visitor_request)visitAtFigObjGroup:(PSFigObjGroup *)group
{GYVE_SUBCLASS_RESPONSIBILITY(return visitor_stop);}
- (enum gyve_visitor_request)visitAtMaskedFigObjGroup:(PSMaskedFigObjGroup *)masked_group
{GYVE_SUBCLASS_RESPONSIBILITY(return visitor_stop);}
- (enum gyve_visitor_request)visitAtCompoundPaths: (PSCompoundPaths *)compound_paths
{GYVE_SUBCLASS_RESPONSIBILITY(return visitor_stop);}
- (enum gyve_visitor_request)visitAtProxy: (PSFigObjSelectedProxy *)proxy
{GYVE_SUBCLASS_RESPONSIBILITY(return visitor_stop);}
- (void)leaveFromProxy: (PSFigObjSelectedProxy *)proxy
{}
- (void)leaveFromBuffer:(GyveBuffer *)buffer
{}
- (void)leaveFromLayer:(GyveLayer *)layer
{}
- (void)leaveFromPath:(PSPath *)path
{}
- (void)leaveFromText:(NSObject<PSText>*)text
{}
- (void)leaveFromImage:(PSImage *)image
{}
- (void)leaveFromFigObjGroup:(PSFigObjGroup *)group
{}
- (void)leaveFromMaskedFigObjGroup:(PSMaskedFigObjGroup *)masked_group
{}
- (void)leaveFromCompoundPaths: (PSCompoundPaths *)compound_paths
{}
- initWithVisitingTarget: (NSObject <FigObjsContaining>*)_target
{
  [super init];
  filter = id_predicator_yes;
  target = _target;
  return self;
}
- (NSObject <FigObjsContaining> *)visitingTarget
{
  return target;
}
- (void)setEnumerationFilter:(id_predicator_t *)_filter
{
  filter = _filter;
}

- (void)visitUpward
{
  visiting_direction = visiting_upward;
  [self visit];
}
- (void)visitDownward 
{
  visiting_direction = visiting_downward;
  [self visit];
}
@end

@implementation GyveVisitor(Protected)
- (void)visit
{
  if (nil == target) 
    return ;
  else if (NO == filter(target))
    return ;
  else if (YES == [target isKindOfClass: [GyveBuffer class]])
    [self _visitAtBuffer: (GyveBuffer *)target];
  else if (YES == [target isKindOfClass: [GyveLayer class]])
    [self _visitAtLayer: (GyveLayer *)target];
  else if (YES == [target conformsToProtocol: @protocol(PSFigObj)])
    [self _visitAtFigObj: (id<PSFigObj>)target];
  else if (YES == ([(NSObject *)target isKindOfClass: [PSFigObjSelectedProxy class]]))
    [self _visitAtFigObj: 
	    (id<PSFigObj>)[(PSFigObjSelectedProxy *)target 
						    targetForFigObjProxy]];
  return ;
}
- (enum gyve_visitor_request)_visitAtBuffer: (GyveBuffer *)buffer
{
  enum gyve_visitor_request buf_req = [self visitAtBuffer: buffer];
  GyveLayer * layer;
  BOOL array_loop_continue 	  = YES; 
  enum gyve_visitor_request result = visitor_continue;
  enum gyve_visitor_request lay_req;
  switch (buf_req)
    {
    case visitor_detail: 
      if (visiting_upward == visiting_direction)
	{
	  FOR_ARRAY(buffer, layer)
	    {
	      if ((YES == array_loop_continue) && (YES == filter (layer)))
		{
		  lay_req = [self _visitAtLayer: layer];
		  switch (lay_req)
		    {
		    case visitor_stop:
		      array_loop_continue = NO;
		      result 	      = visitor_stop;
		      break;
		    case visitor_continue:
		      array_loop_continue = YES;
		      result 	      = visitor_continue;
		      break;
		    default:
		      NSCAssert(0, @"_visitAtLayer returns wrong value.");
		      result = visitor_stop;
		    }
		}
	    }
	  END_FOR_ARRAY(buffer);
	}
      else
	{
	  FOR_ARRAY_REVERSE(buffer, layer)
	    {
	      if ((YES == array_loop_continue) && (YES == filter (layer)))
		{
		  lay_req = [self _visitAtLayer: layer];
		  switch (lay_req)
		    {
		    case visitor_stop:
		      array_loop_continue = NO;
		      result 	      = visitor_stop;
		      break;
		    case visitor_continue:
		      array_loop_continue = YES;
		      result 	      = visitor_continue;
		      break;
		    default:
		      NSCAssert(0, @"_visitAtLayer returns wrong value.");
		      result = visitor_stop;
		    }
		}
	    }
	  END_FOR_ARRAY(buffer);
	}
      break;
    case visitor_stop:
      result = visitor_stop;
      break;
    case visitor_continue:
      result = visitor_continue;
      break;
    case visitor_rough:
      result = visitor_continue;
      break;			// Do nothing;
    }
  [self leaveFromBuffer: buffer];
  return result;
}
- (enum gyve_visitor_request)_visitAtLayer: (GyveLayer *)layer
{
  enum gyve_visitor_request result = visitor_continue;
  enum gyve_visitor_request lay_req = [self visitAtLayer: layer]; 
  id<PSFigObj>figobj;
  enum gyve_visitor_request figobj_req;
  BOOL array_loop_continue = YES; 
  switch (lay_req)
    {
    case visitor_detail:
      if (visiting_upward == visiting_direction)
	{
	  FOR_ARRAY(layer, figobj)
	    {
	      if ((YES == array_loop_continue) && (YES == filter (figobj)))
		{
		  figobj_req = [self _visitAtFigObj: figobj];
		  switch (figobj_req)
		    {
		    case visitor_continue:
		      result = visitor_continue;
		      break;
		    case visitor_stop:
		      array_loop_continue = NO;
		      result = visitor_stop;
		      break;
		    case visitor_rough:
		      array_loop_continue = NO;
		      result = visitor_continue;
		      break;
		    case visitor_detail:
		      NSCAssert((visitor_detail != figobj_req), 
				@"_visitAtFigObj returns wrong value.");
		      result = visitor_stop;
		      break;
		    }
		}
	    }
	  END_FOR_ARRAY(layer);
	}
      else
	{
	  FOR_ARRAY_REVERSE(layer, figobj)
	    {
	      if ((YES == array_loop_continue) && (YES == filter (figobj)))
		{
		  figobj_req = [self _visitAtFigObj: figobj];
		  switch (figobj_req)
		    {
		    case visitor_continue:
		      result = visitor_continue;
		      break;
		    case visitor_stop:
		      array_loop_continue = NO;
		      result = visitor_stop;
		      break;
		    case visitor_rough:
		      array_loop_continue = NO;
		      result = visitor_continue;
		      break;
		    case visitor_detail:
		      NSCAssert((visitor_detail != figobj_req), 
				@"_visitAtFigObj returns wrong value.");
		      result = visitor_stop;
		      break;
		    }
		}
	    }
	  END_FOR_ARRAY(layer);
	}
      break;
    case visitor_stop:
      result = visitor_stop;
      break;
    case   visitor_rough:
      result = visitor_continue;
      break;
    case visitor_continue:
      result =  visitor_continue;
      break;
    default: 
      // ERROR
      result =  visitor_stop;
    }
  [self leaveFromLayer: layer];
  return result;
}
- (enum gyve_visitor_request)_visitAtFigObj: (id<PSFigObj>)figobj
{
  if ([(NSObject *)figobj isKindOfClass: [PSFigObjSelectedProxy class]])
    return [self _visitAtProxy: (PSFigObjSelectedProxy *)figobj];
  else if ([(NSObject *)figobj isKindOfClass: [PSPath class]])
    return [self _visitAtPath: (PSPath *)figobj];
  else if ([(NSObject *)figobj conformsToProtocol: @protocol(PSText)])
    return [self _visitAtText: (id<PSText>)figobj];
  else if ([(NSObject *)figobj isKindOfClass: [PSImage class]])
    return [self _visitAtImage: (PSImage *)figobj];
  else if ([(NSObject *)figobj isMemberOfClass: [PSFigObjGroup class]])
    return [self _visitAtFigObjGroup: (PSFigObjGroup *)figobj];
  else if ([(NSObject *)figobj isMemberOfClass: [PSMaskedFigObjGroup class]])
    return [self _visitAtMaskedFigObjGroup: (PSMaskedFigObjGroup *)figobj];
  else if ([(NSObject *)figobj isMemberOfClass: [PSCompoundPaths class]])
    return [self _visitAtCompoundPaths: (PSCompoundPaths *)figobj];
  else
    return YES;
}
- (enum gyve_visitor_request)_visitAtProxy: (PSFigObjSelectedProxy *)proxy
{
  enum gyve_visitor_request proxy_req;
  enum gyve_visitor_request figobj_req = visitor_continue;
  enum gyve_visitor_request result;
  if (YES == filter (proxy))
    proxy_req = [self visitAtProxy: proxy];   
  else
    {
      result = visitor_continue;
      goto end_of_method;
    }

  switch (proxy_req)
    {
    case visitor_continue:
      result = visitor_continue;
      goto end_of_method;
    case visitor_stop:
      result = visitor_stop;
      goto end_of_method;
    case visitor_rough:
      result = visitor_continue;
      goto end_of_method;
    case visitor_detail:
      figobj_req = [self _visitAtFigObj:  (id<PSFigObj>)[proxy targetForFigObjProxy]];
      break;
    }
  switch (figobj_req)
    {
    case visitor_continue:
      result = visitor_continue;
      break;
    case visitor_stop:
      result = visitor_stop;
      break;
    case visitor_rough:
      result = visitor_continue;
      break;
//    case visitor_detail:
    default: 
      NSCAssert((visitor_detail != figobj_req), 
		@"_visitAtProxy returns wrong value.");
      result = visitor_stop;
    }
  end_of_method:
  [self leaveFromProxy: proxy];
  return result;
}
- (enum gyve_visitor_request)_visitAtImage:(PSImage *)image
{
  enum gyve_visitor_request image_req = [self visitAtImage: image];   
  enum gyve_visitor_request result;
  switch (image_req)
    {
    case visitor_continue:
      result = visitor_continue;
      break;
    case visitor_detail:
      result =visitor_continue;	// error?
      break;
    case visitor_rough:
      result = visitor_rough;
      break;
    case visitor_stop:
      result = visitor_stop;
      break;
    default:
      // error
      result = visitor_stop;
    }
  [self leaveFromImage: image];
  return result;
}
- (enum gyve_visitor_request)_visitAtPath: (PSPath *)path
{
  enum gyve_visitor_request path_req = [self visitAtPath: path];   
  enum gyve_visitor_request result;
  switch (path_req)
    {
    case visitor_continue:
      result = visitor_continue;
      break;
    case visitor_detail:
      result =visitor_continue;	// error?
      break;
    case visitor_rough:
      result = visitor_rough;
      break;
    case visitor_stop:
      result = visitor_stop;
      break;
    default:
      // error
      result = visitor_stop;
    }
  [self leaveFromPath: path];
  return result;
}
- (enum gyve_visitor_request)_visitAtText: (NSObject<PSText> *)text;
{
  enum gyve_visitor_request text_req = [self visitAtText: text];   
  enum gyve_visitor_request result;
  switch (text_req)
    {
    case visitor_continue:
      result = visitor_continue;
      break;
    case visitor_detail:
      result =  visitor_continue;		// error?
      break;
    case visitor_rough:
      result = visitor_rough;
      break;
    case visitor_stop:
      result = visitor_stop;
      break;
    default :
      // error
      result =visitor_stop;
    }
  [self leaveFromText: text];
  return result;
}
- (enum gyve_visitor_request)_visitAtFigObjGroup: (PSFigObjGroup *)group
{
  enum gyve_visitor_request result = visitor_continue;
  enum gyve_visitor_request group_req = [self visitAtFigObjGroup: group];
  enum gyve_visitor_request figobj_req;
  BOOL array_loop_continue = YES; 
  id<PSFigObj>figobj;
  switch (group_req)
    {
    case visitor_detail:
      if (visiting_upward == visiting_direction)
	{
	  FOR_ARRAY(group, figobj)
	    {
	      if ((YES == array_loop_continue) && (YES == filter (figobj)))
		{
		  figobj_req = [self _visitAtFigObj: figobj];
		  switch (figobj_req)
		    {
		    case visitor_continue:
		      result = visitor_continue;
		      break;
		    case visitor_stop:
		      array_loop_continue = NO;
		      result = visitor_stop;
		      break;
		    case visitor_rough:
		      array_loop_continue = NO;
		      result = visitor_continue;
		      break;
		    case visitor_detail:
		      NSCAssert((visitor_detail != figobj_req), 
				@"_visitAtFigObj returns wrong value.");
		      result = visitor_stop;
		      break;
		    }
		}
	    }
	  END_FOR_ARRAY(group);
	}
      else
	{
	  FOR_ARRAY_REVERSE(group, figobj)
	    {
	      if ((YES == array_loop_continue) && (YES == filter (figobj)))
		{
		  figobj_req = [self _visitAtFigObj: figobj];
		  switch (figobj_req)
		    {
		    case visitor_continue:
		      result = visitor_continue;
		      break;
		    case visitor_stop:
		      array_loop_continue = NO;
		      result = visitor_stop;
		      break;
		    case visitor_rough:
		      array_loop_continue = NO;
		      result = visitor_continue;
		      break;
		    case visitor_detail:
		      NSCAssert((visitor_detail != figobj_req), 
				@"_visitAtFigObj returns wrong value.");
		      result = visitor_stop;
		      break;
		    }
		}
	    }
	  END_FOR_ARRAY(group);
	}
      break;
    case visitor_stop:
      result = visitor_stop;
      break;
    case visitor_rough:
      result = visitor_continue;
      break;
    case visitor_continue:
      result = visitor_continue;
      break;
    default:
      // ERROR
      result = visitor_stop;
    }
  [self leaveFromFigObjGroup: group];
  return result;
}
- (enum gyve_visitor_request)_visitAtMaskedFigObjGroup: 
  (PSMaskedFigObjGroup *)masked_group
{
  enum gyve_visitor_request result     = visitor_continue;
  enum gyve_visitor_request group_req  = [self visitAtMaskedFigObjGroup: masked_group];
  enum gyve_visitor_request figobj_req = visitor_continue;
  BOOL array_loop_continue = YES; 
  id<PSFigObj>figobj;
  id<PSFigObj>mask_path = [masked_group maskPath];
  
  switch (group_req)
    {
    case visitor_detail:
      if (visiting_upward == visiting_direction)
	{
	  FOR_ARRAY(masked_group, figobj)
	    {
	      if ((YES == array_loop_continue) && (YES == filter (figobj)))
		{
		  figobj_req = [self _visitAtFigObj: figobj];
		  switch (figobj_req)
		    {
		    case visitor_continue:
		      result = visitor_continue;
		      break;
		    case visitor_stop:
		      array_loop_continue = NO;
		      result = visitor_stop;
		      break;
		    case visitor_rough:
		      array_loop_continue = NO;
		      result = visitor_continue;
		      break;
		    case visitor_detail:
		      NSCAssert((visitor_detail != figobj_req), 
				@"_visitAtFigObj returns wrong value.");
		      array_loop_continue = NO;
		      result = visitor_stop;
		      break;
		    }
		}
	    }
	  END_FOR_ARRAY(masked_group);

	  if ((array_loop_continue == YES) &&
	      (visitor_continue == figobj_req) &&
	      (YES == filter (mask_path)))
	    {
	      figobj_req = [self _visitAtFigObj: mask_path];
	      switch (figobj_req)
		{
		case visitor_continue:
		  result = visitor_continue;
		  break;
		case visitor_stop:
		  result = visitor_stop;
		  break;
		case visitor_rough:
		  result = visitor_continue;
		  break;
		case visitor_detail:
		  NSCAssert((visitor_detail != figobj_req), 
			    @"_visitAtFigObj returns wrong value.");
		  result = visitor_stop;
		  break;
		}
	    }
	}
      else
	{
	  if (YES == filter (mask_path))
	    {
	      figobj_req = [self _visitAtFigObj: mask_path];
	      switch (figobj_req)
		{
		case visitor_continue:
		  break;
		case visitor_stop:
		  result = visitor_stop;
		  goto end_of_method;
		case visitor_rough:
		  result = visitor_continue;
		  goto end_of_method;
		case visitor_detail:
		  NSCAssert((visitor_detail != figobj_req), 
			    @"_visitAtFigObj returns wrong value.");
		  result = visitor_stop;
		  break;
		}
	    }
	  FOR_ARRAY(masked_group, figobj)
	    {
	      if ((YES == array_loop_continue) && (YES == filter (figobj)))
		{
		  figobj_req = [self _visitAtFigObj: figobj];
		  switch (figobj_req)
		    {
		    case visitor_continue:
		      result = visitor_continue;
		      break;
		    case visitor_stop:
		      array_loop_continue = NO;
		      result = visitor_stop;
		      break;
		    case visitor_rough:
		      array_loop_continue = NO;
		      result = visitor_continue;
		      break;
		    case visitor_detail:
		      NSCAssert((visitor_detail != figobj_req), 
				@"_visitAtFigObj returns wrong value.");
		      array_loop_continue = NO;
		      result = visitor_stop;
		      break;
		    }
		}
	    }
	  END_FOR_ARRAY(masked_group);
	}
      break;
    case visitor_stop:
      result = visitor_stop;
      break;
    case visitor_rough:
      result = visitor_continue;
      break;
    case visitor_continue:
      result = visitor_continue;
      break;
    default:
      // ERROR
      result = visitor_stop;
      break; 
    }
end_of_method:
  [self leaveFromMaskedFigObjGroup: masked_group];
  return result;
}
- (enum gyve_visitor_request)_visitAtCompoundPaths: (PSCompoundPaths *)compound_paths
{
  enum gyve_visitor_request result = visitor_continue;
  enum gyve_visitor_request compound_paths_req 
    = [self visitAtCompoundPaths: compound_paths];
  enum gyve_visitor_request figobj_req;
  BOOL array_loop_continue = YES; 
  id<PSFigObj>figobj;
  switch (compound_paths_req)
    {
    case visitor_detail:
      if (visiting_upward == visiting_direction)
	{
	  FOR_ARRAY(compound_paths, figobj)
	    {
	      if ((YES == array_loop_continue) && (YES == filter (figobj)))
		{
		  figobj_req = [self _visitAtFigObj: figobj];
		  switch (figobj_req)
		    {
		    case visitor_continue:
		      result = visitor_continue;
		      break;
		    case visitor_stop:
		      array_loop_continue = NO;
		      result = visitor_stop;
		      break;
		    case visitor_rough:
		      array_loop_continue = NO;
		      result = visitor_continue;
		      break;
		    case visitor_detail:
		      NSCAssert((visitor_detail != figobj_req), 
				@"_visitAtFigObj returns wrong value.");
		      result = visitor_stop;
		      break;
		    }
		}
	    }
	  END_FOR_ARRAY(compound_paths);
	}
      else
	{
	  FOR_ARRAY_REVERSE(compound_paths, figobj)
	    {
	      if ((YES == array_loop_continue) && (YES == filter (figobj)))
		{
		  figobj_req = [self _visitAtFigObj: figobj];
		  switch (figobj_req)
		    {
		    case visitor_continue:
		      result = visitor_continue;
		      break;
		    case visitor_stop:
		      array_loop_continue = NO;
		      result = visitor_stop;
		      break;
		    case visitor_rough:
		      array_loop_continue = NO;
		      result = visitor_continue;
		      break;
		    case visitor_detail:
		      NSCAssert((visitor_detail != figobj_req), 
				@"_visitAtFigObj returns wrong value.");
		      result = visitor_stop;
		      break;
		    }
		}
	    }
	  END_FOR_ARRAY(compound_paths);
	}
      break;
    case visitor_stop:
      result = visitor_stop;
      break;
    case visitor_rough:
      result = visitor_continue;
      break;
    case visitor_continue:
      result = visitor_continue;
      break; 
    default:
      // ERROR
      result = visitor_stop;
    }
  [self leaveFromCompoundPaths: compound_paths];
  return result;
}
@end

@implementation GyveDebugVisitor
- (enum gyve_visitor_request)visitAtBuffer:(GyveBuffer *)buffer
{
  return visitor_detail;
}
- (enum gyve_visitor_request)visitAtLayer:(GyveLayer *)layer
{
   return visitor_detail;
}
- (enum gyve_visitor_request)visitAtPath:(PSPath *)path
{
  return visitor_continue;
}
- (enum gyve_visitor_request)visitAtText:(NSObject<PSText>*)text
{
  return visitor_continue;
}
- (enum gyve_visitor_request)visitAtImage:(PSImage *)image
{
  return visitor_continue;
}
- (enum gyve_visitor_request)visitAtFigObjGroup:(PSFigObjGroup *)group
{
  return visitor_detail;
}
- (enum gyve_visitor_request)visitAtMaskedFigObjGroup:(PSMaskedFigObjGroup *)masked_group
{
  return visitor_detail;
}
- (enum gyve_visitor_request)visitAtCompoundPaths: (PSCompoundPaths *)compound_paths
{
  return visitor_detail;
}

@end
