/* GyveWindow.h --- The definition for top level document window

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "GyveWindow.h"
#include "GyveScrolledCanvas.h"
#include "GyveBuffer.h"
#include "Gyve.h"
#include "public.h"
#include "GyveNil.h"
#include <Foundation/NSSet.h>
#include <Foundation/NSArray.h>


@implementation GyveWindow
static NSMutableArray * windows_array = nil;
static int current_window_ID 	      = 0;
static int last_clicked_window_ID     = 0;
+ (int)issueWindowIDFor: (GyveWindow *)window
{
  if (nil == windows_array)
    {
      windows_array  = [[NSMutableArray array] retain];
      [windows_array addObject: [GyveNil gyvenil]];
    }
  [windows_array addObject: window];
  return [windows_array indexOfObject: window];
}
+ (GyveWindow *)windowFromID:(int)windowID
{
  GyveWindow * result;
  if (nil == windows_array) return nil;	// Exception?
  if ([windows_array count] > windowID)
    {
      result = [windows_array objectAtIndex: windowID];
      if (result == [GyveNil gyvenil]) return nil;
      else return result;
    }
  else
    return nil;
}
- (int)windowID
{
  return window_ID;
}
+ (void)setCurrentWindow:(GyveWindow *)window
{
  /* fprintf(stderr, "->%s\n",
	  [[[window contentBuffer] bufferName] cString]); */
  current_window_ID = [window windowID];
  [GyveBuffer setCurrentBuffer: [window contentBuffer]];
  [window becomeCurrentWindow];
}
+ (GyveWindow * )currentWindow
{
  GyveWindow * result = [windows_array objectAtIndex: current_window_ID];
  if (result == [GyveNil gyvenil])
    return nil;
  else
    {
      return result;
    }
}
+ (void)setLastClickedWindow:(GyveWindow *)window
{
  last_clicked_window_ID = [window windowID];
}
+ (GyveWindow * )lastClickedWindow
{
  GyveWindow * result = [windows_array objectAtIndex:
					 last_clicked_window_ID];
  if (result == [GyveNil gyvenil])
    return nil;
  else
    {
      return result;
    }
}
+ (GyveWindow *)lookUpWindowByCanvas:(GyveCanvas *)_canvas
{
  id tmp;
  int i, max = [windows_array count];
  for (i = 0; i < max; i++)
    {
      tmp = [windows_array objectAtIndex: i];
      if (tmp == [GyveNil gyvenil]) 
	continue ;
      else if (_canvas == [(GyveWindow *)tmp contentCanvas])
	return (GyveWindow *)tmp;
    }
  return nil;
}
+ (NSSet *)lookUpWindowByBuffer: (GyveBuffer * )_buffer
{
  id tmp;
  NSSet * set     = [NSMutableSet set];
  int i, max;
  max = [windows_array count];
  for (i = 0; i < max ; i++)
    {
      tmp = [windows_array objectAtIndex: i];
      if (tmp == [GyveNil gyvenil])
	continue ;
      else if (_buffer == [(GyveWindow *)tmp contentBuffer])
	[set addObject: tmp];
    }
  
  if (0 == [set count])
    return nil;
  else
    return set;
}

- initWithBuffer: (GyveBuffer *)b
{
  NSObject<GyveDrawingEngine>* drawing_engine;
  NSObject<GyveEventHandler>* event_handler;

  [super init];
  
  if (nil == b)
    {
      [self release];
      return nil;		//  should throw an exception.
    }
  buffer = [b retain];

  /* Canvas setup */
  canvas = [[GyveScrolledCanvas alloc] 
	     initWithViewWidth: [buffer bufferWidth]
	     height: [buffer bufferHeight]];
  drawing_engine = [[[GyveDrawingEngine alloc] 
		      initWithBuffer: buffer] autorelease];
  event_handler = [[[GyveEventHandler alloc] 
			      initWithBuffer: buffer] autorelease];
  [canvas setDrawingEngine: drawing_engine];
  [canvas setEventHandler: event_handler];

  widget = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_container_add(GTK_CONTAINER(widget), [canvas gtkWidget]);
  [self setTitle: [[[buffer bufferName] copy] autorelease]];
  window_ID = [GyveWindow issueWindowIDFor: self];

  return self; 
}
- (void)setTitle:(NSString *)title
{
  gtk_window_set_title(GTK_WINDOW(widget), [title cString]);
}
- (void)show
{
  [canvas show];
  [super show];
}
- (void)becomeCurrentWindow
{
  gtk_window_set_focus (GTK_WINDOW(widget), [canvas dpsAreaWidget]);
}
- (void)dealloc
{
  [buffer release], buffer = nil;
  [canvas release], canvas = nil;
  window_ID 		   = 0;
  [super dealloc];
}
- (GyveBuffer *)contentBuffer
{
  return buffer;
}
- (GyveCanvas *)contentCanvas
{
  return canvas;
}
// Redraw
- (void)redrawAll
{
  [(GyveView *)[self contentCanvas] redrawAll];
}
- redrawLockRetain
{
  [(GyveView *)[self contentCanvas] redrawLockRetain];
  return self ;
}
- (void)redrawLockRelease
{
  [(GyveView *)[self contentCanvas] redrawLockRelease];
}
- (void)redrawFigObj: (id<PSFigObj>)figobj
{
  [(GyveView *)[self contentCanvas] redrawFigObj: figobj];
}
- (void)redrawRect: (NSRect *)rect
{
  [(GyveView *)[self contentCanvas] redrawRect: rect];
}
- (void)redrawBBox: (id<PSBBox>)dps_bboxobj
{
  [(GyveView *)[self contentCanvas] redrawBBox: dps_bboxobj];
}
- (void)redrawBBox: (id<PSBBox>)dps_bboxobj expandBy: (float)delta
{
  [(GyveView *)[self contentCanvas] redrawBBox: dps_bboxobj expandBy: delta];
}
@end

@implementation GyveWindow(PublicGuile)
- (void)redrawRectValue: (NSRectValue *)rectvalue
{
  NSRect rect = [rectvalue rectValue];
  [(GyveView *)[self contentCanvas] redrawRect: &rect];
}
@end
