/* PSCompoundPaths.h ---

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "PSCompoundPaths.h"
#include "PSPath.h"
#include "containing.h"
#include "enumerating.h"

@interface PSCompoundPathsAbstractPointsEnumerator: NSObject<PointsEnumerating>
{
  id_predicator_t * filter;
  NSObject<FigObjsPrimitiveEnumerating> * primitive_enumerator;
  NSObject<PointsEnumerating> * inner_points_enumerator;
}
- (void)setEnumerationFilter: (id_predicator_t *)filter;
- nextObject;
- (NSPointValue *)nextPointValue;
- (void)dealloc;
@end

@interface PSCompoundPathsObversePointsEnumerator: 
  PSCompoundPathsAbstractPointsEnumerator
- initWithPSCompoundPaths: (PSCompoundPaths *)cpaths;
- (const NSPoint *)nextPoint;
@end

@interface PSCompoundPathsReversePointsEnumerator:
  PSCompoundPathsAbstractPointsEnumerator
- initWithPSCompoundPaths: (PSCompoundPaths *)cpaths;
- (const NSPoint *)nextPoint;
@end

@interface PSCompoundPathsAbstractSegmentsEnumerator: NSObject<SegmentsEnumerating>
{
  id_predicator_t * filter;
  NSObject<FigObjsPrimitiveEnumerating> * primitive_enumerator;
  NSObject<SegmentsEnumerating> * inner_segments_enumerator;
}
- (void)setEnumerationFilter: (id_predicator_t *)filter;
- nextObject;
- (void)dealloc;
@end

@interface PSCompoundPathsObverseSegmentsEnumerator: 
  PSCompoundPathsAbstractSegmentsEnumerator
- initWithPSCompoundPaths: (PSCompoundPaths *)cpaths;
- (PSSegment *)nextSegment;
@end

@interface PSCompoundPathsReverseSegmentsEnumerator:
  PSCompoundPathsAbstractSegmentsEnumerator
- initWithPSCompoundPaths: (PSCompoundPaths *)cpaths;
- (PSSegment *)nextSegment;
@end

 // PSCompoundPaths
@implementation PSCompoundPaths
- init
{
  closed = NO;
  [super init];
  return self ;
}
- (BOOL)isClosedPath
{
  return closed;
}
- (void)closePath
{
  closed = YES;
}
- (void)openPath
{
  closed = NO;
}
- (int)countSegments
{
  int i = 0;
  id<SegmentsEnumerating> e = [self segmentsEnumerator];
  id tmp;
  while (tmp = [e nextSegment], tmp) i++;
  return i;
}
- (int)countPoints
{
  int i = 0;
  id<PointsEnumerating> e = [self pointsEnumerator];
  const NSPoint * tmp;
  while (tmp = [e nextPoint], tmp) i++;
  return i;
}

- (id<PointsEnumerating>)pointsEnumerator
{
  
  return [[[PSCompoundPathsObversePointsEnumerator alloc]
	    initWithPSCompoundPaths: self] autorelease];
}
- (id<PointsEnumerating>)reversePointsEnumerator
{
  return [[[PSCompoundPathsReversePointsEnumerator alloc]
	    initWithPSCompoundPaths: self] autorelease];
}
- (id<SegmentsEnumerating>)segmentsEnumerator
{
  return [[[PSCompoundPathsObverseSegmentsEnumerator alloc]
	    initWithPSCompoundPaths: self] autorelease];
}
- (id<SegmentsEnumerating>)reverseSegmentsEnumerator
{
  return [[[PSCompoundPathsReverseSegmentsEnumerator alloc]
	    initWithPSCompoundPaths: self] autorelease];
}
- (PSPaintStyle *)paintStyle
{
  id<PSFigObj>result = [self firstPrimitiveFigObj];
  if ([(NSObject *)result isKindOfClass: [PSPath class]])
    return [(PSPath *)result paintStyle];
  else
    return nil;	
}
- (void)setPaintStyle: (PSPaintStyle *) style
{
  id<PSFigObj>result = [self firstPrimitiveFigObj];
  if ([(NSObject *)result isKindOfClass: [PSPath class]])
    [(PSPath *)result setPaintStyle: style];
}
- (NSPoint *)pointAtIndex: (int)n
{GYVE_SHOULD_NOT_IMPLEMENT(return NULL);}
- (NSPointValue *) pointValueAtIndex:(int)index
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (NSPointValue *)firstPointValue
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (const NSPoint *)firstPoint
{GYVE_SHOULD_NOT_IMPLEMENT(return NULL);}
- (const NSPoint *)lastPoint
{GYVE_SHOULD_NOT_IMPLEMENT(return NULL);}
- (NSPointValue *)lastPointValue
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (BOOL)containsPoint: (const NSPoint *)point
{GYVE_SHOULD_NOT_IMPLEMENT(return NO);}
- (BOOL)containsPointValue: (const NSPointValue *)point
{GYVE_SHOULD_NOT_IMPLEMENT(return NO);}
- (BOOL)containsSegment: (PSSegment *)segment
{GYVE_SHOULD_NOT_IMPLEMENT(return NO);}
- (PSSegment *)lastSegment
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (int)indexOfSegment: (PSSegment *)seg
{GYVE_SHOULD_NOT_IMPLEMENT(return 0);}
- (PSSegment *)segmentAtIndex: (int)n
{
  GYVE_SHOULD_NOT_IMPLEMENT(return nil);
}
- (PSSegment *)firstSegment
{GYVE_SHOULD_NOT_IMPLEMENT(return nil);}
- (float)deltaForExpandingBBox
{
  id<PSFigObj>result = [self firstPrimitiveFigObj];
  if ([(NSObject *)result isKindOfClass: [PSPath class]])
    return [(PSPath *)result deltaForExpandingBBox];
  else
    return 0;			// TODO
}
- (Class)selectedProxyClass
{
  return Nil;
}
@end

// PSCompoundPathsPointsEnumerator

@implementation PSCompoundPathsAbstractPointsEnumerator
- (void)setEnumerationFilter: (id_predicator_t *)f
{ 
  filter = f;
}
- nextObject
{
  return [self nextPointValue];
}
- (NSPointValue *)nextPointValue
{
  const NSPoint * point = [self nextPoint];
  if (NULL == point)
    return nil;
  else
    return (NSPointValue *)[NSValue valueWithPoint: *point];
}
- (void)dealloc
{
  if (nil != inner_points_enumerator)
    [inner_points_enumerator release], inner_points_enumerator = nil;
  if (nil != primitive_enumerator)
    [primitive_enumerator release], primitive_enumerator = nil;
  [super dealloc];
}
- (const NSPoint *)nextPoint
{
  [self subclassResponsibility:_cmd];
  return NULL;
}
@end

@implementation PSCompoundPathsObversePointsEnumerator
- initWithPSCompoundPaths: (PSCompoundPaths *)cpaths
{
  [super init];
  filter 	   	  = id_predicator_yes;
  primitive_enumerator 	  = [[cpaths primitiveFigObjsEnumerator] retain];
  inner_points_enumerator = nil;
  return self; 
}
- (const NSPoint *)nextPoint
{
  id <PSFigObj> figobj;
  const NSPoint * point;
  if (nil == inner_points_enumerator)
    {
      figobj = [primitive_enumerator nextPrimitiveFigObj];
      if (nil == figobj) return NULL;
      inner_points_enumerator = [figobj pointsEnumerator];
      return [self nextPoint];
    }
  else
    {
      point = [inner_points_enumerator nextPoint];
      if (point == NULL)
	{
	  inner_points_enumerator = nil;
	  return [self nextPoint];
	}
      else if (YES == filter([NSValue valueWithPoint: *point]))
	{
	  return point;
	}
      else
	{
	  return [self nextPoint];
	}
    }
}

@end

@implementation PSCompoundPathsReversePointsEnumerator
- initWithPSCompoundPaths: (PSCompoundPaths *)cpaths
{
  [super init];
  filter 	   	  = id_predicator_yes;
  primitive_enumerator 	  = [[cpaths reversePrimitiveFigObjsEnumerator] retain];
  inner_points_enumerator = nil;
  return self; 
}
- (const NSPoint *)nextPoint
{
  if (nil == inner_points_enumerator)
    {
      id <PSFigObj> figobj;
      figobj 		      = [primitive_enumerator nextPrimitiveFigObj];
      if (nil == figobj) return NULL;
      inner_points_enumerator = [figobj reversePointsEnumerator];
      return [self nextPoint];
    }
  else
    {
      const NSPoint * point;
      point = [inner_points_enumerator nextPoint];
      if (point == NULL)
	{
	  inner_points_enumerator = nil;
	  return [self nextPoint];
	}
      else if (YES == filter([NSValue valueWithPoint: *point]))
	{
	  return point;
	}
      else
	{
	  return [self nextPoint];
	}
    }
}
@end

// PSCompoundPathsSegmentsEnumerator

@implementation PSCompoundPathsAbstractSegmentsEnumerator
- (void)setEnumerationFilter: (id_predicator_t *)f
{ 
  filter = f;
}
- nextObject
{
  return [self nextSegment];
}
- (PSSegment *)nextSegment
{
  [self subclassResponsibility:_cmd];
  return nil;
}
- (void)dealloc
{
  if (nil != inner_segments_enumerator)
    [inner_segments_enumerator release], inner_segments_enumerator = nil;
  if (nil != primitive_enumerator)
    [primitive_enumerator release], primitive_enumerator = nil;
  [super dealloc];
}
@end


@implementation PSCompoundPathsObverseSegmentsEnumerator
- initWithPSCompoundPaths: (PSCompoundPaths *)cpaths
{
  [super init];
  filter 	   	  = id_predicator_yes;
  primitive_enumerator 	  = [[cpaths primitiveFigObjsEnumerator] retain];
  inner_segments_enumerator = nil;
  return self; 
}
- (PSSegment *)nextSegment
{
  id <PSFigObj> figobj;
  PSSegment * segment;
  if (nil == inner_segments_enumerator)
    {
      figobj = [primitive_enumerator nextPrimitiveFigObj];
      if (nil == figobj) return NULL;
      inner_segments_enumerator = [figobj segmentsEnumerator];
      return [self nextSegment];
    }
  else
    {
      segment = [inner_segments_enumerator nextSegment];
      if (segment == NULL)
	{
	  inner_segments_enumerator = nil;
	  return [self nextSegment];
	}
      else if (YES == filter(segment))
	{
	  return segment;
	}
      else
	{
	  return [self nextSegment];
	}
    }
}
@end

@implementation PSCompoundPathsReverseSegmentsEnumerator
- initWithPSCompoundPaths: (PSCompoundPaths *)cpaths
{
  [super init];
  filter 	   	  = id_predicator_yes;
  primitive_enumerator 	  = [[cpaths reversePrimitiveFigObjsEnumerator] retain];
  inner_segments_enumerator = nil;
  return self; 
}
- (PSSegment *)nextSegment
{
  if (nil == inner_segments_enumerator)
    {
      id <PSFigObj> figobj;
      figobj 		      = [primitive_enumerator nextPrimitiveFigObj];
      if (nil == figobj) return NULL;
      inner_segments_enumerator = [figobj reverseSegmentsEnumerator];
      return [self nextSegment];
    }
  else
    {
      PSSegment * segment;
      segment = [inner_segments_enumerator nextSegment];
      if (segment == NULL)
	{
	  inner_segments_enumerator = nil;
	  return [self nextSegment];
	}
      else if (YES == filter(segment))
	{
	  return segment;
	}
      else
	{
	  return [self nextSegment];
	}
    }
}
@end
