/* PSFigObjGroup.m --- group of figure objects

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "PSFigObjGroup.h"
#include "PSSegment.h"
#include "utilities.h"

@interface PSFigObjGroupAbstractFigObjsEnumerator: NSObject<FigObjsEnumerating>
{
  id_predicator_t * filter;
  id<Enumerating>inner_enumerator;
}
- init;
- (void)setEnumerationFilter: (id_predicator_t *)filter;
- nextObject;
- (id<PSFigObj>)nextFigObj;
@end

@interface PSFigObjGroupObverseFigObjsEnumerator: 
  PSFigObjGroupAbstractFigObjsEnumerator
- initWithPSFigObjGroup: (PSFigObjGroup *)group;
@end

@interface PSFigObjGroupReverseFigObjsEnumerator: 
  PSFigObjGroupAbstractFigObjsEnumerator
- initWithPSFigObjGroup: (PSFigObjGroup *)group;
@end

@interface PSFigObjGroupAbstractPrimitiveFigObjsEnumerator: 
  NSObject<FigObjsPrimitiveEnumerating>
{
  id_predicator_t * filter;
  id<Enumerating>currenet_inner_enumerator;
  GyveStack * enumerators_stack;
}
- (void)setEnumerationFilter: (id_predicator_t *)filter;
- nextObject;
@end

@interface PSFigObjGroupObversePrimitiveFigObjsEnumerator:
  PSFigObjGroupAbstractPrimitiveFigObjsEnumerator
- initWithPSFigObjGroup: (PSFigObjGroup *)group;
- (id<PSFigObj>)nextPrimitiveFigObj;
@end

@interface PSFigObjGroupReversePrimitiveFigObjsEnumerator:
  PSFigObjGroupAbstractPrimitiveFigObjsEnumerator
- initWithPSFigObjGroup: (PSFigObjGroup *)group;
- (id<PSFigObj>)nextPrimitiveFigObj;
@end

//  PSFigObjGroup
@implementation  PSFigObjGroup
/*
 * Creating and destroying
 */
- init
{
  [super init];
  [self zeroBBox];
  return self ;
}
- (void)dealloc
{
  [super dealloc];
}
- copyWithZone: (NSZone *)zone
{
  id figobj;
  int i, max;
  PSFigObjGroup * new_group = [[[self class] allocWithZone: zone] init];
  max 			    = [self countFigObjs];
  for (i = 0; i < max; i++)
    {
      figobj = [self figObjAtIndex: i];
      [new_group appendObject: [[figobj copyWithZone: zone] autorelease]];
    }
  [new_group calcBBox];
  return new_group;
}

/*
 * BBox
 */
- (void)calcBBox
{
  [self zeroBBox];
  [self makeObjectsPerform: @selector(reportBBoxTo:) withObject: self];
}
- (struct bbox *)bboxCStructure
{
  return &bbox;
}
- (void)zeroBBox
{
  bbox_copy(&zero_bbox, [self bboxCStructure]);
}
- (BOOL)isZeroBBox
{
  return bbox_is_zero_bbox([self bboxCStructure]);
}
- (BOOL)expandBBoxByBBox: (id<PSBBox>)b
{
  return bbox_expand_by_bbox([self bboxCStructure], [b bboxCStructure]);
}
- (void)reportBBoxTo: (id<PSBBox>)b
{
  [b expandBBoxByBBox: self];
}
- (BOOL)isPointInBBox: (NSPoint *)p
{
  return bbox_contains_point([self bboxCStructure], p);
}
- (BOOL)intersectsWithBBox: (id<PSBBox>)b
{
  return bbox_intersects([self bboxCStructure], [b bboxCStructure]);
}
- (BOOL)intersectsWithRect: (NSRect *)r
{
  struct bbox b;
  NSRect_to_bbox (r, &b);
  return bbox_intersects([self bboxCStructure], &b);
}

/*
 * FigObj protocol 
 */
- (BOOL)isFigObjProxy
{
  return NO;
}
- (NSObject<PSFigObj> *)targetForFigObjProxy
{
  return self ;
}
- markAsSelected
{
  selected = YES;
  return self ;
}
- (BOOL)isSelected
{
  return selected;
}
- (void)unMarkAsSelected
{
  selected = NO;
}
- (BOOL)isLockedFigObj
{
  return locked;
}
- (void)setFigObjLock: (BOOL)l
{
  locked = l;
}
- (float)deltaForExpandingBBox
{
  id<PSFigObj> figobj;
  int i, max;
  float delta = 0.0;
  float x;
  max = [self countFigObjs];
  for (i = 0; i < max; i++)
    {
      figobj = [self figObjAtIndex: i];
      x = [figobj deltaForExpandingBBox];
      if (x > delta)
	delta = x;
    }
  return delta;
}
- (Class)selectedProxyClass
{
  return Nil;
}

/*
 * FigObj containing protocol 
 */ 
- (id<PSFigObj>)firstPrimitiveFigObj
{
  id<PSFigObj>result = [self firstFigObj];
  if (result == nil) return nil;
  // Check when the text class are implemented.
  while (NO == id_predicator_primitive_figobj(result))
    result = [result firstFigObj]; 
  return result;
}
- (id<PSFigObj>)lastPrimitiveFigObj
{
  id<PSFigObj>result = [self lastFigObj];
  if (result == nil) return nil;
  // Check when the text class are implemented.
  while (NO == id_predicator_primitive_figobj(result))
    result = [result lastFigObj];
  return result;
}
- (id<PSFigObj>)firstFigObj
{
  if ([self count] > 0) 
    return  (id<PSFigObj>)[self firstObject];
  else
    return nil;
}
- (id<PSFigObj>)lastFigObj
{
  if ([self count] > 0) 
    return  (id<PSFigObj>)[self lastObject];
  else
    return nil;
}
- (BOOL)containsFigObj: (id<PSFigObj>)figobj
{
  return [self containsObject: figobj];
}
- (int)countFigObjs
{
  return [self count];
}
- (id<PSFigObj>)figObjAtIndex: (int)n
{
  return (id<PSFigObj>)[self objectAtIndex: n];
}

- (int)indexOfFigObj: (id<PSFigObj>)figobj
{
  return [self indexOfObject: figobj];
}
- (id<FigObjsEnumerating>)reverseFigObjsEnumerator
{
  return [[[PSFigObjGroupReverseFigObjsEnumerator alloc]
	    initWithPSFigObjGroup: self] autorelease];
}
- (id<FigObjsPrimitiveEnumerating>)reversePrimitiveFigObjsEnumerator
{
  return [[[PSFigObjGroupReversePrimitiveFigObjsEnumerator alloc]
	    initWithPSFigObjGroup: self] autorelease];
}
- (id<FigObjsEnumerating>)figObjsEnumerator
{
  return [[[PSFigObjGroupObverseFigObjsEnumerator alloc]
	    initWithPSFigObjGroup: self] autorelease];
}
- (id<FigObjsPrimitiveEnumerating>)primitiveFigObjsEnumerator
{
  return [[[PSFigObjGroupObversePrimitiveFigObjsEnumerator alloc]
	    initWithPSFigObjGroup: self] autorelease];
}
- (void)addFigObj: (NSObject<PSFigObj> *)figobj
{
  [self appendObject: figobj];
}
@end

 // PSFigObjGroupFigObjEnumerator
@implementation PSFigObjGroupAbstractFigObjsEnumerator
- init
{
  [super init];  
  filter 	   = id_predicator_yes;
  inner_enumerator = nil;
  return self ;
}
- (void)setEnumerationFilter: (id_predicator_t *)f
{
  filter = f;
}
- (id<PSFigObj>)nextFigObj
{
  id<PSFigObj> figobj = [inner_enumerator nextObject];
  if (nil == figobj) return nil;
  else if (YES == filter(figobj))
    return figobj;
  else
    return [self nextFigObj];
}
- nextObject
{
  return [self nextFigObj];
}
- (void)dealloc
{
  [inner_enumerator release], inner_enumerator = nil;
  [super dealloc];
}
@end

@implementation  PSFigObjGroupObverseFigObjsEnumerator
- initWithPSFigObjGroup: (PSFigObjGroup *)group
{
  [super init];
  inner_enumerator = [[group objectEnumerator] retain];  
  return self ;
}
@end

@implementation PSFigObjGroupReverseFigObjsEnumerator
- initWithPSFigObjGroup: (PSFigObjGroup *)group
{
  [super init];
  inner_enumerator = [[group reverseObjectEnumerator] retain];  
  return self ;
}
@end

 // PSFigObjGroupPrimitiveFigObjEnumerator
@implementation PSFigObjGroupAbstractPrimitiveFigObjsEnumerator
- init
{
  [super init];
  currenet_inner_enumerator = nil;
  filter 		    = id_predicator_yes;
  enumerators_stack = [[GyveStack alloc] init];
  return self;
}
- (void)dealloc
{
  [currenet_inner_enumerator release], currenet_inner_enumerator = nil;
  [enumerators_stack release], enumerators_stack = nil;
  [super dealloc];
}
- (void)setEnumerationFilter: (id_predicator_t *)f
{
  filter = f;
}
- nextObject
{
  return [self nextPrimitiveFigObj];
}
- (id<PSFigObj>)nextPrimitiveFigObj
{
  [self subclassResponsibility:_cmd];
  return nil;
}
@end

@implementation PSFigObjGroupObversePrimitiveFigObjsEnumerator
- initWithPSFigObjGroup: (PSFigObjGroup *)group
{
  [super init];
  currenet_inner_enumerator = [[group figObjsEnumerator] retain];
  return self ;
}
- (id<PSFigObj>)nextPrimitiveFigObj
{
  id <PSFigObj> result;
  if (currenet_inner_enumerator == nil) return nil;

  result = [currenet_inner_enumerator nextObject];

  if (nil == result)
    {
      [currenet_inner_enumerator release];
      if ([enumerators_stack count] > 0)
	{
	  currenet_inner_enumerator = [enumerators_stack popObject];
	  return [self nextPrimitiveFigObj];
	}
      else
	{
	  currenet_inner_enumerator = nil;
	  return nil;
	}
    }
  else if (YES == id_predicator_primitive_figobj(result))
    {
      if (YES == filter(result))
	return result;
      else 
	return [self nextPrimitiveFigObj];
    }
  else
    {
      [enumerators_stack pushObject: currenet_inner_enumerator];
      currenet_inner_enumerator = [[result figObjsEnumerator] retain];
      return [self nextPrimitiveFigObj];
    }
}
@end

@implementation PSFigObjGroupReversePrimitiveFigObjsEnumerator
- initWithPSFigObjGroup: (PSFigObjGroup *)group
{
  [super init];
  currenet_inner_enumerator = [[group reverseFigObjsEnumerator] retain];
  return self ;
}
- (id<PSFigObj>)nextPrimitiveFigObj
{
  id <PSFigObj> result;
  if (currenet_inner_enumerator == nil) return nil;

  result = [currenet_inner_enumerator nextObject];

  if (nil == result)
    {
      [currenet_inner_enumerator release];
      if ([enumerators_stack count] > 0)
	{
	  currenet_inner_enumerator = [enumerators_stack popObject];
	  return [self nextPrimitiveFigObj];
	}
      else
	{
	  currenet_inner_enumerator = nil;
	  return nil;
	}
    }
  else if (YES == id_predicator_primitive_figobj(result))
    {
      if (YES == filter(result))
	return result;
      else 
	return [self nextPrimitiveFigObj];
    }
  else
    {
      [enumerators_stack pushObject: currenet_inner_enumerator];
      currenet_inner_enumerator = [[result reverseFigObjsEnumerator] retain];
      return [self nextPrimitiveFigObj];
    }
}
@end
