/* PSImage.h --- Image object

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "PSImage.h"
#include "geometry.h"
#include "geometry.h"
#include "PSMatrix.h"
#include "GyveGUI.h"

#ifdef HAVE_LIBXPM
#include <X11/xpm.h>
#endif /* Def: HAVE_LIBXPM */

@implementation PSImage
+ allocWithZone: (NSZone*)z
{
  return NSAllocateObject([PSImage class], 0, z);
}
+ alloc
{
  return [self allocWithZone: NULL];
}
+ randomImageWithWidth: (int)w height: (int)h
{
  id tmp;
  int i;
  int size = sizeof(unsigned char)*h*w*3;
  unsigned char * pixels = malloc(size);
  for (i = 0; i < size ; i += 3)
    {
      pixels[i] = random()%255;
      pixels[i+1] = random()%255;
      pixels[i+2] = random()%255;
    }
  tmp = [[[self alloc] initWithWidth: w height: h pixels: pixels] 
	  autorelease];
  free(pixels);
  return tmp;
}
+ imageFromXpmFile: (char *)file_name
{
#ifdef HAVE_LIBXPM
  float * matrix;
  unsigned int color_index;
  id result;
  int w, h;
  int status;
  XpmImage image;
  unsigned char* pixels;
  int src, dist;
  unsigned int r, g, b;

  status = XpmReadFileToXpmImage(file_name,
				 &image,
				 NULL);
  if (status != XpmSuccess) return nil;

  w   	 = image.width;
  h   	 = image.height;
  
  pixels = malloc(3*w*h);

  for  (src = 0, dist=0;
	dist < w*h*3;
	dist += 3, src++)
  {
    color_index = image.data[src];
    sscanf((image.colorTable[color_index].c_color),
	   "#%02X%02X%02X", 
	   &r, &g, &b);
	   pixels[dist+RED_INDEX] = r;
	   pixels[dist+GREEN_INDEX] = g;
	   pixels[dist+BLUE_INDEX] = b;
  }

  result = [[[self alloc] initWithWidth: w height: h pixels: pixels] autorelease];
  free (pixels);
  XpmFreeXpmImage(&image);

  matrix = [result mappingMatrix];
  matrix[MATRIX_A] = w;
  matrix[MATRIX_D] = h;
  matrix[MATRIX_TY] = 0;
  return result;
#else 
  gyve_message("Importing Xpm is not supported.");
  return nil;
#endif /* Def: HAVE_LIBXPM */
}
- initWithWidth: (int)w height: (int)h pixels: (unsigned char*) pixels
{
  int length_local;
  width  = w;
  height = h;
  length_local = 3*width * height;
  matrix_init(mapping_matrix);
  mapping_matrix[MATRIX_A] = w;
  mapping_matrix[MATRIX_D] = -h;
  mapping_matrix[MATRIX_TY] = h;
  locked  		    = NO;
  selected 		    = NO;
  
  if (pixels)
    [super initWithBytes: (void *)pixels length: length_local];
  else
    [super initWithLength: length_local];

  [self calcBBox];

  return self ;
}
- (BOOL) isEqual: anObject
{
  if (self == anObject)
    return YES;
  else
    return NO;
}
- (int)imageWidth
{
  return width;
}
- (int)imageHeight
{
  return height;
}
- initWithWidth: (int)w height: (int)h
{
  return [self initWithWidth: w height: h pixels: NULL];
}
- (float *)mappingMatrix
{
  return mapping_matrix;
}
- (void)setMappingMatrix: (float *)new_mapping_matrix
{
  matrix_copy(new_mapping_matrix, mapping_matrix);
}
- (BOOL)isFigObjProxy
{
  return NO;
}
- (NSObject<PSFigObj> *)targetForFigObjProxy
{
  return self ;
}
- markAsSelected
{
  selected = YES;
  return self ;
}
- (BOOL)isSelected
{
  return selected;
}
- (void)unMarkAsSelected
{
  selected = NO;
}
- (BOOL)isLockedFigObj
{
  return locked;
}
- (void)setFigObjLock: (BOOL)l
{
  locked = l;
}
- (void)calcBBox
{
  NSPoint zero = {0.0, 0.0};
  NSPoint one  = {1.0, 1.0};
  bbox_from_points(&bbox, &zero, &one);
  matrix_apply_on_bbox(mapping_matrix, &bbox);
}
- (struct bbox *)bboxCStructure
{
  return &bbox;
}
- (void)zeroBBox;
{
  bbox_copy(&zero_bbox, [self bboxCStructure]);
}
- (BOOL)isZeroBBox;
{
  return bbox_is_zero_bbox([self bboxCStructure]);
}
- (BOOL)expandBBoxByBBox: (id<PSBBox>)b;
{
  return bbox_expand_by_bbox([self bboxCStructure], [b bboxCStructure]);
}
- (void)reportBBoxTo: (id<PSBBox>)b ;
{
  [b expandBBoxByBBox: self];
}
- (BOOL)isPointInBBox: (NSPoint *)p;
{
  return bbox_contains_point([self bboxCStructure], p);
}
- (BOOL)intersectsWithBBox: (id<PSBBox>)b;
{
  return bbox_intersects([self bboxCStructure], [b bboxCStructure]);
}
- (unsigned char *)mutablePixelsOfLine: (int)line
{
  char * pixels;
  BOOL is_wrong_line = NO;
  if (line < 0) is_wrong_line      = YES;
  if (line >= height) is_wrong_line = YES;

  if (is_wrong_line) return NULL;	// Should raise an exception
  else pixels = [self mutableBytes];
  pixels += (3*width)*line;
  return pixels;
}
- (unsigned char *)mutablePixelInLine: (int)line column: (int)column
{
  unsigned char * p = [self mutablePixelsOfLine: line];
  if (NULL != p)
    {
      p += (3*column);
    }
  return p;
}
- copy
{
  return [self copyWithZone: NULL];
}
- copyWithZone: (NSZone *)z
{
  PSImage * new_img = [[[self class] allocWithZone: z] 
			initWithWidth: [self imageWidth]
			height: [self imageHeight]
			pixels: [self mutableBytes]];
  [new_img setMappingMatrix: [self mappingMatrix]];
  return new_img;
}
- (BOOL)intersectsWithRect: (NSRect *)r
{
  struct bbox b;
  NSRect_to_bbox (r, &b);
  return bbox_intersects([self bboxCStructure], &b);
}
- (float)deltaForExpandingBBox
{
  return 0.0;
}
- (Class)selectedProxyClass
{
  return Nil;
}
- (void)bindToUserobject: (userobject)i inContext:(DPSContext)ctxt
{
  // TODO
}
- (userobject)userobject
{
  return userobject_unbound;
}
@end

@implementation PSImage(Guile)
- (void)setMappingMatrixValue: (NSValue *) matrix
{
  [self setMappingMatrix: [matrix matrixValue]];
}
- (PSMatrix *)mappingMatrixValue
{
  float * m = [self mappingMatrix];
  return (PSMatrix *)[PSMatrix valueWithMatrix: m];
}
@end
