/* PSMatrix.m --- linear operator 

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "PSMatrix.h"
#include "PSPath.h"
#include "PSFigObjGroup.h"
#include "PSMaskedFigObjGroup.h"
#include "PSCompoundPaths.h"
#include "PSImage.h"

@implementation PSMatrix: NSConcreteValue
- init
{
  float m[6];
  matrix_init(m);
  return [self initWithMatrix: m];
}
- initWithMatrix: (float *)matrix
{
  [super initValue: matrix withObjCType: @encode(float[6])];
  return self ;
}
- initWithMatrixElementA: (float)a
		       B: (float)b
		       C: (float)c
		       D: (float)d
		      TX: (float)tx
		      TY: (float)ty
{
  float m[6];
  m[MATRIX_A]  = a;
  m[MATRIX_B]  = b;
  m[MATRIX_C]  = c;
  m[MATRIX_D]  = d;
  m[MATRIX_TX] = tx;
  m[MATRIX_TY] = ty;
  return [self initWithMatrix: m];
}
- (float *)matrixValue
{
  return (float *)data;
}
@end

@implementation NSValue(PSMatrix)
+ (NSValue *)valueWithMatrix: (float *)matrix
{
  return [[[self alloc] initWithMatrix: matrix] autorelease];
}
- (float *)matrixValue
{
  return NULL;			
}
- (float)matrixElementAtIndex: (enum matrix_element_index)index
{
  return *([self matrixValue] + index);
  
}
- (void)setMatrixElement: (float)element
		 atIndex: (enum matrix_element_index)index
{
  *([self matrixValue] + index) = element;
}
- (void)setMatrix: (void *)m ofType: (enum matrix_type) t
{
  if (t == C_MATRIX)
    {
      [self setMatrixElement: ((float *)m)[MATRIX_A] atIndex: MATRIX_A];
      [self setMatrixElement: ((float *)m)[MATRIX_B] atIndex: MATRIX_B];
      [self setMatrixElement: ((float *)m)[MATRIX_C] atIndex: MATRIX_C];
      [self setMatrixElement: ((float *)m)[MATRIX_D] atIndex: MATRIX_D];
      [self setMatrixElement: ((float *)m)[MATRIX_TX] atIndex: MATRIX_TX];
      [self setMatrixElement: ((float *)m)[MATRIX_TY] atIndex: MATRIX_TY];
    }
  else if (t == OBJC_MATRIX)
    {
      [self setMatrixElement: [(id)m matrixElementAtIndex: MATRIX_A] atIndex: MATRIX_A];
      [self setMatrixElement: [(id)m matrixElementAtIndex: MATRIX_B] atIndex: MATRIX_B];
      [self setMatrixElement: [(id)m matrixElementAtIndex: MATRIX_C] atIndex: MATRIX_C];
      [self setMatrixElement: [(id)m matrixElementAtIndex: MATRIX_D] atIndex: MATRIX_D];
      [self setMatrixElement: [(id)m matrixElementAtIndex: MATRIX_TX] atIndex: MATRIX_TX];
      [self setMatrixElement: [(id)m matrixElementAtIndex: MATRIX_TY] atIndex: MATRIX_TY];
    }
}
- (void)getMatrixValue: (float *)m
{
  matrix_copy([self matrixValue], m);
}
- (void)applyMatrixTo: (NSPoint *)target result: (NSPoint *) result
{
  float * m = [self matrixValue];
  NSPoint local_target;
  
  NSPoint_copy(target, &local_target);
  matrix_apply_on_point(m, &local_target);
  NSPoint_copy(&local_target, result);
  
}
@end

@implementation PSMatrixApplier

- initWithVisitingTarget: (NSObject <FigObjsContaining>*)_target
{
  [super initWithVisitingTarget: _target];
  matrix_init(matrix.c);
  type 		      = C_MATRIX;
  delta_from_origin.x = 0.0;
  delta_from_origin.y = 0.0;
  return self;
}
- (void)dealloc
{
  if (type == OBJC_MATRIX)
    [matrix.objc release];
  [super dealloc];
}

/*
 * Matrix Protocol
 */
- (void)setMatrix: (void *)m ofType: (enum matrix_type) t
{
  if (t == C_MATRIX)
    {
      if (type == OBJC_MATRIX) [matrix.objc release];
      matrix_copy((float *)m, matrix.c);
      type = t;
    }
  else if (t == OBJC_MATRIX)
    {
      if (type == OBJC_MATRIX) [matrix.objc release];
      matrix.objc = [(id)m copy];
      type 	  = t;
      
    }
}
  
- (void)setMatrixElement: (float)element
		 atIndex: (enum matrix_element_index)index 
{
  if (type == C_MATRIX)
    matrix.c[index] = element;
  else if (type == OBJC_MATRIX)
    [matrix.objc setMatrixElement: element atIndex: index];
}
- (float)matrixElementAtIndex: (enum matrix_element_index)index
{
  if (type == C_MATRIX)
    return matrix.c[index];
  else if (type == OBJC_MATRIX)
    return [matrix.objc matrixElementAtIndex: index];
  else
    return 0.0;
}
- (void)getMatrixValue: (float *)m
{
  if (type == C_MATRIX)
    matrix_copy(matrix.c, m);
  else if (type == OBJC_MATRIX)
    [matrix.objc getMatrixValue: m];
}
- (void)applyMatrixTo: (NSPoint *)src result: (NSPoint *) result
{
  float * m;
  float _m[6];
  NSPoint local_src;
  
  if (result == NULL || src == NULL) return;

  if (type == C_MATRIX)
    m = matrix.c;
  else if (type == OBJC_MATRIX)
    {
      m = _m;
      [matrix.objc getMatrixValue: m];
    }
  
  NSPoint_copy(src, &local_src);
  NSPoint_diff(&local_src, &delta_from_origin, &local_src);
  matrix_apply_on_point(m, &local_src);
  NSPoint_add(&local_src, &delta_from_origin, &local_src);
  NSPoint_copy(&local_src, result);
  
}
/*
 * Delta
 */
- (void)setDeltaFromOriginX: (float)x Y: (float)y
{
  delta_from_origin.x = x;
  delta_from_origin.y = y;
}
- (NSPoint *)deltaFromOrigin
{
  return &delta_from_origin;
}

/*
 * GyveVisitor
 */
- (enum gyve_visitor_request)visitAtBuffer:(GyveBuffer *)buffer
{
  return visitor_detail;
}
- (enum gyve_visitor_request)visitAtLayer:(GyveLayer *)layer
{
   return visitor_detail;
}
- (enum gyve_visitor_request)visitAtPath:(PSPath *)path
{
  id<PointsEnumerating> e = [path pointsEnumerator];
  NSPoint * p;
  BOOL delta_calc = !NSEqualPoints(delta_from_origin,NSZeroPoint);
  float * m;
  float _m[6];

  if (type == C_MATRIX)
    m = matrix.c;
  else if (type == OBJC_MATRIX)
    {
      m = _m;
      [matrix.objc getMatrixValue: m];
    }
  
  while (p = (NSPoint * )[e nextPoint], p)
    {
      if (YES == delta_calc)
	{
	  NSPoint_diff(p, &delta_from_origin, p);
	  matrix_apply_on_point(m, p);
	  NSPoint_add(p, &delta_from_origin, p);
	}
      else
	{
	  matrix_apply_on_point(m, (NSPoint *)p);
	}
    }
  return visitor_continue;
}
- (enum gyve_visitor_request)visitAtText:(NSObject<PSText>*)text
{
  float * m;
  float _m[6];
  BOOL delta_calc = !NSEqualPoints(delta_from_origin,NSZeroPoint);

  if (type == C_MATRIX)
    m = matrix.c;
  else if (type == OBJC_MATRIX)
    {
      m = _m;
      [matrix.objc getMatrixValue: m];
    }

  if (YES == delta_calc)
    {
      float * local_m = [text mappingMatrix];
      local_m[MATRIX_TX] -= delta_from_origin.x;
      local_m[MATRIX_TY] -= delta_from_origin.y;
      matrix_apply_on_matrix(m, local_m);
      local_m[MATRIX_TX] += delta_from_origin.x;
      local_m[MATRIX_TY] += delta_from_origin.y;
    }
  else
    {
      matrix_apply_on_matrix(m, [text mappingMatrix]);
    }
  [text zeroBBox];
  return visitor_continue;
}
- (enum gyve_visitor_request)visitAtImage:(PSImage *)image
{
  float * m;
  float _m[6];
  BOOL delta_calc = !NSEqualPoints(delta_from_origin,NSZeroPoint);

  if (type == C_MATRIX)
    m = matrix.c;
  else if (type == OBJC_MATRIX)
    {
      m = _m;
      [matrix.objc getMatrixValue: m];
    }

  if (YES == delta_calc)
    {
      float * local_m = [image mappingMatrix];
      local_m[MATRIX_TX] -= delta_from_origin.x;
      local_m[MATRIX_TY] -= delta_from_origin.y;
      matrix_apply_on_matrix(m, local_m);
      local_m[MATRIX_TX] += delta_from_origin.x;
      local_m[MATRIX_TY] += delta_from_origin.y;
    }
  else
    {
      matrix_apply_on_matrix(m, [image mappingMatrix]);
    }
  [image calcBBox];
  return visitor_continue;
}
- (enum gyve_visitor_request)visitAtFigObjGroup:(PSFigObjGroup *)group
{
  return visitor_detail;
}
- (enum gyve_visitor_request)visitAtMaskedFigObjGroup:(PSMaskedFigObjGroup *)masked_group
{
  return visitor_detail;
}
- (enum gyve_visitor_request)visitAtCompoundPaths: (PSCompoundPaths *)compound_paths
{
  return visitor_detail;
}
- (enum gyve_visitor_request)visitAtProxy: (PSFigObjSelectedProxy *)proxy
{
  return visitor_detail;
}
- (void)leaveFromProxy: (PSFigObjSelectedProxy *)proxy
{
  
}
- (void)leaveFromBuffer:(GyveBuffer *)buffer {}
- (void)leaveFromLayer:(GyveLayer *)layer {}
- (void)leaveFromPath:(PSPath *)path 
{
  [path calcBBox];
}
- (void)leaveFromText:(NSObject<PSText>*)text
{
  
}
- (void)leaveFromImage:(PSImage *)image
{
}
- (void)leaveFromFigObjGroup:(PSFigObjGroup *)group
{
  [group calcBBox];
}
- (void)leaveFromMaskedFigObjGroup:(PSMaskedFigObjGroup *)masked_group
{
  [masked_group calcBBox];
}
- (void)leaveFromCompoundPaths: (PSCompoundPaths *)compound_paths
{
  [compound_paths calcBBox];
}
@end
