/* PSPath.m --- The definition for paths

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "PSPath.h"
#include "PSSegment.h"
#include "public.h"
#include "Gyve.h"
#include "PSPaintStyle.h"
#include "PaintStylePallet.h"
#include "utilities.h"

#include <Foundation/NSObject.h>
#include <Foundation/NSValue.h>
#include <Foundation/NSException.h>

@interface PSPathPointsEnumerator: NSObject<PointsEnumerating>
{
  id<SegmentsEnumerating>inner_enumerator;
  PSSegment * segment;
  int current_index;
  id_predicator_t * filter;
}
- initWithPSPath: (PSPath *)path;
@end

@interface PSPathReversePointsEnumerator: NSObject<PointsEnumerating>
{
  id<SegmentsEnumerating>inner_enumerator;
  PSSegment * segment;
  int current_index;
  id_predicator_t * filter;
}
- initWithPSPath: (PSPath *)path;
@end

@interface PSPathSegmentsEnumerator: NSObject<SegmentsEnumerating>
{
  id<Enumerating> inner_enumerator;
  id_predicator_t * filter;
}
- initWithPSPath: (PSPath *)path;
@end

@interface PSPathReverseSegmentsEnumerator: NSObject<SegmentsEnumerating>
{
  id<Enumerating> inner_enumerator;
  id_predicator_t * filter;
}
- initWithPSPath: (PSPath *)path;
@end



@implementation  PSPath
/* 
 * Constructing and destroying
 */
+ (PSPath *)rectPathFromRect: (NSRect *)rect
{
  PSPath * path;
  NSPoint point;
  point.x = rect->origin.x;
  point.y = rect->origin.y;
  path    = [[[self alloc] initWithMovetoPoint: &point]
	      autorelease];
  
  point.x = rect->origin.x + rect->size.width;
  point.y = rect->origin.y;
  [path addLinetoPoint: &point];

  point.x = rect->origin.x + rect->size.width;
  point.y = rect->origin.y + rect->size.height;
  [path addLinetoPoint: &point];

  point.x = rect->origin.x;
  point.y = rect->origin.y + rect->size.height;
  [path addLinetoPoint: &point];
  
  point.x = rect->origin.x;
  point.y = rect->origin.y;
  [path addLinetoPoint: &point];

  [path closePath];
  return path;
}
+ (PSPath *)roundRectPathFromRect: (NSRect *)rect
		    withRoundSeed: (float)roundseed
{
  NSPoint point0;
  NSPoint point[3];
  PSPath * path;
  
  NSPoint center = NSRect_center(rect);
  float radiusx  = (rect->size.width)/2.0 , radiusy = (rect->size.height)/2.0;
  float centerx = center.x, centery = center.y;
  float roundseedx = radiusx * roundseed, roundseedy = radiusy * roundseed;

  // X
  point0 = NSMakePoint(centerx, centery + radiusy);
  path    = [[[self alloc] initWithMovetoPoint: &point0]
	      autorelease];  
  
  // A
  point[0] = NSMakePoint((centerx + roundseedx), (centery + radiusy));
  point[1] = NSMakePoint((centerx + radiusx), (centery + roundseedy));
  point[2] = NSMakePoint((centerx + radiusx), centery);
  [path addCurvetoPoints: point];

  // B
  point[0] = NSMakePoint((centerx + radiusx), (centery - roundseedy));
  point[1] = NSMakePoint((centerx + roundseedx), (centery - radiusy));
  point[2] = NSMakePoint(centerx, (centery - radiusy));
  [path addCurvetoPoints: point];
  
  // C
  point[0] = NSMakePoint((centerx - roundseedx), (centery - radiusy));
  point[1] = NSMakePoint((centerx - radiusx), (centery - roundseedy));
  point[2] = NSMakePoint( (centerx - radiusx), centery);
  [path addCurvetoPoints: point];

  // D
  point[0] = NSMakePoint((centerx - radiusx), (centery + roundseedy));
  point[1] = NSMakePoint((centerx - roundseedx), (centery + radiusy));
  point[2] = NSMakePoint(centerx, (centery + radiusy));
  [path addCurvetoPoints: point];

  [path closePath];
  return path;
}
+ (PSPath *)ovalPathFromRect: (NSRect *)rect
{
  return [self roundRectPathFromRect: rect
	       withRoundSeed: 0.54];
}
- initWithMovetoPoint: (const NSPoint *)point
{
  PSSegment * moveto = nil;

  [super init];

  moveto = [[[PSSegment alloc] initWithDPSOperation: dps_moveto
			       points: point] autorelease];
  [self appendObject: moveto];

  bbox_make(&bbox, point, point);
 
  locked      = NO;
  order       = YES;
  selected    = NO;
  closed      = NO;
  paint_style = [[PSPaintStyle alloc] initWithPaintStyle: 
					(PaintStylePallet *)[PaintStylePallet sharedObject]];
  return self ;
}
- copyWithZone: (NSZone *)zone
{
  PSPath * copy_path 	    = [[self class] allocWithZone: zone];
  id<SegmentsEnumerating> e = [self segmentsEnumerator];
  PSSegment * seg 	    = [e nextSegment];
  DPSUserPathOp op;

  copy_path 		    = [copy_path initWithMovetoPoint: [seg firstPoint]];

  while (seg = [e nextSegment], seg)
    {
      op = [seg dpsOperation];
      if (dps_lineto == op)
	[copy_path addLinetoPoint: [seg firstPoint]];
      else if (dps_curveto == op)
	{
	  int i;
	  NSPoint points[3];
	  for (i = 0; i < 3; i++)
	    NSPoint_copy([seg pointAtIndex: i], &(points[i]));
	  [copy_path addCurvetoPoints: points];
	}
      else
	{
	  NSAssert(0, @"Wrong dps operation.");
	}
    }

  [copy_path setFigObjLock: [self isLockedFigObj]];
  if ([self isRversed])
    ;				// TODO
  if ([self isClosedPath])
    [copy_path closePath];
  [[copy_path paintStyle] installPaintStyle: [self paintStyle]];
  return copy_path;
}
- (void)addLinetoPoint:  (const NSPoint *)p
{
  PSSegment * lineto = nil;
  BOOL move_point_is_on_zero = NO;
  lineto = [[[PSSegment alloc] initWithDPSOperation: dps_lineto
			      points: p] autorelease];
  [self appendObject: lineto];
  
  if (NSEqualPoints(NSZeroPoint, *[self firstPoint]))
      move_point_is_on_zero = YES;
  
  bbox_expand_by_point(&bbox, p);
  
  if (move_point_is_on_zero)
    bbox_expand_by_point(&bbox, &NSZeroPoint);
}
- (void)addCurvetoPoints: (const NSPoint *)p
{
  PSSegment * curveto = nil;
  BOOL move_point_is_on_zero = NO;

  curveto = [[[PSSegment alloc] initWithDPSOperation: dps_curveto
			       points: p] autorelease];
  [self appendObject: curveto];
  
  if (NSEqualPoints(NSZeroPoint, *[self firstPoint]))
      move_point_is_on_zero = YES;
  bbox_expand_by_point(&bbox, &(p[0]));
  bbox_expand_by_point(&bbox, &(p[1]));
  bbox_expand_by_point(&bbox, &(p[2]));

  if (move_point_is_on_zero)
    bbox_expand_by_point(&bbox, &NSZeroPoint);
}
- (void)dealloc
{
  if (paint_style)
    [paint_style release], paint_style = nil;
  [super dealloc];
}

/*
 * Modification
 */
- (PSSegment *)curveSegmentAtIndex: (int) i
{
  PSSegment * prev_segment;
  PSSegment * tmp;
  PSSegment * result;
  NSPoint points[3];
  if (i == 0)
    {
      return nil;
    }
  tmp = [self segmentAtIndex: i];
  if (nil == tmp)
    {
      return nil;
    }
  else if (dps_curveto == [tmp dpsOperation])
    {
      return tmp;
    }
  else if (dps_lineto == [tmp dpsOperation])
    {
      prev_segment = [self segmentAtIndex: i - 1];
      points[0]    = *[prev_segment lastPoint];
      points[1]    = *[tmp firstPoint];
      points[2]    = *[tmp firstPoint];
      result = [[[PSSegment alloc] 
		  initWithDPSOperation: dps_curveto
		  points: points]
		 autorelease];
      [self replaceObject: tmp withObject: result];
      tmp = nil;
      [self calcBBox];
      return result;
    }
  else
    {
      return nil;		// ???
    }
}
/*
 * Properties
 */
- (PSPaintStyle *)paintStyle
{
  return paint_style;
}
- (void)setPaintStyle: (BYREF PSPaintStyle *) style
{
  ASSGIN_NSOBJECT(paint_style, style);
}
- (BOOL)isClosedPath
{
  return closed;
}
- (void)closePath
{
  closed = YES;
}
- (void)openPath
{
  closed = NO;
}
- (BOOL)isLockedFigObj
{
  return locked;
}
- (void)setFigObjLock: (BOOL)l
{
  locked = l;
}
- (void)reverseOrder
{
  int i, max;
  PSPath * reversed_path;
  NSPoint points_to_pass[3];
  id<SegmentsEnumerating> segment_enum = [self reverseSegmentsEnumerator];
  id<PointsEnumerating> point_enum = [self reversePointsEnumerator];
  PSSegment * current_segment;
  
  NSPoint_copy([point_enum nextPoint], points_to_pass);
  reversed_path = [[[PSPath alloc] initWithMovetoPoint: points_to_pass] 
		    autorelease];

  while (current_segment = [segment_enum nextSegment], current_segment)
    {
      switch([current_segment dpsOperation])
	{
	case dps_curveto:
	  NSPoint_copy([point_enum nextPoint], &points_to_pass[0]);
	  NSPoint_copy([point_enum nextPoint], &points_to_pass[1]);
	  NSPoint_copy([point_enum nextPoint], &points_to_pass[2]);
	  [reversed_path addCurvetoPoints: points_to_pass];
	  break;
	case dps_lineto:
	  NSPoint_copy([point_enum nextPoint], &points_to_pass[0]);
	  [reversed_path addLinetoPoint: points_to_pass];
	case dps_moveto:
	  break;
	default:
	  break;
	}
    }
  NSAssert([point_enum nextPoint] == NULL, @"");

  [self empty];
  max = [reversed_path countSegments];
  for (i = 0; i < max; i++)
    {
      current_segment = [reversed_path segmentAtIndex: i];
      [self appendObject: current_segment];
    }
}
- (BOOL)isRversed
{
  return order;
}
/*
 * Selection
 */
- markAsSelected
{
  selected = YES;
  return self ;
}
- (BOOL)isSelected
{
  return selected;
}
- (void)unMarkAsSelected
{
  selected = NO;
}

/*
 * BBox
 */
- (void)calcBBox
{
  int i, max;
  PSSegment * segment = nil;

  BOOL move_point_is_on_zero = NO;
  const NSPoint * moveto_point = NULL;

  max = [self countSegments];
  for (i = 0; i < max; i++)
    {
      segment = [self segmentAtIndex: i];
      switch ([segment dpsOperation])
	{
	case dps_curveto:
	  bbox_expand_by_point(&bbox, [segment pointAtIndex: 0]);
	  bbox_expand_by_point(&bbox, [segment pointAtIndex: 1]);
	  bbox_expand_by_point(&bbox, [segment pointAtIndex: 2]);
	  if (YES == move_point_is_on_zero)
	    {
	      bbox_expand_by_point(&bbox, moveto_point);
	      move_point_is_on_zero = NO; // DONE
	      moveto_point = NULL;
	    }
	  break;
	case dps_lineto:
	  bbox_expand_by_point(&bbox, [segment pointAtIndex: 0]);
	  if (YES == move_point_is_on_zero)
	    {
	      bbox_expand_by_point(&bbox, moveto_point);
	      move_point_is_on_zero = NO; // DONE
	      moveto_point = NULL;
	    }
	  break;
	case dps_moveto:
	  moveto_point = [segment pointAtIndex: 0];
	  bbox_make(&bbox, 
		    moveto_point,
		    moveto_point);
	  if (NSEqualPoints(NSZeroPoint, *moveto_point))
	    move_point_is_on_zero = YES;
	  break;
	}
    }
  segment = nil;
}
- (struct bbox *)bboxCStructure
{
  return &bbox;
}
- (void)zeroBBox
{
  // I think this method is not useful for anyone.
  // TODO: Throw a exception or someting...
  bbox = zero_bbox;
}
- (BOOL)isZeroBBox
{
  return bbox_is_zero_bbox([self bboxCStructure]);
}
- (BOOL)expandBBoxByBBox:(id<PSBBox>)b
{
  return bbox_expand_by_bbox([self bboxCStructure], [b bboxCStructure]);
}
- (void)reportBBoxTo: (id<PSBBox>)b
{
  [b expandBBoxByBBox: self];
}

/*
 * Points containing protocol
 */
- (NSPoint *)firstPoint
{
  return [[self firstSegment] firstPoint];
}
- (NSPointValue *)firstPointValue
{
  return (NSPointValue *)[NSValue valueWithPoint: 
				    *[self firstPoint]];
}
- (NSPoint *)lastPoint
{
  return [[self lastSegment] lastPoint];
}
- (NSPointValue *)lastPointValue
{
  return (NSPointValue *)[NSValue valueWithPoint: 
				    *[self lastPoint]];
}
- (int)countPoints
{
  int result = 0;
  int i, max;
  PSSegment * seg;
  
  max = [self countSegments];
  for (i = 0; i < max; i++)
    {
      seg = [self segmentAtIndex: i];
      result += [seg countPoints];
    }
  return result;
}
- (id<PointsEnumerating>)pointsEnumerator
{
  return [[[PSPathPointsEnumerator alloc] initWithPSPath: self]
	   autorelease];
}
- (id<PointsEnumerating>)reversePointsEnumerator
{
  return [[[PSPathReversePointsEnumerator alloc] initWithPSPath: self]
	   autorelease];
}
- (BOOL)containsPoint: (const NSPoint *)point
{
  GYVE_SHOULD_NOT_IMPLEMENT(return NO);
}
- (BOOL)containsPointValue: (const NSPointValue *)point
{
  GYVE_SHOULD_NOT_IMPLEMENT(return NO);
}
- (NSPoint *)pointAtIndex: (int)n
{
  GYVE_SHOULD_NOT_IMPLEMENT(return NULL);
}
- (NSPointValue *) pointValueAtIndex:(int)index
{
  GYVE_SHOULD_NOT_IMPLEMENT(return nil);
}

/*
 * Segments containing protocol 
 */
- (int)countSegments
{
  return [self count];
}
- (id<SegmentsEnumerating>)segmentsEnumerator
{
  return [[[PSPathSegmentsEnumerator alloc] initWithPSPath: self]
	   autorelease];
}
- (id<SegmentsEnumerating>)reverseSegmentsEnumerator
{
  return [[[PSPathReverseSegmentsEnumerator alloc] initWithPSPath: self]
	   autorelease];
}
- (PSSegment *)firstSegment
{
  return (PSSegment *)[self objectAtIndex: 0];
}
- (PSSegment *)lastSegment
{
  return (PSSegment *)[self lastObject];
}

/*
 * FigObj protocol 
 */
- (BOOL)isFigObjProxy
{
  return NO;
}
- (NSObject<PSFigObj> *)targetForFigObjProxy
{
  return self;
}
- (float)deltaForExpandingBBox
{
  /* TODO */
  return [paint_style lineWidth]; 
}

- (BOOL)isPointInBBox: (NSPoint *)p;
{
  return bbox_contains_point([self bboxCStructure], p);
}
- (BOOL)intersectsWithBBox: (id<PSBBox>)b;
{
  return bbox_intersects([self bboxCStructure], [b bboxCStructure]);
}
- (BOOL)intersectsWithRect: (NSRect *)r
{
  struct bbox b;
  NSRect_to_bbox (r, &b);
  return bbox_intersects([self bboxCStructure], &b);
}
- (PSSegment *)segmentAtIndex: (int)n
{
  return (PSSegment *) [self objectAtIndex: n];
}
- (int)indexOfSegment: (PSSegment *)seg
{
  return [self indexOfObject:seg];
}
- (BOOL)containsSegment: (PSSegment *)segment
{
  return [self containsObject: segment];
}
- (Class)selectedProxyClass
{
  return Nil;
}
@end				/* PSPath */

@implementation PSPath(PublicGuile)
- initWithMovetoSegmentByPointValue: (NSPointValue *)point
{
  NSPoint p = [point pointValue];
  return [self initWithMovetoPoint: &p];
}
- (void)addLinetoSegmentByPointValue:  (NSPointValue *)point
{
  NSPoint p = [point pointValue];
  [self addLinetoPoint: &p];
}
- (void)addCurvetoSegmentByPointValue: (NSPointValue *)point0 
			       : (NSPointValue *)point1 
			       : (NSPointValue *)point2
{
  NSPoint points[3];
  points[0] = [point0 pointValue];
  points[1] = [point1 pointValue];
  points[2] = [point2 pointValue];
  [self addCurvetoPoints: points];
}
@end

@implementation PSPathPointsEnumerator
- (void)setEnumerationFilter: (id_predicator_t *)f
{ 
  filter = f;
}
- initWithPSPath: (PSPath *)path
{
  [super init];
  inner_enumerator = (id<Enumerating>)[(NSObject *)[path segmentsEnumerator]
						   retain];
  filter = id_predicator_yes;
  segment    	= [[inner_enumerator nextSegment] retain];
  current_index = 0;
  return self ;
}
- (const NSPoint *)nextPoint
{
  const NSPoint * p;
  if (inner_enumerator == nil)
    {
      return NULL;
    }
  else if (current_index < [segment countPoints])
    {
      p = [segment pointAtIndex: current_index++];
      if (p == NULL)
	return NULL;
      else if (YES == filter([NSValue valueWithPoint: *p])) // 
	return p;
      else
	return [self nextPoint];
    }
  else
    {
      current_index = 0;

      [segment release];
      segment = [inner_enumerator nextSegment];
      if (nil == segment)
	inner_enumerator = nil;
      else
	[segment retain];
      return [self nextPoint];
    }
}
- (NSPointValue *)nextPointValue
{
  return (NSPointValue *)[NSValue valueWithPoint: 
				    *[self nextPoint]];
}
- nextObject
{
  return [self nextPointValue];
}
- (void)dealloc
{
  if (inner_enumerator)
    [(NSObject *)inner_enumerator release], inner_enumerator = nil;
  if (segment)
    [segment release], segment = nil;
  [super dealloc];
}
@end


@implementation PSPathReversePointsEnumerator
- (void)setEnumerationFilter: (id_predicator_t *)f
{ 
  filter = f;
}
- initWithPSPath: (PSPath *)path
{
  [super init];
  inner_enumerator = (id<Enumerating>)[(NSObject *)[path 
						     reverseSegmentsEnumerator]
						   retain];
  filter = id_predicator_yes;
  segment 	   = [[inner_enumerator nextSegment] retain];
  current_index    = [[path lastObject] countPoints] - 1;
  return self ;
}
- (const NSPoint *)nextPoint
{
  const NSPoint * p;
  if (inner_enumerator == nil)
    {
      return NULL;
    }
  else if (0 <= current_index)
    {
      p = [segment pointAtIndex: current_index--];
      
      if (NULL == p)
	return NULL;
      else if (YES == filter([NSValue valueWithPoint: *p]))
	return p;
      else
	return [self nextPoint];
    }
  else
    {
      [segment release];
      segment = [inner_enumerator nextSegment];
      if (nil == segment)
	inner_enumerator = nil;
      else
	{
	  [segment retain];
	  current_index = [segment countPoints] - 1;
	}
      return [self nextPoint];
    }
}
- (NSPointValue *)nextPointValue
{
  return (NSPointValue *)[NSValue valueWithPoint: *[self nextPoint]];
}
- nextObject
{
  return [self nextPointValue];
}
- (void)dealloc
{
  if (segment)
    [segment release], segment = nil;
  if (inner_enumerator)
    [(NSObject *)inner_enumerator release], inner_enumerator = nil;
}
@end

@implementation PSPathSegmentsEnumerator
- (void)setEnumerationFilter: (id_predicator_t *)f
{ 
  filter = f;
}
- initWithPSPath: (PSPath *)path
{
  [super init];
  inner_enumerator = [[path objectEnumerator] retain];
  filter 	   = id_predicator_yes;
  return self ;
}
- (PSSegment *)nextSegment
{
  PSSegment * seg = [inner_enumerator nextObject];
  if (nil == seg) return nil;
  else if (YES == filter(seg))
    return seg;
  else
    return [self nextSegment];
}
- nextObject
{
  return [self nextSegment];
}
- (void)dealloc
{
  [inner_enumerator release], inner_enumerator = nil;
  [super dealloc];
}
@end

@implementation PSPathReverseSegmentsEnumerator
- (void)setEnumerationFilter: (id_predicator_t *)f
{ 
  filter = f;
}
- initWithPSPath: (PSPath *)path
{
  [super init];
  inner_enumerator = [[path reverseObjectEnumerator] retain];
  filter 	   = id_predicator_yes;
  return self ;
}
- (PSSegment *)nextSegment
{
  PSSegment * seg = (PSSegment *)[inner_enumerator nextObject];
  if (nil == seg) return nil;
  else if (YES == (filter(seg)))
    return seg;
  else
    return [self nextSegment];
}
- nextObject
{
  return [self nextSegment];
}
- (void)dealloc
{
  [inner_enumerator release], inner_enumerator = nil;
  [super dealloc];
}

@end
