/* PSSegment.m --- The segment definition for paths

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "PSSegment.h"
#include "geometry.h"
#include <Foundation/NSException.h>

int 
count_points_from_path_op (DPSUserPathOp op)
{
  switch(op)
    {
    case dps_curveto:
      return 3;
    case dps_moveto:
    case dps_lineto:
      return 1;
    default:
      return PS_SEGMENT_WRONG_PATH_OP;
    }
}

NSString * PSSegmentWrongOpException = @"PSSegmentWrongOpException";

@interface PSSegmentPointsEnumerator: NSObject<PointsEnumerating>
{
  int current_index;
  PSSegment * segment;
  id_predicator_t * filter;
}
- initWithPSSegment: (PSSegment *)seg;
@end

@interface PSSegmentReversePointsEnumerator: NSObject<PointsEnumerating>
{
  int current_index;
  PSSegment * segment;
  id_predicator_t * filter;
}
- initWithPSSegment: (PSSegment *)segment;
@end

@implementation PSSegment
- initWithDPSOperation: (DPSUserPathOp)o points: (const NSPoint *)c
{
  int length;

  [super init];

  length  = count_points_from_path_op(o);
  if (PS_SEGMENT_WRONG_PATH_OP == length)
    {
      [self release], self = nil;
      [NSException raise: 
		     PSSegmentWrongOpException
		   format: 
		     @"Wrong operation is ginve as an argumet for PSSegment"];
    }
  op = o;
  
  points = objc_malloc(length * sizeof (NSPoint));
  if (NULL == c)
    {
      NSPoint zero[3] = {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}};
      NSPoint_ncopy(zero, points, length);
    }
  else
    NSPoint_ncopy(c, points, length);

  return self ;
}
- copyWithZone:(NSZone *)zone
{
  PSSegment * copy_seg = [[self class] allocWithZone: zone];
  [copy_seg initWithDPSOperation: [self dpsOperation] points: points];
  return copy_seg ;
}

- (void)dealloc
{
  objc_free(points);
  [super dealloc];
}
- (DPSUserPathOp)dpsOperation 
{
  return op; 
}
- (int) countPoints 
{ 
  return (int)count_points_from_path_op(op);
}
- (void)setPoint: (NSPoint*)c atIndex:(int)index
{
  if (index < [self countPoints])
    NSPoint_copy(c, &(points[index]));
  else
    [NSException raise: NSRangeException format: @"Wrong index for PSSegment"]; 
 
}
- (NSPoint *) firstPoint
{
  return [self pointAtIndex: 0];
}
- (NSPointValue *)firstPointValue
{
  return (NSPointValue *)[NSValue valueWithPoint: *[self firstPoint]];
}
- (NSPoint *) lastPoint
{
  int index = 0;

  if (dps_curveto == [self dpsOperation])
    index = 2;
  
  return [self pointAtIndex: index];
}
- (NSPointValue *)lastPointValue
{
  return (NSPointValue *)[NSValue valueWithPoint: *[self lastPoint]];
}
- (NSPoint *) pointAtIndex:(int)index
{
  if (index < [self countPoints])
    {
      return &(points[index]);
    }
  else
    {
      [NSException raise: NSRangeException 
		   format: @"Wrong index for PSSegment"]; 
      return NULL;
    }
}
- (int)indexOfPoint: (NSPoint *)p
{
  NSPoint * point;
  int i;
  int length = [self countPoints];
  for (i = 0; i < length; i++)
    {
      point = [self pointAtIndex: i];
      if (YES == NSPoint_equal(point, p))
	{
	  return i;
	}
    }
  return -1;
}
- (NSPointValue *) pointValueAtIndex:(int)index
{
  const NSPoint * p =  [self pointAtIndex: index];
  if (NULL == p) return nil;
  else return (NSPointValue *)[NSValue valueWithPoint: *p];
}
- (id<PointsEnumerating>)pointsEnumerator
{
  return [[[PSSegmentPointsEnumerator alloc] initWithPSSegment: self]
	   autorelease];
}
- (id<PointsEnumerating>)reversePointsEnumerator
{
  return [[[PSSegmentReversePointsEnumerator alloc] initWithPSSegment: self]
	   autorelease];
}
- (BOOL)containsPoint: (const NSPoint *)point
{
  GYVE_SHOULD_NOT_IMPLEMENT(return NO);
}
- (BOOL)containsPointValue: (const NSPointValue *)point
{
  GYVE_SHOULD_NOT_IMPLEMENT(return NO);
}

@end				/* PSSegment */

@implementation PSSegment(Guile)
- (NSPointValue *) pointValueAtIndex:(int)index
{
  return (NSPointValue *)[NSValue valueWithPoint: *[self pointAtIndex: index]];
}
+ (DPSUserPathOp)movetoDPSOperation { return dps_moveto;}
+ (DPSUserPathOp)linetoDPSOperation { return dps_lineto;}
+ (DPSUserPathOp)curvetoDPSOperation { return dps_curveto;}
@end


@implementation PSSegmentPointsEnumerator
- (void)setEnumerationFilter: (id_predicator_t *)f
{ 
  filter = f;
}
- initWithPSSegment: (PSSegment *)seg
{
  segment 	= [seg retain];
  current_index = 0;
  filter 	= id_predicator_yes;
  [super init];
  return self ;
}
- (const NSPoint *)nextPoint
{
  const NSPoint * p;
  if (current_index < [segment countPoints])
    {
      p = [segment pointAtIndex: current_index++];
      if (NULL == p)
	return NULL;
      else if (YES == filter ((id)p))
	return p;
      else
	return [self nextPoint];
    }
  else
    return NULL;
}
- (NSPointValue *)nextPointValue
{
  const NSPoint * p = [self nextPoint];
  if (p)
    return (NSPointValue *)[NSValue valueWithPoint: *p];
  else
    return nil;
}
- nextObject
{
  return [self nextPointValue];
}
- (void)dealloc
{
  [segment release], segment = nil;
  [super dealloc];
}
@end

@implementation PSSegmentReversePointsEnumerator
- (void)setEnumerationFilter: (id_predicator_t *)f
{ 
  filter = f;
}
- initWithPSSegment: (PSSegment *)seg
{
  segment 	= [seg retain];
  current_index = [segment countPoints] - 1;
  filter 	= id_predicator_yes;
  [super init];
  return self ;
}
- (const NSPoint *)nextPoint
{
  const NSPoint * p;
  if (current_index >= 0)
    {
      p = [segment pointAtIndex: current_index--];
      if (NULL == p)
	return NULL;
      else if (YES == filter ((id) p))
	return p;
      else
	return [self nextPoint];
    }
  else
    return NULL;
}
- (NSPointValue *)nextPointValue
{
  const NSPoint * p = [self nextPoint];
  if (p)
    return (NSPointValue *)[NSValue valueWithPoint: *p];
  else
    return nil;
}
- nextObject
{
  return [self nextPointValue];
}
- (void)dealloc
{
  [segment release], segment = nil;
  [super dealloc];
}
@end
