/* PSText.m --- The definition for text 
   
   Copyright (C) 1998 Free Software Foundation, Inc.
   
   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "PSText.h"
#include "PSTextElement.h"
#include "utilities.h"
#include "geometry.h"
#include "PSPaintStyle.h"
#include "TextStylePallet.h"
#include "PaintStylePallet.h"

#include <Foundation/NSException.h>
#include <Foundation/NSData.h>

// PSTextAtPoint
@implementation  PSTextAtPoint
- initWithString: (NSString *)str point: (BYCOPY const NSPoint *)p
{
  const char * cstr;
  [super init];

  if (nil != str)
    {
      cstr = [str cString];
      while (*cstr != '\0')
	{
	  [self addTextElement:
		  [PSTextElement textElementWithCharacter: *cstr++]];
	}
    }

  mapping_matrix[MATRIX_A] = 1.0;
  mapping_matrix[MATRIX_B] = 0.0;
  mapping_matrix[MATRIX_C] = 0.0;
  mapping_matrix[MATRIX_D] = 1.0;
  if (NULL != p)
    {
      mapping_matrix[MATRIX_TX] = p->x;
      mapping_matrix[MATRIX_TY] = p->y;
    }
  else
    {  
      mapping_matrix[MATRIX_TX] = 0.0;
      mapping_matrix[MATRIX_TY] = 0.0;
    }
  lock 		= NO;
  selected  	= NO;
  [self zeroBBox];
  return self ;
}

- initWithString: (NSString *)str pointValue: (BYCOPY NSPointValue *)p
{
  NSPoint point = [p pointValue];
  return [self initWithString: str point: &point];
}
- initWithPoint: (const NSPoint *)p
{
  return [self initWithString: nil point: p];
}
- initWithPointValue: (NSPointValue *)p
{
  NSPoint point = [p pointValue];
  return [self initWithPoint: &point];
}
- copyWithZone: (NSZone *)zone
{
  int i, max;
  PSTextAtPoint * new_text  = [[[self class] allocWithZone: zone] 
				initWithPoint: NULL];
  max = [self countTextElements];
  for (i = 0; i < max; i++)
    [new_text addTextElement:
		[[[self textElementAtIndex: i] copyWithZone: zone] autorelease]];
  bbox_copy([self bboxCStructure], [new_text bboxCStructure]);
  matrix_copy([self mappingMatrix], [new_text mappingMatrix]);
  return new_text;
}

- (const char*) cString
{
  int i, max = [self countTextElements];
  char * cstr_base = objc_malloc(max + 1);
  char * cstr 	     = cstr_base;
  PSTextElement * elt;

  for (i = 0; i < max ; i++)
    {
      elt = [self textElementAtIndex: i];
      *cstr = (char)[elt character];
      cstr++;
    }
  *cstr = '\0';

  [NSData dataWithBytesNoCopy: cstr_base length: max + 1];
  return cstr_base;
}
- (NSString*) string
{
  return [NSString stringWithCString: [self cString]];
}
- (NSString*)substringFromRange: (NSRange)range
{
  return [[NSString stringWithCString: [self cString]]
	   substringFromRange: range];
}
- (float *)mappingMatrix
{
  return mapping_matrix;
}
- (void)setMappingMatrix: (float *)new_mapping_matrix
{
  matrix_copy(new_mapping_matrix, mapping_matrix);
}
- (BOOL)hasZeroBBoxTextElement
{
  int i, max;
  PSTextElement * elt;
  max = [self countTextElements];
  for (i = 0; i < max; i++)
    {
      elt = [self textElementAtIndex: i];
      if ((NO == [elt isNewLine] && YES == [elt isZeroBBox]))
	{
	  return YES;
	}
    }
  return NO;
}

// PSText
- (int)countTextElements
{
  return [self count];
}
- (PSTextElement *)textElementAtIndex: (int)n
{
  return (PSTextElement *)[self objectAtIndex: n];
}
- (PSTextElement *)firstTextElement
{
  if ([self countTextElements] < 1)
    {
      return nil;
    }
  else
    {
      return [self textElementAtIndex: 0];
    }
}
- (PSTextElement *)lastTextElement
{
  if ([self countTextElements] < 1)
    {
      return nil;
    }
  else
    {
      return [self textElementAtIndex: ([self countTextElements] - 1)];
    }
}
- (int)indexOfTextElement: (BYREF PSTextElement *)e
{
  return (int)[self indexOfObject: e];
}
- (void)addTextElement: (BYREF PSTextElement *)e
{
  if ((nil != e)
       &&  ([self countTextElements] == 0))
    {
      if (nil == [e textStyle])
	{
	  [e setTextStyle: [TextStylePallet textStylePallet]];
	}
      if (nil == [e paintStyle])
	{
	  [e setPaintStyle: [PaintStylePallet paintStylePallet]];
	}
    }
  [self addObject: e];
  [self zeroBBox];
}
- (void)insertTextElement: (BYREF PSTextElement *)e atIndex: (int)n
{
  if ((nil != e) 
      && (n == 0))
    {
      if (nil == [e textStyle])
	{
	  [e setTextStyle: [TextStylePallet textStylePallet]];
	}
      if (nil == [e paintStyle])
	{
	  [e setPaintStyle: [PaintStylePallet paintStylePallet]];
	}
    }
  [self insertObject: e atIndex: n];
  [self zeroBBox];
}
- (void)insertString: (BYCOPY NSString *)str atIndex: (int)n
{
  const char * cstr;
  cstr = [str cString];
  while (*cstr != '\0')
    {
      [self insertTextElement: 
	      [PSTextElement textElementWithCharacter: *cstr++]
	    atIndex: n++];
    }
  [self zeroBBox];
}
- (void)removeLastTextElement
{
  [self removeLastObject];
  [self zeroBBox];
}
- (void)removeTextElementAtIndex: (int)n
{
  [self removeObjectAtIndex: n];
  [self zeroBBox];
}
- (void)removeTextElementsInRange: (NSRange *)range
{
  // [self removeObjectsInRange: *range];
  [self removeRange: *range];
  [self zeroBBox];
}
- (BOOL)isFigObjProxy
{
  return NO;
}
- (NSObject<PSFigObj> *)targetForFigObjProxy
{
  return self;
}
- markAsSelected
{
  selected = YES;
  return self ;
}
- (BOOL)isSelected
{
  return selected;
}
- (void)unMarkAsSelected
{
  selected = NO;
}
- (float)deltaForExpandingBBox
{
  return 0.0;			
}
- (BOOL)isLockedFigObj
{
  return lock;
}
- (void)setFigObjLock: (BOOL)l
{
  lock = l;
}

// BBox
- (void)calcBBox
{
  [self zeroBBox];
  [self makeObjectsPerform: @selector(reportBBoxTo:) withObject: self];
}
- (struct bbox *)bboxCStructure
{
  return &bbox;
}
- (void)zeroBBox
{
  bbox_copy(&zero_bbox, [self bboxCStructure]);
}
- (BOOL)isZeroBBox
{
  return bbox_is_zero_bbox([self bboxCStructure]);
}
- (BOOL)expandBBoxByBBox:(id<PSBBox>)b
{
  return bbox_expand_by_bbox([self bboxCStructure], [b bboxCStructure]);
}
- (void)reportBBoxTo: (id<PSBBox>)b
{
  [b expandBBoxByBBox: self];
}
- (BOOL)isPointInBBox: (NSPoint *)p
{
  return bbox_contains_point([self bboxCStructure], p);
}
- (BOOL)intersectsWithBBox: (id<PSBBox>)b
{
  return bbox_intersects([self bboxCStructure], [b bboxCStructure]);
}
- (BOOL)intersectsWithRect: (NSRect *)r
{
  struct bbox b;
  NSRect_to_bbox (r, &b);
  return bbox_intersects([self bboxCStructure], &b);
}
- (Class)selectedProxyClass
{
  return [PSTextSelectedProxy class];
}
@end

@implementation PSTextSelectedProxy
- initForFigObj:  (NSObject<PSFigObj> *)figobj 
      fromLayer:  (GyveLayer *)layer
       ofBuffer: (GyveBuffer *)_buffer
{
  [super initForFigObj: figobj fromLayer: layer ofBuffer: _buffer];
  has_insertion_index = NO;
  return self; 
}
- (id<PSText>)targetForTextProxy
{
  return (id<PSText>)[self targetForFigObjProxy];
}
- (void)setInsertionIndex:(int)i
{
  selected_range      = NSMakeRange(i, 0);
  has_insertion_index = YES;
}
- (void)setSelectionRange:(BYCOPY NSRange *)r
{
  if (r == NULL)
    {
      selected_range.length = 0;
    }
  else
    {
      has_insertion_index = YES;
      NSRange_copy(r, &selected_range);
    }  
}
- (void)removeTextElements
{
  if (YES == [self hasSelectionRange])
    {
      [self removeTextElementsInRange: [self selectionRange]];
      [self clearSelectionRange];
    }
  else if ((YES  == [self hasInsertionIndex])
	   && (0 != [self insertionIndex]))
    {
      [self removeTextElementAtIndex: 
	      [self insertionIndex] - 1];
      [self setInsertionIndex: [self insertionIndex] - 1];
    }
}
- (void)insertString: (BYCOPY NSString *)str
{
  int i;
  if (YES == [self hasSelectionRange]) 
    {
      [self removeTextElements];
    }
  if ([self hasInsertionIndex])
    {
       i = [self insertionIndex];
       [self insertString: str atIndex: i];
       i += [str length];
       [self setInsertionIndex: i];
    }
}
- (void)insertTextElement: (BYREF PSTextElement *)e
{
  int i;
  if (YES == [self hasSelectionRange]) 
    {
      [self removeTextElements];
    }
  if ([self hasInsertionIndex])
    {
       i = [self insertionIndex];
       [self insertTextElement: e atIndex: i++];
       [self setInsertionIndex: i];
    }
}
- (NSString*)substring
{
  if (YES == [self hasSelectionRange])
    {
      return [self substringFromRange: *[self selectionRange]];
    }
  else
    {
      return nil;
    }
}
- (void)clearSelectionRange
{
  [self setSelectionRange: NULL];
}
- (void)clearInsertionIndex
{
  has_insertion_index = NO;
}
- (NSRange *)selectionRange
{
  if (YES == has_insertion_index)
    {
      return &selected_range;
    }
  else
    {
      return NULL;
    }
}
- (int)selectionLength
{
  if (YES == has_insertion_index)
    {
      return selected_range.length;
    }
  else
    {
      return -1;
    }
}
- (int)selectionLocation
{
  if (YES == has_insertion_index)
    {
      return selected_range.location;
    }
  else
    {
      return -1;
    }
}
- (int)insertionIndex
{
  return [self selectionLocation];
}
- (BOOL)hasSelectionRange
{
  if (NULL == [self selectionRange])
    {
      return NO;
    }
  else if (0 == [self selectionLength] )
    {
      return NO;
    }
  else
    {
      return YES;
    }
}
- (BOOL)hasInsertionIndex
{
  if (NULL == [self selectionRange])
    {
      return NO;
    }
  else
    {
      return YES;
    }
}
- (BOOL)isIndexInSelectionRange: (int)n
{
  if (NO == [self hasSelectionRange])
    {
      return NO;
    }
  else
    {
      return NSLocationInRange(n, *[self selectionRange]);

    }
}
- (BOOL)isTextElementInSelectionRange: (PSTextElement *)e
{
  int i = [(id<PSText>)[self targetForFigObjProxy]
		       indexOfTextElement: e];
  if (YES == [self isIndexInSelectionRange: i])
    {
      return YES;
    }
  else
    {
      i = [(id<PSText>)[self originalTargetForFigObjProxy]
		       indexOfTextElement: e];
      return [self isIndexInSelectionRange: i];
    }
}
@end

@implementation PSTextSelectedProxy(PSText)
- (int)countTextElements
{
  return [[self targetForTextProxy] countTextElements];
}
- (PSTextElement *)firstTextElement
{
  return [[self targetForTextProxy] firstTextElement];
}
- (PSTextElement *)lastTextElement
{
  return [[self targetForTextProxy] lastTextElement];
}
- (PSTextElement *)textElementAtIndex: (int)n
{
  return [[self targetForTextProxy] textElementAtIndex: n];
}
- (int)indexOfTextElement: (BYREF PSTextElement *)e
{
  return [[self targetForTextProxy] indexOfTextElement: e];
}

// Adding and inserting
- (void)addTextElement: (BYREF PSTextElement *)e
{
  [[self targetForTextProxy] addTextElement: e];
}
- (void)insertTextElement: (BYREF PSTextElement *)e atIndex: (int)n
{
  [[self targetForTextProxy] insertTextElement: e atIndex: n];
}
- (void)insertString: (BYCOPY NSString *)str atIndex: (int)n
{
  [[self targetForTextProxy] insertString: str atIndex: n];
}

// Removing
- (void)removeLastTextElement
{
  [[self targetForTextProxy] removeLastTextElement];
}
- (void)removeTextElementAtIndex: (int)n
{
  [[self targetForTextProxy] removeTextElementAtIndex: n];
}
- (void)removeTextElementsInRange: (NSRange *)range
{
  [[self targetForTextProxy] removeTextElementsInRange: range];
}

- (BOOL)hasZeroBBoxTextElement
{
  return [[self targetForTextProxy] hasZeroBBoxTextElement];
}

// 
- (NSString*) string
{
  return [[self targetForTextProxy] string];
}
- (NSString*)substringFromRange: (NSRange)range
{
  return [[self targetForTextProxy] substringFromRange: range];
}
@end
