/* geometry.m --- Implementation for geometry releated routines for GYVE

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "geometry.h"
#include <stdio.h>
#include <Foundation/NSException.h>
#include <math.h>
#include <float.h>
#include "utilities.h"


/* NSPoint */
void
NSPoint_copy(const NSPoint * src, NSPoint * dist)
{
  if (dist == src) return ;

  dist->x = src->x;
  dist->y = src->y;
}

void 
NSPoint_ncopy(const NSPoint * src, NSPoint * dist, const int n)
{
  int num_to_copy = n;
  if (src == dist) return ;
  while (num_to_copy > 0)
    {
      NSPoint_copy(src, dist);
      src++;
      dist++;
      num_to_copy--;
    }
}

BOOL
NSPoint_in_bbox(const NSPoint * point, const struct bbox * bbox)
{
  if ((bbox_element(bbox, BBOX_LLX) < point->x) &&
      (point->x < bbox_element(bbox, BBOX_URX)) &&
      (bbox_element(bbox, BBOX_LLY) < point->y) &&
      (point->y < bbox_element(bbox, BBOX_URY)))
    return YES;
  else
    return NO;
}

float 
NSPoint_distance(const NSPoint * a, const NSPoint * b)
{
  float d = NSPoint_power_of_distance(a, b);

  /* TODO:
     Solaris2 doesn't have sqrtf but I think I should use sqrtf
     if the system has... -Masatake */
  return sqrt(d);
}

float
NSPoint_power_of_distance(const NSPoint * a, const NSPoint * b)
{
  float dx = a->x - b->x;
  float dy = a->y - b->y;
  return dx*dx + dy*dy;
}

BOOL
NSPoint_equal(const NSPoint * p, const NSPoint * q)
{
  if ((p->x == q->x) 
      && (p->y == q->y))
    {
      return YES;
    }
  else
    {
      return NO;
    }
}

BOOL
NSPoint_is_zero_point(const NSPoint * p)
{
  return NSPoint_equal(p, &NSZeroPoint);
}

void 
NSPoint_diff (NSPoint * a, NSPoint * b, NSPoint * result)
{
  float tmp;
  tmp 	    = a->x - b->x;
  result->x = tmp;
  tmp = a->y - b->y;
  result->y = tmp;
}

void
NSPoint_add (NSPoint * a, NSPoint * b, NSPoint * result)
{
  float tmp;
  tmp 	    = a->x + b->x;
  result->x = tmp;
  tmp = a->y + b->y;
  result->y = tmp;
}

void
NSPoint_from_gtkAllocation(NSPoint * start, 
			   NSPoint * end, 
			   GtkAllocation * allocation)
{
  if (NULL != start)
    {
      start->x = allocation->x;
      start->y = allocation->y;
    }
  if (NULL != end)
    {
      end->x = allocation->x + allocation->width;
      end->y = allocation->y + allocation->height;
    }
}

void
NSPoint_print(NSPoint * p, FILE * stream)
{
  if (NULL == stream)
    {
      stream = stderr;
    }

  if (NULL != p)
    {
      fprintf(stderr, "NSPoint: [%f %f]\n", p->x, p->y);
    }
  else
    {
      fprintf(stderr, "NSPoint: NULL\n");
    }
}


/* BBox */

void
bbox_copy(const struct bbox * src, struct bbox * dist)
{
  NSPoint_copy(&(src->lower_left), &(dist->lower_left));
  NSPoint_copy(&(src->upper_right), &(dist->upper_right));
}

float 
bbox_element(const struct bbox * bbox, const enum bbox_element_index index)
{
  switch(index)
    {
    case BBOX_LLX:
      return bbox->lower_left.x;
    case BBOX_LLY:
      return bbox->lower_left.y;
    case BBOX_URX:
      return bbox->upper_right.x;
    case BBOX_URY:
      return bbox->upper_right.y;
    default: {
	NSCAssert(0, @"Wrong index in bbox_element.");
	return 0;		/*  NOT REACHED */
      }
    }
}

void
bbox_set(struct bbox * bbox, enum bbox_element_index index, float value)
{
  switch (index)
    {
    case BBOX_LLX:
      bbox->lower_left.x = value;
      return ;
    case BBOX_LLY:
      bbox->lower_left.y = value;
      return ;
    case BBOX_URX:
      bbox->upper_right.x = value;
      return ;
    case BBOX_URY:
      bbox->upper_right.y = value;
      return ;
    }
}

void 
bbox_make(struct bbox * bbox, const NSPoint * a, const NSPoint * b)
{
  if (a->x < b->x)
    {
      bbox_set(bbox, BBOX_LLX, a->x);
      bbox_set(bbox, BBOX_URX, b->x);
    }
  else
    {
      bbox_set(bbox, BBOX_LLX, b->x);
      bbox_set(bbox, BBOX_URX, a->x);
    }
  
  if (a->y < b->y)
    {
      bbox_set(bbox, BBOX_LLY, a->y);
      bbox_set(bbox, BBOX_URY, b->y);
    }
  else
    {
      bbox_set(bbox, BBOX_LLY, b->y);
      bbox_set(bbox, BBOX_URY, a->y);
    }
}

BOOL
bbox_in_bbox(const struct bbox * big, const struct bbox * small)
{
  if ((bbox_element(big, BBOX_LLX) < bbox_element(small, BBOX_LLX)) &&
      (bbox_element(big, BBOX_LLY) < bbox_element(small, BBOX_LLY)) &&
      (bbox_element(big, BBOX_URX) > bbox_element(small, BBOX_URX)) &&
      (bbox_element(big, BBOX_URY) > bbox_element(small, BBOX_URY)))
    return YES;
  else
    return NO;
}

void
bbox_expand_by_delta(struct bbox * bbox, float delta)
{
  enum bbox_element_index index;
  index = BBOX_LLX;
  bbox_set(bbox, index, bbox_element(bbox, index) - delta);
  index = BBOX_LLY;
  bbox_set(bbox, index, bbox_element(bbox, index) - delta);

  index = BBOX_URX;
  bbox_set(bbox, index, bbox_element(bbox, index) + delta);
  index = BBOX_URY;
  bbox_set(bbox, index, bbox_element(bbox, index) + delta);
}

BOOL
bbox_expand_by_point(struct bbox * bbox, const NSPoint * point)
{
  BOOL is_expanded = NO;
  enum bbox_element_index index;

  if (bbox_is_zero_bbox(bbox))
    {
      struct bbox point_bbox;
      point_bbox.lower_left = *point;
      point_bbox.upper_right = *point;
      bbox_copy(&point_bbox, bbox);
      is_expanded = YES;
    }
  else
    {
      if (index = BBOX_LLX, (bbox_element(bbox, index) > point->x))
	{
	  bbox_set(bbox, index, point->x);
	  is_expanded = YES;
	}
      else if (index = BBOX_URX, (bbox_element(bbox, index) < point->x))
	{
	  bbox_set(bbox, index, point->x);
	  is_expanded = YES;
	}

      if (index = BBOX_LLY, (bbox_element(bbox, index) > point->y))
	{
	  bbox_set(bbox, index, point->y);
	  is_expanded = YES;
	}
      else if (index = BBOX_URY, (bbox_element(bbox, index) < point->y))
	{
	  bbox_set(bbox, index, point->y);
	  is_expanded = YES;
	}
    }
  return is_expanded;
}

BOOL
bbox_expand_by_bbox(struct bbox * master, const struct bbox * expander)
{
  BOOL is_expanded  = NO;
  if (bbox_is_zero_bbox(master) && (!bbox_is_zero_bbox(expander)))
    {
      bbox_copy(expander, master);
      is_expanded = YES;
    }
  else if (!bbox_is_zero_bbox(expander))
    {
      if (bbox_expand_by_point(master, &(expander->lower_left)))
	is_expanded = YES;
      if (bbox_expand_by_point(master, &(expander->upper_right)))
	is_expanded = YES;
    }
  return is_expanded;
}

void
bbox_to_rect(const struct bbox * from_bbox, NSRect * to_rect)
{
  float x, y, h, w;
  x = bbox_element(from_bbox, BBOX_LLX);
  y = bbox_element(from_bbox, BBOX_LLY);
  w = bbox_element(from_bbox, BBOX_URX) - x;
  h = bbox_element(from_bbox, BBOX_URY) - y;
  to_rect->origin.x = x;
  to_rect->origin.y = y;
  to_rect->size.width = w;
  to_rect->size.height = h;
}

const struct bbox zero_bbox = {{0.0, 0.0}, {0.0, 0.0}};

BOOL 
bbox_is_zero_bbox (const struct bbox * bbox)
{
  return bbox_equal(bbox, &zero_bbox);
}

BOOL 
bbox_intersects (const struct bbox * a, const struct bbox * b)
{
  // TODO:
  // This function uses NSRect.
  // Write BBox native version.
  NSRect a_rect, b_rect;
  bbox_to_rect(a, &a_rect);
  bbox_to_rect(b, &b_rect);
  return NSIntersectsRect(a_rect, b_rect);
}

BOOL 
bbox_contains_point (const struct bbox * bbox, const NSPoint * point)
{
  // TODO:
  // This function uses NSRect.
  // Write BBox native version.
  NSRect rect;
  bbox_to_rect(bbox, &rect);
  return NSPointInRect(*point, rect);
}

void
bbox_print(const struct bbox * bbox, FILE *stream)
{
  if (NULL == stream)
    {
      stream = stderr;
    }

  if (NULL != bbox)
    {
      fprintf(stream, 
	      "bbox: [%f %f %f %f]\n",
	      bbox_element(bbox, BBOX_LLX),
	      bbox_element(bbox, BBOX_LLY),
	      bbox_element(bbox, BBOX_URX),
	      bbox_element(bbox, BBOX_URY));
    }
  else
    {
      fprintf(stream, "bbox: NULL\n");
    }
}

void 
bbox_from_points(struct bbox * bbox, NSPoint * a, NSPoint * b)
{
  // TODO
  // Write BBox native version.
  NSRect rect;
  NSRect_from_points(&rect, a, b);
  NSRect_to_bbox(&rect, bbox);
}

NSPoint
bbox_center(struct bbox * bbox)
{
  NSPoint result;
  result.x = bbox_element(bbox, BBOX_URX) - bbox_element(bbox, BBOX_LLX);
  result.x /= 2.0;
  result.x += bbox_element(bbox, BBOX_LLX);
  result.y = bbox_element(bbox, BBOX_URY) - bbox_element(bbox, BBOX_LLY);
  result.y /= 2.0;
  result.y += bbox_element(bbox, BBOX_LLY);
  return result;
}

BOOL
bbox_equal(const struct bbox * a, const struct bbox * b)
{
  if (a == NULL || b == NULL)
    {
      return NO;
    }
  else if (a == b)
    {
      return YES;
    }
  else if ((bbox_element(a, BBOX_LLX) == bbox_element(b, BBOX_LLX))
      && (bbox_element(a, BBOX_LLY) == bbox_element(b, BBOX_LLY))
      && (bbox_element(a, BBOX_URX) == bbox_element(b, BBOX_URX))
      && (bbox_element(a, BBOX_URY) == bbox_element(b, BBOX_URY)))
  {
    return YES;
  }
  else
  {
    return NO;
  }
}

/* Size */
void
NSSize_copy(const NSSize * src, NSSize * dist)
{
  if (src == dist) return ;
  
  dist->width  = src->width;
  dist->height = src->height;
}

void
NSSize_from_points(NSSize *size, NSPoint * p0, NSPoint * p1)
{
  size->width  = p0->x - p1->x;
  size->height = p0->y - p1->y;
}

void NSSize_from_gtkAllocation
(NSSize * size, GtkAllocation * allocation)
{
  size->width  = allocation->width;
  size->height = allocation->height;
}

void
NSSize_diff (NSSize * a, NSSize * b, NSSize * result)
{
  NSSize tmp;
  tmp.width = a->width - b->width;
  tmp.height = a->height - b->height;
  *result    = tmp;
}

BOOL
NSSize_is_zero_size(NSSize * size)
{
  if ((size->width == NSZeroSize.width) &&
      (size->height == NSZeroSize.height))
    return YES;
  else
    return NO;
}


/* Rect */
void
NSRect_copy(const NSRect * src, NSRect * dist)
{
  if (dist == src) return ;
  
  NSPoint_copy(&(src->origin), &(dist->origin));
  NSSize_copy(&(src->size), &(dist->size));
}

void 
NSRect_to_bbox(const NSRect * from_rect, struct bbox * to_bbox)
{
  float x = from_rect->origin.x;
  float y = from_rect->origin.y;
  bbox_set(to_bbox, BBOX_LLX, x);
  bbox_set(to_bbox, BBOX_LLY, y);
  bbox_set(to_bbox, BBOX_URX, from_rect->size.width + x);
  bbox_set(to_bbox, BBOX_URY, from_rect->size.height + y);
}

void
NSRect_from_points(NSRect * rect, const NSPoint * a, const NSPoint * b)
{
  struct bbox bbox;
  bbox_make(&bbox, a, b);
  bbox_to_rect(&bbox, rect);
}

void 
NSRect_expand_by_delta(NSRect * rect, float delta)
{
  rect->origin.x -= delta;
  rect->origin.y -= delta;
  rect->size.width += (2* delta);
  rect->size.height += (2 * delta);
}

/* YES is (equal? rect NSZeroRect) */
BOOL
NSRect_is_zero_rect(const NSRect * rect)
{
  if ((rect->origin.x == 0.0) &&
      (rect->origin.y == 0.0) &&
      (rect->size.width == 0.0) &&
      (rect->size.height == 0.0))
    return YES;
  else
    return NO;
}

void
NSRect_print(const NSRect * rect, FILE * stream)
{
  if (NULL == stream)
    {
      stream = stderr;
    }

  if (NULL != rect)
    {
      fprintf(stream, "NSRect: [origin: [%f %f], size: [%f %f]]\n",
	      rect->origin.x, rect->origin.y, 
	      rect->size.width, rect->size.height);
    }
  else
    {
      fprintf(stream, "NSRect: NULL\n");
    }
}

NSPoint
NSRect_origin(NSRect * rect)
{
  return rect->origin;
}

NSPoint
NSRect_end(NSRect * rect)
{
  NSPoint point;
  point.x = rect->origin.x + rect->size.width;
  point.y = rect->origin.y + rect->size.height;
  return point;
}

void
NSRect_from_gtkAllocation(NSRect * rect, GtkAllocation * allocation)
{
  rect->origin.x   = allocation->x;
  rect->origin.y   = allocation->y;
  rect->size.width = allocation->width;
  rect->size.height = allocation->height;
}

void
NSRect_from_NSSize(NSRect * rect, NSSize * size)
{
  rect->origin = NSZeroPoint;
  rect->size   = *size;
}

void
NSRect_from_center(NSRect * rect, NSPoint * p, float h, float w)
{
  rect->origin.x   = ((p->x) - (w/2.0));
  rect->origin.y   = ((p->y) - (h/2.0));
  rect->size.width = w;
  rect->size.height = h;
}

NSPoint
NSRect_center(NSRect * rect)
{
  NSPoint point;
  point.x = NSMidX(*rect);
  point.y = NSMidY(*rect);
  return point;
}


/* Matrix */
void
matrix_init(float * matrix)
{
  matrix[MATRIX_A] = 1.0; matrix[MATRIX_B] = 0.0; matrix[MATRIX_TX] = 0.0;
  matrix[MATRIX_C] = 0.0; matrix[MATRIX_D] = 1.0; matrix[MATRIX_TY] = 0.0;
}

void
matrix_apply_on_point(float * matrix, NSPoint * point)
{
  float x, y;
  x = matrix[MATRIX_A]*point->x + matrix[MATRIX_C]*point->y + matrix[MATRIX_TX];
  y = matrix[MATRIX_B]*point->x + matrix[MATRIX_D]*point->y + matrix[MATRIX_TY];
  point->x = x;
  point->y = y;
}

void
matrix_apply_on_matrix(float * matrix, float * target)
{
  float tmp[6];
  matrix_copy(target, tmp);
  // 
  target[MATRIX_A] = 
    tmp[MATRIX_A]*matrix[MATRIX_A] + tmp[MATRIX_B]*matrix[MATRIX_C];
  target[MATRIX_B] = 
    tmp[MATRIX_A]*matrix[MATRIX_B] + tmp[MATRIX_B]*matrix[MATRIX_D];
  target[MATRIX_C] = 
    tmp[MATRIX_C]*matrix[MATRIX_A] + tmp[MATRIX_D]*matrix[MATRIX_C];
  target[MATRIX_D] = 
    tmp[MATRIX_C]*matrix[MATRIX_B] + tmp[MATRIX_D]*matrix[MATRIX_D];
  
  target[MATRIX_TX] = 
    matrix[MATRIX_A]*tmp[MATRIX_TX] + matrix[MATRIX_C]*tmp[MATRIX_TY] + matrix[MATRIX_TX];
  target[MATRIX_TY] = 
    matrix[MATRIX_B]*tmp[MATRIX_TX] + matrix[MATRIX_D]*tmp[MATRIX_TY] + matrix[MATRIX_TY];
}

void
matrix_apply_on_bbox(float * matrix, struct bbox * target)
{
  /*
     d------------c
     |            |
     |            |
     |            |
     |            |
     |            |
     a------------b
   */
  
  NSPoint a, b, c, d;
  
  a.x = bbox_element(target, BBOX_LLX);
  a.y = bbox_element(target, BBOX_LLY);
  matrix_apply_on_point(matrix, &a);

  b.x = bbox_element(target, BBOX_URX);
  b.y = bbox_element(target, BBOX_LLY);
  matrix_apply_on_point(matrix, &b);

  c.x = bbox_element(target, BBOX_URX);
  c.y = bbox_element(target, BBOX_URY);
  matrix_apply_on_point(matrix, &c);

  d.x = bbox_element(target, BBOX_LLX);
  d.y = bbox_element(target, BBOX_URY);
  matrix_apply_on_point(matrix, &d);  
  
  *target = zero_bbox;
  bbox_expand_by_point(target, &a);
  bbox_expand_by_point(target, &b);
  bbox_expand_by_point(target, &c);
  bbox_expand_by_point(target, &d);
}

void
matrix_copy(float * src, float * dist)
{
  dist[MATRIX_A] = src[MATRIX_A];
  dist[MATRIX_B] = src[MATRIX_B];
  dist[MATRIX_C] = src[MATRIX_C];
  dist[MATRIX_D] = src[MATRIX_D];
  dist[MATRIX_TX] = src[MATRIX_TX];
  dist[MATRIX_TY] = src[MATRIX_TY];
}

void
matrix_print(float * matrix, FILE * stream)
{
  if (NULL == stream)
    {
      stream = stderr;
    }

  if (NULL != matrix)
    {
      fprintf(stream, "matrix: [%f %f %f %f %f %f]\n",
	      matrix[MATRIX_A],
	      matrix[MATRIX_B],
	      matrix[MATRIX_C],
	      matrix[MATRIX_D],
	      matrix[MATRIX_TX],
	      matrix[MATRIX_TY]);
    }
  else
    {
      fprintf(stream, "matrix: NULL\n");
    }
}

BOOL
matrix_invert(float * base_matrix, float * new_matrix)
{
  BOOL result = YES;
  float delta;			/* ad-bc */
  float ad = base_matrix[MATRIX_A] * base_matrix[MATRIX_D];
  float bc = base_matrix[MATRIX_B] * base_matrix[MATRIX_C];
  delta = ad - bc;
  
  
  if (YES == float_zero(delta))
    {
      return NO;
    }
  if (NULL != new_matrix)
    {
      new_matrix[MATRIX_A] = base_matrix[MATRIX_D]/delta;
      new_matrix[MATRIX_B] = - base_matrix[MATRIX_B]/delta;
      new_matrix[MATRIX_C] = - base_matrix[MATRIX_C]/delta;
      new_matrix[MATRIX_D] = base_matrix[MATRIX_A]/delta;
      new_matrix[MATRIX_TX] = 
	(base_matrix[MATRIX_B]*base_matrix[MATRIX_TY]
       - base_matrix[MATRIX_D]*base_matrix[MATRIX_TX]) /delta;
      new_matrix[MATRIX_TY] = 
	(base_matrix[MATRIX_C]*base_matrix[MATRIX_TX]
       - base_matrix[MATRIX_A]*base_matrix[MATRIX_TY]) /delta;
    }
  return result;
}

BOOL
matrix_equal(const float * a, const float * b)
{
  if (a == NULL || b == NULL)
    {
      return NO;
    }
  else if (a == b)
    {
      return YES;
    }
  else if ((a[MATRIX_A] == b[MATRIX_A])
	   && (a[MATRIX_B] == b[MATRIX_B])
	   && (a[MATRIX_B] == b[MATRIX_B])
	   && (a[MATRIX_C] == b[MATRIX_C])
	   && (a[MATRIX_D] == b[MATRIX_D])
	   && (a[MATRIX_TX] == b[MATRIX_TX])
	   && (a[MATRIX_TY] == b[MATRIX_TY]))
    {
      return YES;
    }
  else
    {
      return NO;
    }
}
