/* layers_pallet.h

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "layers_pallet.h"
#include "GyveBuffer.h"
#include "GyveWindow.h"
#include "GyveLayer.h"
#include "PSProcessColor.h"
#include "popup_menu.h"

#include <Foundation/NSString.h>
#include <Foundation/NSSet.h>

static void layers_menu_popup (GtkWidget *label, 
			       GdkEventButton *event,
			       struct layers_pallet * pallet);

static void layers_info_update(struct layers_pallet * pallet);
static GtkWidget * layers_info_item_new(struct layers_pallet * pallet);

static void layers_info_item_visibility_changed(GtkWidget * button, 
						 struct layers_pallet * pallet);
static void layers_info_item_writablity_changed(GtkWidget * button,
						 struct layers_pallet * pallet);
static void layers_info_item_printablity_changed(GtkWidget * button,
						  struct layers_pallet * pallet);

static GtkWidget * buffers_menu_new(struct layers_pallet* pallet);
static void buffers_menu_select(GtkWidget * menu_item, struct layers_pallet * pallet);

static void layer_op_new  (GtkWidget *widget, struct layers_pallet *);
static void layer_op_move_to_top (GtkWidget *widget, struct layers_pallet *);
static void layer_op_move_up (GtkWidget *widget, struct layers_pallet *);
static void layer_op_move_down (GtkWidget *widget, struct layers_pallet *);
static void layer_op_move_to_bottom (GtkWidget *widget, struct layers_pallet *);
static void layer_op_dup (GtkWidget *widget, struct layers_pallet *);
static void layer_op_delete (GtkWidget *widget, struct layers_pallet *);
static void layer_op_rename (GtkWidget *widget, struct layers_pallet *);
static void layer_op_color (GtkWidget *widget, struct layers_pallet *);
#define make_menu_func(func) (void (*)(GtkWidget *widget, gpointer))(func)

GtkWidget *
layers_pallet_new(struct layers_pallet * pallet)
{
  GtkMenuEntry layers_menu_items[] =
  {
    { "<Layers>/New", NULL, make_menu_func(layer_op_new), pallet},
    { "<Layers>/Dup", NULL, make_menu_func(layer_op_dup), pallet},
    { "<Layers>/Delete", NULL, make_menu_func(layer_op_delete), pallet},
    { "<Layers>/Figures/Cut", NULL, NULL, pallet},
    { "<Layers>/Figures/Copy", NULL, NULL, pallet},
    { "<Layers>/Figures/Paste", NULL, NULL, pallet},
    { "<Layers>/Flatten", NULL, NULL, pallet},
    { "<Layers>/Merge visible layers", NULL, NULL, pallet},
    { "<Layers>/Position/Top", NULL, make_menu_func(layer_op_move_to_top), pallet},
    { "<Layers>/Position/Up", NULL, make_menu_func(layer_op_move_up), pallet},
    { "<Layers>/Position/Down", NULL, make_menu_func(layer_op_move_down), pallet},
    { "<Layers>/Position/Bottom", NULL, make_menu_func(layer_op_move_to_bottom), pallet},
    { "<Layers>/Rename", NULL, make_menu_func(layer_op_rename), pallet},
    { "<Layers>/Color", NULL, make_menu_func(layer_op_color), pallet},
  };

  GtkWidget *window;
  GtkWidget *box;
  GtkWidget *separator;
  GtkWidget *hbox;
  GtkWidget *buffers_menu;
  GtkWidget * buffers_menu_label;
  GtkWidget * button;
  gint nmenu_items = sizeof(layers_menu_items)/sizeof(layers_menu_items[0]);
  gint old_mask;

  /* Layers Operations Popup Menu */
  pallet->layers_menu = popup_menu_new(layers_menu_items, 
				       nmenu_items,
				       "<Layers>");
  
  /* Top level window */
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), "Layers");
  gtk_container_border_width (GTK_CONTAINER (window), 0);

  /* Added mode switch */
  old_mask = gtk_widget_get_events(GTK_WIDGET(window));
  gtk_widget_set_events(GTK_WIDGET(window), old_mask| GDK_ENTER_NOTIFY_MASK);
  gtk_signal_connect(GTK_OBJECT(window), "enter_notify_event",
		     GTK_SIGNAL_FUNC (gui_enter_interactive_mode),
		     NULL);
  
  box = gtk_vbox_new (FALSE, 0);  
  gtk_container_add (GTK_CONTAINER (window), box);
  gtk_widget_show (box);

  /* Buffers Options Menu */
  hbox 	= gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (box), hbox, FALSE, TRUE, 0);
  gtk_container_border_width (GTK_CONTAINER (hbox), 10);
  gtk_widget_show(hbox);

  buffers_menu_label =  gtk_label_new("Buffer: ");
  gtk_box_pack_start (GTK_BOX (hbox), buffers_menu_label, FALSE, TRUE, 0);
  gtk_widget_show (buffers_menu_label);

  buffers_menu = buffers_menu_new(pallet);
  gtk_box_pack_start (GTK_BOX(hbox), buffers_menu, TRUE, TRUE, 0);
  gtk_widget_show (buffers_menu);
  
  separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (box), separator, FALSE, TRUE, 0);
  gtk_widget_show(separator);

  /* Layers box */
  pallet->layers_box = gtk_vbox_new (FALSE, 0);  
  gtk_box_pack_start (GTK_BOX (box), pallet->layers_box, FALSE, TRUE, 0);
  gtk_widget_show(pallet->layers_box);
  
  separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (box), separator, FALSE, TRUE, 0);
  
  /* Close button */
  button = gtk_button_new_with_label ("Close");
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (layers_pallet_hide),
			     GTK_OBJECT (window));
  gtk_box_pack_start (GTK_BOX (box), button, TRUE, TRUE, 0);
  gtk_widget_show (button);
  return window;
}

static void
layers_info_update(struct layers_pallet * pallet)
{
  int max, i;
  gtk_container_remove_all(GTK_CONTAINER(pallet->layers_box));
  max = [pallet->current_buffer count];
  for (i = max; i > 0; i--)
    {
      pallet->current_layer = [pallet->current_buffer objectAtIndex: i-1];
      gtk_box_pack_start(GTK_BOX(pallet->layers_box),
			 layers_info_item_new(pallet),
			 TRUE,
			 TRUE,
			 0);
    }  
}

static GtkWidget *
buffers_menu_new(struct layers_pallet * pallet)
{
  GtkWidget *menu;
  /* Menu */
  menu = gtk_option_menu_new ();
  pallet->buffers_menu = [[[BuffersMenu alloc] 
			    initWithHandler: buffers_menu_select
			    data: pallet] autorelease];
  gtk_option_menu_set_menu (GTK_OPTION_MENU (menu), 
			    [pallet->buffers_menu gtkWidget]);
  gtk_option_menu_set_history (GTK_OPTION_MENU (menu), 0);
  return menu;
} 


static void
buffers_menu_select(GtkWidget * menu_item, struct layers_pallet * pallet)
{
  char * name;
  GList * list;
  GtkWidget * label;
  list = gtk_container_children(GTK_CONTAINER (menu_item));
  if (list)
    {
      label = (GtkWidget *)g_list_nth_data(list, 0);
      name = GTK_LABEL(label)->label;
      pallet->current_buffer = [GyveBuffer lookUpBufferByName:
					     [NSString stringWithCString: name]];
      layers_info_update (pallet);
    }
}

static GtkWidget *
layers_info_item_new(struct layers_pallet * pallet)
{
  GyveLayer * layer = pallet->current_layer;
  GtkWidget * hbox;
  GtkWidget * label;
  GtkWidget * button;
  GtkWidget * ebox;
  const char * name = [[layer layerName] cString];

  hbox  = gtk_hbox_new (FALSE, 10);
  gtk_container_border_width (GTK_CONTAINER (hbox), 2);
  gtk_widget_show(hbox);

  ebox = gtk_event_box_new();
  gtk_widget_set_events(ebox, GDK_BUTTON_PRESS_MASK);
  gtk_object_set_data(GTK_OBJECT(ebox), "layer", layer);
  gtk_signal_connect(GTK_OBJECT(ebox), 
		     "button_press_event",
		     GTK_SIGNAL_FUNC (layers_menu_popup),
		      pallet);
  
  gtk_box_pack_start(GTK_BOX (hbox), ebox,
		     TRUE, TRUE, 0);
  gtk_widget_show(ebox);
  
  label = gtk_label_new(name);
  gtk_container_add(GTK_CONTAINER(ebox), label);
  gtk_widget_show(label);
  
  button = gtk_toggle_button_new_with_label("visible");
  gtk_box_pack_start(GTK_BOX (hbox), button,
		     TRUE, TRUE, 0);
  if ([layer isVisible])
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button), 1);
  else
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button), 0);
  gtk_object_set_data(GTK_OBJECT(button), "layer", layer);
  gtk_signal_connect(GTK_OBJECT(button), "toggled",
		      GTK_SIGNAL_FUNC (layers_info_item_visibility_changed),
		       pallet);
  gtk_widget_show(button);

  button = gtk_toggle_button_new_with_label("writable");
  gtk_box_pack_start(GTK_BOX (hbox), button,
		     TRUE, TRUE, 0);
  if ([layer isWritable])
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button), 1);
  else
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button), 0);
  gtk_object_set_data(GTK_OBJECT(button), "layer", layer);
  gtk_widget_show(button);
  gtk_signal_connect(GTK_OBJECT(button), "toggled",
		      GTK_SIGNAL_FUNC (layers_info_item_writablity_changed),
		      pallet);

  button = gtk_toggle_button_new_with_label("printable");
  gtk_box_pack_start(GTK_BOX (hbox), button,
		     TRUE, TRUE, 0);
  if ([layer isPrintable])
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button), 1);
  else
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button), 0);
  gtk_object_set_data(GTK_OBJECT(button), "layer", layer);
  gtk_widget_show(button);
  gtk_signal_connect(GTK_OBJECT(button), "toggled",
		      GTK_SIGNAL_FUNC (layers_info_item_printablity_changed),
		      pallet);

  return hbox ;
}

static void
layers_menu_popup (GtkWidget *event_box, 
		   GdkEventButton *event, 
		   struct layers_pallet * pallet)
{
  pallet->current_layer = gtk_object_get_data(GTK_OBJECT(event_box), "layer");
  popup_menu_popup(pallet->layers_menu, event->button, event->time);  
}


static void
layers_info_item_visibility_changed(GtkWidget * button,
				     struct layers_pallet * pallet)
{
  NSSet * set;
  GyveLayer * layer = gtk_object_get_data(GTK_OBJECT(button), "layer");
  if (GTK_TOGGLE_BUTTON(button)->active)
    [layer setVisible: YES];
  else
    {
      [layer setVisible: NO];
    }
  set = [GyveWindow lookUpWindowByBuffer: pallet->current_buffer];
  if (set) [set makeObjectsPerform: @selector(redrawAll)];
}
static void
layers_info_item_writablity_changed(GtkWidget * button,
				     struct layers_pallet * pallet)
{
  GyveLayer * layer = gtk_object_get_data(GTK_OBJECT(button), "layer");
  if (GTK_TOGGLE_BUTTON(button)->active)
    [layer setWritable: YES];
  else
    [layer setWritable: NO];
}

static void
layers_info_item_printablity_changed(GtkWidget * button,
				      struct layers_pallet * pallet)
{
  GyveLayer * layer = gtk_object_get_data(GTK_OBJECT(button), "layer");
  if (GTK_TOGGLE_BUTTON(button)->active)
    [layer setPrintable: YES];
  else
    [layer setPrintable: NO];
}

static void
layer_op_new  (GtkWidget *widget, struct layers_pallet * pallet)
{
  [pallet->current_buffer addNewLayerOnTop];
  layers_info_update(pallet);
		     
}

static void
layer_op_move_to_top (GtkWidget *widget, struct layers_pallet * pallet)
{
  int i = [pallet->current_buffer indexOfLayer: pallet->current_layer];
  [pallet->current_buffer moveToTopLayerAtIndex: i];
  layers_info_update(pallet);
		     
}
static void
layer_op_move_up (GtkWidget *widget, struct layers_pallet * pallet)
{
  int i = [pallet->current_buffer indexOfLayer: pallet->current_layer];
  [pallet->current_buffer moveUpLayerAtIndex: i];
  layers_info_update(pallet);
		     
}

static void
layer_op_move_down (GtkWidget *widget, struct layers_pallet * pallet)
{
  int i = [pallet->current_buffer indexOfLayer: pallet->current_layer];
  [pallet->current_buffer moveDownLayerAtIndex: i];
  layers_info_update(pallet);
		     
}

static void
layer_op_move_to_bottom (GtkWidget *widget, struct layers_pallet * pallet)
{
  int i = [pallet->current_buffer indexOfLayer: pallet->current_layer];
  [pallet->current_buffer moveToBottomLayerAtIndex: i];
  layers_info_update(pallet);
		     
}

static void
layer_op_dup (GtkWidget *widget, struct layers_pallet * pallet)
{
  int i = [pallet->current_buffer indexOfLayer: pallet->current_layer];
  [pallet->current_buffer duplicateLayerAtIndex: i];
  layers_info_update(pallet);
		     
}

static void
layer_op_delete (GtkWidget *widget, struct layers_pallet * pallet)
{
  NSSet * set;
  if (NO == [pallet->current_buffer containsObject: pallet->current_layer])
    {
      gyve_message("The buffer doesn't contain the layer");
      return ;
    }
  else if (1 == [pallet->current_buffer count]) 
    {
      gyve_message("Buffer should have one layer at least");
      return ;
    }
  [pallet->current_buffer removeLayer: pallet->current_layer];
  pallet->current_layer = nil;
  
  set = [GyveWindow lookUpWindowByBuffer: pallet->current_buffer];
  if (set) [set makeObjectsPerform: @selector(redrawAll)];
  layers_info_update(pallet);
}


static GtkWidget * rename_dialog_new(struct layers_pallet * pallet);
static void rename_dialog_rename(GtkWidget * button, GtkWidget * entry);
static void rename_dialog_update(GtkWidget * button, struct layers_pallet * pallet);
static void rename_dialog_destroy(GtkWidget * button, GtkWidget * dialog);
void
layer_op_rename (GtkWidget *widget, struct layers_pallet * pallet)
{
  GtkWidget * rename_dialog = rename_dialog_new(pallet);
  gtk_widget_show(rename_dialog);
}


static GtkWidget *
rename_dialog_new(struct layers_pallet * pallet)
{
  GyveLayer * layer = pallet->current_layer;
  GtkWidget * rename_dialog = gtk_dialog_new ();
  GtkWidget * button;
  GtkWidget * entry;
  gint old_mask;
  gtk_signal_connect (GTK_OBJECT (rename_dialog), "destroy",
		      GTK_SIGNAL_FUNC(gtk_widget_destroyed),
		      &rename_dialog);
  gtk_window_set_title (GTK_WINDOW (rename_dialog), "Layer name");
  gtk_container_border_width (GTK_CONTAINER (rename_dialog), 0);

  /* Added mode switch */
  old_mask = gtk_widget_get_events(GTK_WIDGET(rename_dialog));
  gtk_widget_set_events(GTK_WIDGET(rename_dialog), old_mask| GDK_ENTER_NOTIFY_MASK);
  gtk_signal_connect(GTK_OBJECT(rename_dialog), "enter_notify_event",
		     GTK_SIGNAL_FUNC (gui_enter_interactive_mode),
		     NULL);
  
  entry = gtk_entry_new ();
  
  gtk_entry_set_text (GTK_ENTRY (entry), [[layer layerName] cString]); 
  gtk_editable_select_region (GTK_EDITABLE (entry), 0, -1);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (rename_dialog)->vbox), 
		      entry, TRUE, TRUE, 0);
  gtk_object_set_data(GTK_OBJECT(entry), "layer", layer);
  gtk_widget_show (entry);
  
  button = gtk_button_new_with_label ("Rename");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (rename_dialog)->action_area), 
		      button, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (rename_dialog_rename),
		      entry);
  gtk_signal_connect_after (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (rename_dialog_update),
		      pallet);

  gtk_signal_connect_after (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (rename_dialog_destroy),
		      rename_dialog);

  gtk_widget_show (button);

  button = gtk_button_new_with_label ("Cancel");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (rename_dialog_destroy),
		      rename_dialog);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (rename_dialog)->action_area),
		      button, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);
  return rename_dialog;
}

static void
rename_dialog_rename(GtkWidget * button, GtkWidget * entry)
{
  GyveLayer * layer = gtk_object_get_data(GTK_OBJECT(entry), "layer");
  unsigned char * new_name   = gtk_entry_get_text(GTK_ENTRY(entry));
  if (*new_name != '\0')
    [layer setLayerName: [NSString stringWithCString: new_name]];
}

static void
rename_dialog_update(GtkWidget * button, struct layers_pallet * pallet)
{
  layers_info_update(pallet);
}

static void
rename_dialog_destroy(GtkWidget * button, GtkWidget * dialog)
{
  gtk_widget_destroy(dialog);
}


static void layer_op_color_changed(GtkWidget * button, GtkWidget * dialog);

static void
layer_op_color (GtkWidget *widget, struct layers_pallet * pallet)
{
  gdouble color[3];
  NSString * title = [@"Color for Layer: " stringByAppendingString:
			 [pallet->current_layer layerName]];
  GtkWidget * dialog 	= gtk_color_selection_dialog_new([title cString]);
  GtkWidget * ccsel  	= GTK_COLOR_SELECTION_DIALOG(dialog)->colorsel;
  GtkWidget * ok_button = GTK_COLOR_SELECTION_DIALOG(dialog)->ok_button;
  gint old_mask;

  color[0] 	     = [[pallet->current_layer layerColor] red];
  color[1] 	     = [[pallet->current_layer layerColor] green];
  color[2]           = [[pallet->current_layer layerColor] blue];
  
  gtk_color_selection_set_color(GTK_COLOR_SELECTION(ccsel),
				color);
  gtk_object_set_data(GTK_OBJECT(ok_button), "layer", pallet->current_layer);
  gtk_object_set_data(GTK_OBJECT(ok_button), "buffer", pallet->current_buffer);
  gtk_signal_connect(GTK_OBJECT(ok_button), "clicked",
		     GTK_SIGNAL_FUNC(layer_op_color_changed),
		     dialog);

  /* Added mode switch */
  old_mask = gtk_widget_get_events(GTK_WIDGET(dialog));
  gtk_widget_set_events(GTK_WIDGET(dialog), old_mask| GDK_ENTER_NOTIFY_MASK);
  gtk_signal_connect(GTK_OBJECT(dialog), "enter_notify_event",
		     GTK_SIGNAL_FUNC (gui_enter_interactive_mode),
		     NULL);

  gtk_widget_show(dialog);
}

static void
layer_op_color_changed(GtkWidget * button, GtkWidget * dialog)
{
  NSSet * set;
  gdouble color[3];
  PSProcessColor * color_obj;
  GyveLayer * layer   = gtk_object_get_data(GTK_OBJECT(button), "layer");
  GyveBuffer * buffer = gtk_object_get_data(GTK_OBJECT(button), "Buffer");
  GtkWidget * ccsel  	= GTK_COLOR_SELECTION_DIALOG(dialog)->colorsel;
  gtk_color_selection_get_color(GTK_COLOR_SELECTION(ccsel), color);
  color_obj = [[[PSProcessColor alloc]
		initWithRed: color[0] green: color[1] blue: color[2]]
		autorelease];
  [layer setLayerColor: color_obj];
  set = [GyveWindow lookUpWindowByBuffer: buffer];
  if (set)
    {
      [set makeObjectsPerform: @selector(redrawAll)];
    }
  gtk_widget_destroy(dialog);
}
