/* painting_preview.m

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by: Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
   
   This file is part of the GNU Yellow Vector Editor

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */ 

#include "painting_preview.h"
#include <gyve/GyveCanvas.h>
#include "GyveDrawingEngine.h"
#include "GyveEventHandler.h"

#include "PaintStylePallet.h"

#include <DPS/psops.h>
#include <Foundation/NSObject.h>
#include <stdio.h>

#define preview_size 80

@interface PaintingPreviewDrawingEngine: NSObject<GyveDrawingEngine>
{
  BOOL selected;
  NSObject * painting;
}
- (void)select;
- (void)unSelect;
- (void)setPainting: (NSObject *)p;
@end

@interface PaintingPreviewFillingDrawingEngine :PaintingPreviewDrawingEngine
@end
@interface PaintingPreviewStrokingDrawingEngine :PaintingPreviewDrawingEngine
@end

@interface PaintingPreviewEventHandler: NSObject<GyveEventHandler>
{
  struct painting_preview * preview;
}
- initWithPaintingPreview: (struct painting_preview *) p;
@end

GtkWidget *
painting_preview_new(struct painting_preview * preview)
{
  GtkWidget * box;
  GtkWidget * frame;
  NSObject <GyveDrawingEngine> *e;
  if (preview == NULL)
    {
      return NULL;
    }

  box = gtk_vbox_new (FALSE, 10);
  gtk_container_border_width (GTK_CONTAINER (box), 10);  

  /* 
   * Filling 
   */
  preview->filling_canvas = [[GyveCanvas alloc] 
			    initWithViewWidth: preview_size 
			    height: preview_size];
  e = [[[PaintingPreviewFillingDrawingEngine alloc] init] autorelease] ;
  [e select];
  [preview->filling_canvas setDrawingEngine: e];
  [preview->filling_canvas setEventHandler:
	    [[[PaintingPreviewEventHandler alloc]
	       initWithPaintingPreview: preview]
	      autorelease]];
  
  frame = gtk_frame_new("Filling");
  gtk_container_add (GTK_CONTAINER (frame), [preview->filling_canvas dpsAreaWidget]);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
  gtk_widget_show(frame);

  gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 0);
  [preview->filling_canvas show];

  /*
   * Stroking
   */ 
  preview->stroking_canvas = [[GyveCanvas alloc] 
			    initWithViewWidth: preview_size 
			    height: preview_size];
  e = [[[PaintingPreviewStrokingDrawingEngine alloc] init] autorelease];
  [e unSelect];
  [preview->stroking_canvas setDrawingEngine: e];
  [preview->stroking_canvas setEventHandler:
	    [[[PaintingPreviewEventHandler alloc]
	       initWithPaintingPreview: preview]
	      autorelease]];
  
  frame = gtk_frame_new("Stroking");
  gtk_container_add (GTK_CONTAINER (frame), [preview->stroking_canvas dpsAreaWidget]);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
  gtk_widget_show(frame);

  gtk_box_pack_start (GTK_BOX (box), frame, 
		      FALSE, FALSE, 0);
  [preview->stroking_canvas show];

  preview->is_filling_active = YES;

  preview->preview_switch_relay = NULL;
  
  gtk_widget_show(box);
 return box; 
}

void
painting_preview_set_relay(struct painting_preview * preview, 
			   void (*relay)(GyveCanvas * canvas, 
					 DPSUserPathAction action))
{
  preview->preview_switch_relay = relay;
}

DPSUserPathAction
painting_preview_get_active(struct painting_preview * preview)
{
  if (YES == preview->is_filling_active)
    {
      return dps_ufill;
    }
  else
    {
      return dps_ustroke;
    }
}

void
painting_preview_switch(struct painting_preview * preview,
			DPSUserPathAction action)
{
  if (preview->is_filling_active == YES)
    {
      if (action == dps_ustroke)
	{
	  preview->is_filling_active = NO;
	  [[preview->filling_canvas drawingEngine] unSelect];
	  [preview->filling_canvas redrawAll];
	  [[preview->stroking_canvas drawingEngine] select];
	  [preview->stroking_canvas redrawAll];
	  if (NULL != preview->preview_switch_relay)
	    {
	      preview->preview_switch_relay(preview->stroking_canvas,
					    action);
	    }

	}
      else
	{
	}
    }
  else
    {
      if (action == dps_ufill)
	{
	  preview->is_filling_active = YES;
	  [[preview->filling_canvas drawingEngine] select];
	  [preview->filling_canvas redrawAll];
	  [[preview->stroking_canvas drawingEngine] unSelect];
	  [preview->stroking_canvas redrawAll];
	  if (NULL != preview->preview_switch_relay)
	    {
	      preview->preview_switch_relay(preview->filling_canvas,
					    action);
	    }
	}
      else
	{
	}
    }
}

void
painting_preview_set_painting(struct painting_preview * preview,
			      NSObject * painting,
			      DPSUserPathAction action)
{
  if (action == dps_ustroke)
    {
      [[preview->stroking_canvas drawingEngine]
	setPainting: painting];
      [preview->stroking_canvas redrawAll];
    }
  else
    { [[preview->filling_canvas drawingEngine]
	setPainting: painting];
    [preview->filling_canvas redrawAll];
    }
}

void
painting_preview_update(struct painting_preview * preview)
{
  [preview->filling_canvas redrawAll];
  [preview->stroking_canvas redrawAll];
}

@implementation PaintingPreviewDrawingEngine
- init
{
  [super init];
  selected = NO;
  painting = nil;
  return self ;
}
- (void)dealloc
{
  DESTROY(painting);
  [super dealloc];
}
- (void)select
{
  selected = YES;
}
- (void)unSelect
{
  selected = NO;
}
- (void)setPainting: (NSObject *)p
{
  ASSIGN(painting, p);
}
- (void)setAbsoluteScale:(float) scale
{
}
- (float)absoluteScale
{
  return 1.0;
}
- (enum drawing_mode)drawingMode
{
  return PREVIEW_MODE;
}
  
- (void)setDrawingMode: (enum drawing_mode)mode
{
}
- (void)imageCacheRequestWithArea: (GtkDPSArea * )area
			  coordTr: (struct coord_tr *)_coord_tr
{
}
- (void)coordTrChangesTo: (struct coord_tr *)coord_tr
{
}
- (void)setDocumentSize: (NSSize *) size
{
}
- (NSSize)documentSize
{
  return NSZeroSize;
}
- (BOOL)calcBBoxOfFigObj: (NSObject<PSFigObj>*)figobj
{
  return NO;
}
@end

@implementation PaintingPreviewEventHandler
- initWithPaintingPreview: (struct painting_preview *) p
{
  [super init];
  preview = p;
  return self ;
}
- (void)buttonPressEvent:  (GdkEventButton * )event onCanvas: (GyveCanvas *)canvas{}
- (void)buttonReleaseEvent:(GdkEventButton *)event onCanvas: (GyveCanvas *)canvas
{
  if (preview->stroking_canvas == canvas)
    {
      painting_preview_switch(preview, dps_ustroke);
    }
  else
    {
      painting_preview_switch(preview, dps_ufill);
    }
}
- (void)motionNotifyEvent: (GdkEventMotion *)event onCanvas: (GyveCanvas *)canvas{}
- (void)enterNotifyEvent:  (GdkEventCrossing *)event onCanvas: (GyveCanvas *)canvas{}
- (void)leaveNotifyEvent:  (GdkEventCrossing *)event onCanvas: (GyveCanvas *)canvas{}
- (void)keyPressEvent:     (GdkEventKey *)event      onCanvas: (GyveCanvas *)canvas{}
- (void)keyReleaseEvent:   (GdkEventKey *)event      onCanvas: (GyveCanvas *)canvas{}
@end


@implementation PaintingPreviewFillingDrawingEngine
- (void)redrawRect: (NSRect *)rect
{
  PSgsave();
  if (painting)
    {
      PSsetrgbcolor([(id<PSColor>)painting red],
		    [(id<PSColor>)painting green],
		    [(id<PSColor>)painting blue]);
    }
  else
    {
      PSsetrgbcolor(0.0, 0.0, 0.0);
    }
  PSrectfill(rect->origin.x, rect->origin.y, rect->size.width, rect->size.height);
  if (selected == YES)
    {
      PSsetrgbcolor(0.5, 0.5, 0.5);
      PSmoveto(0.0, 0.0);
      PSlineto(12.0, 0.0);
      PSlineto(0.0, 12.0);
      PSlineto(0.0, 0.0);
      PSclosepath();
      PSfill();
    }
  PSgrestore();
}
@end

@implementation  PaintingPreviewStrokingDrawingEngine
- (void)redrawRect: (NSRect *)rect
{
  PSgsave();
  PSsetrgbcolor(1.0, 1.0, 1.0);
  PSrectfill(rect->origin.x, rect->origin.y, rect->size.width, rect->size.height);
  if (painting)
    {
      PSsetrgbcolor([(id<PSColor>)painting red],
		    [(id<PSColor>)painting green],
		    [(id<PSColor>)painting blue]);
      PSsetlinewidth([[PaintStylePallet paintStylePallet]
		       lineWidth]);
      PSsetlinejoin([[PaintStylePallet paintStylePallet]
		      lineJoin]);
      PSsetlinecap([[PaintStylePallet paintStylePallet]
		     lineCap]);
    }
  else
    {
      PSsetrgbcolor(0.0, 0.0, 0.0);
      PSsetlinewidth(1.0);
    }
  PSmoveto(10.0, 10.0);
  PSlineto(40.0, 40.0);
  PSlineto(0.0, 100.0);
  PSstroke();
  if (selected == YES)
    {
      PSsetrgbcolor(0.5, 0.5, 0.5);
      PSmoveto(0.0, 0.0);
      PSlineto(12.0, 0.0);
      PSlineto(0.0, 12.0);
      PSlineto(0.0, 0.0);
      PSclosepath();
      PSfill();
    }
  PSgrestore();
}
@end

