/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
  PSConsole -- postscript console widget
  Author: Masatake YAMATO <masata-y@is.aist-nara.ac.jp>
  
  Almost all of PSConsole comes from GConsole.
  Author: Eiichi Takamori <taka@ma1.seikyou.ne.jp>

  based on script-fu's console code
  idea is taken from simple guile console widget by
  Robert Havoc Pennington <rhpennin@midway.uchicago.edu> */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include <DPS/dpsclient.h>
#include <DPS/dpsops.h>

#include "psconsole.h"

#include "GyveWindow.h"
#include "GyveCanvas.h"

#define TEXT_WIDTH	400
#define TEXT_HEIGHT	400
#define	BUFSIZE		1024
#define FLUSH_TIMEOUT	200

static void psconsole_class_init(PSConsoleClass *class);
static void psconsole_init(PSConsole *con);
static void psconsole_destroy (GtkObject *object);
static gint psconsole_entry_event_keypress(GtkWidget *widget, GdkEventKey *event, gpointer data);
static void psconsole_create_ps_interp(PSConsole *con);
static gchar *psconsole_read_eval_print_with_string(PSConsole *con, gchar *string);
static void psconsole_add_to_history(PSConsole *con, gchar *str);
static gchar *psconsole_history_previous(PSConsole *con);
static gchar *psconsole_history_next(PSConsole *con);
static void psconsole_entry_handle_rparen (PSConsole *con);
static gint psconsole_entry_start_flush (gpointer data);
static gint psconsole_entry_end_flush (gpointer data);


guint
psconsole_get_type (void)
{
  static guint psconsole_type = 0;

  if (!psconsole_type)
    {
      GtkTypeInfo psconsole_info =
      {
	"PSConsole",
	sizeof (PSConsole),
	sizeof (PSConsoleClass),
	(GtkClassInitFunc) psconsole_class_init,
	(GtkObjectInitFunc) psconsole_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL
      };
      psconsole_type = gtk_type_unique (gtk_window_get_type (), &psconsole_info);
    }
  return psconsole_type;
}

static void
psconsole_class_init (PSConsoleClass *klass)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) klass;
  object_class->destroy = psconsole_destroy;
}

static void
psconsole_init (PSConsole * con)
{
  GtkWidget *window;
  GtkWidget *box1;
  GtkWidget *box2;
  GtkWidget *button;
  GtkWidget *separator;
  GtkWidget *table;
  GtkWidget *hscrollbar;
  GtkWidget *vscrollbar;
  GtkWidget *text;
  GtkWidget *entry;


  con->prompt = NULL;
  con->history = NULL;
  con->history_current = NULL;
  memset (&con->flush, 0, sizeof (con->flush));
  
  window = GTK_WIDGET (con);
  gtk_container_border_width (GTK_CONTAINER (window), 0);

  box1 = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), box1);
  gtk_widget_show (box1);


  box2 = gtk_vbox_new (FALSE, 10);
  gtk_container_border_width (GTK_CONTAINER (box2), 10);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 0);
  gtk_widget_show (box2);
  table = gtk_table_new (2, 2, FALSE);
  gtk_table_set_row_spacing (GTK_TABLE (table), 0, 2);
  gtk_table_set_col_spacing (GTK_TABLE (table), 0, 2);
  gtk_box_pack_start (GTK_BOX (box2), table, TRUE, TRUE, 0);
  gtk_widget_show (table);

  con->text =
    text = gtk_text_new (NULL, NULL);
  gtk_widget_set_name (text, "text");
  gtk_widget_set_usize (text, TEXT_WIDTH, TEXT_HEIGHT);
  gtk_text_set_editable (GTK_TEXT (text), TRUE);
  gtk_table_attach_defaults (GTK_TABLE (table), text, 0, 1, 0, 1);
  gtk_widget_show (text);

  hscrollbar = gtk_hscrollbar_new (GTK_TEXT (text)->hadj);
  gtk_table_attach (GTK_TABLE (table), hscrollbar, 0, 1, 1, 2,
		    GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show (hscrollbar);

  vscrollbar = gtk_vscrollbar_new (GTK_TEXT (text)->vadj);
  gtk_table_attach (GTK_TABLE (table), vscrollbar, 1, 2, 0, 1,
		    GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_widget_show (vscrollbar);

  con->vadj = GTK_TEXT (text)->vadj;

  separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, TRUE, 0);
  gtk_widget_show (separator);


  box2 = gtk_vbox_new (FALSE, 10);
  gtk_container_border_width (GTK_CONTAINER (box2), 10);
  gtk_box_pack_start (GTK_BOX (box1), box2, FALSE, TRUE, 0);
  gtk_widget_show (box2);


  con->entry =
    entry = gtk_entry_new ();
  gtk_widget_set_name (entry, "entry");
  GTK_WIDGET_SET_FLAGS (entry, GTK_CAN_FOCUS);
  gtk_box_pack_start (GTK_BOX (box2), entry, TRUE, TRUE, 0);
  gtk_widget_show (entry);
  gtk_signal_connect (GTK_OBJECT (entry), "key_press_event",
		      GTK_SIGNAL_FUNC (psconsole_entry_event_keypress),
		      con);

  button = gtk_button_new_with_label ("close");
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_hide),
			     GTK_OBJECT (window));
  gtk_box_pack_start (GTK_BOX (box2), button, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  /* need to be after grab_default for button? */
  gtk_widget_grab_focus (entry);

  gtk_widget_show (window);
  gtk_widget_map (window);

  /* make sure window is mapped */
  while (gtk_events_pending ())
    gtk_main_iteration_do (TRUE);

  psconsole_create_ps_interp (con);
}

	    
GtkWidget *
psconsole_new (gchar * prompt, gchar *initial_message)
{
  PSConsole *con = PSCONSOLE (gtk_type_new (psconsole_get_type ()));

  if (prompt)
      con->prompt = g_strdup (prompt);
  else
      con->prompt = g_strdup ("DPS> ");

  gtk_widget_realize (con->text);
  if (initial_message)
      psconsole_output (con, initial_message);
  psconsole_output (con, con->prompt);

  return GTK_WIDGET (con);
}

void
psconsole_output (PSConsole * con, gchar *string)
{
  if (GTK_WIDGET_REALIZED (con->text))
    {
      gtk_text_freeze (GTK_TEXT (con->text));
      gtk_text_insert (GTK_TEXT (con->text), NULL, NULL, NULL,
		       string, -1);
      gtk_text_thaw (GTK_TEXT (con->text));
      con->vadj->value = con->vadj->upper - con->vadj->page_size;
      gtk_signal_emit_by_name (GTK_OBJECT (con->vadj), "changed");
    }
}

static void
psconsole_destroy (GtkObject *object)
{
  PSConsole *con = PSCONSOLE (object);
  GList	   *tmp;
  
  g_free (con->prompt), con->prompt = NULL;
  for (tmp = con->history; tmp; tmp = tmp->next)
      g_free (tmp->data), tmp->data = NULL;	/* duplicated string */
  g_list_free (con->history), con->history = NULL;
  con->history_current = NULL;
}


#define CTRL(c)		((c) & 0x1f)

static gint
psconsole_entry_event_keypress (GtkWidget * widget,
			       GdkEventKey * event,
			       gpointer data)
{
  PSConsole	*con = PSCONSOLE (data);
  char		*tmp;
  char		*buffer;
  char		*retstr;
  int		keyval;

  if (con->flush.flushing)
    {
      psconsole_entry_end_flush (con);
    }
  
  keyval = event->keyval;
  if ((event->state & GDK_CONTROL_MASK) && keyval >= 0x40 && keyval < 0x80)
    keyval &= 0x1f;		/* not too good */
	    
  switch (keyval)
    {
    case GDK_Return:
    case CTRL('m'):
      gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");
      tmp = gtk_entry_get_text (GTK_ENTRY (con->entry));
      buffer = g_new0 (gchar, strlen (tmp) + 2);
      strcpy (buffer, tmp);
      gtk_entry_set_text (GTK_ENTRY (con->entry), ""), tmp = NULL;
      psconsole_add_to_history (con, buffer);
      strcat (buffer, "\n");

      /* local echo */
      psconsole_output (con, buffer);

      gdk_flush ();

      retstr = psconsole_read_eval_print_with_string (con, buffer);
      psconsole_output (con, retstr);
      psconsole_output (con, con->prompt);

      free (retstr);
      g_free (buffer);
      return TRUE;
    case CTRL('p'):
      tmp = psconsole_history_previous (con);
      if (tmp)
	gtk_entry_set_text (GTK_ENTRY (con->entry), tmp);
      return TRUE;
    case CTRL('n'):
      tmp = psconsole_history_next (con);
      if (tmp)
	gtk_entry_set_text (GTK_ENTRY (con->entry), tmp);
      return TRUE;
    case ')':
      psconsole_entry_handle_rparen (con);
      return TRUE;
    default:
      return FALSE;
    }
}

static void
psconsole_text_backstop(DPSContext ctxt, char *buf,
			long unsigned int count)
{
  char *str;
  static PSConsole *con;
  if (ctxt == NULL)
    {
      con = (PSConsole *) buf;
    }
  else
    {
      str = malloc(count+2);
      memcpy(str, buf, count);
      str[count] = '\n';
      str[count+1] = '\0';
      psconsole_output(con, str);
      free(str);
    }
}

static void
psconsole_create_ps_interp (PSConsole *con)
{
}

static gchar *
psconsole_read_eval_print_with_string (PSConsole * con, gchar * string)
{
  DPSContext ctxt;
  GyveWindow * w = [GyveWindow currentWindow];
  GyveView * v;
  DPSTextProc tmp;
  if (w != nil)
    {
      v   = [w contentCanvas];
      ctxt = [v dpsContextBegin];
      tmp = DPSGetCurrentTextBackstop(ctxt);
      psconsole_text_backstop(NULL, (void *)con, 0);
      DPSSetTextBackstop(psconsole_text_backstop);
      DPSPrintf(ctxt, "%s", string);
      DPSflush(ctxt);
      DPSSetTextBackstop(tmp);
      [v dpsContextEnd];
    }
  return strdup("");
}

/* duplicate STR and add it to the history.
   current history ptr is also set to it */
static void
psconsole_add_to_history (PSConsole *con, gchar *str)
{
  con->history = g_list_prepend (con->history, g_strdup (str));
  con->history_current = NULL;
  while (g_list_length (con->history) > 32)
    {
      GList *last = g_list_last (con->history);
      g_free (last->data), last->data = NULL;
      g_list_remove_link (con->history, last);
      g_list_free_1 (last);
    }
}

/* returns previous history, or NULL if first item */
static gchar *
psconsole_history_previous (PSConsole * con)
{
  if (con->history_current)
    {
      if (con->history_current->next)
	{
	  con->history_current = con->history_current->next;
	  return con->history_current->data;
	}
      else
	return NULL;
    }
  else
    {
      /* MODIFICAION BY JET 
	 ҥȥ꤬ξ֤ C-pȥå夹Х */
      if (con->history)
	{
	  con->history_current = con->history;
	  return con->history_current->data;
	}
      else
	return NULL;
    }
}

/* returns next history, or NULL if last item */
static gchar *
psconsole_history_next (PSConsole *con)
{
  if (con->history_current && con->history_current->prev)
    {
      con->history_current = con->history_current->prev;
      return con->history_current->data;
    }
  else
    return NULL;
}

static void
psconsole_entry_handle_rparen (PSConsole *con)
{
  gchar		*str;
  int		pos;		/* current position */
  int		match;		/* matched rparen */
  int		level;		/* paren level */

  str = gtk_entry_get_text (GTK_ENTRY (con->entry));
  pos = GTK_EDITABLE (con->entry)->current_pos;

  g_return_if_fail (pos >= 0 && pos <= strlen (str));

  /* find position before matched lparen, or -1 if not found */
  for (level = 1, match = pos - 1; match >= 0; match--)
    {
      if (str[match] == ')')
	level++;
      else if (str[match] == '(')
	level--;
      if (level == 0)
	break;
    }

  if (match >= 0)
    {
      memset (&con->flush, 0, sizeof (con->flush));
      con->flush.flushing = TRUE;
      con->flush.flush_pos = match;
      con->flush.idle_tag = gtk_idle_add (psconsole_entry_start_flush, con);
    }
}

static gint
psconsole_entry_start_flush (gpointer data)
{
  PSConsole *con = PSCONSOLE (data);

  g_return_val_if_fail (con->flush.flushing, FALSE);

  con->flush.saved_pos = GTK_EDITABLE (con->entry)->current_pos;
  gtk_entry_set_position (GTK_ENTRY (con->entry), con->flush.flush_pos);
  con->flush.timeout_tag = gtk_timeout_add (FLUSH_TIMEOUT, psconsole_entry_end_flush, con);
  gtk_widget_draw (con->entry, NULL);

  return FALSE;
}

static gint
psconsole_entry_end_flush (gpointer data)
{
  PSConsole *con = PSCONSOLE (data);
  
  g_return_val_if_fail (con->flush.flushing, FALSE);

  gtk_entry_set_position (GTK_ENTRY (con->entry), con->flush.saved_pos);
  con->flush.saved_pos = 0;
  if (con->flush.timeout_tag)
    {
      gtk_timeout_remove (con->flush.timeout_tag);
      con->flush.timeout_tag = 0;
    }
  if (con->flush.idle_tag)
    {
      gtk_idle_remove (con->flush.idle_tag);
      con->flush.idle_tag = 0;
    }
  gtk_widget_draw (con->entry, NULL);
  con->flush.flushing = FALSE;

  return FALSE;
}




/*
 * Local Variables:
 * Mode: c
 * eval: (c-set-style "gnu")
 * End:
 */
