/*
 * $Id: io.c,v 1.3 1999/07/29 20:27:33 malte Exp $
 */
/*
 * Copyright (C) 1994-1999  Malte Uhl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/*
 * This module contains all IO functions dealing with encryption data
 * and the IO redirection for data (de)compression
 *
 * The format of valid input or output files is:
 *		Beginning with arbitrarily many bytes of data, say n.
 *		If n is a multiple of BLKSZ the file contains only data bytes.
 *		If it's not, random bytes are used to stuff up to ( n / 8 + 1 ) * 8.
 *		and another byte a appended specifying how many bytes of the last
 *		block where databytes.
 *
 *		---------------------------------------
 *		  | data | data | rand | rand | count |
 *		---------------------------------------
 *									  ^
 *									x * BLKSZ, count = 6
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "dea.h"
#include "comm.h"

#define	BLKSZ	sizeof( deablk )

static	FILE	*infd, *outfd;
static	size_t	done, rest;

int		(*input)( deablk ), (*output)( const deablk );

static
int		inputec( deablk into )
{
	if(( done = fread( into, 1, BLKSZ, infd )) == BLKSZ)
		return done > 0;
	else if( done > 0 )
	{
		size_t	n;
		char	*p = (char *)into;

		srand48( (long)time( NULL ));

		for( n = done; n < BLKSZ; n++ )
			p[ n ] = (char)lrand48();

		return (rest = done) > 0;
	}
	else if( ferror( infd ))
	{
		perror( prgname );
		return 0;
	}
	else
		return 0;
}

static
int		outputec( const deablk from )
{
	(void)fwrite( from, 1, BLKSZ, outfd );
	if( rest != 0 )
	{
		char	tmp;

		tmp = (char)rest;
		(void)fwrite( &tmp, 1, 1, outfd );

		return 0;
	}
	return 0;
}

static
int		inputdc( deablk into )
{
	char	*p;

	if(( done = fread( into, 1, BLKSZ, infd )) == BLKSZ)
		return done > 0;
	else switch( (unsigned)done )
	{
		case 0:
			rest = BLKSZ;
			(void)output( NULL );
			return 0;
		case 1:
			p    = (char *)into;
			rest = (size_t)p[ 0 ];
			(void)output( NULL );
			return 0;
		default:
			(void)fprintf( stderr, "%s: input corrupted\n", prgname );
			return 1;
	}
}

static	deablk	outbuff;

static
int		outputdc2( const deablk from )
{
	if( rest == 0 )
	{
		(void)fwrite( &outbuff, 1, BLKSZ, outfd );
		outbuff[ 0 ] = from[ 0 ], outbuff[ 1 ] = from[ 1 ];
	}
	else
	{
		(void)fwrite( &outbuff, 1, rest, outfd );
		return 0;
	}

	return 0;
}

static
int		outputdc( const deablk from )
{
	outbuff[ 0 ] = from[ 0 ], outbuff[ 1 ] = from[ 1 ];
	output  = outputdc2;

	return 0;
}

int		ioinit( FILE *in, FILE *out, int how, const char *compress )
{
	int		error = 0;

	infd = in, outfd = out;

	if( how == DEAENCRYPT )
	{
		input = inputec, output = outputec;

		if( compress )
			error = filter( 0, compress, fileno( infd ));
	}
	else
	{
		input = inputdc, output = outputdc;

		if( compress )
			error = filter( 1, compress, fileno( out ));
	}

	done = rest = 0;

	return error;
}
