/*
 * $Id: main.c,v 1.4 1999/07/29 20:27:33 malte Exp $
 */
/*
 * Copyright (C) 1994-1999  Malte Uhl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/time.h>

extern	int		opterr, optind;
extern	char	*optarg;
extern	int		getopt();

#include "dea.h"
#include "comm.h"

static	int		compflag, force;
		int		verbose;
		char	*prgname = "dea";
		char	*compress;

int		usage( void )
{
	(void)fprintf( stderr, "usage:\t%s [-cdevz] [-k key] [file ..]\n", prgname );
	return( 1 );
}

#if	!defined(basename) && !defined(__linux)
char	*basename( char *str )
{
	char	*p;

	return (p = strrchr( str, '/' )) ? p + 1 : str;
}
#endif

static
struct	timeval	*amtime( struct stat sb )
{
	static	struct	timeval	tv[ 2 ];

	tv[ 0 ].tv_sec  = sb.st_atime;
	tv[ 1 ].tv_sec  = sb.st_mtime;
	tv[ 0 ].tv_usec = tv[ 1 ].tv_usec = 0;

	return tv;
}

static
char	*newname( const char *old, const char *suffix )
{
	static	char	*tempfile;
	static	size_t	slen;
			size_t	len;

	len = strlen( old ) + strlen( suffix ) + 1;

	if( slen == 0 )
		tempfile = malloc( slen = len );
	else if( slen <= len )
		tempfile = realloc( tempfile, slen = len );

	if( tempfile == NULL )
	{
		(void)fprintf( stderr, "%s: out of memory\n", prgname );
		exit( 2 );
	}

	(void)sprintf( tempfile, "%s%s", old, suffix );

	return tempfile;
}

static	char	*infile, *outfile;
static	struct	stat	filemode;

static
FILE	*findinp( char *file, int how )
{
	FILE	*fp;
	char	*ptr;

	if( how == DEAENCRYPT )
	{
		if( stat( file, &filemode ) != 0 )
			perror( file );
		else if( ! S_ISREG( filemode.st_mode ))
			(void)fprintf( stderr, "%s: not a regular file\n", file );
		else if(( fp = fopen( file, "r" )) == NULL )
			perror( file );
		else
		{
			infile  = file;
			outfile = newname( file, compflag ? COMPFIX : SUFFIX );

			return fp;
		}
	}
	else
	{
		if(( ptr = strrchr( file, '.' )) != NULL &&
		   ( strcmp( ptr, SUFFIX ) == 0 || strcmp( ptr, COMPFIX ) == 0 ))
		{
			infile  = file;
			outfile = newname( file, "" );
			*strrchr( outfile, '.' ) = 0;
		}
		else
		{
			infile  = newname( file, compflag ? COMPFIX : SUFFIX );
			outfile = file;
		}

		if( stat( infile, &filemode ) != 0 )
			perror( infile );
		else if( ! S_ISREG( filemode.st_mode ))
			(void)fprintf( stderr, "%s: not a regular file\n", infile );
		else if(( fp = fopen( infile, "r" )) == NULL )
			perror( infile );
		else
			return fp;
	}
	return NULL;
}

static
FILE	*findoutp( const char *file )
{
	struct	stat	sbuf;
	int		fd, flags;

	flags = O_WRONLY | O_CREAT | O_NOCTTY;
	if( force == 0 )
		flags |= O_EXCL;

	if(( fd = open( file, flags, filemode.st_mode )) >= 0 )
		if( fstat( fd, &sbuf ) != 0 )
		{
			perror( file );
			(void)close( fd );
			return NULL;
		}
		else if( ! S_ISREG( sbuf.st_mode ))
		{
			(void)fprintf( stderr, "%s: not a regular file\n", file );
			(void)close( fd );
			return NULL;
		}
		else
			return fdopen( fd, "w" );
	else if( errno == EEXIST )
		(void)fprintf( stderr, "%s: file already exists, not overwritten\n", file );
	else
		perror( file );

	return NULL;
}

static
void	cleanup( int signal )
{
	if( outfile != NULL )
		(void)unlink( outfile );

	exit( 1 );
}

static
void	setsignals( void )
{
	struct	sigaction	act;

	(void)sigfillset( &act.sa_mask );
	act.sa_flags = 0;
	act.sa_handler = cleanup;

	if( sigaction( SIGHUP,  &act, NULL )
	 || sigaction( SIGINT,  &act, NULL )
	 || sigaction( SIGTERM, &act, NULL ))
	{
		perror( "sigaction" );
		exit( 5 );
	}
}

static
int		deacrypt( int how, void *keys, FILE *infd, FILE *outfd )
{
	deablk	data;
	int		err = 0;

	if( ioinit( infd, outfd, how, compflag ? compress : NULL ) != 0 )
		return 1;

	while( input( data ))
	{
		dea( data, keys );
		if( output( data ))
		{
			err = 1;
			break;
		}
	}
	(void)fclose( infd ); (void)fclose( outfd );

	return err | filtwait();
}

int		main( int argc, char **argv )
{
	int		opt,
			how = DEAENCRYPT,
			errors = 0,
			clear = 1;

	void	*keytab;
	char	*env = ENVCOMPRESS;
	FILE	*inp, *outp = NULL;
	static	char	keystr[ 8 ];

	prgname = basename( *argv );
	opterr  = 0;

	while(( opt = getopt( argc, argv, "cdefk:vz" )) != -1 )
		switch( opt )
		{
			case 'c': clear = 0;
				break;
			case 'd': how = DEADECRYPT, env = ENVDECOMPRESS;
				break;
			case 'e': how = DEAENCRYPT, env = ENVCOMPRESS;
				break;
			case 'f': force = 1;
				break;
			case 'k': (void)strncpy( keystr, optarg, sizeof( keystr ));
					  (void)memset( optarg, 0, strlen( optarg ));
				break;
			case 'v': verbose = 1;
				break;
			case 'z': compflag = 1;
				break;
			default :
				return usage();
		}
	argc -= optind; argv += optind;

	if( *keystr != 0 )
	{
		if( keycheck( keystr ))
			return 2;
	}
	else if( readkey( keystr, how == DEAENCRYPT ))
		return 2;

	if(( compress = getenv( env )) == NULL )
		compress = ( how == DEAENCRYPT ) ? COMPRESS : UNCOMPRESS;

	if(( keytab = dea_keytab()) == NULL )
	{
		(void)fprintf( stderr, "%s: no space for a new keytab\n", prgname );
		return 2;
	}
	dea_schedule( (unsigned char *)keystr, how, keytab );

	if( argc == 0 )
		return deacrypt( how, keytab, stdin, stdout );
	else
	{
		setsignals();

		for( ;argc > 0; argc--, argv++ )
			if(( inp = findinp( *argv, how )) == NULL ||
			   ( outp = findoutp( outfile )) == NULL )
				errors = 1;
			else
			{
				if( verbose )
					(void)fprintf( stderr, "%s:\n", *argv );

				if( deacrypt( how, keytab, inp, outp ) == 0 )
				{
					if( clear != 0 )
					{
						if( unlink( infile ) != 0 )
							perror( infile ), errors = 1;
						if( utimes( outfile, amtime( filemode )) != 0 )
							perror( outfile ), errors = 1;
					}
				}
				else
				{
					(void)fprintf( stderr, "%s: error %scrypting %s\n",
						prgname, ( how == DEAENCRYPT ) ? "en" : "de", infile );
					unlink( outfile ), errors = 1;
				}
			}
	}

	return errors ? 1 : 0;
}
